
//
// File: benchmark_mzcbor.cpp
// Created by: Olivier Langella
// Created on: 22/12/2025
//
/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// ./tests/benchmarks/catch2-only-benchmarks [mzcbor] -s
// ./tests/benchmarks/catch2-only-benchmarks [mzcbor] -s --benchmark-samples 5
// ./tests/benchmarks/catch2-only-benchmarks [mzcborrandom] -s --benchmark-samples 5
// ./tests/benchmarks/catch2-only-benchmarks [mzcborxic] -s --benchmark-samples 5


#include <catch2/catch_test_macros.hpp>
#include <catch2/benchmark/catch_benchmark.hpp>
#include <pappsomspp/core/processing/uimonitor/uimonitorvoid.h>
#include <pappsomspp/core/msfile/msfileaccessor.h>
#include <pappsomspp/core/msrun/private/mzcbormsrunreader.h>
#include <pappsomspp/core/xicextractor/msrunxicextractorfactory.h>
#include <QDebug>

TEST_CASE("cbor psm map test suite.", "[mzcbor]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  SECTION("..:: PSM cbor features::..", "[mzml]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzML",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");

    BENCHMARK("build TIC chromatogram from mzML")
    {
      msrunA01->getTicChromatogram();
    };


    BENCHMARK("build retention timeline from mzML")
    {
      msrunA01->getRetentionTimeLine();
    };
  }

  SECTION("..:: PSM cbor features::..", "[mzcbor]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/mzcbor/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzcbor",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");


    BENCHMARK("build TIC chromatogram from mzcbor")
    {
      msrunA01->getTicChromatogram();
    };
    BENCHMARK("build retention timeline from mzcbor")
    {
      msrunA01->getRetentionTimeLine();
    };
  }
}

TEST_CASE("cbor psm map test suite.", "[mzcborrandom]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  std::vector<double> relative_positions({0.9, 0.1, 0.5});

  SECTION("..:: PSM cbor features::..", "[mzml]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzML",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");
    std::size_t total_index          = msrunA01.get()->spectrumListSize();

    BENCHMARK("Random access in mzML")
    {
      for(double relative_position : relative_positions)
        {
          std::size_t index = relative_position * total_index;
          pappso::QualifiedMassSpectrum qmass_spec =
            msrunA01.get()->qualifiedMassSpectrum(index, true);
        }
    };
  }

  SECTION("..:: PSM cbor features::..", "[mzcbor]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/mzcbor/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzcbor",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");

    std::size_t total_index = msrunA01.get()->spectrumListSize();

    BENCHMARK("Random access in mzcbor")
    {
      for(double relative_position : relative_positions)
        {
          std::size_t index = relative_position * total_index;
          pappso::QualifiedMassSpectrum qmass_spec =
            msrunA01.get()->qualifiedMassSpectrum(index, true);
        }
    };
  }
}

TEST_CASE("benchmark XIC extractor", "[mzcborxic]")
{
  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));


  // pappso::MzRange mass_range(600, pappso::PrecisionFactory::getPpmInstance(10));

  pappso::MsRunXicExtractorFactory::getInstance().setMsRunXicExtractorFactoryType(
    pappso::MsRunXicExtractorFactoryType::direct);

  std::vector<pappso::XicCoordSPtr> xic_coord_list;
  pappso::XicCoord xic_coord;
  for(double mz = 700; mz < 1200; mz += 10)
    {
      xic_coord_list.push_back(xic_coord.initializeAndClone());
      xic_coord_list.back().get()->mzRange =
        pappso::MzRange(mz, pappso::PrecisionFactory::getPpmInstance(10));
    }
  WARN(QString("benchmark for %1 XICs").arg(xic_coord_list.size()).toStdString());

  SECTION("..:: benchmark XIC extractor mzML ::..", "[mzml]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzML",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");

    auto direct_extractor =
      pappso::MsRunXicExtractorFactory::getInstance().buildMsRunXicExtractorSp(msrunA01);

    pappso::UiMonitorVoid mon_void;
    BENCHMARK("XIC extractor on mzML")
    {
      direct_extractor.get()->extractXicCoordSPtrList(mon_void, xic_coord_list);
    };
  }

  SECTION("..:: benchmark XIC extractor mzcbor ::..", "[mzcbor]")
  {
    pappso::MsFileAccessor file_access(
      "/gorgone/pappso/moulon/raw/20250521_proteobench_dda_astral/mzcbor/"
      "LFQ_Astral_DDA_15min_50ng_Condition_A_REP1.mzcbor",
      "");
    pappso::MsRunReaderSPtr msrunA01 = file_access.getMsRunReaderSPtrByRunId("", "runa01");

    auto direct_extractor =
      pappso::MsRunXicExtractorFactory::getInstance().buildMsRunXicExtractorSp(msrunA01);

    pappso::UiMonitorVoid mon_void;
    BENCHMARK("XIC extractor on mzcbor")
    {
      direct_extractor.get()->extractXicCoordSPtrList(mon_void, xic_coord_list);
    };
  }
}
