<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
declare(strict_types=1);

namespace eZ\Publish\Core\Persistence\Legacy\Content\UrlAlias;

/**
 * UrlAlias Gateway.
 *
 * @internal For internal use by Persistence Handlers.
 */
abstract class Gateway
{
    /**
     * Default database table.
     */
    public const TABLE = 'ezurlalias_ml';

    public const INCR_TABLE = 'ezurlalias_ml_incr';
    public const INCR_TABLE_SEQ = 'ezurlalias_ml_incr_id_seq';

    public const NOP = 'nop';
    public const NOP_ACTION = self::NOP . ':';

    /**
     * Changes the gateway database table.
     *
     * @internal
     *
     * @param string $name
     */
    abstract public function setTable(string $name): void;

    /**
     * Loads all list of aliases by given $locationId.
     */
    abstract public function loadAllLocationEntries(int $locationId): array;

    /**
     * Load list of aliases by given $locationId.
     */
    abstract public function loadLocationEntries(
        int $locationId,
        bool $custom = false,
        ?int $languageId = null
    ): array;

    /**
     * Load paged list of global aliases.
     *
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException if the given language doesn't exist
     */
    abstract public function listGlobalEntries(
        ?string $languageCode = null,
        int $offset = 0,
        int $limit = -1
    ): array;

    /**
     * Return boolean indicating if the row with given $id is special root entry.
     *
     * Special root entry entry will have parentId=0 and text=''.
     * In standard installation this entry will point to location with id=2.
     */
    abstract public function isRootEntry(int $id): bool;

    /**
     * Update single row data matched by composite primary key.
     *
     * @param array $values associative array with column names as keys and column values as values
     */
    abstract public function updateRow(int $parentId, string $textMD5, array $values): void;

    /**
     * Insert new row into urlalias_ml table.
     *
     * @param array $values
     */
    abstract public function insertRow(array $values): int;

    /**
     * Load single row matched by composite primary key.
     */
    abstract public function loadRow(int $parentId, string $textMD5): array;

    /**
     * Downgrade autogenerated entry matched by given $action and $languageId and negatively matched by
     * composite primary key.
     *
     * If language mask of the found entry is composite (meaning it consists of multiple language ids) given
     * $languageId will be removed from mask. Otherwise entry will be marked as history.
     */
    abstract public function cleanupAfterPublish(
        string $action,
        int $languageId,
        int $newId,
        int $parentId,
        string $textMD5
    ): void;

    /**
     * Archive entry with $action by $languageMask.
     *
     * Used when swapping Location aliases, this ensures that given $languageMask matches a
     * single entry (database row).
     */
    abstract public function historizeBeforeSwap(string $action, int $languageMask): void;

    /**
     * Mark all entries with given $id as history entries.
     *
     * This method is used by Handler::locationMoved(). Each row is separately historized
     * because future publishing needs to be able to take over history entries safely.
     */
    abstract public function historizeId(int $id, int $link): void;

    /**
     * Update parent id of autogenerated entries.
     *
     * Update includes history entries.
     */
    abstract public function reparent(int $oldParentId, int $newParentId): void;

    /**
     * Load path data identified by given $id.
     *
     * @throws \eZ\Publish\API\Repository\Exceptions\BadStateException
     */
    abstract public function loadPathData(int $id): array;

    /**
     * Load path data identified by given ordered array of hierarchy data.
     *
     * The first entry in $hierarchyData corresponds to the top-most path element in the path, the second entry the
     * child of the first path element and so on.
     * This method is faster than self::getPath() since it can fetch all elements using only one query, but can be used
     * only for autogenerated paths.
     *
     * @param array $hierarchyData hierarchy-ordered URL alias entries data
     */
    abstract public function loadPathDataByHierarchy(array $hierarchyData): array;

    /**
     * Load complete URL alias data by given array of path hashes.
     *
     * @param string[] $urlHashes URL string hashes
     */
    abstract public function loadUrlAliasData(array $urlHashes): array;

    /**
     * Load autogenerated entry id by given $action and optionally $parentId.
     */
    abstract public function loadAutogeneratedEntry(string $action, ?int $parentId = null): array;

    /**
     * Delete single custom alias row matched by composite primary key.
     */
    abstract public function removeCustomAlias(int $parentId, string $textMD5): bool;

    /**
     * Delete all rows with given $action and optionally $id.
     *
     * If $id is set only autogenerated entries will be removed.
     */
    abstract public function remove(string $action, ?int $id = null): void;

    /**
     * Load all autogenerated entries with given $parentId with optionally included history entries.
     */
    abstract public function loadAutogeneratedEntries(
        int $parentId,
        bool $includeHistory = false
    ): array;

    /**
     * Return next value for "id" column.
     */
    abstract public function getNextId(): int;

    /**
     * Return main language ID of the Content on the Location with given $locationId.
     */
    abstract public function getLocationContentMainLanguageId(int $locationId): int;

    /**
     * Remove languageId of removed translation from lang_mask and deletes single language rows for multiple Locations.
     *
     * @param string[] $actions actions for which to perform the update
     */
    abstract public function bulkRemoveTranslation(int $languageId, array $actions): void;

    /**
     * Archive (remove or historize) URL aliases for removed Translations.
     *
     * @param int $parentId Parent alias used for linking historized entries
     * @param int[] $languageIds Language IDs of removed Translations
     */
    abstract public function archiveUrlAliasesForDeletedTranslations(
        int $locationId,
        int $parentId,
        array $languageIds
    ): void;

    /**
     * Delete URL aliases pointing to non-existent Locations.
     *
     * @return int Number of affected rows.
     */
    abstract public function deleteUrlAliasesWithoutLocation(): int;

    /**
     * Delete URL aliases pointing to non-existent parent nodes.
     *
     * @return int Number of affected rows.
     */
    abstract public function deleteUrlAliasesWithoutParent(): int;

    /**
     * Delete URL aliases which do not link to any existing URL alias node.
     *
     * Note: Typically link column value is used to determine original alias for an archived entries.
     *
     * @return int Number of deleted rows.
     */
    abstract public function deleteUrlAliasesWithBrokenLink(): int;

    /**
     * Delete "nop" type actions URL aliases that don't have children.
     */
    abstract public function deleteUrlNopAliasesWithoutChildren(): int;

    /**
     * Return aliases which are connected with provided parentId.
     */
    abstract public function getAllChildrenAliases(int $parentId): array;

    /**
     * Attempt repairing data corruption for broken archived URL aliases for Location,
     * assuming there exists restored original (current) entry.
     */
    abstract public function repairBrokenUrlAliasesForLocation(int $locationId): void;
}
