<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
namespace eZ\Publish\API\Repository\Tests\FieldType;

use Doctrine\DBAL\ParameterType;
use DOMDocument;
use eZ\Publish\API\Repository\Values\Content\Content;
use eZ\Publish\API\Repository\Values\Content\Field;
use eZ\Publish\Core\FieldType\Image\Value as ImageValue;
use eZ\Publish\Core\Persistence\Legacy\Content\Gateway;

/**
 * Integration test for use field type.
 *
 * @group integration
 * @group field-type
 */
class ImageIntegrationTest extends FileSearchBaseIntegrationTest
{
    /**
     * Stores the loaded image path for copy test.
     */
    protected static $loadedImagePath;

    /**
     * IOService storage prefix for the tested Type's files.
     *
     * @var string
     */
    protected static $storagePrefixConfigKey = 'image_storage_prefix';

    protected function getStoragePrefix()
    {
        return $this->getConfigValue(self::$storagePrefixConfigKey);
    }

    /**
     * Sets up fixture data.
     *
     * @return array
     */
    protected function getFixtureData()
    {
        return [
            'create' => [
                'fileName' => 'Icy-Night-Flower.jpg',
                'inputUri' => ($path = __DIR__ . '/_fixtures/image.jpg'),
                'alternativeText' => 'My icy flower at night',
                'fileSize' => filesize($path),
            ],
            'update' => [
                'fileName' => 'Blue-Blue-Blue.png',
                'inputUri' => ($path = __DIR__ . '/_fixtures/image.png'),
                'alternativeText' => 'Such a blue …',
                'fileSize' => filesize($path),
            ],
        ];
    }

    /**
     * Get name of tested field type.
     *
     * @return string
     */
    public function getTypeName()
    {
        return 'ezimage';
    }

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    public function getSettingsSchema()
    {
        return [];
    }

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    public function getValidFieldSettings()
    {
        return [];
    }

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidFieldSettings()
    {
        return [
            'somethingUnknown' => 0,
        ];
    }

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    public function getValidatorSchema()
    {
        return [
            'FileSizeValidator' => [
                'maxFileSize' => [
                    'type' => 'int',
                    'default' => false,
                ],
            ],
            'AlternativeTextValidator' => [
                'required' => [
                    'type' => 'bool',
                    'default' => false,
                ],
            ],
        ];
    }

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    public function getValidValidatorConfiguration()
    {
        return [
            'FileSizeValidator' => [
                'maxFileSize' => 2 * 1024 * 1024, // 2 MB
            ],
            'AlternativeTextValidator' => [
                'required' => true,
            ],
        ];
    }

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidValidatorConfiguration()
    {
        return [
            'StringLengthValidator' => [
                'minStringLength' => new \stdClass(),
            ],
        ];
    }

    /**
     * Get initial field data for valid object creation.
     *
     * @return mixed
     */
    public function getValidCreationFieldData()
    {
        $fixtureData = $this->getFixtureData();

        return new ImageValue($fixtureData['create']);
    }

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    public function getFieldName()
    {
        return 'My icy flower at night';
    }

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     */
    public function assertFieldDataLoadedCorrect(Field $field): void
    {
        self::assertInstanceOf(
            'eZ\\Publish\\Core\\FieldType\\Image\\Value',
            $field->value
        );

        $fixtureData = $this->getFixtureData();
        $expectedData = $fixtureData['create'];

        // Will be nullified by external storage
        $expectedData['inputUri'] = null;

        // Will be changed by external storage as fileName will be decorated with a hash
        $expectedData['fileName'] = $field->value->fileName;

        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );

        self::assertTrue(
            $this->uriExistsOnIO($field->value->uri),
            "Asserting that {$field->value->uri} exists."
        );

        self::$loadedImagePath = $field->value->id;
    }

    /**
     * Get field data which will result in errors during creation.
     *
     * This is a PHPUnit data provider.
     *
     * The returned records must contain of an error producing data value and
     * the expected exception class (from the API or SPI, not implementation
     * specific!) as the second element. For example:
     *
     * <code>
     * array(
     *      array(
     *          new DoomedValue( true ),
     *          'eZ\\Publish\\API\\Repository\\Exceptions\\ContentValidationException'
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array[]
     */
    public function provideInvalidCreationFieldData()
    {
        return [
            // will fail because the provided file doesn't exist, and fileSize/fileName won't be set
            [
                new ImageValue(
                    [
                        'inputUri' => __DIR__ . '/_fixtures/nofile.png',
                    ]
                ),
                'eZ\\Publish\\Core\\Base\\Exceptions\\InvalidArgumentException',
            ],
        ];
    }

    /**
     * Get update field externals data.
     *
     * @return array
     */
    public function getValidUpdateFieldData()
    {
        $fixtureData = $this->getFixtureData();

        return new ImageValue($fixtureData['update']);
    }

    /**
     * Get externals updated field data values.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function assertUpdatedFieldDataLoadedCorrect(Field $field)
    {
        self::assertInstanceOf(
            'eZ\\Publish\\Core\\FieldType\\Image\\Value',
            $field->value
        );

        $fixtureData = $this->getFixtureData();
        $expectedData = $fixtureData['update'];

        // Will change during storage
        $expectedData['inputUri'] = null;

        // Will change during storage as fileName will be decorated with a hash
        $expectedData['fileName'] = $field->value->fileName;

        $expectedData['uri'] = $field->value->uri;

        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );

        self::assertTrue(
            $this->uriExistsOnIO($field->value->uri),
            "Asserting that file {$field->value->uri} exists"
        );
    }

    /**
     * Get field data which will result in errors during update.
     *
     * This is a PHPUnit data provider.
     *
     * The returned records must contain of an error producing data value and
     * the expected exception class (from the API or SPI, not implementation
     * specific!) as the second element. For example:
     *
     * <code>
     * array(
     *      array(
     *          new DoomedValue( true ),
     *          'eZ\\Publish\\API\\Repository\\Exceptions\\ContentValidationException'
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array[]
     */
    public function provideInvalidUpdateFieldData()
    {
        return $this->provideInvalidCreationFieldData();
    }

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \eZ\Publish\API\Repository\Values\Content\Field $field
     */
    public function assertCopiedFieldDataLoadedCorrectly(Field $field)
    {
        $this->assertFieldDataLoadedCorrect($field);

        $this->assertEquals(
            self::$loadedImagePath,
            $field->value->id
        );
    }

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    public function provideToHashData()
    {
        return [
            [
                new ImageValue(
                    [
                        'inputUri' => ($path = __DIR__ . '/_fixtures/image.jpg'),
                        'fileName' => 'Icy-Night-Flower.jpg',
                        'alternativeText' => 'My icy flower at night',
                    ]
                ),
                [
                    'inputUri' => $path,
                    'path' => $path,
                    'fileName' => 'Icy-Night-Flower.jpg',
                    'alternativeText' => 'My icy flower at night',
                    'fileSize' => null,
                    'id' => null,
                    'imageId' => null,
                    'uri' => null,
                    'width' => null,
                    'height' => null,
                    'additionalData' => [],
                ],
            ],
            [
                new ImageValue(
                    [
                        'id' => $path = 'var/test/storage/images/file.png',
                        'fileName' => 'Icy-Night-Flower.jpg',
                        'alternativeText' => 'My icy flower at night',
                        'fileSize' => 23,
                        'imageId' => '1-2',
                        'uri' => "/$path",
                        'width' => 123,
                        'height' => 456,
                    ]
                ),
                [
                    'id' => $path,
                    'path' => $path,
                    'fileName' => 'Icy-Night-Flower.jpg',
                    'alternativeText' => 'My icy flower at night',
                    'fileSize' => 23,
                    'inputUri' => null,
                    'imageId' => '1-2',
                    'uri' => "/$path",
                    'width' => 123,
                    'height' => 456,
                    'additionalData' => [],
                ],
            ],
        ];
    }

    /**
     * Get expectations for the fromHash call on our field value.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function provideFromHashData()
    {
        $fixture = $this->getFixtureData();

        return [
            [
                $fixture['create'],
                $this->getValidCreationFieldData(),
            ],
        ];
    }

    public function testInherentCopyForNewLanguage()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $type = $this->createContentType(
            $this->getValidFieldSettings(),
            $this->getValidValidatorConfiguration(),
            [],
            // Causes a copy of the image value for each language in legacy
            // storage
            ['isTranslatable' => false]
        );

        $draft = $this->createContent($this->getValidCreationFieldData(), $type);

        $updateStruct = $contentService->newContentUpdateStruct();
        $updateStruct->initialLanguageCode = 'ger-DE';
        $updateStruct->setField('name', 'Sindelfingen');

        // Automatically creates a copy of the image field in the back ground
        $updatedDraft = $contentService->updateContent($draft->versionInfo, $updateStruct);

        $paths = [];
        foreach ($updatedDraft->getFields() as $field) {
            if ($field->fieldDefIdentifier === 'data') {
                $paths[$field->languageCode] = $field->value->uri;
            }
        }

        $this->assertTrue(
            isset($paths['eng-US']) && isset($paths['ger-DE']),
            'Failed asserting that file path for all languages were found in draft'
        );

        $this->assertEquals(
            $paths['eng-US'],
            $paths['ger-DE']
        );

        $contentService->deleteContent($updatedDraft->contentInfo);

        foreach ($paths as $uri) {
            self::assertFalse(
                $this->uriExistsOnIO($uri),
                "$uri has not been removed"
            );
        }
    }

    public function providerForTestIsEmptyValue()
    {
        return [
            [new ImageValue()],
        ];
    }

    public function providerForTestIsNotEmptyValue()
    {
        return [
            [
                $this->getValidCreationFieldData(),
            ],
        ];
    }

    /**
     * Covers EZP-23080.
     */
    public function testUpdatingImageMetadataOnlyWorks()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $type = $this->createContentType(
            $this->getValidFieldSettings(),
            $this->getValidValidatorConfiguration(),
            []
        );

        $draft = $this->createContent($this->getValidCreationFieldData(), $type);

        /** @var \eZ\Publish\Core\FieldType\Image\Value $imageFieldValue */
        $imageFieldValue = $draft->getFieldValue('data');
        $initialValueImageUri = $imageFieldValue->uri;

        // update alternative text
        $imageFieldValue->alternativeText = __METHOD__;
        $updateStruct = $contentService->newContentUpdateStruct();
        $updateStruct->setField('data', $imageFieldValue);
        $updatedDraft = $contentService->updateContent($draft->versionInfo, $updateStruct);

        /** @var \eZ\Publish\Core\FieldType\Image\Value $updatedImageValue */
        $updatedImageValue = $updatedDraft->getFieldValue('data');

        self::assertEquals($initialValueImageUri, $updatedImageValue->uri);
        self::assertEquals(__METHOD__, $updatedImageValue->alternativeText);
    }

    /**
     * @see https://jira.ez.no/browse/EZP-23152
     *
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    public function testThatRemovingDraftDoesntRemovePublishedImages(): void
    {
        $repository = $this->getRepository();

        // Load services
        $contentService = $repository->getContentService();

        // create content and publish image
        $content = $this->publishNewImage(
            'EZP23152_1',
            $this->getValidCreationFieldData(),
            [2]
        );
        $originalFileUri = $this->getImageURI($content);

        self::assertTrue(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting image file $originalFileUri exists."
        );

        // Create a new draft and update it
        $updatedDraft = $contentService->createContentDraft($content->contentInfo);
        $contentUpdateStruct = $contentService->newContentUpdateStruct();
        $contentUpdateStruct->initialLanguageCode = 'eng-GB';
        $contentUpdateStruct->setField('name', 'EZP23152_2');
        $updatedDraft = $contentService->updateContent($updatedDraft->versionInfo, $contentUpdateStruct);

        // remove the newly published content version, verify that the original file exists
        $contentService->deleteVersion($updatedDraft->versionInfo);
        self::assertTrue(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting original image file $originalFileUri exists."
        );

        // delete content
        $contentService->deleteContent($content->contentInfo);
        self::assertFalse(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting image file $originalFileUri has been removed."
        );
    }

    /**
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    public function testUpdateImageAltTextOnly(): void
    {
        $content = $this->publishNewImage(
            __METHOD__,
            new ImageValue(
                [
                    'inputUri' => __DIR__ . '/_fixtures/image.jpg',
                    'fileName' => 'image.jpg',
                    'fileSize' => filesize(__DIR__ . '/_fixtures/image.jpg'),
                    'alternativeText' => 'Initial alternative text',
                ]
            ),
            [2]
        );

        /** @var \eZ\Publish\Core\FieldType\Image\Value $imageField */
        $imageField = $content->getFieldValue('image');
        $updatedAlternativeText = 'Updated alternative text';
        $imageField->alternativeText = $updatedAlternativeText;

        $content = $this->updateImage($content, $imageField);

        self::assertSame(
            $updatedAlternativeText,
            $content->getFieldValue('image')->alternativeText
        );
    }

    protected function getValidSearchValueOne()
    {
        return new ImageValue(
            [
                'fileName' => '1234eeee1234-cafe-terrace-at-night.jpg',
                'inputUri' => ($path = __DIR__ . '/_fixtures/1234eeee1234-image.jpg'),
                'alternativeText' => 'café terrace at night, also known as the cafe terrace on the place du forum',
                'fileSize' => filesize($path),
            ]
        );
    }

    protected function getValidSearchValueTwo()
    {
        return new ImageValue(
            [
                'fileName' => '2222eeee1111-thatched-cottages-at-cordeville.png',
                'inputUri' => ($path = __DIR__ . '/_fixtures/2222eeee1111-image.png'),
                'alternativeText' => 'chaumes de cordeville à auvers-sur-oise',
                'fileSize' => filesize($path),
            ]
        );
    }

    protected function getSearchTargetValueOne()
    {
        $value = $this->getValidSearchValueOne();

        // ensure case-insensitivity
        return strtoupper($value->fileName);
    }

    protected function getSearchTargetValueTwo()
    {
        $value = $this->getValidSearchValueTwo();
        // ensure case-insensitivity
        return strtoupper($value->fileName);
    }

    protected function getAdditionallyIndexedFieldData()
    {
        return [
            [
                'alternative_text',
                $this->getValidSearchValueOne()->alternativeText,
                $this->getValidSearchValueTwo()->alternativeText,
            ],
            [
                'file_size',
                $this->getValidSearchValueOne()->fileSize,
                $this->getValidSearchValueTwo()->fileSize,
            ],
            [
                'mime_type',
                // ensure case-insensitivity
                'IMAGE/JPEG',
                'IMAGE/PNG',
            ],
        ];
    }

    /**
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    public function testRemovingContentRemovesImages(): void
    {
        $repository = $this->getRepository();

        // Load services
        $contentService = $repository->getContentService();

        $content = $this->publishNewImage('My Image', $this->getValidCreationFieldData());
        $originalFileUri = $this->getImageURI($content);

        // sanity check
        self::assertTrue(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting image file $originalFileUri exists"
        );

        $content = $this->updateImage($content, $this->getValidUpdateFieldData());
        $updatedFileUri = $this->getImageURI($content);

        // sanity check
        self::assertNotEquals($originalFileUri, $updatedFileUri);

        $contentService->deleteContent($content->contentInfo);

        self::assertFalse(
            $this->uriExistsOnIO($updatedFileUri),
            "Asserting updated image file $updatedFileUri has been removed"
        );

        self::assertFalse(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting original image file $originalFileUri has been removed"
        );
    }

    /**
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    public function testRemovingDraftRemovesOldImage(): void
    {
        $repository = $this->getRepository();

        // Load services
        $contentService = $repository->getContentService();

        $contentVersion1 = $this->publishNewImage('My Image', $this->getValidCreationFieldData());
        $originalFileUri = $this->getImageURI($contentVersion1);

        // sanity check
        self::assertTrue(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting image file $originalFileUri exists"
        );

        $contentVersion2 = $this->updateImage($contentVersion1, $this->getValidUpdateFieldData());
        $updatedFileUri = $this->getImageURI($contentVersion2);

        // delete 1st version with original image
        $contentService->deleteVersion(
            // reload 1st version (its state changed) to delete
            $contentService->loadVersionInfo(
                $contentVersion1->contentInfo,
                $contentVersion1->getVersionInfo()->versionNo
            )
        );

        // updated image should be available, but original image should be gone now
        self::assertTrue(
            $this->uriExistsOnIO($updatedFileUri),
            "Asserting image file {$updatedFileUri} exists"
        );

        self::assertFalse(
            $this->uriExistsOnIO($originalFileUri),
            "Asserting image file {$originalFileUri} has been removed"
        );
    }

    public function testDeleteImageWithCorruptedName(): void
    {
        $content = $this->publishNewImage(
            __METHOD__,
            new ImageValue(
                [
                    'inputUri' => __DIR__ . '/_fixtures/image.jpg',
                    'fileName' => 'image.jpg',
                    'fileSize' => filesize(__DIR__ . '/_fixtures/image.jpg'),
                    'alternativeText' => 'Alternative',
                ]
            ),
            [2]
        );

        $imageFieldDefinition = $content->getContentType()->getFieldDefinition('image');

        $record = $this->fetchXML(
            $content->id,
            $content->getVersionInfo()->versionNo,
            $imageFieldDefinition->id
        );

        $document = $this->corruptImageFieldXML($record);

        $this->updateXML(
            $content->id,
            $content->getVersionInfo()->versionNo,
            $imageFieldDefinition->id,
            $document
        );

        $repository = $this->getRepository(false);
        $contentService = $repository->getContentService();

        $contentService->deleteContent($content->getVersionInfo()->getContentInfo());

        // Expect no League\Flysystem\CorruptedPathDetected thrown
    }

    /**
     * @return array<string,mixed>
     */
    private function fetchXML(int $contentId, int $versionNo, int $fieldDefinitionId): array
    {
        $connection = $this->getRawDatabaseConnection();

        $query = $connection->createQueryBuilder();
        $query
            ->select('data_text')
            ->from(Gateway::CONTENT_FIELD_TABLE)
            ->andWhere('contentclassattribute_id = :contentclassattribute_id')
            ->andWhere('version = :version')
            ->andWhere('contentobject_id = :contentobject_id')
            ->setParameter('contentclassattribute_id', $fieldDefinitionId, ParameterType::INTEGER)
            ->setParameter('version', $versionNo, ParameterType::INTEGER)
            ->setParameter('contentobject_id', $contentId, ParameterType::INTEGER);
        $result = $query->execute();

        return $result->fetchAssociative();
    }

    /**
     * @param array<string,mixed> $row
     */
    private function corruptImageFieldXML(array $row): DOMDocument
    {
        $corruptedChar = '­';

        $document = new DOMDocument('1.0', 'utf-8');
        $document->loadXML($row['data_text']);
        $elements = $document->getElementsByTagName('ezimage');
        $element = $elements->item(0);
        $element->setAttribute('filename', $element->getAttribute('filename') . $corruptedChar);
        $element->setAttribute('url', $element->getAttribute('url') . $corruptedChar);

        return $document;
    }

    private function updateXML(
        int $contentId,
        int $versionNo,
        int $fieldDefinitionId,
        DOMDocument $document
    ): void {
        $connection = $this->getRawDatabaseConnection();

        $query = $connection->createQueryBuilder();
        $query
            ->update(Gateway::CONTENT_FIELD_TABLE)
            ->set('data_text', ':data_text')
            ->setParameter('data_text', $document->saveXML(), ParameterType::STRING)
            ->andWhere('contentclassattribute_id = :contentclassattribute_id')
            ->andWhere('version = :version')
            ->andWhere('contentobject_id = :contentobject_id')
            ->setParameter('contentclassattribute_id', $fieldDefinitionId, ParameterType::INTEGER)
            ->setParameter('version', $versionNo, ParameterType::INTEGER)
            ->setParameter('contentobject_id', $contentId, ParameterType::INTEGER);

        $query->execute();
    }

    /**
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    private function publishNewImage(
        string $name,
        ImageValue $imageValue,
        array $parentLocationIDs = []
    ): Content {
        $repository = $this->getRepository(false);
        $contentService = $repository->getContentService();
        $locationService = $repository->getLocationService();
        $contentTypeService = $repository->getContentTypeService();

        $contentCreateStruct = $contentService->newContentCreateStruct(
            $contentTypeService->loadContentTypeByIdentifier('image'),
            'eng-GB'
        );
        $contentCreateStruct->setField('name', $name);
        $contentCreateStruct->setField('image', $imageValue);

        $locationCreateStructList = [];
        foreach ($parentLocationIDs as $parentLocationID) {
            $locationCreateStructList[] = $locationService->newLocationCreateStruct(
                $parentLocationID
            );
        }

        return $contentService->publishVersion(
            $contentService
                ->createContent($contentCreateStruct, $locationCreateStructList)
                ->getVersionInfo()
        );
    }

    /**
     * @throws \eZ\Publish\API\Repository\Exceptions\ForbiddenException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     * @throws \eZ\Publish\API\Repository\Exceptions\NotFoundException
     */
    private function updateImage(Content $publishedImageContent, ImageValue $newImageValue): Content
    {
        $repository = $this->getRepository(false);
        $contentService = $repository->getContentService();

        $contentDraft = $contentService->createContentDraft($publishedImageContent->contentInfo);
        $contentUpdateStruct = $contentService->newContentUpdateStruct();
        $contentUpdateStruct->setField('image', $newImageValue);
        $contentService->updateContent($contentDraft->getVersionInfo(), $contentUpdateStruct);

        $content = $contentService->publishVersion($contentDraft->getVersionInfo());

        // reload Content to make sure proper data has been persisted
        return $contentService->loadContentByContentInfo($content->contentInfo);
    }

    private function getImageURI(Content $content): string
    {
        return $content->getFieldValue('image')->uri;
    }
}
