<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
namespace eZ\Publish\API\Repository\Tests\FieldType;

use eZ\Publish\API\Repository;
use eZ\Publish\API\Repository\Exceptions\ContentTypeFieldDefinitionValidationException;
use eZ\Publish\API\Repository\Exceptions\NotFoundException;
use eZ\Publish\API\Repository\Tests;
use eZ\Publish\API\Repository\Values\Content\Content;
use eZ\Publish\API\Repository\Values\Content\Field;
use eZ\Publish\API\Repository\Values\ContentType\FieldDefinition;

/**
 * Integration test for legacy storage field types.
 *
 * This abstract base test case is supposed to be the base for field type
 * integration tests. It basically calls all involved methods in the field type
 * ``Converter`` and ``Storage`` implementations. Fo get it working implement
 * the abstract methods in a sensible way.
 *
 * The following actions are performed by this test using the custom field
 * type:
 *
 * - Create a new content type with the given field type
 * - Load created content type
 * - Create content object of new content type
 * - Load created content
 * - Publish created content
 * - Update content
 * - Copy created content
 * - Remove copied content
 * - Test toHash
 * - Test fromHash
 *
 * @group integration
 * @group field-type
 *
 * @todo Finalize dependencies to other tests (including groups!)
 */
abstract class BaseIntegrationTest extends Tests\BaseTest
{
    /**
     * Content version archive limit (default).
     * Note: currently there is no way to retrieve this setting from the ContentService.
     */
    public const VERSION_ARCHIVE_LIMIT = 5;

    /**
     * Identifier of the custom field.
     *
     * @var string
     */
    protected $customFieldIdentifier = 'data';

    /**
     * Get name of tested field type.
     *
     * @return string
     */
    abstract public function getTypeName();

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    abstract public function getSettingsSchema();

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    abstract public function getValidFieldSettings();

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    abstract public function getInvalidFieldSettings();

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    abstract public function getValidatorSchema();

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    abstract public function getValidValidatorConfiguration();

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    abstract public function getInvalidValidatorConfiguration();

    /**
     * Get initial field data for valid object creation.
     *
     * @return mixed
     */
    abstract public function getValidCreationFieldData();

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    abstract public function getFieldName();

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     *
     * @param \eZ\Publish\API\Repository\Values\Content\Field $field
     */
    abstract public function assertFieldDataLoadedCorrect(Field $field);

    /**
     * Get field data which will result in errors during creation.
     *
     * This is a PHPUnit data provider.
     *
     * The returned records must contain of an error producing data value and
     * the expected exception class (from the API or SPI, not implementation
     * specific!) as the second element. For example:
     *
     * <code>
     * array(
     *      array(
     *          new DoomedValue( true ),
     *          'eZ\\Publish\\API\\Repository\\Exceptions\\ContentValidationException'
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array[]
     */
    abstract public function provideInvalidCreationFieldData();

    /**
     * Get valid field data for updating content.
     *
     * @return mixed
     */
    abstract public function getValidUpdateFieldData();

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidUpdateFieldData()}
     * was stored and loaded correctly.
     *
     * @param \eZ\Publish\API\Repository\Values\Content\Field $field
     */
    abstract public function assertUpdatedFieldDataLoadedCorrect(Field $field);

    /**
     * Get field data which will result in errors during update.
     *
     * This is a PHPUnit data provider.
     *
     * The returned records must contain of an error producing data value and
     * the expected exception class (from the API or SPI, not implementation
     * specific!) as the second element. For example:
     *
     * <code>
     * array(
     *      array(
     *          new DoomedValue( true ),
     *          'eZ\\Publish\\API\\Repository\\Exceptions\\ContentValidationException'
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array[]
     */
    abstract public function provideInvalidUpdateFieldData();

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \eZ\Publish\API\Repository\Values\Content\Field $field
     */
    abstract public function assertCopiedFieldDataLoadedCorrectly(Field $field);

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    abstract public function provideToHashData();

    /**
     * Get hashes and their respective converted values.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the input hash assigned to the
     * first index and the expected value result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          array( 'myValue' => true ),
     *          new MyValue( true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    abstract public function provideFromHashData();

    /**
     * Method called after content creation.
     *
     * Useful, if additional stuff should be executed (like creating the actual
     * user).
     *
     * We cannot just overwrite the testCreateContent method, since this messes
     * up PHPUnits @depends sorting of tests, so everything will be skipped.
     *
     * @param \eZ\Publish\API\Repository $repository
     * @param \eZ\Publish\API\Repository\Values\Content\Content $content
     */
    public function postCreationHook(Repository\Repository $repository, Repository\Values\Content\Content $content)
    {
        // Do nothing by default
    }

    public function getValidContentTypeConfiguration(): array
    {
        return [];
    }

    public function getValidFieldConfiguration(): array
    {
        return [];
    }

    public function testCreateContentType()
    {
        $contentType = $this->createContentType(
            $this->getValidFieldSettings(),
            $this->getValidValidatorConfiguration(),
            $this->getValidContentTypeConfiguration(),
            $this->getValidFieldConfiguration()
        );

        $this->assertNotNull($contentType->id);

        return $contentType;
    }

    /**
     * For checking if field type can be used in name/url schema (pattern).
     *
     * @return bool
     */
    protected function checkSupportGetName()
    {
        return true;
    }

    /**
     * Creates a content type under test with $fieldSettings and
     * $validatorConfiguration.
     *
     * $typeCreateOverride and $fieldCreateOverride can be used to selectively
     * override settings on the type create struct and field create struct.
     *
     * @param mixed $fieldSettings
     * @param mixed $validatorConfiguration
     * @param array $typeCreateOverride
     * @param array $fieldCreateOverride
     *
     * @return \eZ\Publish\API\Repository\Values\ContentType\ContentType
     */
    protected function createContentType($fieldSettings, $validatorConfiguration, array $typeCreateOverride = [], array $fieldCreateOverride = [])
    {
        $repository = $this->getRepository();
        $contentTypeService = $repository->getContentTypeService();

        $contentTypeIdentifier = 'test-' . $this->getTypeName();

        try {
            return $contentTypeService->loadContentTypeByIdentifier($contentTypeIdentifier);
        } catch (NotFoundException $e) {
            // Move on to creating Content Type
        }

        $createStruct = $contentTypeService->newContentTypeCreateStruct(
            $contentTypeIdentifier
        );
        $createStruct->mainLanguageCode = $this->getOverride('mainLanguageCode', $typeCreateOverride, 'eng-GB');
        $createStruct->remoteId = $this->getTypeName();
        $createStruct->names = $this->getOverride('names', $typeCreateOverride, ['eng-GB' => 'Test']);
        $createStruct->creatorId = 14;
        $createStruct->creationDate = $this->createDateTime();

        if ($this->checkSupportGetName()) {
            $createStruct->nameSchema = '<name> <data>';
            $createStruct->urlAliasSchema = '<data>';
        }

        $nameFieldCreate = $contentTypeService->newFieldDefinitionCreateStruct('name', 'ezstring');
        $nameFieldCreate->names = ['eng-GB' => 'Title'];
        $nameFieldCreate->fieldGroup = 'main';
        $nameFieldCreate->position = 1;
        $nameFieldCreate->isTranslatable = true;
        $createStruct->addFieldDefinition($nameFieldCreate);

        $dataFieldCreate = $contentTypeService->newFieldDefinitionCreateStruct('data', $this->getTypeName());
        $dataFieldCreate->names = $this->getOverride('names', $fieldCreateOverride, ['eng-GB' => 'Title']);
        $dataFieldCreate->fieldGroup = 'main';
        $dataFieldCreate->position = 2;
        $dataFieldCreate->isTranslatable = $this->getOverride('isTranslatable', $fieldCreateOverride, false);

        // Custom settings
        $dataFieldCreate->fieldSettings = $fieldSettings;
        $dataFieldCreate->validatorConfiguration = $validatorConfiguration;

        $createStruct->addFieldDefinition($dataFieldCreate);

        $contentGroup = $contentTypeService->loadContentTypeGroupByIdentifier('Content');
        $contentTypeDraft = $contentTypeService->createContentType($createStruct, [$contentGroup]);

        $contentTypeService->publishContentTypeDraft($contentTypeDraft);
        $contentType = $contentTypeService->loadContentType($contentTypeDraft->id);

        return $contentType;
    }

    /**
     * Retrieves a value for $key from $overrideValues, falling back to
     * $default.
     *
     * @param string $key
     * @param array $overrideValues
     * @param mixed $default
     *
     * @return mixed
     */
    protected function getOverride($key, array $overrideValues, $default)
    {
        return isset($overrideValues[$key]) ? $overrideValues[$key] : $default;
    }

    /**
     * @covers \eZ\Publish\Core\FieldType\FieldType::isEmptyValue
     * @dataProvider providerForTestIsEmptyValue
     */
    public function testIsEmptyValue($value)
    {
        $this->assertTrue($this->getRepository()->getFieldTypeService()->getFieldType($this->getTypeName())->isEmptyValue($value));
    }

    abstract public function providerForTestIsEmptyValue();

    /**
     * @covers \eZ\Publish\Core\FieldType\FieldType::isEmptyValue
     * @dataProvider providerForTestIsNotEmptyValue
     */
    public function testIsNotEmptyValue($value)
    {
        $this->assertFalse($this->getRepository()->getFieldTypeService()->getFieldType($this->getTypeName())->isEmptyValue($value));
    }

    abstract public function providerForTestIsNotEmptyValue();

    /**
     * @depends testCreateContentType
     */
    public function testContentTypeField($contentType)
    {
        $this->assertSame(
            $this->getTypeName(),
            $contentType->fieldDefinitions[1]->fieldTypeIdentifier
        );
    }

    /**
     * @depends testCreateContentType
     */
    public function testLoadContentTypeField()
    {
        $contentType = $this->testCreateContentType();

        $repository = $this->getRepository();
        $contentTypeService = $repository->getContentTypeService();

        return $contentTypeService->loadContentType($contentType->id);
    }

    /**
     * @depends testLoadContentTypeField
     */
    public function testLoadContentTypeFieldType($contentType)
    {
        $this->assertSame(
            $this->getTypeName(),
            $contentType->fieldDefinitions[1]->fieldTypeIdentifier
        );

        return $contentType->fieldDefinitions[1];
    }

    public function testSettingsSchema()
    {
        $repository = $this->getRepository();
        $fieldTypeService = $repository->getFieldTypeService();
        $fieldType = $fieldTypeService->getFieldType($this->getTypeName());

        $this->assertEquals(
            $this->getSettingsSchema(),
            $fieldType->getSettingsSchema()
        );
    }

    /**
     * @depends testLoadContentTypeFieldType
     */
    public function testLoadContentTypeFieldData(FieldDefinition $fieldDefinition)
    {
        $this->assertEquals(
            $this->getTypeName(),
            $fieldDefinition->fieldTypeIdentifier,
            'Loaded fieldTypeIdentifier does not match.'
        );
        $this->assertEquals(
            $this->getValidFieldSettings(),
            $fieldDefinition->fieldSettings,
            'Loaded fieldSettings do not match.'
        );
        $this->assertEquals(
            $this->getValidValidatorConfiguration(),
            $fieldDefinition->validatorConfiguration,
            'Loaded validatorConfiguration does not match.'
        );
    }

    /**
     * @depends testCreateContentType
     */
    public function testCreateContentTypeFailsWithInvalidFieldSettings()
    {
        $this->expectException(ContentTypeFieldDefinitionValidationException::class);

        $this->createContentType(
            $this->getInvalidFieldSettings(),
            $this->getValidValidatorConfiguration()
        );
    }

    public function testValidatorSchema()
    {
        $repository = $this->getRepository();
        $fieldTypeService = $repository->getFieldTypeService();
        $fieldType = $fieldTypeService->getFieldType($this->getTypeName());

        $this->assertEquals(
            $this->getValidatorSchema(),
            $fieldType->getValidatorConfigurationSchema()
        );
    }

    /**
     * @depends testCreateContentType
     */
    public function testCreateContentTypeFailsWithInvalidValidatorConfiguration()
    {
        $this->expectException(ContentTypeFieldDefinitionValidationException::class);

        $this->createContentType(
            $this->getValidFieldSettings(),
            $this->getInvalidValidatorConfiguration()
        );
    }

    /**
     * @depends testLoadContentTypeField
     */
    public function testCreateContent()
    {
        return $this->createContent($this->getValidCreationFieldData());
    }

    /**
     * Creates content with $fieldData.
     *
     * @param mixed $fieldData
     *
     * @return \eZ\Publish\API\Repository\Values\Content\Content
     */
    protected function createContent($fieldData, $contentType = null)
    {
        if ($contentType === null) {
            $contentType = $this->testCreateContentType();
        }

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $createStruct = $contentService->newContentCreateStruct($contentType, 'eng-US');
        $createStruct->setField('name', 'Test object');
        $createStruct->setField(
            'data',
            $fieldData
        );

        $createStruct->remoteId = 'abcdef0123456789abcdef0123456789';
        $createStruct->alwaysAvailable = true;

        return $contentService->createContent($createStruct);
    }

    /**
     * Create multilingual content of given name and FT-specific data.
     *
     * @param array $names Content names in the form of <code>[languageCode => name]</code>
     * @param array $fieldData FT-specific data in the form of <code>[languageCode => data]</code>
     * @param \eZ\Publish\API\Repository\Values\Content\LocationCreateStruct[] $locationCreateStructs
     *
     * @return \eZ\Publish\API\Repository\Values\Content\Content
     *
     * @throws \eZ\Publish\API\Repository\Exceptions\ContentFieldValidationException
     * @throws \eZ\Publish\API\Repository\Exceptions\ContentValidationException
     * @throws \eZ\Publish\API\Repository\Exceptions\InvalidArgumentException
     * @throws \eZ\Publish\API\Repository\Exceptions\UnauthorizedException
     */
    protected function createMultilingualContent(array $names, array $fieldData, array $locationCreateStructs = [])
    {
        self::assertEquals(array_keys($names), array_keys($fieldData), 'Languages passed to names and data differ');

        $contentType = $this->createContentType(
            $this->getValidFieldSettings(),
            $this->getValidValidatorConfiguration(),
            $this->getValidContentTypeConfiguration(),
            array_merge(
                $this->getValidFieldConfiguration(),
                ['isTranslatable' => true]
            )
        );

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $createStruct = $contentService->newContentCreateStruct($contentType, 'eng-US');
        foreach ($names as $languageCode => $name) {
            $createStruct->setField('name', $name, $languageCode);
        }
        foreach ($fieldData as $languageCode => $value) {
            $createStruct->setField('data', $value, $languageCode);
        }

        $createStruct->remoteId = md5(uniqid('', true) . microtime());
        $createStruct->alwaysAvailable = true;

        return $contentService->createContent($createStruct, $locationCreateStructs);
    }

    /**
     * @depends testCreateContent
     */
    public function testCreatedFieldType($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testCreateContent
     */
    public function testPublishContent()
    {
        $draft = $this->testCreateContent();

        if (!$draft->getVersionInfo()->isDraft()) {
            $this->markTestSkipped('Provided content object is not a draft.');
        }

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        return $contentService->publishVersion($draft->getVersionInfo());
    }

    /**
     * @depends testPublishContent
     */
    public function testPublishedFieldType($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testPublishContent
     */
    public function testPublishedName(Content $content)
    {
        $this->assertEquals(
            $content->getFieldValue('name') . ' ' . $this->getFieldName(),
            $content->contentInfo->name
        );
    }

    /**
     * @depends testCreateContent
     */
    public function testLoadField()
    {
        $content = $this->testCreateContent();

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        return $contentService->loadContent($content->contentInfo->id);
    }

    /**
     * @depends testLoadField
     */
    public function testLoadFieldType()
    {
        $content = $this->testCreateContent();

        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testLoadFieldType
     */
    public function testLoadExternalData()
    {
        $this->assertFieldDataLoadedCorrect($this->testLoadFieldType());
    }

    public function testCreateContentWithEmptyFieldValue()
    {
        /** @var \eZ\Publish\Core\FieldType\FieldType $fieldType */
        $fieldType = $this->getRepository()->getFieldTypeService()->getFieldType($this->getTypeName());

        return $this->createContent($fieldType->getEmptyValue());
    }

    /**
     * Test that publishing (and thus indexing) content with an empty field value does not fail.
     *
     * @depends testCreateContentWithEmptyFieldValue
     *
     * @param \eZ\Publish\API\Repository\Values\Content\Content $contentDraft
     */
    public function testPublishContentWithEmptyFieldValue(Content $contentDraft)
    {
        $this->getRepository(false)->getContentService()->publishVersion(
            $contentDraft->versionInfo
        );
    }

    /**
     * @depends testCreateContentWithEmptyFieldValue
     */
    public function testCreatedEmptyFieldValue($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testCreateContentWithEmptyFieldValue
     * @group xx
     */
    public function testLoadEmptyFieldValue()
    {
        $content = $this->testCreateContentWithEmptyFieldValue();

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        return $contentService->loadContent($content->contentInfo->id);
    }

    /**
     * @depends testLoadEmptyFieldValue
     */
    public function testLoadEmptyFieldValueType($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testLoadEmptyFieldValueType
     */
    public function testLoadEmptyFieldValueData($field)
    {
        /** @var \eZ\Publish\Core\FieldType\FieldType $fieldType */
        $fieldType = $this->getRepository()->getFieldTypeService()->getFieldType($this->getTypeName());

        // @todo either test this not using acceptValue, or add to API (but is not meant for high level API, so..)
        $refObject = new \ReflectionObject($fieldType);
        $refProperty = $refObject->getProperty('internalFieldType');
        $refProperty->setAccessible(true);
        $spiFieldType = $refProperty->getValue($fieldType);

        $this->assertEquals(
            $fieldType->getEmptyValue(),
            $spiFieldType->acceptValue($field->value)
        );
    }

    /**
     * @depends testLoadFieldType
     */
    public function testUpdateField()
    {
        return $this->updateContent($this->getValidUpdateFieldData());
    }

    /**
     * Updates the standard published content object with $fieldData.
     *
     * @param mixed $fieldData
     * @param bool $setField If false the update struct will be empty (field value will not be set)
     *
     * @return \eZ\Publish\API\Repository\Values\Content\Content
     */
    public function updateContent($fieldData, $setField = true)
    {
        $content = $this->testPublishContent();

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $draft = $contentService->createContentDraft($content->contentInfo);

        $updateStruct = $contentService->newContentUpdateStruct();
        if ($setField) {
            $updateStruct->setField(
                $this->customFieldIdentifier,
                $fieldData
            );
        }

        return $contentService->updateContent($draft->versionInfo, $updateStruct);
    }

    /**
     * @depends testUpdateField
     */
    public function testUpdateTypeFieldStillAvailable($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testUpdateTypeFieldStillAvailable
     */
    public function testUpdatedDataCorrect(Field $field)
    {
        $this->assertUpdatedFieldDataLoadedCorrect($field);
    }

    /**
     * Tests creating a new Version keeps the existing value.
     */
    public function testUpdateFieldNoNewContent()
    {
        return $this->updateContent(null, false);
    }

    /**
     * @depends testUpdateFieldNoNewContent
     */
    public function testUpdateNoNewContentTypeFieldStillAvailable($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testUpdateNoNewContentTypeFieldStillAvailable
     */
    public function testUpdatedNoNewContentDataCorrect(Field $field)
    {
        $this->assertFieldDataLoadedCorrect($field);
    }

    /**
     * @depends testCreateContent
     */
    public function testCopyField($content)
    {
        $content = $this->testCreateContent();

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $locationService = $repository->getLocationService();
        $parentLocationId = $this->generateId('location', 2);
        $locationCreate = $locationService->newLocationCreateStruct($parentLocationId);

        $copied = $contentService->copyContent($content->contentInfo, $locationCreate);

        $this->assertNotSame(
            $content->contentInfo->id,
            $copied->contentInfo->id
        );

        return $contentService->loadContent($copied->id);
    }

    /**
     * @depends testCopyField
     */
    public function testCopiedFieldType($content)
    {
        foreach ($content->getFields() as $field) {
            if ($field->fieldDefIdentifier === $this->customFieldIdentifier) {
                return $field;
            }
        }

        $this->fail('Custom field not found.');
    }

    /**
     * @depends testCopiedFieldType
     */
    public function testCopiedExternalData(Field $field)
    {
        $this->assertCopiedFieldDataLoadedCorrectly($field);
    }

    /**
     * @depends testCopyField
     */
    public function testDeleteContent($content)
    {
        $this->expectException(NotFoundException::class);

        $content = $this->testPublishContent();

        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $contentService->deleteContent($content->contentInfo);

        $contentService->loadContent($content->contentInfo->id);
    }

    /**
     * Tests failing content creation.
     *
     * @param mixed $failingValue
     *
     * @dataProvider provideInvalidCreationFieldData
     */
    public function testCreateContentFails($failingValue, ?string $expectedException): void
    {
        $this->expectException($expectedException);
        $this->createContent($failingValue);
    }

    /**
     * Tests failing content update.
     *
     * @param mixed $failingValue
     * @param string $expectedException
     *
     * @dataProvider provideInvalidUpdateFieldData
     */
    public function testUpdateContentFails($failingValue, $expectedException)
    {
        $this->expectException($expectedException);
        $this->updateContent($failingValue);
    }

    protected function removeFieldDefinition()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();
        $contentTypeService = $repository->getContentTypeService();
        $content = $this->testPublishContent();

        $contentType = $contentTypeService->loadContentType($content->contentInfo->contentTypeId);
        $contentTypeDraft = $contentTypeService->createContentTypeDraft($contentType);
        $fieldDefinition = $contentTypeDraft->getFieldDefinition('data');

        $contentTypeService->removeFieldDefinition($contentTypeDraft, $fieldDefinition);
        $contentTypeService->publishContentTypeDraft($contentTypeDraft);

        return $contentService->loadContent($content->id);
    }

    /**
     * Tests removal of field definition from the ContentType of the Content.
     */
    public function testRemoveFieldDefinition()
    {
        $content = $this->removeFieldDefinition();

        $this->assertCount(1, $content->getFields());
        $this->assertNull($content->getFieldValue('data'));
    }

    protected function addFieldDefinition()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();
        $contentTypeService = $repository->getContentTypeService();
        $content = $this->removeFieldDefinition();

        $contentType = $contentTypeService->loadContentType($content->contentInfo->contentTypeId);
        $contentTypeDraft = $contentTypeService->createContentTypeDraft($contentType);

        $fieldDefinitionCreateStruct = $contentTypeService->newFieldDefinitionCreateStruct(
            'data',
            $this->getTypeName()
        );

        $fieldDefinitionCreateStruct->names = $this->getOverride('names', $this->getValidFieldConfiguration(), [$contentType->mainLanguageCode => $this->getTypeName()]);
        $fieldDefinitionCreateStruct->validatorConfiguration = $this->getValidValidatorConfiguration();
        $fieldDefinitionCreateStruct->fieldSettings = $this->getValidFieldSettings();
        $fieldDefinitionCreateStruct->defaultValue = null;

        $contentTypeService->addFieldDefinition($contentTypeDraft, $fieldDefinitionCreateStruct);
        $contentTypeService->publishContentTypeDraft($contentTypeDraft);

        return $contentService->loadContent($content->id);
    }

    /**
     * Tests addition of field definition from the ContentType of the Content.
     */
    public function testAddFieldDefinition()
    {
        $content = $this->addFieldDefinition();

        $this->assertCount(2, $content->getFields());

        $this->assertTrue(
            $this->getRepository()->getFieldTypeService()->getFieldType(
                $this->getTypeName()
            )->isEmptyValue(
                $content->getFieldValue('data')
            )
        );
    }

    /**
     * @dataProvider provideToHashData
     */
    public function testToHash($value, $expectedHash)
    {
        $repository = $this->getRepository();
        $fieldTypeService = $repository->getFieldTypeService();
        $fieldType = $fieldTypeService->getFieldType($this->getTypeName());

        $this->assertEquals(
            $expectedHash,
            $fieldType->toHash($value)
        );
    }

    /**
     * @depends testCreateContent
     * @dataProvider provideFromHashData
     * @todo: Requires correct registered FieldTypeService, needs to be
     *        maintained!
     */
    public function testFromHash($hash, $expectedValue)
    {
        $repository = $this->getRepository();
        $fieldTypeService = $repository->getFieldTypeService();
        $fieldType = $fieldTypeService->getFieldType($this->getTypeName());

        $this->assertEquals(
            $expectedValue,
            $fieldType->fromHash($hash)
        );
    }

    /**
     * Test that exceeding default version archive limit has no effect on a published content.
     */
    public function testExceededVersionArchiveLimitHasNoEffectOnContent()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();
        $contentDraft = $this->createContent($this->getValidCreationFieldData());
        $publishedContent = $contentService->publishVersion($contentDraft->versionInfo);
        // update and publish content to exceed version archive limit
        $contentUpdateStruct = $contentService->newContentUpdateStruct();
        $contentUpdateStruct->setField('data', $this->getValidUpdateFieldData());
        for ($i = 0; $i < static::VERSION_ARCHIVE_LIMIT + 1; ++$i) {
            $contentDraft = $contentService->createContentDraft($publishedContent->contentInfo);
            $contentService->updateContent($contentDraft->versionInfo, $contentUpdateStruct);
            $publishedContent = $contentService->publishVersion($contentDraft->versionInfo);
        }

        $loadedContent = $contentService->loadContent(
            $publishedContent->contentInfo->id,
            ['eng-US']
        );
        $this->assertUpdatedFieldDataLoadedCorrect($loadedContent->getField('data'));
    }

    /**
     * Test that deleting new draft does not affect data of published version.
     */
    public function testDeleteDraftOfPublishedContentDoesNotDeleteData()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();
        $fieldType = $repository->getFieldTypeService()->getFieldType($this->getTypeName());

        $contentDraft = $this->testCreateContent();
        $publishedContent = $contentService->publishVersion($contentDraft->versionInfo);

        $contentDraft = $contentService->createContentDraft($publishedContent->contentInfo);

        $contentService->deleteVersion($contentDraft->versionInfo);
        $loadedContent = $contentService->loadContent($publishedContent->contentInfo->id, ['eng-US']);

        self::assertFalse(
            $fieldType->isEmptyValue($loadedContent->getField('data')->value)
        );
    }

    /**
     * Test creating new translation from existing content with empty field.
     */
    public function testUpdateContentWithNewTranslationOnEmptyField()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $content = $this->testCreateContentWithEmptyFieldValue();
        $publishedContent = $contentService->publishVersion($content->versionInfo);

        $contentDraft = $contentService->createContentDraft($publishedContent->contentInfo);
        $updateStruct = $contentService->newContentUpdateStruct();
        $updateStruct->setField(
            'data',
            $publishedContent->getFieldValue('data', 'eng-US'),
            'eng-US'
        );
        $updateStruct->initialLanguageCode = 'eng-GB';
        $updatedContentDraft = $contentService->updateContent($contentDraft->versionInfo, $updateStruct);
        $contentService->publishVersion($updatedContentDraft->versionInfo);
    }

    /**
     * Get proper multilingual FT-specific Values. It Can be overridden by a Field Type test case.
     *
     * @param string[] $languageCodes List of languages to create data for
     *
     * @return array an array in the form of <code>[languageCode => data]</code>
     */
    public function getValidMultilingualFieldData(array $languageCodes)
    {
        $data = [];
        foreach ($languageCodes as $languageCode) {
            $data[$languageCode] = $this->getValidCreationFieldData();
        }

        return $data;
    }

    /**
     * Test that removing Translation from all Versions works for data from a Field Type.
     *
     * @covers \eZ\Publish\API\Repository\ContentService::deleteTranslation
     */
    public function testDeleteTranslation()
    {
        $repository = $this->getRepository();
        $contentService = $repository->getContentService();

        $languageCodes = ['eng-US', 'ger-DE'];

        $fieldName = $this->getFieldName();
        $names = [];
        foreach ($languageCodes as $languageCode) {
            $names[$languageCode] = "{$languageCode} {$fieldName}";
        }

        $fieldData = $this->getValidMultilingualFieldData($languageCodes);

        $content = $contentService->publishVersion(
            $this->createMultilingualContent($names, $fieldData)->versionInfo
        );

        // create one more Version
        $publishedContent = $contentService->publishVersion(
            $contentService->createContentDraft($content->contentInfo)->versionInfo
        );

        // create Draft
        $contentService->createContentDraft($content->contentInfo);

        // create copy of content in all Versions to use it for comparision later on
        $contentByVersion = [];
        foreach ($contentService->loadVersions($content->contentInfo) as $versionInfo) {
            $contentByVersion[$versionInfo->versionNo] = $contentService->loadContent(
                $content->id,
                null,
                $versionInfo->versionNo
            );
        }

        // delete Translation from all available Versions
        $contentService->deleteTranslation($publishedContent->contentInfo, 'ger-DE');

        // check if are Versions have valid Translation
        foreach ($contentService->loadVersions($publishedContent->contentInfo) as $versionInfo) {
            // check if deleted Translation does not exist
            self::assertEquals(['eng-US'], array_keys($versionInfo->getNames()));
            self::assertEquals(['eng-US'], $versionInfo->languageCodes);

            // load Content of a Version to access other fields data
            $versionContent = $contentService->loadContent(
                $content->id,
                null,
                $versionInfo->versionNo
            );
            // check if deleted Translation for Field Type data does not exist
            self::assertEmpty($versionContent->getFieldsByLanguage('ger-DE'));
            self::assertEmpty($versionContent->getField('data', 'ger-DE'));

            // check if the remaining Translation is still valid
            $expectedContent = $contentByVersion[$versionContent->versionInfo->versionNo];
            self::assertNotEmpty($versionContent->getFieldsByLanguage('eng-US'));
            self::assertEquals(
                $expectedContent->getField('name', 'eng-US'),
                $versionContent->getField('name', 'eng-US')
            );
            self::assertEquals(
                $expectedContent->getField('data', 'eng-US'),
                $versionContent->getField('data', 'eng-US')
            );
        }
    }
}
