/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2007-11-27
 * Description : a bar to filter album contents
 *
 * Copyright (C) 2007-2009 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqwhatsthis.h>
#include <tqtooltip.h>

// KDE includes.

#include <tdelocale.h>
#include <kdialog.h>

// Local includes.

#include "ddebug.h"
#include "statusled.h"
#include "albumsettings.h"
#include "searchtextbar.h"
#include "ratingfilter.h"
#include "mimefilter.h"
#include "albumiconviewfilter.h"
#include "albumiconviewfilter.moc"

namespace Digikam
{

class AlbumIconViewFilterPriv
{
public:

    AlbumIconViewFilterPriv()
    {
        textFilter       = 0;
        mimeFilter       = 0;
        ratingFilter     = 0;
        led              = 0;
    }

    StatusLed     *led;

    SearchTextBar *textFilter;

    MimeFilter    *mimeFilter;

    RatingFilter  *ratingFilter;
};

AlbumIconViewFilter::AlbumIconViewFilter(TQWidget* parent)
                   : TQHBox(parent)
{
    d = new AlbumIconViewFilterPriv;

    d->led = new StatusLed(this);
    d->led->installEventFilter(this);
    d->led->setLedColor(StatusLed::Gray);
    TQWhatsThis::add(d->led, i18n("This LED indicates the global image filter status, "
                                 "encompassing all status-bar filters and all tag filters from the right sidebar.\n\n"
                                 "GRAY: no filter is active, all items are visible.\n"
                                 "RED: filtering is on, but no items match.\n"
                                 "GREEN: filter(s) matches at least one item.\n\n"
                                 "Any mouse button click will reset all filters."));

    d->textFilter = new SearchTextBar(this, "AlbumIconViewFilterTextFilter");
    d->textFilter->setEnableTextQueryCompletion(true);
    TQToolTip::add(d->textFilter, i18n("Text quick filter (search)"));
    TQWhatsThis::add(d->textFilter, i18n("Enter search patterns to quickly filter this view on file names, "
                                        "captions (comments), and tags"));

    d->mimeFilter   = new MimeFilter(this);
    d->ratingFilter = new RatingFilter(this);

    setSpacing(KDialog::spacingHint());
    setMargin(0);

    connect(d->ratingFilter, TQ_SIGNAL(signalRatingFilterChanged(int, AlbumLister::RatingCondition)),
            this, TQ_SLOT(slotRatingFilterChanged(int, AlbumLister::RatingCondition)));

    connect(d->mimeFilter, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotMimeTypeFilterChanged(int)));

    connect(d->textFilter, TQ_SIGNAL(signalTextChanged(const TQString&)),
            this, TQ_SLOT(slotTextFilterChanged(const TQString&)));

    connect(AlbumLister::instance(), TQ_SIGNAL(signalItemsTextFilterMatch(bool)),
            d->textFilter, TQ_SLOT(slotSearchResult(bool)));

    connect(AlbumLister::instance(), TQ_SIGNAL(signalItemsFilterMatch(bool)),
            this, TQ_SLOT(slotItemsFilterMatch(bool)));
}

AlbumIconViewFilter::~AlbumIconViewFilter()
{
    delete d;
}

void AlbumIconViewFilter::readSettings()
{
    AlbumSettings *settings = AlbumSettings::instance();
    d->ratingFilter->setRatingFilterCondition((Digikam::AlbumLister::RatingCondition)
                                              (settings->getRatingFilterCond()));
    /*
    Bug 181705: always enable filters
    d->ratingFilter->setEnabled(settings->getIconShowRating());
    d->textFilter->setEnabled(settings->getIconShowName()     ||
                              settings->getIconShowComments() ||
                              settings->getIconShowTags());
    */
}

void AlbumIconViewFilter::saveSettings()
{
    AlbumSettings::instance()->setRatingFilterCond(d->ratingFilter->ratingFilterCondition());
}

void AlbumIconViewFilter::slotRatingFilterChanged(int rating, AlbumLister::RatingCondition cond)
{
    AlbumLister::instance()->setRatingFilter(rating, cond);
}

void AlbumIconViewFilter::slotMimeTypeFilterChanged(int mimeTypeFilter)
{
    AlbumLister::instance()->setMimeTypeFilter(mimeTypeFilter);
}

void AlbumIconViewFilter::slotTextFilterChanged(const TQString& text)
{
    AlbumLister::instance()->setTextFilter(text);
}

void AlbumIconViewFilter::slotItemsFilterMatch(bool match)
{
    TQStringList filtersList;
    TQString     message;

    if (!d->textFilter->text().isEmpty())
        filtersList.append(i18n("<br><nobr><i>Text</i></nobr>"));

    if (d->mimeFilter->mimeFilter() != MimeFilter::AllFiles)
        filtersList.append(i18n("<br><nobr><i>Mime Type</i></nobr>"));

    if (d->ratingFilter->rating() != 0 || d->ratingFilter->ratingFilterCondition() != AlbumLister::GreaterEqualCondition)
        filtersList.append(i18n("<br/><nobr><i>Rating</i></nobr>"));

    if (AlbumLister::instance()->tagFiltersIsActive())
        filtersList.append(i18n("<br><nobr><i>Tags</i></nobr>"));

    if (filtersList.count() > 1)
        message = i18n("<nobr><b>Active filters:</b></nobr>");
    else
        message = i18n("<nobr><b>Active filter:</b></nobr>");

    message.append(filtersList.join(TQString()));

    if (filtersList.isEmpty())
    {
        TQToolTip::add(d->led, i18n("No active filter"));
        d->led->setLedColor(StatusLed::Gray);
    }
    else
    {
        TQToolTip::add(d->led, message);
        d->led->setLedColor(match ? StatusLed::Green : StatusLed::Red);
    }
}

bool AlbumIconViewFilter::eventFilter(TQObject *object, TQEvent *e)
{
    TQWidget *widget = static_cast<TQWidget*>(object);

    if (e->type() == TQEvent::MouseButtonRelease)
    {
        TQMouseEvent* event = static_cast<TQMouseEvent*>(e);
        if ( widget->rect().contains(event->pos()) && d->led->ledColor() != StatusLed::Gray)
        {
            // Reset all filters settings.
            d->textFilter->setText(TQString());
            d->ratingFilter->setRating(0);
            d->ratingFilter->setRatingFilterCondition(AlbumLister::GreaterEqualCondition);
            d->mimeFilter->setMimeFilter(MimeFilter::AllFiles);
            emit signalResetTagFilters();
        }
    }

    return false;
}

}  // namespace Digikam
