/* This file is part of the KDE project
   Copyright (C) 2004 Till Adam <adam@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "testresource.h"
#include "testincidencegenerator.h"

#include <config.h>

#include <tdeabc/stdaddressbook.h>
#include <kurl.h>
#include <tdeapplication.h>
#include <tdeio/netaccess.h>
#include <tdeio/job.h>
#include <kdebug.h>
#include <tdecmdlineargs.h>
#include <kinputdialog.h>
#include <tderesources/factory.h>

#include <tqdir.h>
#include <tqfileinfo.h>
#include <tqstringlist.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>

#include "calendarresources.h"
#include "resourcecalendar.h"
#include "icalformat.h"
#include "event.h"

static const TDECmdLineOptions options[] =
{
  { "resource <type>", "The resource to test", 0 },
  { "configfile <file>", "Location of a config file for the resource", 0 },
  TDECmdLineLastOption // End of options
};

int main(int argc, char *argv[])
{
    // Use another directory than the real one, just to keep things clean
    // TDEHOME needs to be writable though, for a tdesycoca database
    setenv( "TDEHOME", TQFile::encodeName( TQDir::homeDirPath() + "/.tde-testresource" ), true );
    setenv( "TDE_FORK_SLAVES", "yes", true ); // simpler, for the final cleanup

    TDEApplication::disableAutoDcopRegistration();
    TDECmdLineArgs::init(argc,argv,"testresource", 0, 0, 0, 0);
    TDECmdLineArgs::addCmdLineOptions( options );
 
    TDEApplication app;
    TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();
    TQString type = TQString();
    if ( args->getOption( "resource" ) )
      type = TQString::fromLocal8Bit( args->getOption( "resource" ) );
    TDEConfig *config = 0;
    if ( args->getOption( "configfile" ) )
      config = new TDEConfig( KURL( args->getOption( "configfile" ) ).url() );
    kdDebug() << KURL( args->getOption( "configfile" ) ).url() << endl;
    KCal::TestResource test( type, config );
    test.setup();
    test.runAll();
    test.cleanup();
    kdDebug() << "All tests OK." << endl;
    return 0;
}

namespace KCal {

TestResource::TestResource( const TQString &type, TDEConfig *config )
 :m_resource_type( type ), m_config( config ), m_res( 0 )
{}
  
void TestResource::setup()
{
  CalendarResourceManager *manager = new CalendarResourceManager( "calendar" );
  manager->readConfig();

  TQStringList resources = manager->resourceTypeNames();

  if ( m_resource_type.isNull() ) {

    const TQString & chosen = KInputDialog::getItem( "Select Resource",
        "Select the resource you wish to test. Test data will be used.", 
        resources );

    kdDebug() << "Selected Resource: " << chosen << endl;
    if ( !chosen.isNull() )
      m_resource_type = chosen;
  }
  assert( !m_resource_type.isNull() );
  /* Either read one from the config file, or create a default one. */
  if ( m_config ) {
    kdDebug() << "Reading config from file" << endl;
    KRES::Factory *factory = KRES::Factory::self( "calendar" );
    m_res = dynamic_cast<ResourceCalendar*>( factory->resource( m_resource_type, m_config ) );
  } else {
    kdDebug() << "Creating blank resource" << endl;
    m_res = manager->createResource( m_resource_type );
  }
  assert( m_res );
}


void TestResource::runAll()
{
  testOpenAndClose();
  /* now we can trust it to open correctly */
  m_res->open();
  testResourceAttributes();
  testResourceCalendarAttributes();
  testEventAddRemove();
  testTodoAddRemove();
  testJournalAddRemove();
  m_res->close();
}

bool TestResource::check(const TQString& txt, TQString a, TQString b)
{
    if (a.isEmpty())
        a = TQString();
    if (b.isEmpty())
        b = TQString();
    if (a == b) {
        kdDebug() << txt << " : checking '" << a << "' against expected value '" << b << "'... " << "ok" << endl;
    }
    else {
        kdDebug() << txt << " : checking '" << a << "' against expected value '" << b << "'... " << "KO !" << endl;
        cleanup();
        exit(1);
    }
    return true;
}

void TestResource::testOpenAndClose()
{
  kdDebug() << k_funcinfo << endl;
  assert( m_res->open() );
  assert( m_res->isOpen() );
  m_res->close();
  assert( !m_res->isOpen() );
}

void TestResource::testResourceAttributes()
{
  kdDebug() << k_funcinfo << endl;
  
  check( "type", m_res->type(), m_resource_type );
  
  m_res->setReadOnly( true );
  assert( m_res->readOnly() );
  m_res->setReadOnly( false );
  assert( !m_res->readOnly() );
 
  m_res->setResourceName( "Margarete" );
  check( "name", m_res->resourceName(), "Margarete" );

  m_res->setActive( false );
  assert( !m_res->isActive() );
  m_res->setActive( true );
  assert( m_res->isActive() );
  m_res->dump();
}

void TestResource::testResourceCalendarAttributes()
{
  kdDebug() << k_funcinfo << endl;
}


void TestResource::testEventAddRemove()
{
  ICalFormat f;
  kdDebug() << k_funcinfo << endl;
  
  int oldcount = m_res->rawIncidences().count();
  Event *event = makeTestEvent();
  const TQString origString = f.toString( event );
  m_res->addEvent( event );
  Event *fromRes = m_res->event( event->uid() );
  assert( fromRes == event );
  const TQString fromResString = f.toString( fromRes );
  check( "add", origString, fromResString );
  m_res->deleteEvent( event );
  assert( !m_res->event( event->uid() ) );
  int newcount = m_res->rawIncidences().count();
  assert( oldcount == newcount );
  delete event;
}

void TestResource::testTodoAddRemove()
{
  ICalFormat f;
  kdDebug() << k_funcinfo << endl;
  
  int oldcount = m_res->rawIncidences().count();
  Todo *todo = makeTestTodo();
  const TQString origString = f.toString( todo );
  m_res->addTodo( todo );
  Todo *fromRes = m_res->todo( todo->uid() );
  assert( fromRes == todo );
  const TQString fromResString = f.toString( fromRes );
  check( "add", origString, fromResString );
  m_res->deleteTodo( todo );
  assert( !m_res->todo( todo->uid() ) );
  int newcount = m_res->rawIncidences().count();
  assert( oldcount == newcount );
  delete todo;
}

void TestResource::testJournalAddRemove()
{
  ICalFormat f;
  kdDebug() << k_funcinfo << endl;
  
  int oldcount = m_res->rawIncidences().count();
  Journal *journal = makeTestJournal();
  const TQString origString = f.toString( journal );
  m_res->addJournal( journal );
  Journal *fromRes = m_res->journal( journal->uid() );
  assert( fromRes == journal );
  const TQString fromResString = f.toString( fromRes );
  check( "add", origString, fromResString );
  m_res->deleteJournal( journal );
  assert( !m_res->journal( journal->uid() ) );
  int newcount = m_res->rawIncidences().count();
  assert( oldcount == newcount );
  delete journal;
}

void TestResource::cleanup()
{
  kdDebug() << k_funcinfo << endl;
}

}

#include "testresource.moc"
