/*
  KPF - Public fileserver for KDE

  Copyright 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <tqpainter.h>
#include <tqtimer.h>
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqtoolbutton.h>
#include <tqpopupmenu.h>
#include <tqfileinfo.h>
#include <tqcursor.h>

#include <dcopclient.h>
#include <kiconloader.h>
#include <tdeglobal.h>
#include <tdeconfig.h>
#include <tdeaboutapplication.h>
#include <tdeaboutdata.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <tdeapplication.h>
#include <kurldrag.h>

#include "System.h"
#include "Defines.h"
#include "Applet.h"
#include "AppletItem.h"
#include "WebServerManager.h"
#include "WebServer.h"
#include "ServerWizard.h"

static const char kpfVersion[] = "1.0.1";

extern "C"
{
  TDE_EXPORT  KPanelApplet *
  init(TQWidget * parent, const TQString & configFile)
  {
    if (0 == kpf::userId() || 0 == kpf::effectiveUserId())
    {
      // Don't run as root.
      KMessageBox::detailedError
        ( 0,
          i18n("You cannot run KPF as root."),
          i18n("Running as root exposes the whole system to "
               "external attackers."),
          i18n("Running as root.")
        );
      return NULL;
    }
    else
    {
      kpf::blockSigPipe();

      TDEGlobal::locale()->insertCatalogue("kpf");

      return new KPF::Applet
        (
         configFile,
         KPanelApplet::Normal,
         KPanelApplet::About|KPanelApplet::Help,
         parent,
         "kpf"
        );
    }
  }
}

namespace KPF
{
  Applet::Applet
  (
   const TQString  & configFile,
   Type             type,
   int              actions,
   TQWidget        * parent,
   const char     * name
  )
    : KPanelApplet  (configFile, type, actions, parent, name),
      wizard_       (0L),
      popup_        (0L),
      dcopClient_   (0L)
  {
    setAcceptDrops(true);

    //setFrameStyle(TQFrame::Panel | TQFrame::Sunken);
    //setLineWidth(1);

    connect
      (
       WebServerManager::instance(),
       TQ_SIGNAL(serverCreated(WebServer *)),
       TQ_SLOT(slotServerCreated(WebServer *))
      );

    connect
      (
       WebServerManager::instance(),
       TQ_SIGNAL(serverDisabled(WebServer *)),
       TQ_SLOT(slotServerDisabled(WebServer *))
      );

    WebServerManager::instance()->loadConfig();

    popup_ = new TQPopupMenu(this);

    popup_->insertItem
      (BarIcon("document-new"), i18n("New Server..."), NewServer, NewServer);

//    popup_->insertItem
//      (BarIcon("quit"), i18n("Quit"), Quit, Quit);

    dcopClient_ = new DCOPClient;
    dcopClient_->registerAs("kpf", false);
  }

  Applet::~Applet()
  {
    delete dcopClient_;
    WebServerManager::instance()->shutdown();
  }

    int
  Applet::widthForHeight(int h) const
  {
    uint serverCount = itemList_.count();

    if (0 == serverCount)
      serverCount = 1;

    if (TQt::Vertical == orientation())
      return h / serverCount;
    else
      return h * serverCount;
  }

    int
  Applet::heightForWidth(int w) const
  {
    uint serverCount = itemList_.count();

    if (0 == serverCount)
      serverCount = 1;

    if (TQt::Vertical == orientation())
      return w * serverCount;
    else
      return w / serverCount;
  }

    void
  Applet::help()
  {
    tdeApp->invokeHelp( TQString(), "kpf" );
  }

    void
  Applet::about()
  {
    TDEAboutData about
      (
       "kpf",
       I18N_NOOP("kpf"),
       kpfVersion,
       I18N_NOOP("TDE public fileserver"),
       TDEAboutData::License_Custom,
       "(C) 2001 Rik Hemsley (rikkus) <rik@kde.org>",
       I18N_NOOP(
         "File sharing applet, using the HTTP (Hyper Text Transfer Protocol)"
         " standard to serve files."
       ),
       "http://rikkus.info/kpf.html"
     );

    about.setLicenseText
      (
        I18N_NOOP
        (
"Permission is hereby granted, free of charge, to any person obtaining a copy\n"
"of this software and associated documentation files (the \"Software\"), to\n"
"deal in the Software without restriction, including without limitation the\n"
"rights to use, copy, modify, merge, publish, distribute, sublicense, and/or\n"
"sell copies of the Software, and to permit persons to whom the Software is\n"
"furnished to do so, subject to the following conditions:\n"
"\n"
"The above copyright notice and this permission notice shall be included in\n"
"all copies or substantial portions of the Software.\n"
"\n"
"THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\n"
"IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n"
"FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE\n"
"AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN\n"
"ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION\n"
"WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n"
        )
      );

    TDEAboutApplication a(&about, this);
    a.exec();
  }

    void
  Applet::orientationChange(TQt::Orientation)
  {
    resetLayout();
  }

    void
  Applet::resizeEvent(TQResizeEvent *)
  {
    resetLayout();
  }
    
    void
  Applet::moveEvent(TQMoveEvent *)
  {
    TQPtrListIterator<AppletItem> it(itemList_);

    for (uint i = 0; it.current(); ++it, ++i)
        it.current()->setBackground();
  }

    void
  Applet::resetLayout()
  {
    if (0 == itemList_.count())
      return;

    switch (orientation())
    {
      case TQt::Vertical:
        {
          uint itemHeight = height() / itemList_.count();

          TQPtrListIterator<AppletItem> it(itemList_);

          for (uint i = 0; it.current(); ++it, ++i)
          {
            it.current()->resize(width(), itemHeight);
            it.current()->move(0, i * itemHeight);
          }
        }
        break;

      case TQt::Horizontal:
        {
          uint itemWidth = width() / itemList_.count();

          TQPtrListIterator<AppletItem> it(itemList_);

          for (uint i = 0; it.current(); ++it, ++i)
          {
            it.current()->resize(itemWidth, height());
            it.current()->move(i * itemWidth, 0);
          }
        }
      default:
        break;
    }
  }

    void
  Applet::mousePressEvent(TQMouseEvent * ev)
  {
    if (TQt::RightButton != ev->button() && TQt::LeftButton != ev->button())
      return;

    switch (popup_->exec(TQCursor::pos()))
    {
      case NewServer:
        slotNewServer();
        break;

      case Quit:
        slotQuit();
        break;

      default:
        break;
    }
  }

    void
  Applet::slotNewServerAtLocation(const TQString & location)
  {
    if (0 != wizard_)
    {
      wizard_->setLocation(location);
      wizard_->show();
    }

    else
    {
      wizard_ = new ServerWizard;

      connect
        (
         wizard_,
         TQ_SIGNAL(dying(ServerWizard *)),
         TQ_SLOT(slotWizardDying(ServerWizard *))
        );

      wizard_->setLocation(location);
      wizard_->show();
    }
  }

    void
  Applet::slotNewServer()
  {
    if (0 != wizard_)
      wizard_->show();

    else
    {
      wizard_ = new ServerWizard;

      connect
        (
         wizard_,
         TQ_SIGNAL(dying(ServerWizard *)),
         TQ_SLOT(slotWizardDying(ServerWizard *))
        );

      wizard_->show();
    }
  }

    void
  Applet::slotWizardDying(ServerWizard * wiz)
  {
    if (TQDialog::Accepted == wiz->result())
    {
      WebServerManager::instance()->createServerLocal
        (
         wiz->root(),
         wiz->listenPort(),
         wiz->bandwidthLimit(),
         wiz->connectionLimit(),
	 Config::DefaultFollowSymlinks,
	 wiz->serverName()
        );
    }

    delete wizard_;
    wizard_ = 0;
  }

    void
  Applet::drawContents(TQPainter * p)
  {
    TQPixmap px;

    if (width() > 48)
      px = TDEGlobal::iconLoader()->loadIcon("kpf", TDEIcon::Panel, 48);
    else if (width() > 32)
      px = TDEGlobal::iconLoader()->loadIcon("kpf", TDEIcon::Panel, 32);
    else if (width() > 16)
      px = TDEGlobal::iconLoader()->loadIcon("kpf", TDEIcon::Panel, 16);
    else
      return;

    TQRect r(contentsRect());

    p->drawPixmap
      (
       r.x() + r.width()  / 2 - px.width()  / 2,
       r.y() + r.height() / 2 - px.height() / 2,
       px
      );
  }

    void
  Applet::dragEnterEvent(TQDragEnterEvent * e)
  {
    KURL::List l;

    if (!KURLDrag::decode(e, l))
      return;

    if (l.count() != 1)
      return;

    const KURL &url = l[0];

    if (!url.isLocalFile() || !TQFileInfo(url.path()).isDir())
      return;

    e->accept();
  }

    void
  Applet::dropEvent(TQDropEvent * e)
  {
    KURL::List l;

    if (!KURLDrag::decode(e, l))
      return;

    if (l.count() != 1)
      return;

    const KURL &url = l[0];

    if (!url.isLocalFile() || !TQFileInfo(url.path()).isDir())
      return;

    e->accept();

    slotNewServerAtLocation(url.path());
  }

    void
  Applet::slotServerCreated(WebServer * server)
  {
    AppletItem * i = new AppletItem(server, this);

    connect
      (
       i,
       TQ_SIGNAL(newServer()),
       TQ_SLOT(slotNewServer())
      );

    connect
      (
       i,
       TQ_SIGNAL(newServerAtLocation(const TQString &)),
       TQ_SLOT(slotNewServerAtLocation(const TQString &))
      );

    itemList_.append(i);
    i->show();
    emit(updateLayout());
    resetLayout();
  }

    void
  Applet::slotServerDisabled(WebServer * server)
  {
    TQPtrListIterator<AppletItem> it(itemList_);

    for (; it.current(); ++it)
    {
      AppletItem * i = it.current();

      if (i->server() == server)
      {
        itemList_.removeRef(i);
        delete i;
        emit(updateLayout());
        resetLayout();
        return;
      }
    }
  }

    void
  Applet::slotQuit()
  {
    // How ?
  }

} // End namespace KPF

#include "Applet.moc"
