/*
    This file is part of the TDE games library
    Copyright (C) 2002 Nicolas Hadacek (hadacek@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kexthighscore_tab.h"
#include "kexthighscore_tab.moc"

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqvgroupbox.h>
#include <tqgrid.h>
#include <tqheader.h>

#include <kdialogbase.h>
#include <tdelistview.h>
#include <kdebug.h>
#include <tdeglobal.h>

#include "kexthighscore.h"
#include "kexthighscore_internal.h"


namespace KExtHighscore
{

//-----------------------------------------------------------------------------
PlayersCombo::PlayersCombo(TQWidget *parent, const char *name)
    : TQComboBox(parent, name)
{
    const PlayerInfos &p = internal->playerInfos();
    for (uint i = 0; i<p.nbEntries(); i++)
        insertItem(p.prettyName(i));
    insertItem(TQString("<") + i18n("all") + '>');
    connect(this, TQ_SIGNAL(activated(int)), TQ_SLOT(activatedSlot(int)));
}

void PlayersCombo::activatedSlot(int i)
{
    const PlayerInfos &p = internal->playerInfos();
    if ( i==(int)p.nbEntries() ) emit allSelected();
    else if ( i==(int)p.nbEntries()+1 ) emit noneSelected();
    else emit playerSelected(i);
}

void PlayersCombo::load()
{
    const PlayerInfos &p = internal->playerInfos();
    for (uint i = 0; i<p.nbEntries(); i++)
        changeItem(p.prettyName(i), i);
}

//-----------------------------------------------------------------------------
AdditionalTab::AdditionalTab(TQWidget *parent, const char *name)
    : TQWidget(parent, name)
{
    TQVBoxLayout *top = new TQVBoxLayout(this, KDialogBase::marginHint(),
                                       KDialogBase::spacingHint());

    TQHBoxLayout *hbox = new TQHBoxLayout(top);
    TQLabel *label = new TQLabel(i18n("Select player:"), this);
    hbox->addWidget(label);
    _combo = new PlayersCombo(this);
    connect(_combo, TQ_SIGNAL(playerSelected(uint)),
            TQ_SLOT(playerSelected(uint)));
    connect(_combo, TQ_SIGNAL(allSelected()), TQ_SLOT(allSelected()));
    hbox->addWidget(_combo);
    hbox->addStretch(1);
}

void AdditionalTab::init()
{
    uint id = internal->playerInfos().id();
    _combo->setCurrentItem(id);
    playerSelected(id);
}

void AdditionalTab::allSelected()
{
    display(internal->playerInfos().nbEntries());
}

TQString AdditionalTab::percent(uint n, uint total, bool withBraces)
{
    if ( n==0 || total==0 ) return TQString();
    TQString s =  TQString("%1%").arg(100.0 * n / total, 0, 'f', 1);
    return (withBraces ? TQString("(") + s + ")" : s);
}

void AdditionalTab::load()
{
    _combo->load();
}


//-----------------------------------------------------------------------------
const char *StatisticsTab::COUNT_LABELS[Nb_Counts] = {
    I18N_NOOP("Total:"), I18N_NOOP("Won:"), I18N_NOOP("Lost:"),
    I18N_NOOP("Draw:")
};
const char *StatisticsTab::TREND_LABELS[Nb_Trends] = {
    I18N_NOOP("Current:"), I18N_NOOP("Max won:"), I18N_NOOP("Max lost:")
};

StatisticsTab::StatisticsTab(TQWidget *parent)
    : AdditionalTab(parent, "statistics_tab")
{
    // construct GUI
    TQVBoxLayout *top = static_cast<TQVBoxLayout *>(layout());

    TQHBoxLayout *hbox = new TQHBoxLayout(top);
    TQVBoxLayout *vbox = new TQVBoxLayout(hbox);
    TQVGroupBox *group = new TQVGroupBox(i18n("Game Counts"), this);
    vbox->addWidget(group);
    TQGrid *grid = new TQGrid(3, group);
    grid->setSpacing(KDialogBase::spacingHint());
    for (uint k=0; k<Nb_Counts; k++) {
        if ( Count(k)==Draw && !internal->showDrawGames ) continue;
        (void)new TQLabel(i18n(COUNT_LABELS[k]), grid);
        _nbs[k] = new TQLabel(grid);
        _percents[k] = new TQLabel(grid);
    }

    group = new TQVGroupBox(i18n("Trends"), this);
    vbox->addWidget(group);
    grid = new TQGrid(2, group);
    grid->setSpacing(KDialogBase::spacingHint());
    for (uint k=0; k<Nb_Trends; k++) {
        (void)new TQLabel(i18n(TREND_LABELS[k]), grid);
        _trends[k] = new TQLabel(grid);
    }

    hbox->addStretch(1);
    top->addStretch(1);
}

void StatisticsTab::load()
{
    AdditionalTab::load();
    const PlayerInfos &pi = internal->playerInfos();
    uint nb = pi.nbEntries();
    _data.resize(nb+1);
    for (uint i=0; i<_data.size()-1; i++) {
        _data[i].count[Total] = pi.item("nb games")->read(i).toUInt();
        _data[i].count[Lost] = pi.item("nb lost games")->read(i).toUInt()
                       + pi.item("nb black marks")->read(i).toUInt(); // legacy
        _data[i].count[Draw] = pi.item("nb draw games")->read(i).toUInt();
        _data[i].count[Won] = _data[i].count[Total] - _data[i].count[Lost]
                              - _data[i].count[Draw];
        _data[i].trend[CurrentTrend] =
            pi.item("current trend")->read(i).toInt();
        _data[i].trend[WonTrend] = pi.item("max won trend")->read(i).toUInt();
        _data[i].trend[LostTrend] =
            -(int)pi.item("max lost trend")->read(i).toUInt();
    }

    for (uint k=0; k<Nb_Counts; k++) _data[nb].count[k] = 0;
    for (uint k=0; k<Nb_Trends; k++) _data[nb].trend[k] = 0;
    for (uint i=0; i<_data.size()-1; i++) {
        for (uint k=0; k<Nb_Counts; k++)
            _data[nb].count[k] += _data[i].count[k];
        for (uint k=0; k<Nb_Trends; k++)
            _data[nb].trend[k] += _data[i].trend[k];
    }
    for (uint k=0; k<Nb_Trends; k++)
        _data[nb].trend[k] /= (_data.size()-1);

    init();
}

TQString StatisticsTab::percent(const Data &d, Count count) const
{
    if ( count==Total ) return TQString();
    return AdditionalTab::percent(d.count[count], d.count[Total], true);
}

void StatisticsTab::display(uint i)
{
    const Data &d = _data[i];
    for (uint k=0; k<Nb_Counts; k++) {
        if ( Count(k) && !internal->showDrawGames ) continue;
        _nbs[k]->setText(TQString::number(d.count[k]));
        _percents[k]->setText(percent(d, Count(k)));
    }
    for (uint k=0; k<Nb_Trends; k++) {
        TQString s;
        if ( d.trend[k]>0 ) s = '+';
        int prec = (i==internal->playerInfos().nbEntries() ? 1 : 0);
        _trends[k]->setText(s + TQString::number(d.trend[k], 'f', prec));
    }
}

//-----------------------------------------------------------------------------
HistogramTab::HistogramTab(TQWidget *parent)
    : AdditionalTab(parent, "histogram_tab")
{
    // construct GUI
    TQVBoxLayout *top = static_cast<TQVBoxLayout *>(layout());

    _list = new TDEListView(this);
    _list->setSelectionMode(TQListView::NoSelection);
    _list->setItemMargin(3);
    _list->setAllColumnsShowFocus(true);
    _list->setSorting(-1);
    _list->header()->setClickEnabled(false);
    _list->header()->setMovingEnabled(false);
    top->addWidget(_list);

    _list->addColumn(i18n("From"));
    _list->addColumn(i18n("To"));
    _list->addColumn(i18n("Count"));
    _list->addColumn(i18n("Percent"));
    for (uint i=0; i<4; i++) _list->setColumnAlignment(i, AlignRight);
    _list->addColumn(TQString());

    const Item *sitem = internal->scoreInfos().item("score")->item();
    const PlayerInfos &pi = internal->playerInfos();
    const TQMemArray<uint> &sh = pi.histogram();
    for (uint k=1; k<pi.histoSize(); k++) {
        TQString s1 = sitem->pretty(0, sh[k-1]);
        TQString s2;
        if ( k==sh.size() ) s2 = "...";
        else if ( sh[k]!=sh[k-1]+1 ) s2 = sitem->pretty(0, sh[k]);
        (void)new TDEListViewItem(_list, s1, s2);
    }
}

void HistogramTab::load()
{
    AdditionalTab::load();
    const PlayerInfos &pi = internal->playerInfos();
    uint n = pi.nbEntries();
    uint s = pi.histoSize() - 1;
    _counts.resize((n+1) * s);
    _data.fill(0, n+1);
    for (uint k=0; k<s; k++) {
        _counts[n*s + k] = 0;
        for (uint i=0; i<n; i++) {
            uint nb = pi.item(pi.histoName(k+1))->read(i).toUInt();
            _counts[i*s + k] = nb;
            _counts[n*s + k] += nb;
            _data[i] += nb;
            _data[n] += nb;
        }
    }

    init();
}

void HistogramTab::display(uint i)
{
    const PlayerInfos &pi = internal->playerInfos();
    TQListViewItem *item = _list->firstChild();
    uint s = pi.histoSize() - 1;
    for (int k=s-1; k>=0; k--) {
        uint nb = _counts[i*s + k];
        item->setText(2, TQString::number(nb));
        item->setText(3, percent(nb, _data[i]));
        uint width = (_data[i]==0 ? 0 : tqRound(150.0 * nb / _data[i]));
        TQPixmap pixmap(width, 10);
        pixmap.fill(blue);
        item->setPixmap(4, pixmap);
        item = item->nextSibling();
    }
}

} // namespace
