/* This file is part of the KDE libraries
    Copyright (C) 1997, 1998 Stephan Kulow (coolo@kde.org)
              (C) 1997, 1998 Sven Radej (radej@kde.org)
              (C) 1997, 1998 Mark Donohoe (donohoe@kde.org)
              (C) 1997, 1998 Matthias Ettrich (ettrich@kde.org)
              (C) 2000 Kurt Granroth (granroth@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

// $Id: tdetoolbarbutton.h,v 1.21.2.2 2002/05/26 13:09:06 carewolf Exp $
#ifndef _TDETOOLBARBUTTON_H
#define _TDETOOLBARBUTTON_H

#include <tqpixmap.h>
#include <tqtoolbutton.h>
#include <tqintdict.h>
#include <tqstring.h>
#include <tdeglobal.h>

class TDEToolBar;
class TDEToolBarButtonPrivate;
class TDEInstance;
class TQEvent;
class TQPopupMenu;
class TQPainter;

/**
 * A toolbar button. This is used internally by @ref TDEToolBar, use the
 * TDEToolBar methods instead.
 * @internal
 */
class TDEToolBarButton : public TQToolButton
{
  TQ_OBJECT

public:
  /**
   * Construct a button with an icon loaded by the button itself.
   * This will trust the button to load the correct icon with the
   * correct size.
   *
   * @param icon   Name of icon to load (may be absolute or relative)
   * @param id     Id of this button
   * @param parent This button's parent
   * @param name   This button's internal name
   * @param txt    This button's text (in a tooltip or otherwise)
   */
  TDEToolBarButton(const TQString& icon, int id, TQWidget *parent,
                 const char *name=0L, const TQString &txt=TQString::null,
                 TDEInstance *_instance = TDEGlobal::instance());

  /**
   * Construct a button with an existing pixmap.  It is not
   * recommended that you use this as the internal icon loading code
   * will almost always get it "right".
   *
   * @param icon   Name of icon to load (may be absolute or relative)
   * @param id     Id of this button
   * @param parent This button's parent
   * @param name   This button's internal name
   * @param txt    This button's text (in a tooltip or otherwise)
   */
  TDEToolBarButton(const TQPixmap& pixmap, int id, TQWidget *parent,
                 const char *name=0L, const TQString &txt=TQString::null);

  /**
   * Construct a separator button
   *
   * @param parent This button's parent
   * @param name   This button's internal name
   */
  TDEToolBarButton(TQWidget *parent=0L, const char *name=0L);

  /**
   * Standard destructor
   */
  ~TDEToolBarButton();

#ifndef KDE_NO_COMPAT
  /**
   * @deprecated
   * Set the pixmap directly for this button.  This pixmap should be
   * the active one... the dimmed and disabled pixmaps are constructed
   * based on this one.  However, don't use this function unless you
   * are positive that you don't want to use @ref setIcon.
   *
   * @param pixmap The active pixmap
   */
  // this one is from TQButton, so #ifdef-ing it out doesn't break BC
  virtual void setPixmap(const TQPixmap &pixmap);

  /**
   * @deprecated
   * Force the button to use this pixmap as the default one rather
   * then generating it using effects.
   *
   * @param pixmap The pixmap to use as the default (normal) one
   */
  void setDefaultPixmap(const TQPixmap& pixmap);

  /**
   * @deprecated
   * Force the button to use this pixmap when disabled one rather then
   * generating it using effects.
   *
   * @param pixmap The pixmap to use when disabled
   */
  void setDisabledPixmap(const TQPixmap& pixmap);
#endif

  /**
   * Set the text for this button.  The text will be either used as a
   * tooltip (IconOnly) or will be along side the icon
   *
   * @param text The button (or tooltip) text
   */
  virtual void setText(const TQString &text);

  /**
   * Set the icon for this button. The icon will be loaded internally
   * with the correct size. This function is preferred over @ref setIconSet
   *
   * @param icon The name of the icon
   */
  virtual void setIcon(const TQString &icon);

  virtual void setIcon( const TQPixmap &pixmap )
  { TQToolButton::setIcon( pixmap ); }

  /**
   * Set the pixmaps for this toolbar button from a TQIconSet.
   * If you call this you don't need to call any of the other methods
   * that set icons or pixmaps.
   * @param iconset  The iconset to use
   */
  virtual void setIconSet( const TQIconSet &iconset );

#ifndef KDE_NO_COMPAT
  /**
   * @deprecated
   * Set the active icon for this button.  The pixmap itself is loaded
   * internally based on the icon size...  .. the disabled and default
   * pixmaps, however will only be constructed if @ref #generate is
   * true.  This function is preferred over @ref setPixmap
   *
   * @param icon     The name of the active icon
   * @param generate If true, then the other icons are automagically
   *                 generated from this one
   */
  void setIcon(const TQString &icon, bool /*generate*/ ) { setIcon( icon ); }

  /**
   * @deprecated
   * Force the button to use this icon as the default one rather
   * then generating it using effects.
   *
   * @param icon The icon to use as the default (normal) one
   */
  void setDefaultIcon(const TQString& icon);

  /**
   * @deprecated
   * Force the button to use this icon when disabled one rather then
   * generating it using effects.
   *
   * @param icon The icon to use when disabled
   */
  void setDisabledIcon(const TQString& icon);
#endif

  /**
   * Turn this button on or off
   *
   * @param flag true or false
   */
  void on(bool flag = true);

  /**
   * Toggle this button
   */
  void toggle();

  /**
   * Turn this button into a toggle button or disable the toggle
   * aspects of it.  This does not toggle the button itself.  Use @ref
   * toggle for that.
   *
   * @param toggle true or false
   */
  void setToggle(bool toggle = true);

  /**
   * Return a pointer to this button's popup menu (if it exists)
   */
  TQPopupMenu *popup();

  /**
   * Give this button a popup menu.  There will not be a delay when
   * you press the button.  Use @ref setDelayedPopup if you want that
   * behavior. You can also make the popup-menu
   * "sticky", i.e. visible until a selection is made or the mouse is
   * clikced elsewhere, by simply setting the second argument to true.
   * This "sticky" button feature allows you to make a selection without
   * having to press and hold down the mouse while making a selection.
   *
   * @param p The new popup menu
   * @param toggle if true, makes the button "sticky" (toggled)
   */
  void setPopup (TQPopupMenu *p, bool toggle = false);

  /**
   * Gives this button a delayed popup menu.
   *
   * This function allows you to add a delayed popup menu to the button.
   * The popup menu is then only displayed when the button is pressed and
   * held down for about half a second.  You can also make the popup-menu
   * "sticky", i.e. visible until a selection is made or the mouse is
   * clikced elsewhere, by simply setting the second argument to true.
   * This "sticky" button feature allows you to make a selection without
   * having to press and hold down the mouse while making a selection.
   *
   * @param p the new popup menu
   * @param toggle if true, makes the button "sticky" (toggled)
   */
  void setDelayedPopup(TQPopupMenu *p, bool toggle = false);

  /**
   * Turn this button into a radio button
   *
   * @param f true or false
   */
  void setRadio(bool f = true);

  /**
   * Toolbar buttons naturally will assume the global styles
   * concerning icons, icons sizes, etc.  You can use this function to
   * explicitely turn this off, if you like.
   *
   * @param no_style Will disable styles if true
   */
  void setNoStyle(bool no_style = true);

signals:
  void clicked(int);
  void doubleClicked(int);
  void pressed(int);
  void released(int);
  void toggled(int);
  void highlighted(int, bool);

public slots:
  /**
   * This slot should be called whenever the toolbar mode has
   * potentially changed.  This includes such events as text changing,
   * orientation changing, etc.
   */
   void modeChange();
   virtual void setTextLabel(const TQString&, bool tipToo);

protected:
  void paletteChange(const TQPalette &);
  void leaveEvent(TQEvent *e);
  void enterEvent(TQEvent *e);
  void drawButton(TQPainter *p);
  bool eventFilter (TQObject *o, TQEvent *e);
  void showMenu();
  TQSize sizeHint() const;
  TQSize minimumSizeHint() const;
  TQSize minimumSize() const;

  bool isRaised() const;
  bool isActive() const;
  int iconTextMode() const;

protected slots:
  void slotClicked();
  void slotPressed();
  void slotReleased();
  void slotToggled();
  void slotDelayTimeout();

protected:
  virtual void virtual_hook( int id, void* data );
private:
  TDEToolBarButtonPrivate *d;
};

/**
* List of @ref TDEToolBarButton objects.
* @internal
* @version $Id: tdetoolbarbutton.h,v 1.21.2.2 2002/05/26 13:09:06 carewolf Exp $
*/
class TDEToolBarButtonList : public TQIntDict<TDEToolBarButton>
{
public:
   TDEToolBarButtonList();
   ~TDEToolBarButtonList() {}
};

#endif
