/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Jaroslav Reznik <jreznik@redhat.com>
 * Copyright (C) 2010 Dario Freddi <drf@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef POLKIT_TQT_ACTION_DESCRIPTION_H
#define POLKIT_TQT_ACTION_DESCRIPTION_H

#include "polkit-tqt-export.h"


typedef struct _PolkitActionDescription PolkitActionDescription;

class TQString;
template<typename> class TQValueList;


namespace PolkitTQt
{
/**
 * \class ActionDescription polkit-tqt-actiondescription.h ActionDescription
 * \author Jaroslav Reznik <jreznik@redhat.com>
 * \author Dario Freddi <drf@kde.org>
 *
 * \brief Class used to encapsulate a registered action.
 */
class POLKIT_TQT_EXPORT ActionDescription
{
  public:
    enum ImplicitAuthorization
    {
      /** Unknown whether the subject is authorized, never returned in any public API. **/
      Unknown = -1,
      /** Subject is not authorized. **/
      NotAuthorized = 0,
      /** Authentication is required. **/
      AuthenticationRequired = 1,
      /** Authentication as an administrator is required. **/
      AdministratorAuthenticationRequired = 2,
      /** Authentication is required. If the authorization is obtained, it is retained. **/
      AuthenticationRequiredRetained = 3,
      /** Authentication as an administrator is required. If the authorization is obtained, it is retained. **/
      AdministratorAuthenticationRequiredRetained = 4,
      /** The subject is authorized. **/
      Authorized = 5
    };

    typedef TQValueList<ActionDescription> List;

    ActionDescription();
    ActionDescription(const ActionDescription &other);
    ~ActionDescription();

    /**
     * \brief Constructor of ActionDescription object from PolkitActionDescription
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param actionDesciption PolkitActionDescription
     */
    explicit ActionDescription(PolkitActionDescription *pkActionDescription);

    ActionDescription& operator=(const ActionDescription &other);

    /**
     * \brief Gets the action id for ActionDescription
     *
     * \return id of the action
     */
    TQString actionId() const;

    /**
     * \brief Gets the description used of ActionDescription
     *
     * \return description of the action
     */
    TQString description() const;

    /**
     * \brief Gets the message used for ActionDescription
     *
     * \return action message
     */
    TQString message() const;

    /**
     * \brief Gets the vendor name for ActionDescription, if any
     *
     * \return vendor name
     */
    TQString vendorName() const;

    /**
     * \brief Gets the vendor URL for ActionDescription, if any
     *
     * \return vendor URL or empty TQString if there is no vendor URL
     */
    TQString vendorUrl() const;

    /**
     * \brief Gets the icon name for ActionDescription, if any
     *
     * \return icon name or empty TQString if there is no icon
     */
    TQString iconName() const;

    /**
     * \brief Gets the implicit authorization for ActionDescription used
     *        for any subject
     *
     * \return A value from ImplicitAuthorization enumeration
     */
    ImplicitAuthorization implicitAny() const;

    /**
     * \brief Gets the implicit authorization for ActionDescription used
     *        for subjects in inactive session on a local console
     *
     * \return A value from ImplicitAuthorization enumeration
     */
    ImplicitAuthorization implicitInactive() const;

    /**
     * \brief Gets the implicit authorization for ActionDescription used
     *        for subjects in active session on a local console
     *
     * \return A value from ImplicitAuthorization enumeration
     */
    ImplicitAuthorization implicitActive() const;

  private:
    class Data;
    Data *d;
};

}

#endif
