#ifndef _KVI_IRCSERVER_H_
#define _KVI_IRCSERVER_H_

//=============================================================================
//
//   File : kvi_ircserver.h
//   Creation date : Mon Jul 10 2000 03:24:11 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=============================================================================

#include "kvi_settings.h"
#include "kvi_heapobject.h"
#include "kvi_qstring.h"
#include "kvi_pointerlist.h"

#include <tqstringlist.h>

class KviConfig;
class KviNickServRuleSet;
class KviProxy;
class KviProxyDataBase;
class KviIrcServer;

#define KVI_IRCSERVER_FLAG_IPV6 1
#define KVI_IRCSERVER_FLAG_CACHEIP 2
#define KVI_IRCSERVER_FLAG_SSL 4

class KVILIB_API KviIrcServerReconnectInfo {
public:
	TQString               m_szNick;
	TQString               m_szAwayReason;
	TQString               m_szJoinChannels;
	TQStringList           m_szOpenQueryes;
	bool                  m_bIsAway;
};

class KVILIB_API KviIrcServer : public KviHeapObject
{
public:
	KviIrcServer();
	KviIrcServer(const KviIrcServer &serv);
	~KviIrcServer();
public:
	KviIrcServerReconnectInfo *m_pReconnectInfo;
	TQString            m_szHostname;          // the server hostname (or ip eventually)
	TQString            m_szIp;                // the server's cached ip (if we're caching)
	TQString            m_szDescription;       // the server description
	kvi_u32_t          m_uPort;               // the server's port
	unsigned short int m_uFlags;              // flags

	// Extended data
	TQString            m_szUserIdentityId;    // The user identity to use for this server: if empty
	                                          // Then use the network identity instead
	
	TQString            m_szUser;              // special username
	TQString            m_szPass;              // special password
	TQString            m_szNick;              // special nickname
	TQString            m_szRealName;          // special real name
	TQString            m_szInitUMode;         // special user mode
	TQString            m_szOnConnectCommand;  // the command to run on connect
	TQString            m_szOnLoginCommand;    // the command to run after login
	TQString            m_szLinkFilter;        // the link filter object
	TQString            m_szEncoding;          // if empty, use network encoding
	TQStringList      * m_pChannelList;        // Channels to auto join
	bool               m_bAutoConnect;        // autoconnect
	TQString            m_szId;                // the server's may-be-unique id, may be auto-generated
	int                m_iProxy;              // proxy server's id
public:
	int proxy() { return m_iProxy; };
	KviProxy* proxyServer(KviProxyDataBase * pDb);
	
	kvi_u32_t port() const { return m_uPort; };
	const TQString & ipAddress() const { return m_szIp; };
	const TQString & password() const { return m_szPass; };
	const TQString & nickName() const { return m_szNick; };
    const TQString & initUMode() const { return m_szInitUMode; };
	const TQString & hostName() const { return m_szHostname; };
	const TQString & ip() const { return m_szIp; };
	const TQString & onLoginCommand() const { return m_szOnLoginCommand; };
	const TQString & onConnectCommand() const { return m_szOnConnectCommand; };
	const TQString & userName() const { return m_szUser; };
	const TQString & realName() const { return m_szRealName; };
	const TQString & linkFilter() const { return m_szLinkFilter; };
	const TQString & description() const { return m_szDescription; };
	const TQString & encoding() const { return m_szEncoding; };
	const TQString & id() const { return m_szId; };
	const TQString & userIdentityId() const { return m_szUserIdentityId; };
	bool autoConnect() const { return m_bAutoConnect; };
	TQStringList* autoJoinChannelList(){ return m_pChannelList; };
	bool isIpV6() const { return (m_uFlags & KVI_IRCSERVER_FLAG_IPV6); };
	bool useSSL() const { return (m_uFlags & KVI_IRCSERVER_FLAG_SSL); };
	bool cacheIp() const { return (m_uFlags & KVI_IRCSERVER_FLAG_CACHEIP); };
	
	void setProxy(int p){ m_iProxy = p; };
	void setIpAddress(const TQString &a){ m_szIp = a; };
	void setPort(kvi_u32_t p){ m_uPort = p; };
	void setHostName(const TQString &n){ m_szHostname = n; };
	void setDescription(const TQString &d){ m_szDescription = d; };
	void setUserName(const TQString &u){ m_szUser = u; };
	void setPassword(const TQString &p){ m_szPass = p; };
	void setNickName(const TQString &n){ m_szNick = n; };
	void setRealName(const TQString &r){ m_szRealName = r; };
	void setEncoding(const TQString &e){ m_szEncoding = e; };
    void setInitUMode(const TQString &u){ m_szInitUMode = u; };
	void setOnConnectCommand(const TQString &cmd){ m_szOnConnectCommand = cmd; };
	void setOnLoginCommand(const TQString &cmd){ m_szOnLoginCommand = cmd; };
	void setLinkFilter(const TQString &f){ m_szLinkFilter = f; };
	// the channel list must be allocated with new!
	void setAutoJoinChannelList(TQStringList * pNewChannelList);
	void setAutoConnect(bool autoconnect) { m_bAutoConnect = autoconnect; };
	void setUserIdentityId(const TQString &szUserIdentityId){ m_szUserIdentityId = szUserIdentityId; };
	void setIpV6(bool bSet)
	{
		if(bSet)m_uFlags |= KVI_IRCSERVER_FLAG_IPV6;
		else m_uFlags &= ((unsigned short)~KVI_IRCSERVER_FLAG_IPV6);
	};
	void setUseSSL(bool bSet)
	{
		if(bSet)m_uFlags |= KVI_IRCSERVER_FLAG_SSL;
		else m_uFlags &= ((unsigned short)~KVI_IRCSERVER_FLAG_SSL);
	};
	void setCacheIp(bool bSet)
	{
		if(bSet)m_uFlags |= KVI_IRCSERVER_FLAG_CACHEIP;
		else m_uFlags &= ((unsigned short)~KVI_IRCSERVER_FLAG_CACHEIP);
	};
	void operator =(const KviIrcServer &s);

	bool load(KviConfig * cfg,const TQString &prefix);
	void save(KviConfig * cfg,const TQString &prefix);
	
	void generateUniqueId();
	void setId(const TQString &szId){ m_szId = szId; if(m_szId.isEmpty())generateUniqueId(); };

	TQString ircUri();
};

class KVILIB_API KviIrcNetwork : public KviHeapObject
{
	friend class KviIrcServerDataBase;
public:
	KviIrcNetwork(const KviIrcNetwork &src);
	KviIrcNetwork(const TQString &name);
	~KviIrcNetwork();
protected:
	TQString              m_szName;
	TQString              m_szDescription;
	TQString              m_szEncoding;                // if empty, use system default
	TQString              m_szNickName;                // preferred nick name
	TQString              m_szUserName;                // preferred user name
	TQString              m_szRealName;                // preferred real name
	TQString              m_szOnConnectCommand;        // the command to run on connect
	TQString              m_szOnLoginCommand;          // the command to run after login
    TQStringList        * m_pChannelList;              // Channels to auto join
	KviNickServRuleSet * m_pNickServRuleSet;          // set of nick serv rules
	bool                 m_bAutoConnect;              // autoconnect
	TQString              m_szUserIdentityId;          // The user identity to use for this server: if empty
	                                                  // Then use the global primary identity
public:
	const TQString & name() const { return m_szName; };
	const TQString & encoding() const { return m_szEncoding; };
	const TQString & description() const { return m_szDescription; };
	const TQString & nickName() const { return m_szNickName; };
	const TQString & realName() const { return m_szRealName; };
	const TQString & userName() const { return m_szUserName; };
	const TQString & onLoginCommand() const { return m_szOnLoginCommand; };
	const TQString & onConnectCommand() const { return m_szOnConnectCommand; };
	const TQString & userIdentityId() const { return m_szUserIdentityId; };
	bool autoConnect() const { return m_bAutoConnect; };
	TQStringList* autoJoinChannelList(){ return m_pChannelList; };
	KviNickServRuleSet * nickServRuleSet(){ return m_pNickServRuleSet; };
	void setNickServRuleSet(KviNickServRuleSet * s);
	void copyFrom(const KviIrcNetwork &d);
	void setName(const TQString &szName){ m_szName = szName; };
	void setEncoding(const TQString &szEncoding){ m_szEncoding = szEncoding; };
	void setDescription(const TQString &szDescription){ m_szDescription = szDescription; };
	void setOnConnectCommand(const TQString &cmd){ m_szOnConnectCommand = cmd; };
	void setOnLoginCommand(const TQString &cmd){ m_szOnLoginCommand = cmd; };
	void setNickName(const TQString &n){ m_szNickName = n; };
	void setRealName(const TQString &r){ m_szRealName = r; };
	void setUserName(const TQString &u){ m_szUserName = u; };
	void setAutoJoinChannelList(TQStringList * pNewChannelList);
	void setAutoConnect(bool bAutoConnect){ m_bAutoConnect = bAutoConnect; };
	void setUserIdentityId(const TQString &szUserIdentityId){ m_szUserIdentityId = szUserIdentityId; };
};



#endif //_KVI_IRCSERVER_H_
