/*
 * This file is part of Chalk
 *
 * Copyright (c) 1999 Matthias Elter (me@kde.org)
 * Copyright (c) 2001-2002 Igor Jansen (rm@kde.org)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "ko_rgb_widget.h"

#include <tqlayout.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqspinbox.h>
#include <tqtooltip.h>
#include <tqcolor.h>

#include <kdebug.h>
#include <tdelocale.h>

#include <koFrameButton.h>
#include <koColorSlider.h>
#include <kcolordialog.h>

KoRGBWidget::KoRGBWidget(TQWidget *parent, const char *name) : super(parent, name)
{
    m_ColorButton = new KDualColorButton(this);
    m_ColorButton ->  setFixedSize(m_ColorButton->sizeHint());
    TQGridLayout *mGrid = new TQGridLayout(this, 3, 5, 5, 2);

    /* setup color sliders */
    mRSlider = new KoColorSlider(this);
    mRSlider->setMaximumHeight(20);
    mRSlider->slotSetRange(0, 255);
    mRSlider->setFocusPolicy( TQWidget::ClickFocus );

    mGSlider = new KoColorSlider(this);
    mGSlider->setMaximumHeight(20);
    mGSlider->slotSetRange(0, 255);
    mGSlider->setFocusPolicy( TQWidget::ClickFocus );

    mBSlider = new KoColorSlider(this);
    mBSlider->setMaximumHeight(20);
    mBSlider->slotSetRange(0, 255);
    mBSlider->setFocusPolicy( TQWidget::ClickFocus );

    /* setup slider labels */
    mRLabel = new TQLabel("R:", this);
    mRLabel->setFixedWidth(12);
    mRLabel->setFixedHeight(20);
    mGLabel = new TQLabel("G:", this);
    mGLabel->setFixedWidth(12);
    mGLabel->setFixedHeight(20);
    mBLabel = new TQLabel("B:", this);
    mBLabel->setFixedWidth(12);
    mBLabel->setFixedHeight(20);

    /* setup spin box */
    mRIn = new TQSpinBox(0, 255, 1, this);
    mRIn->setFixedWidth(50);
    mRIn->setFixedHeight(20);
    mRIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mRIn, i18n( "Red" ) );

    mGIn = new TQSpinBox(0, 255, 1, this);
    mGIn->setFixedWidth(50);
    mGIn->setFixedHeight(20);
    mGIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mGIn, i18n( "Green" ) );

    mBIn = new TQSpinBox(0, 255, 1, this);
    mBIn->setFixedWidth(50);
    mBIn->setFixedHeight(20);
    mBIn->setFocusPolicy( TQWidget::ClickFocus );
    TQToolTip::add( mBIn, i18n( "Blue" ) );

    mGrid->addMultiCellWidget(m_ColorButton, 0, 3, 0, 0, TQt::AlignTop);
    mGrid->addWidget(mRLabel, 0, 1);
    mGrid->addWidget(mGLabel, 1, 1);
    mGrid->addWidget(mBLabel, 2, 1);
    mGrid->addMultiCellWidget(mRSlider, 0, 0, 2, 3);
    mGrid->addMultiCellWidget(mGSlider, 1, 1, 2, 3);
    mGrid->addMultiCellWidget(mBSlider, 2, 2, 2, 3);
    mGrid->addWidget(mRIn, 0, 4);
    mGrid->addWidget(mGIn, 1, 4);
    mGrid->addWidget(mBIn, 2, 4);

    connect(m_ColorButton, TQ_SIGNAL(fgChanged(const TQColor &)), this, TQ_SLOT(slotFGColorSelected(const TQColor &)));
    connect(m_ColorButton, TQ_SIGNAL(bgChanged(const TQColor &)), this, TQ_SLOT(slotBGColorSelected(const TQColor &)));
    connect(m_ColorButton, TQ_SIGNAL(currentChanged(KDualColorButton::DualColor)), this, TQ_SLOT(currentChanged(KDualColorButton::DualColor)));

    /* connect color sliders */
    connect(mRSlider, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotRChanged(int)));
    connect(mGSlider, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotGChanged(int)));
    connect(mBSlider, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotBChanged(int)));

    /* connect spin box */
    connect(mRIn, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotRChanged(int)));
    connect(mGIn, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotGChanged(int)));
    connect(mBIn, TQ_SIGNAL(valueChanged(int)), this, TQ_SLOT(slotBChanged(int)));

    update(TQt::black, TQt::white);
}

void KoRGBWidget::slotRChanged(int r)
{
    if (m_ColorButton->current() == KDualColorButton::Foreground)
        slotFGColorSelected( TQColor(r, m_fgColor.green(), m_fgColor.blue()));
    else
        slotBGColorSelected( TQColor(r, m_bgColor.green(), m_bgColor.blue()));
}

void KoRGBWidget::slotGChanged(int g)
{
    if (m_ColorButton->current() == KDualColorButton::Foreground)
        slotFGColorSelected( TQColor( m_fgColor.red(), g, m_fgColor.blue()));
    else
        slotBGColorSelected( TQColor( m_bgColor.red(), g, m_bgColor.blue()));;
}

void KoRGBWidget::slotBChanged(int b)
{
    if (m_ColorButton->current() == KDualColorButton::Foreground)
        slotFGColorSelected( TQColor( m_fgColor.red(), m_fgColor.green(), b));
    else
        slotBGColorSelected( TQColor( m_bgColor.red(), m_bgColor.green(), b));
}

void KoRGBWidget::setFgColor(const TQColor & c)
{
    blockSignals(true);
    slotFGColorSelected(c);
    blockSignals(false);
}

void KoRGBWidget::setBgColor(const TQColor & c)
{
    blockSignals(true);
    slotBGColorSelected(c);
    blockSignals(false);
}

void KoRGBWidget::update(const TQColor fgColor, const TQColor bgColor)
{

    m_fgColor = fgColor;
    m_bgColor = bgColor;

    TQColor color = (m_ColorButton->current() == KDualColorButton::Foreground)? m_fgColor : m_bgColor;

    int r = color.red();
    int g = color.green();
    int b = color.blue();

    mRSlider->blockSignals(true);
    mRIn->blockSignals(true);
    mGSlider->blockSignals(true);
    mGIn->blockSignals(true);
    mBSlider->blockSignals(true);
    mBIn->blockSignals(true);

    mRSlider->slotSetColor1(TQColor(0, g, b));
    mRSlider->slotSetColor2(TQColor(255, g, b));
    mRSlider->slotSetValue(r);
    mRIn->setValue(r);

    mGSlider->slotSetColor1(TQColor(r, 0, b));
    mGSlider->slotSetColor2(TQColor(r, 255, b));
    mGSlider->slotSetValue(g);
    mGIn->setValue(g);

    mBSlider->slotSetColor1(TQColor(r, g, 0));
    mBSlider->slotSetColor2(TQColor(r, g, 255));
    mBSlider->slotSetValue(b);
    mBIn->setValue(b);

    mRSlider->blockSignals(false);
    mRIn->blockSignals(false);
    mGSlider->blockSignals(false);
    mGIn->blockSignals(false);
    mBSlider->blockSignals(false);
    mBIn->blockSignals(false);
}

void KoRGBWidget::slotFGColorSelected(const TQColor& c)
{
    m_fgColor = c;
    disconnect(m_ColorButton, TQ_SIGNAL(fgChanged(const TQColor &)), this, TQ_SLOT(slotFGColorSelected(const TQColor &)));
    m_ColorButton->setForeground( m_fgColor );
    connect(m_ColorButton, TQ_SIGNAL(fgChanged(const TQColor &)), this, TQ_SLOT(slotFGColorSelected(const TQColor &)));

    update( m_fgColor, m_bgColor);
    emit sigFgColorChanged(m_fgColor);
}

void KoRGBWidget::slotBGColorSelected(const TQColor& c)
{
    m_bgColor = c;

    disconnect(m_ColorButton, TQ_SIGNAL(bgChanged(const TQColor &)), this, TQ_SLOT(slotBGColorSelected(const TQColor &)));
    m_ColorButton->setBackground( m_bgColor );
    connect(m_ColorButton, TQ_SIGNAL(bgChanged(const TQColor &)), this, TQ_SLOT(slotBGColorSelected(const TQColor &)));

    update(m_fgColor, m_bgColor);
    emit sigBgColorChanged(m_bgColor);
}

void KoRGBWidget::currentChanged(KDualColorButton::DualColor s)
{
   if(s == KDualColorButton::Foreground)
     slotFGColorSelected(m_ColorButton->currentColor());
   else
     slotBGColorSelected(m_ColorButton->currentColor());
   emit sigModeChanged( s );
}

void KoRGBWidget::setMode(KDualColorButton::DualColor s)
{
    m_ColorButton->setCurrent( s );
}

#include "ko_rgb_widget.moc"
