/*
 * This file is part of the KDE project
 *
 * Copyright (c) 2005 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "wavefilter.h"

#include <stdlib.h>
#include <vector>

#include <tqpoint.h>

#include <kcombobox.h>
#include <kdebug.h>
#include <kgenericfactory.h>
#include <kiconloader.h>
#include <tdeinstance.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <knuminput.h>
#include <tdestandarddirs.h>
#include <tdetempfile.h>

#include <kis_image.h>
#include <kis_iterators_pixel.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_layer.h>
#include <kis_random_sub_accessor.h>
#include <kis_types.h>

#include "kis_wdg_wave.h"
#include "wdgwaveoptions.h"

typedef KGenericFactory<ChalkWaveFilter> ChalkWaveFilterFactory;
K_EXPORT_COMPONENT_FACTORY( chalkwavefilter, ChalkWaveFilterFactory( "chalk" ) )

class KisWaveCurve {
    public:
        virtual double valueAt(int x, int y) =0;
};

class KisSinusoidalWaveCurve : public KisWaveCurve {
    public:
        KisSinusoidalWaveCurve(int amplitude, int wavelength, int shift) : m_amplitude(amplitude), m_wavelength(wavelength), m_shift(shift)
        {
        }
        virtual double valueAt(int x, int y)
        {
            return y + m_amplitude * cos( (double) ( m_shift + x) / m_wavelength );
        }
    private:
        int m_amplitude, m_wavelength, m_shift;
};

class KisTriangleWaveCurve : public KisWaveCurve {
    public:
        KisTriangleWaveCurve(int amplitude, int wavelength, int shift) :  m_amplitude(amplitude), m_wavelength(wavelength), m_shift(shift)
        {
        }
        virtual double valueAt(int x, int y)
        {
            return y +  m_amplitude * pow( -1, (m_shift + x) / m_wavelength )  * (0.5 - (double)( (m_shift + x) % m_wavelength ) / m_wavelength );
        }
    private:
        int m_amplitude, m_wavelength, m_shift;
};



ChalkWaveFilter::ChalkWaveFilter(TQObject *parent, const char *name, const TQStringList &)
        : KParts::Plugin(parent, name)
{
    setInstance(ChalkWaveFilterFactory::instance());


    if (parent->inherits("KisFilterRegistry")) {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisFilterWave());
    }
}

ChalkWaveFilter::~ChalkWaveFilter()
{
}

KisFilterWave::KisFilterWave() : KisFilter(id(), "other", i18n("&Wave..."))
{
}

KisFilterConfiguration* KisFilterWave::configuration(TQWidget* w)
{
    KisWdgWave* wN = dynamic_cast<KisWdgWave*>(w);
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(), 1);
    if(wN)
    {
        config->setProperty("horizontalwavelength", wN->widget()->intHWavelength->value() );
        config->setProperty("horizontalshift", wN->widget()->intHShift->value() );
        config->setProperty("horizontalamplitude", wN->widget()->intHAmplitude->value() );
        config->setProperty("horizontalshape", wN->widget()->cbHShape->currentItem() );
        config->setProperty("verticalwavelength", wN->widget()->intVWavelength->value() );
        config->setProperty("verticalshift", wN->widget()->intVShift->value() );
        config->setProperty("verticalamplitude", wN->widget()->intVAmplitude->value() );
        config->setProperty("verticalshape", wN->widget()->cbVShape->currentItem() );
    }
    return config;
}

KisFilterConfigWidget * KisFilterWave::createConfigurationWidget(TQWidget* parent, KisPaintDeviceSP /*dev*/)
{
    return new KisWdgWave((KisFilter*)this, (TQWidget*)parent, i18n("Configuration of wave filter").ascii());
}

void KisFilterWave::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, KisFilterConfiguration* config, const TQRect& rect)
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);

    setProgressTotalSteps(rect.width() * rect.height());

    TQVariant value;
    int horizontalwavelength = (config && config->getProperty("horizontalwavelength", value)) ? value.toInt() : 50;
    int horizontalshift = (config && config->getProperty("horizontalshift", value)) ? value.toInt() : 50;
    int horizontalamplitude = (config && config->getProperty("horizontalamplitude", value)) ? value.toInt() : 4;
    int horizontalshape = (config && config->getProperty("horizontalshape", value)) ? value.toInt() : 0;
    int verticalwavelength = (config && config->getProperty("verticalwavelength", value)) ? value.toInt() : 50;
    int verticalshift = (config && config->getProperty("verticalshift", value)) ? value.toInt() : 50;
    int verticalamplitude = (config && config->getProperty("verticalamplitude", value)) ? value.toInt() : 4;
    int verticalshape = (config && config->getProperty("verticalshape", value)) ? value.toInt() : 0;
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisWaveCurve* verticalcurve;
    if(verticalshape == 1)
        verticalcurve = new KisTriangleWaveCurve(verticalamplitude, verticalwavelength, verticalshift);
    else
        verticalcurve = new KisSinusoidalWaveCurve(verticalamplitude, verticalwavelength, verticalshift);
    KisWaveCurve* horizontalcurve;
    if(horizontalshape == 1)
        horizontalcurve = new KisTriangleWaveCurve(horizontalamplitude, horizontalwavelength, horizontalshift);
    else
        horizontalcurve = new KisSinusoidalWaveCurve(horizontalamplitude, horizontalwavelength, horizontalshift);
    KisRandomSubAccessorPixel srcRSA = src->createRandomSubAccessor();
    while(!dstIt.isDone())
    {
        double xv = horizontalcurve->valueAt( dstIt.y(), dstIt.x() );
        double yv = verticalcurve->valueAt( dstIt.x(), dstIt.y() );
        srcRSA.moveTo( KisPoint( xv, yv ) );
        srcRSA.sampledOldRawData(dstIt.rawData());
        ++dstIt;
        incProgress();
    }
    delete horizontalcurve;
    delete verticalcurve;
    setProgressDone(); // Must be called even if you don't really support progression
}
