/**********************************************************************
** Copyright (C) 2000-2002 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid TQt Enterprise Edition or TQt Professional Edition
** licenses may use this file in accordance with the TQt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about TQt Commercial License Agreements.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "actiondnd.h"
#include "command.h"
#include "defs.h"
#include "formwindow.h"
#include "mainwindow.h"
#include "metadatabase.h"
#include "widgetfactory.h"
#include "hierarchyview.h"

#include <tqaction.h>
#include <tqapplication.h>
#include <tqbitmap.h>
#include <tqdragobject.h>
#include <tqinputdialog.h>
#include <tqlayout.h>
#include <tqmainwindow.h>
#include <tqmenudata.h>
#include <tqmessagebox.h>
#include <tqobjectlist.h>
#include <tqpainter.h>
#include <tqstyle.h>
#include <tqtimer.h>

#include <tdelocale.h>

TQAction *ActionDrag::the_action = 0;

ActionDrag::ActionDrag(TQAction *action, TQWidget *source)
: TQStoredDrag("application/x-designer-actions", source)
{
    Q_ASSERT(the_action == 0);
    the_action = action;
}

ActionDrag::ActionDrag(TQActionGroup *group, TQWidget *source)
: TQStoredDrag("application/x-designer-actiongroup", source)
{
    Q_ASSERT(the_action == 0);
    the_action = group;
}

ActionDrag::ActionDrag(const TQString &type, TQAction *action, TQWidget *source)
: TQStoredDrag(type, source)
{
    Q_ASSERT(the_action == 0);
    the_action = action;
}

bool ActionDrag::canDecode(TQDropEvent *e)
{
    return e->provides( "application/x-designer-actions" ) ||
	   e->provides( "application/x-designer-actiongroup" ) ||
	   e->provides( "application/x-designer-separator" );
}

ActionDrag::~ActionDrag()
{
    the_action = 0;
}

void QDesignerAction::init()
{
    MetaDataBase::addEntry( this );
    int id = WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( this ) );
    WidgetFactory::saveDefaultProperties( this, id );
    WidgetFactory::saveChangedProperties( this, id );
}

void QDesignerActionGroup::init()
{
    MetaDataBase::addEntry( this );
    int id = WidgetDatabase::idFromClassName( WidgetFactory::classNameOf( this ) );
    WidgetFactory::saveDefaultProperties( this, id );
    WidgetFactory::saveChangedProperties( this, id );
}

bool QDesignerAction::addTo( TQWidget *w )
{
    if ( !widgetToInsert )
	return TQAction::addTo( w );

    if ( ::tqt_cast<TQPopupMenu*>(w) )
	return false;

    widgetToInsert->reparent( w, TQPoint( 0, 0 ), false );
    widgetToInsert->show();
    addedTo( widgetToInsert, w );
    return true;
}

bool QDesignerAction::removeFrom( TQWidget *w )
{
    if ( !widgetToInsert )
	return TQAction::removeFrom( w );

    remove();
    return true;
}

void QDesignerAction::remove()
{
    if ( !widgetToInsert )
	return;
    MainWindow::self->formWindow()->selectWidget( widgetToInsert, false );
    widgetToInsert->reparent( 0, TQPoint( 0, 0 ), false );
}

QDesignerToolBarSeparator::QDesignerToolBarSeparator(TQt::Orientation o , TQToolBar *parent,
                                     const char* name )
    : TQWidget( parent, name )
{
    connect( parent, TQ_SIGNAL(orientationChanged(TQt::Orientation)),
             this, TQ_SLOT(setOrientation(TQt::Orientation)) );
    setOrientation( o );
    setBackgroundMode( parent->backgroundMode() );
    setBackgroundOrigin( ParentOrigin );
    setSizePolicy( TQSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Minimum ) );
}

void QDesignerToolBarSeparator::setOrientation( TQt::Orientation o )
{
    orient = o;
}

void QDesignerToolBarSeparator::styleChange( TQStyle& )
{
    setOrientation( orient );
}

TQSize QDesignerToolBarSeparator::sizeHint() const
{
    int extent = style().pixelMetric( TQStyle::PM_DockWindowSeparatorExtent,
				      this );
    if ( orient ==TQt::Horizontal )
	return TQSize( extent, 0 );
    else
	return TQSize( 0, extent );
}

void QDesignerToolBarSeparator::paintEvent( TQPaintEvent * )
{
    TQPainter p( this );
    TQStyle::SFlags flags = TQStyle::Style_Default;

    if ( orientation() ==TQt::Horizontal )
	flags |= TQStyle::Style_Horizontal;

    style().drawPrimitive( TQStyle::PE_DockWindowSeparator, &p, rect(),
			   colorGroup(), flags );
}



QSeparatorAction::QSeparatorAction( TQObject *parent )
    : TQAction( parent, "qt_designer_separator" ), wid( 0 )
{
}

bool QSeparatorAction::addTo( TQWidget *w )
{
    if ( ::tqt_cast<TQToolBar*>(w) ) {
	TQToolBar *tb = (TQToolBar*)w;
	wid = new QDesignerToolBarSeparator( tb->orientation(), tb );
	return true;
    } else if ( ::tqt_cast<TQPopupMenu*>(w) ) {
	idx = ( (TQPopupMenu*)w )->count();
	( (TQPopupMenu*)w )->insertSeparator( idx );
	return true;
    }
    return false;
}

bool QSeparatorAction::removeFrom( TQWidget *w )
{
    if ( ::tqt_cast<TQToolBar*>(w) ) {
	delete wid;
	return true;
    } else if ( ::tqt_cast<TQPopupMenu*>(w) ) {
	( (TQPopupMenu*)w )->removeItemAt( idx );
	return true;
    }
    return false;
}

TQWidget *QSeparatorAction::widget() const
{
    return wid;
}



QDesignerToolBar::QDesignerToolBar( TQMainWindow *mw )
    : TQToolBar( mw ), lastIndicatorPos( -1, -1 )
{
    insertAnchor = 0;
    afterAnchor = true;
    setAcceptDrops( true );
    MetaDataBase::addEntry( this );
    lastIndicatorPos = TQPoint( -1, -1 );
    indicator = new QDesignerIndicatorWidget( this );
    indicator->hide();
    installEventFilter( this );
    widgetInserting = false;
    findFormWindow();
    mw->setDockEnabled( DockTornOff, false );
}

QDesignerToolBar::QDesignerToolBar( TQMainWindow *mw, Dock dock )
    : TQToolBar( TQString(), mw, dock), lastIndicatorPos( -1, -1 )
{
    insertAnchor = 0;
    afterAnchor = true;
    setAcceptDrops( true );
    indicator = new QDesignerIndicatorWidget( this );
    indicator->hide();
    MetaDataBase::addEntry( this );
    installEventFilter( this );
    widgetInserting = false;
    findFormWindow();
    mw->setDockEnabled( DockTornOff, false );
}

void QDesignerToolBar::findFormWindow()
{
    TQWidget *w = this;
    while ( w ) {
	formWindow = ::tqt_cast<FormWindow*>(w);
	if ( formWindow )
	    break;
	w = w->parentWidget();
    }
}

void QDesignerToolBar::addAction( TQAction *a )
{
    actionList.append( a );
    connect( a, TQ_SIGNAL( destroyed() ), this, TQ_SLOT( actionRemoved() ) );
    if ( ::tqt_cast<TQActionGroup*>(a) ) {
	( (QDesignerActionGroup*)a )->widget()->installEventFilter( this );
	actionMap.insert( ( (QDesignerActionGroup*)a )->widget(), a );
    } else if ( ::tqt_cast<QSeparatorAction*>(a) ) {
	( (QSeparatorAction*)a )->widget()->installEventFilter( this );
	actionMap.insert( ( (QSeparatorAction*)a )->widget(), a );
    } else {
	( (QDesignerAction*)a )->widget()->installEventFilter( this );
	actionMap.insert( ( (QDesignerAction*)a )->widget(), a );
    }
}

static void fixObject( TQObject *&o )
{
    while ( o && o->parent() && !::tqt_cast<QDesignerToolBar*>(o->parent()) )
	o = o->parent();
}

bool QDesignerToolBar::eventFilter( TQObject *o, TQEvent *e )
{
    if ( !o || !e || o->inherits( "TQDockWindowHandle" ) || o->inherits( "TQDockWindowTitleBar" ) )
	return TQToolBar::eventFilter( o, e );

    if ( o == this && e->type() == TQEvent::MouseButtonPress &&
	 ( ( TQMouseEvent*)e )->button() == TQt::LeftButton ) {
	mousePressEvent( (TQMouseEvent*)e );
	return true;
    }

    if ( o == this )
	return TQToolBar::eventFilter( o, e );

    if ( e->type() == TQEvent::MouseButtonPress ) {
	TQMouseEvent *ke = (TQMouseEvent*)e;
	fixObject( o );
	if ( !o )
	    return false;
	buttonMousePressEvent( ke, o );
	return true;
    } else if(e->type() == TQEvent::ContextMenu ) {
	TQContextMenuEvent *ce = (TQContextMenuEvent*)e;
	fixObject( o );
	if( !o )
	    return false;
	buttonContextMenuEvent( ce, o );
	return true;
    } else if ( e->type() == TQEvent::MouseMove ) {
	TQMouseEvent *ke = (TQMouseEvent*)e;
	fixObject( o );
	if ( !o )
	    return false;
	buttonMouseMoveEvent( ke, o );
	return true;
    } else if ( e->type() == TQEvent::MouseButtonRelease ) {
	TQMouseEvent *ke = (TQMouseEvent*)e;
	fixObject( o );
	if ( !o )
	    return false;
	buttonMouseReleaseEvent( ke, o );
	return true;
    } else if ( e->type() == TQEvent::DragEnter ) {
	TQDragEnterEvent *de = (TQDragEnterEvent*)e;
	if (ActionDrag::canDecode(de))
	    de->accept();
    } else if ( e->type() == TQEvent::DragMove ) {
	TQDragMoveEvent *de = (TQDragMoveEvent*)e;
	if (ActionDrag::canDecode(de))
	    de->accept();
    }

    return TQToolBar::eventFilter( o, e );
}

void QDesignerToolBar::paintEvent( TQPaintEvent *e )
{
    TQToolBar::paintEvent( e );
    if ( e->rect() != rect() )
	return;
    lastIndicatorPos = TQPoint( -1, -1 );
}

void QDesignerToolBar::contextMenuEvent( TQContextMenuEvent *e )
{
    e->accept();
    TQPopupMenu menu( 0 );
    menu.insertItem( i18n( "Delete Toolbar" ), 1 );
    int res = menu.exec( e->globalPos() );
    if ( res != -1 ) {
	RemoveToolBarCommand *cmd = new RemoveToolBarCommand( i18n( "Delete Toolbar '%1'" ).arg( name() ),
							      formWindow, 0, this );
	formWindow->commandHistory()->addCommand( cmd );
	cmd->execute();
    }
}

void QDesignerToolBar::mousePressEvent( TQMouseEvent *e )
{
    widgetInserting = false;
    if ( e->button() == TQt::LeftButton &&
	 MainWindow::self->currentTool() != POINTER_TOOL &&
	 MainWindow::self->currentTool() != ORDER_TOOL &&
	 MainWindow::self->currentTool() != CONNECT_TOOL &&
	 MainWindow::self->currentTool() != BUDDY_TOOL )
	widgetInserting = true;
}

void QDesignerToolBar::mouseReleaseEvent( TQMouseEvent *e )
{
    if ( widgetInserting )
	doInsertWidget( mapFromGlobal( e->globalPos() ) );
    widgetInserting = false;
}

void QDesignerToolBar::buttonMouseReleaseEvent( TQMouseEvent *e, TQObject *w )
{
    if ( widgetInserting )
	doInsertWidget( mapFromGlobal( e->globalPos() ) );
    else if ( w->isWidgetType() && formWindow->widgets()->find( w ) ) {
	formWindow->clearSelection( false );
	formWindow->selectWidget( w );
    }
    widgetInserting = false;
}

void QDesignerToolBar::buttonContextMenuEvent( TQContextMenuEvent *e, TQObject *o )
{
    e->accept();
    TQPopupMenu menu( 0 );
    const int ID_DELETE = 1;
    const int ID_SEP = 2;
    const int ID_DELTOOLBAR = 3;
    TQMap<TQWidget*, TQAction*>::Iterator it = actionMap.find( (TQWidget*)o );
    if ( it != actionMap.end() && ::tqt_cast<QSeparatorAction*>(*it) )
	menu.insertItem( i18n( "Delete Separator" ), ID_DELETE );
    else
	menu.insertItem( i18n( "Delete Item" ), ID_DELETE );
    menu.insertItem( i18n( "Insert Separator" ), ID_SEP );
    menu.insertSeparator();
    menu.insertItem( i18n( "Delete Toolbar" ), ID_DELTOOLBAR );
    int res = menu.exec( e->globalPos() );
    if ( res == ID_DELETE ) {
	TQMap<TQWidget*, TQAction*>::Iterator it = actionMap.find( (TQWidget*)o );
	if ( it == actionMap.end() )
	    return;
	TQAction *a = *it;
	int index = actionList.find( a );
	RemoveActionFromToolBarCommand *cmd = new RemoveActionFromToolBarCommand(
	    i18n( "Delete Action '%1' From Toolbar '%2'" ).
	    arg( a->name() ).arg( caption() ),
	    formWindow, a, this, index );
	formWindow->commandHistory()->addCommand( cmd );
	cmd->execute();
    } else if ( res == ID_SEP ) {
	calcIndicatorPos( mapFromGlobal( e->globalPos() ) );
	TQAction *a = new QSeparatorAction( 0 );
	int index = actionList.findRef( *actionMap.find( insertAnchor ) );
	if ( index != -1 && afterAnchor )
	    ++index;
	if ( !insertAnchor )
	    index = 0;

	AddActionToToolBarCommand *cmd = new AddActionToToolBarCommand(
	    i18n( "Add Separator to Toolbar '%1'" ).
	    arg( a->name() ),
	    formWindow, a, this, index );
	formWindow->commandHistory()->addCommand( cmd );
	cmd->execute();
    } else if ( res == ID_DELTOOLBAR ) {
	RemoveToolBarCommand *cmd = new RemoveToolBarCommand( i18n( "Delete Toolbar '%1'" ).arg( name() ),
							      formWindow, 0, this );
	formWindow->commandHistory()->addCommand( cmd );
	cmd->execute();
    }
}

void QDesignerToolBar::buttonMousePressEvent( TQMouseEvent *e, TQObject * )
{
    widgetInserting = false;

    if ( e->button() == TQt::MidButton )
	return;

    if ( e->button() == TQt::LeftButton &&
	 MainWindow::self->currentTool() != POINTER_TOOL &&
	 MainWindow::self->currentTool() != ORDER_TOOL &&
	 MainWindow::self->currentTool() != CONNECT_TOOL &&
	 MainWindow::self->currentTool() != BUDDY_TOOL ) {
	    widgetInserting = true;
	    return;
    }

    dragStartPos = e->pos();
}

void QDesignerToolBar::removeWidget( TQWidget *w )
{
    TQMap<TQWidget*, TQAction*>::Iterator it = actionMap.find( w );
    if ( it == actionMap.end() )
	return;
    TQAction *a = *it;
    int index = actionList.find( a );
    RemoveActionFromToolBarCommand *cmd =
	new RemoveActionFromToolBarCommand( i18n( "Delete Action '%1' From Toolbar '%2'" ).
					    arg( a->name() ).arg( caption() ),
					    formWindow, a, this, index );
    formWindow->commandHistory()->addCommand( cmd );
    cmd->execute();
    TQApplication::sendPostedEvents();
    adjustSize();
}

void QDesignerToolBar::buttonMouseMoveEvent( TQMouseEvent *e, TQObject *o )
{
    if ( widgetInserting || ( e->state() & TQt::LeftButton ) == 0 )
	return;
    if ( TQABS( TQPoint( dragStartPos - e->pos() ).manhattanLength() ) < TQApplication::startDragDistance() )
	return;
    TQMap<TQWidget*, TQAction*>::Iterator it = actionMap.find( (TQWidget*)o );
    if ( it == actionMap.end() )
	return;
    TQAction *a = *it;
    if ( !a )
	return;
    int index = actionList.find( a );
    RemoveActionFromToolBarCommand *cmd =
	new RemoveActionFromToolBarCommand( i18n( "Delete Action '%1' From Toolbar '%2'" ).
					    arg( a->name() ).arg( caption() ),
					    formWindow, a, this, index );
    formWindow->commandHistory()->addCommand( cmd );
    cmd->execute();
    TQApplication::sendPostedEvents();
    adjustSize();

    TQString type = ::tqt_cast<TQActionGroup*>(a) ? TQString( "application/x-designer-actiongroup" ) :
	::tqt_cast<QSeparatorAction*>(a) ? TQString( "application/x-designer-separator" ) : TQString( "application/x-designer-actions" );
    TQStoredDrag *drag = new ActionDrag( type, a, this );
    drag->setPixmap( a->iconSet().pixmap() );
    if ( ::tqt_cast<QDesignerAction*>(a) ) {
	if ( formWindow->widgets()->find( ( (QDesignerAction*)a )->widget() ) )
	    formWindow->selectWidget( ( ( (QDesignerAction*)a )->widget()), false );
    }
    if ( !drag->drag() ) {
	AddActionToToolBarCommand *cmd = new AddActionToToolBarCommand( i18n( "Add Action '%1' to Toolbar '%2'" ).
									arg( a->name() ).arg( caption() ),
									formWindow, a, this, index );
	formWindow->commandHistory()->addCommand( cmd );
	cmd->execute();
    }
    lastIndicatorPos = TQPoint( -1, -1 );
    indicator->hide();
}

#ifndef TQT_NO_DRAGANDDROP

void QDesignerToolBar::dragEnterEvent( TQDragEnterEvent *e )
{
    widgetInserting = false;
    lastIndicatorPos = TQPoint( -1, -1 );
    if (ActionDrag::canDecode(e))
	e->accept();
}

void QDesignerToolBar::dragMoveEvent( TQDragMoveEvent *e )
{
    if (ActionDrag::canDecode(e)) {
	e->accept();
	drawIndicator( calcIndicatorPos( e->pos() ) );
    }
}

void QDesignerToolBar::dragLeaveEvent( TQDragLeaveEvent * )
{
    indicator->hide();
    insertAnchor = 0;
    afterAnchor = true;
}

void QDesignerToolBar::dropEvent( TQDropEvent *e )
{
    if (!ActionDrag::canDecode(e))
	return;

    e->accept();

    indicator->hide();
    TQAction *a = 0;
    int index = actionList.findRef( *actionMap.find( insertAnchor ) );
    if ( index != -1 && afterAnchor )
	++index;
    if ( !insertAnchor )
	index = 0;
    if ( e->provides( "application/x-designer-actions" ) ||
	 e->provides( "application/x-designer-separator" ) ) {
	if ( e->provides( "application/x-designer-actions" ) )
	    a = ::tqt_cast<QDesignerAction*>(ActionDrag::action());
	else
	    a = ::tqt_cast<QSeparatorAction*>(ActionDrag::action());
    } else {
	a = ::tqt_cast<QDesignerActionGroup*>(ActionDrag::action());
    }

    if ( actionList.findRef( a ) != -1 ) {
	TQMessageBox::warning( MainWindow::self, i18n( "Insert/Move Action" ),
			      i18n( "Action '%1' has already been added to this toolbar.\n"
				  "An Action may only occur once in a given toolbar." ).
			      arg( a->name() ) );
	return;
    }

    AddActionToToolBarCommand *cmd = new AddActionToToolBarCommand( i18n( "Add Action '%1' to Toolbar '%2'" ).
								    arg( a->name() ).arg( caption() ),
								    formWindow, a, this, index );
    formWindow->commandHistory()->addCommand( cmd );
    cmd->execute();

    lastIndicatorPos = TQPoint( -1, -1 );
}

#endif

void QDesignerToolBar::reInsert()
{
    TQAction *a = 0;
    actionMap.clear();
    clear();
    for ( a = actionList.first(); a; a = actionList.next() ) {
	a->addTo( this );
	if ( ::tqt_cast<TQActionGroup*>(a) ) {
	    actionMap.insert( ( (QDesignerActionGroup*)a )->widget(), a );
	    if ( ( (QDesignerActionGroup*)a )->widget() )
		( (QDesignerActionGroup*)a )->widget()->installEventFilter( this );
	} else if ( ::tqt_cast<QDesignerAction*>(a) ) {
	    actionMap.insert( ( (QDesignerAction*)a )->widget(), a );
	    ( (QDesignerAction*)a )->widget()->installEventFilter( this );
	} else if ( ::tqt_cast<QSeparatorAction*>(a) ) {
	    actionMap.insert( ( (QSeparatorAction*)a )->widget(), a );
	    ( (QSeparatorAction*)a )->widget()->installEventFilter( this );
	}
    }
    TQApplication::sendPostedEvents();
    adjustSize();
}

void QDesignerToolBar::actionRemoved()
{
    actionList.removeRef( (TQAction*)sender() );
}

TQPoint QDesignerToolBar::calcIndicatorPos( const TQPoint &pos )
{
    if ( orientation() ==TQt::Horizontal ) {
	TQPoint pnt( width() - 2, 0 );
	insertAnchor = 0;
	afterAnchor = true;
	TQObjectList clo = childrenListObject();
	if ( clo.isEmpty() )
	    return pnt;
	pnt = TQPoint( 13, 0 );
	TQObjectListIt it( clo );
	TQObject * obj;
	while( (obj=it.current()) != 0 ) {
	    ++it;
	    if ( obj->isWidgetType() &&
		 qstrcmp( "qt_dockwidget_internal", obj->name() ) != 0 ) {
		TQWidget *w = (TQWidget*)obj;
		if ( w->x() < pos.x() ) {
		    pnt.setX( w->x() + w->width() + 1 );
		    insertAnchor = w;
		    afterAnchor = true;
		}
	    }
	}
	return pnt;
    } else {
	TQPoint pnt( 0, height() - 2 );
	insertAnchor = 0;
	afterAnchor = true;
	TQObjectList clo = childrenListObject();
	if ( clo.isEmpty() )
	    return pnt;
	pnt = TQPoint( 0, 13 );
	TQObjectListIt it( clo );
	TQObject * obj;
	while( (obj=it.current()) != 0 ) {
	    ++it;
	    if ( obj->isWidgetType() &&
		 qstrcmp( "qt_dockwidget_internal", obj->name() ) != 0 ) {
		TQWidget *w = (TQWidget*)obj;
		if ( w->y() < pos.y() ) {
		    pnt.setY( w->y() + w->height() + 1 );
		    insertAnchor = w;
		    afterAnchor = true;
		}
	    }
	}
	return pnt;
    }
}

void QDesignerToolBar::drawIndicator( const TQPoint &pos )
{
    if ( lastIndicatorPos == pos )
	return;
    bool wasVsisible = indicator->isVisible();
    if ( orientation() ==TQt::Horizontal ) {
	indicator->resize( 3, height() );
	if ( pos != TQPoint( -1, -1 ) )
	     indicator->move( pos.x() - 1, 0 );
	indicator->show();
	indicator->raise();
	lastIndicatorPos = pos;
    } else {
	indicator->resize( width(), 3 );
	if ( pos != TQPoint( -1, -1 ) )
	     indicator->move( 0, pos.y() - 1 );
	indicator->show();
	indicator->raise();
	lastIndicatorPos = pos;
    }
    if ( !wasVsisible )
	TQApplication::sendPostedEvents();
}

void QDesignerToolBar::doInsertWidget( const TQPoint &p )
{
    if ( formWindow != MainWindow::self->formWindow() )
	return;
    calcIndicatorPos( p );
    TQWidget *w = WidgetFactory::create( MainWindow::self->currentTool(), this, 0, true );
    installEventFilters( w );
    MainWindow::self->formWindow()->insertWidget( w, true );
    QDesignerAction *a = new QDesignerAction( w, parent() );
    int index = actionList.findRef( *actionMap.find( insertAnchor ) );
    if ( index != -1 && afterAnchor )
	++index;
    if ( !insertAnchor )
	index = 0;
    AddActionToToolBarCommand *cmd = new AddActionToToolBarCommand( i18n( "Add Widget '%1' to Toolbar '%2'" ).
								    arg( w->name() ).arg( caption() ),
								    formWindow, a, this, index );
    formWindow->commandHistory()->addCommand( cmd );
    cmd->execute();
    MainWindow::self->resetTool();
}

void QDesignerToolBar::clear()
{
    for ( TQAction *a = actionList.first(); a; a = actionList.next() ) {
	if ( ::tqt_cast<QDesignerAction*>(a) )
	    ( (QDesignerAction*)a )->remove();
    }
    TQToolBar::clear();
}

void QDesignerToolBar::installEventFilters( TQWidget *w )
{
    if ( !w )
	return;
    TQObjectList *l = w->queryList( "TQWidget" );
    for ( TQObject *o = l->first(); o; o = l->next() )
	o->installEventFilter( this );
    delete l;
}

//#include "actiondnd.moc"
