import java.util.*;
import org.trinitydesktop.qt.*;
import org.trinitydesktop.koala.*;

/**
  * The base class for KScribble application windows. It sets up the main
  * window and reads the config file as well as providing a menubar, toolbar
  * and statusbar. In initView(), your main view is created as the MDI child window manager.
  * Child windows are created in createClient(), which gets a document instance as it's document to
  * display whereby one document can have several views.The MDI child is an instance of KScribbleView,
  * the document an instance of KScribbleDoc.
  * KScribbleApp reimplements the methods that TDEMainWindow provides for main window handling and supports
  * full session management as well as keyboard accelerator configuration by using TDEAccel.
  * @see TDEMainWindow
  * @see TDEApplication
  * @see TDEConfig
  * @see TDEAccel
  *
  * @author Source Framework Automatically Generated by KDevelop, (c) The KDevelop Team.
  * @version KDevelop version 1.1 code generation
  */
public class KScribbleApp extends TDEMainWindow implements Resource {

   /** the configuration object of the application */
   private TDEConfig config;
   /** view is the main widget which represents your working area. The View
   * class should handle all events of the view widget.  It is kept empty so
   * you can create your view according to your application's needs by
   * changing the view class.
   */
   private KScribbleView view;
   /** doc represents your actual document and is created only once. It keeps
   * information such as filename and does the serialization of your files.
   */
   private KScribbleDoc doc;

   /** contains the recently used filenames */
   ArrayList recentFiles = null;

   // menus
   private TQPopupMenu pFileMenu;
   private TQPopupMenu pEditMenu;
   private TQPopupMenu pPenMenu;
   private TQPopupMenu pViewMenu;
   private TQPopupMenu pWindowMenu;
   private TQPopupMenu pHelpMenu;
   private TQPopupMenu pRecentFileMenu;

   private TQWorkspace pWorkspace;
   private TQPrinter printer;
   private int untitledCount = 0;
   private ArrayList pDocList;
   private TDEApplication kapp;
   private TDEIconLoader k = new TDEIconLoader();

    /** construtor of KScribbleApp, calls all init functions to create the application.
     */
   public KScribbleApp(TQWidget parent, String name)  {
      super(parent,name, 0);
      kapp = TDEApplication.kApplication();
      config=kapp.config();

      printer = new TQPrinter();
      untitledCount=0;
      pDocList = new ArrayList();
      setAcceptDrops(true);
      ///////////////////////////////////////////////////////////////////
      // call inits to invoke all other construction parts
      initMenuBar();
      initToolBar();
      initStatusBar();
      initKeyAccel();
      initView();

      readOptions();

      ///////////////////////////////////////////////////////////////////
      // disable menu and toolbar items at startup
      disableCommand(ID_EDIT_UNDO);

   }

   public KScribbleApp() {
	   this(null, null);
   }

   /** initializes the TDEActions of the application */
   protected void initKeyAccel() {

      TDEAccel keyAccel = new TDEAccel(this);

      // fileMenu accelerators
	  keyAccel.insert(TDEStdAccel.New, this, TQ_SLOT("slotFileNew()"));
	  keyAccel.insert(TDEStdAccel.Open, this, TQ_SLOT("slotFileOpen()"));
	  keyAccel.insert(TDEStdAccel.Save, this, TQ_SLOT("slotFileSave()"));
	  keyAccel.insert(TDEStdAccel.Close, this, TQ_SLOT("slotFileClose()"));
	  keyAccel.insert(TDEStdAccel.Print, this, TQ_SLOT("slotFilePrint()"));
	  keyAccel.insert(TDEStdAccel.Quit, this, TQ_SLOT("slotFileQuit()"));

      // editMenu accelerators
	  keyAccel.insert(TDEStdAccel.Cut, this, TQ_SLOT("slotEditCut()"));
	  keyAccel.insert(TDEStdAccel.Copy, this, TQ_SLOT("slotEditCopy()"));
	  keyAccel.insert(TDEStdAccel.Paste, this, TQ_SLOT("slotEditPaste()"));

      // help accelerator
	  keyAccel.insert(TDEStdAccel.Help, this, TQ_SLOT("appHelpActivated()"));

      keyAccel.readSettings();

   }


   void initMenuBar() {
     ///////////////////////////////////////////////////////////////////
     // MENUBAR

     pRecentFileMenu = new TQPopupMenu(this);
     connect(pRecentFileMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("slotFileOpenRecent(int)"));


     ///////////////////////////////////////////////////////////////////
     // menuBar entry file-Menu
     pFileMenu = new TQPopupMenu(this);


     pFileMenu.insertItem(KDE.BarIconSet("filenew"),i18n("&New"), ID_FILE_NEW,-1);
     pFileMenu.insertItem(KDE.BarIconSet("fileopen"),i18n("&Open..."), ID_FILE_OPEN,-1);
     pFileMenu.insertItem(i18n("Open &recent"), pRecentFileMenu, ID_FILE_OPEN_RECENT,-1);

     pFileMenu.insertItem(i18n("&Close"), ID_FILE_CLOSE,-1);
     pFileMenu.insertSeparator();
     pFileMenu.insertItem(KDE.BarIconSet("filefloppy") ,i18n("&Save"), ID_FILE_SAVE,-1);
     pFileMenu.insertItem(i18n("Save &As..."), ID_FILE_SAVE_AS,-1);
     pFileMenu.insertSeparator();
     pFileMenu.insertItem(KDE.BarIconSet("fileprint"), i18n("&Print..."), ID_FILE_PRINT,-1);
     pFileMenu.insertSeparator();
     pFileMenu.insertItem(i18n("E&xit"), ID_FILE_QUIT,-1);

     ///////////////////////////////////////////////////////////////////
     // menuBar entry edit-Menu
     pEditMenu = new TQPopupMenu(this);
     pEditMenu.insertItem(KDE.BarIconSet("edit-undo"), i18n("&Undo"), ID_EDIT_UNDO,-1);
     pEditMenu.insertSeparator();
     pEditMenu.insertItem(KDE.BarIconSet("editcut"), i18n("Cu&t"), ID_EDIT_CUT,-1);
     pEditMenu.insertItem(KDE.BarIconSet("editcopy"), i18n("&Copy"), ID_EDIT_COPY,-1);
     pEditMenu.insertItem(KDE.BarIconSet("editpaste"), i18n("&Paste"), ID_EDIT_PASTE,-1);
     pEditMenu.insertItem(KDE.BarIconSet("delete"),i18n("&Clear All"), ID_EDIT_CLEAR_ALL,-1);

     ///////////////////////////////////////////////////////////////////
     // menuBar entry pen-Menu
     pPenMenu = new TQPopupMenu();
     pPenMenu.insertItem(i18n("&Color"), ID_PEN_COLOR,-1);
     pPenMenu.insertItem(i18n("&Brush"), ID_PEN_BRUSH,-1);

     ///////////////////////////////////////////////////////////////////
     // menuBar entry view-Menu
     pViewMenu = new TQPopupMenu(this);
     pViewMenu.setCheckable(true);
     pViewMenu.insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR,-1);
     pViewMenu.insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR,-1);

     ///////////////////////////////////////////////////////////////////
     // menuBar entry window-Menu
     pWindowMenu = new TQPopupMenu(this);
     pWindowMenu.setCheckable(true);


     ///////////////////////////////////////////////////////////////////
     // menuBar entry helpMenu

      TQPopupMenu pHelpMenu = helpMenu(i18n("Java KScribble " + Main.VERSION + "\n\n(c) 2002 by\n" +
                                     "Ralf Nolden\nRalf.Nolden@post.rwth-aachen.de"),true);

     ///////////////////////////////////////////////////////////////////
     // MENUBAR CONFIGURATION
     // insert your popup menus with the according menubar entries in the order
     // they will appear later from left to right
     menuBar().insertItem(i18n("&File"), pFileMenu);
     menuBar().insertItem(i18n("&Edit"), pEditMenu);
     menuBar().insertItem(i18n("&Pen"), pPenMenu);
     menuBar().insertItem(i18n("&View"), pViewMenu);
     menuBar().insertItem(i18n("&Window"), pWindowMenu );
     menuBar().insertItem(i18n("&Help"), pHelpMenu);

     ///////////////////////////////////////////////////////////////////
     // CONNECT THE MENU SLOTS WITH SIGNALS
     // for execution slots and statusbar messages
     connect(pFileMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("commandCallback(int)"));
     connect(pFileMenu, TQ_SIGNAL("highlighted(int)"), TQ_SLOT("statusCallback(int)"));

     connect(pEditMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("commandCallback(int)"));
     connect(pEditMenu, TQ_SIGNAL("highlighted(int)"), TQ_SLOT("statusCallback(int)"));

     connect(pPenMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("commandCallback(int)"));
     connect(pPenMenu, TQ_SIGNAL("highlighted(int)"), TQ_SLOT("statusCallback(int)"));

     connect(pViewMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("commandCallback(int)"));
     connect(pViewMenu, TQ_SIGNAL("highlighted(int)"), TQ_SLOT("statusCallback(int)"));

     connect(pWindowMenu, TQ_SIGNAL("aboutToShow()" ), TQ_SLOT( "windowMenuAboutToShow()" ) );
     connect(pWindowMenu, TQ_SIGNAL("activated(int)"), TQ_SLOT("commandCallback(int)"));
     connect(pWindowMenu, TQ_SIGNAL("highlighted(int)"), TQ_SLOT("statusCallback(int)"));

   }


   private void initToolBar() {

      ///////////////////////////////////////////////////////////////////
      // TOOLBAR

      toolBar().insertButton(KDE.BarIcon("filenew"), ID_FILE_NEW, true, i18n("New File"),-1);
      toolBar().insertButton(KDE.BarIcon("fileopen"), ID_FILE_OPEN, true, i18n("Open File"),-1);
      toolBar().insertButton(KDE.BarIcon("filefloppy"), ID_FILE_SAVE, true, i18n("Save File"),-1);
      toolBar().insertButton(KDE.BarIcon("fileprint"), ID_FILE_PRINT, true, i18n("Print"),-1);
      toolBar().insertSeparator();
      toolBar().insertButton(KDE.BarIcon("editcut"), ID_EDIT_CUT, true, i18n("Cut"),-1);
      toolBar().insertButton(KDE.BarIcon("editcopy"), ID_EDIT_COPY, true, i18n("Copy"),-1);
      toolBar().insertButton(KDE.BarIcon("editpaste"), ID_EDIT_PASTE, true, i18n("Paste"),-1);
      toolBar().insertSeparator();
      toolBar().insertButton(KDE.BarIcon("pencolor"), ID_PEN_COLOR, true, i18n("Color"),-1 );
      toolBar().insertButton(KDE.BarIcon("penwidth"), ID_PEN_BRUSH, true, i18n("Width"),-1 );
      toolBar().insertSeparator();
      toolBar().insertButton(KDE.BarIcon("help"), ID_HELP_CONTENTS, TQ_SIGNAL("clicked()"),
             this, TQ_SLOT("appHelpActivated()"), true,i18n("Help"),-1);

      TQToolButton btnwhat = TQWhatsThis.whatsThisButton(toolBar());
      TQToolTip.add(btnwhat, i18n("What's this...?"));
      toolBar().insertWidget(ID_HELP_WHATS_THIS, btnwhat.sizeHint().width(), btnwhat);

      ///////////////////////////////////////////////////////////////////
      // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)


      ///////////////////////////////////////////////////////////////////
      // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
      // connect for invoking the slot actions
      connect(toolBar(), TQ_SIGNAL("clicked(int)"), TQ_SLOT("commandCallback(int)"));
      // connect for the status help on holing icons pressed with the mouse button
      connect(toolBar(), TQ_SIGNAL("pressed(int)"), TQ_SLOT("statusCallback(int)"));

   }


    /** sets up the kstatusBar for the main window by initialzing a statuslabel.
     */
   protected void initStatusBar() {
      ///////////////////////////////////////////////////////////////////
      // STATUSBAR
      // TODO: add your own items you need for displaying current application status.
      kstatusBar().insertItem(i18n("Ready."), ID_STATUS_MSG);
   }

    /** creates the centerwidget of the KTMainWindow instance and sets it as the view
     */
   protected void initView() {

      ////////////////////////////////////////////////////////////////////
      // here the main view of the KTMainWindow is created by a background box and
      // the TQWorkspace instance for MDI view.
      TQVBox view_back = new TQVBox( this );
      view_back.setFrameStyle( TQFrame.StyledPanel | TQFrame.Sunken );
      pWorkspace = new TQWorkspace( view_back, "" );
      connect(pWorkspace, TQ_SIGNAL("windowActivated(TQWidget)"), this, TQ_SLOT("setWndTitle(TQWidget)"));
//     setView(view_back);
      setCentralWidget(view_back);
   }

   void createClient(KScribbleDoc doc) {
     KScribbleView w = new KScribbleView(doc, pWorkspace,null,WDestructiveClose);
     w.installEventFilter(this);
     doc.addView(w);
     w.setIcon(kapp.miniIcon());
     if ( pWorkspace.windowList().isEmpty() ) // show the very first window in maximized mode
       w.showMaximized();
     else
       w.show();
   }

   void addRecentFile(String file) {


      if(recentFiles != null && recentFiles.contains(file))
       return; // it's already there

      if( recentFiles.size() < 5)
         recentFiles.add(0,file);
      else{
         recentFiles.remove(recentFiles.remove(recentFiles.size()-1));
         recentFiles.add(0,file);
      }

      pRecentFileMenu.clear();

      Iterator it = recentFiles.iterator();
      while (it.hasNext()) {
         pRecentFileMenu.insertItem((String)it.next());
      }

   }

    /** opens a file specified by commandline option
     */
   public void openDocumentFile(KURL url) {
      slotStatusMsg(i18n("Opening file..."));

      KScribbleDoc doc;
      String file = url.directory(false,true) + url.fileName();

      Iterator it = pDocList.iterator();

      while (it.hasNext()) {

         doc = (KScribbleDoc)it.next();
         // check, if document already open. If yes, set the focus to the first view
         if(doc.pathName().equals(file)) {

             KScribbleView view=doc.firstView();
             view.setFocus();
             return;
         }
      }
      doc = new KScribbleDoc();
      pDocList.add(doc);
      doc.newDocument();
      // Creates an untitled window if file is 0
      if(file == null || file.length() == 0) {
         untitledCount+=1;
         String fileName= i18n("Untitled" +untitledCount);
         doc.setPathName(fileName);
         doc.setTitle(fileName);
      }
      // Open the file
      else {

       String format= TQImageIO.imageFormat(file);
       if(!doc.openDocument(file,format))
         KMessageBox.error (this,i18n("Could not open document !"), i18n("Error !"),KMessageBox.Notify);
       addRecentFile(file);
      }
      // create the window
      createClient(doc);
      slotStatusMsg(i18n("Ready."));

   }


   public void openDocumentFile() {
   	openDocumentFile(new KURL());
	   return;
   }

   void windowMenuAboutToShow() {
      pWindowMenu.clear();

      pWindowMenu.insertItem(i18n("&New Window"), ID_WINDOW_NEW_WINDOW);
      pWindowMenu.insertItem(i18n("&Cascade"),
                             pWorkspace, TQ_SLOT("cascade()" ),new TQKeySequence(0) , ID_WINDOW_CASCADE );
      pWindowMenu.insertItem(i18n("&Tile"),
                             pWorkspace, TQ_SLOT("tile()" ),new TQKeySequence(0) , ID_WINDOW_TILE );

      if ( pWorkspace.windowList().isEmpty() ) {

         disableCommand(ID_WINDOW_NEW_WINDOW);
         disableCommand(ID_WINDOW_CASCADE);
         disableCommand(ID_WINDOW_TILE);
      }

      pWindowMenu.insertSeparator();

      ArrayList windows = pWorkspace.windowList();

      for ( int i = 0; i < windows.size(); ++i ) {
          int id = pWindowMenu.insertItem((i+1)+ ((TQWidget)windows.get(i)).caption(),
                                           this, TQ_SLOT( "windowMenuActivated( int )" ) );
          pWindowMenu.setItemParameter( id, i );
          pWindowMenu.setItemChecked( id, pWorkspace.activeWindow() == (TQWidget)windows.get(i) );
      }
   }

   void windowMenuActivated( int id ) {
      TQWidget w = (TQWidget)pWorkspace.windowList().get( id );
      if ( w != null )
         w.setFocus();
   }

   void setWndTitle(TQWidget qw){
      setCaption(pWorkspace.activeWindow() != null ? pWorkspace.activeWindow().caption() : "");
   }

   void enableCommand(int id_) {
     ///////////////////////////////////////////////////////////////////
     // enable menu and toolbar functions by their ID's
     menuBar().setItemEnabled(id_, true);
     toolBar().setItemEnabled(id_, true);
   }

   void disableCommand(int id_) {
     ///////////////////////////////////////////////////////////////////
     // disable menu and toolbar functions by their ID's
     menuBar().setItemEnabled(id_, false);
     toolBar().setItemEnabled(id_, false);
   }

   void commandCallback(int id_) {
      switch (id_) {
       case ID_FILE_NEW:
          slotFileNew();
            break;

       case ID_FILE_OPEN:
            slotFileOpen();
            break;

       case ID_FILE_SAVE:
            slotFileSave();
            break;

       case ID_FILE_SAVE_AS:
            slotFileSaveAs();
            break;

       case ID_FILE_CLOSE:
            slotFileClose();
            break;

       case ID_FILE_PRINT:
            slotFilePrint();
            break;

       case ID_FILE_QUIT:
            slotFileQuit();
            break;

       case ID_EDIT_CUT:
            slotEditCut();
            break;

       case ID_EDIT_COPY:
            slotEditCopy();
            break;

       case ID_EDIT_PASTE:
            slotEditPaste();
            break;

       case ID_EDIT_CLEAR_ALL:
            slotEditClearAll();
            break;

       case ID_PEN_BRUSH:
            slotPenBrush();
            break;

       case ID_PEN_COLOR:
            slotPenColor();
            break;

       case ID_VIEW_TOOLBAR:
            slotViewToolBar();
            break;

       case ID_VIEW_STATUSBAR:
            slotViewStatusBar();
            break;

       case ID_WINDOW_NEW_WINDOW:
          slotWindowNewWindow();
          break;

       default:
            break;
      }
   }

   void statusCallback(int id_) {
      switch (id_) {
         case ID_FILE_NEW:
            slotStatusHelpMsg(i18n("Creates a new document"));
            break;

         case ID_FILE_OPEN:
            slotStatusHelpMsg(i18n("Opens an existing document"));
            break;

         case ID_FILE_OPEN_RECENT:
            slotStatusHelpMsg(i18n("Opens a recently used file"));
            break;

         case ID_FILE_SAVE:
            slotStatusHelpMsg(i18n("Saves the currently active document"));
            break;

         case ID_FILE_SAVE_AS:
            slotStatusHelpMsg(i18n("Saves the currently active document as under a new filename"));
            break;

         case ID_FILE_CLOSE:
            slotStatusHelpMsg(i18n("Closes the currently active document"));
            break;

         case ID_FILE_PRINT:
            slotStatusHelpMsg(i18n("Prints out the actual document"));
            break;

         case ID_FILE_QUIT:
            slotStatusHelpMsg(i18n("Quits the application"));
            break;

         case ID_EDIT_UNDO:
            slotStatusHelpMsg(i18n("Reverts the last editing action"));
            break;

         case ID_EDIT_CUT:
            slotStatusHelpMsg(i18n("Cuts the selected section and puts it to the clipboard"));
            break;

         case ID_EDIT_COPY:
            slotStatusHelpMsg(i18n("Copies the selected section to the clipboard"));
            break;

         case ID_EDIT_PASTE:
            slotStatusHelpMsg(i18n("Pastes the clipboard contents to actual position"));
            break;

         case ID_EDIT_CLEAR_ALL:
            slotStatusHelpMsg(i18n("Clears the document contents"));
            break;

         case ID_PEN_BRUSH:
            slotStatusHelpMsg(i18n("Sets the pen width"));
            break;

         case ID_PEN_COLOR:
            slotStatusHelpMsg(i18n("Sets the current pen color"));
            break;

         case ID_VIEW_TOOLBAR:
            slotStatusHelpMsg(i18n("Enables/disables the toolbar"));
            break;

         case ID_VIEW_STATUSBAR:
            slotStatusHelpMsg(i18n("Enables/disables the statusbar"));
            break;

         case ID_WINDOW_NEW_WINDOW:
            slotStatusHelpMsg(i18n("Opens a new view for the current document"));
            break;

         case ID_WINDOW_CASCADE:
            slotStatusHelpMsg(i18n("Cascades all windows"));
            break;

         case ID_WINDOW_TILE:
            slotStatusHelpMsg(i18n("Tiles all windows"));
            break;

         default:
            break;
      }
   }

   void slotStatusHelpMsg(String text) {
      ///////////////////////////////////////////////////////////////////
      // change status message of whole statusbar temporary (text, msec)
      statusBar().message(text, 2000);
   }


    /** returns a pointer to the current document connected to the KTMainWindow instance and is used by
     * the View class to access the document object's methods
     */
   public KScribbleDoc getDocument() {
      return doc;
   }

    /** save general Options like all bar positions and status as well as the geometry and the recent file list to the configuration
     * file
     */
   protected void saveOptions() {
      config.setGroup("General Options");
      config.writeEntry("Geometry", size());
      config.writeEntry("Show Toolbar", toolBar().isVisible());
      config.writeEntry("Show Statusbar",statusBar().isVisible());
      config.writeEntry("ToolBarPos", (int) toolBar().barPos());
      String[] rf = new String[recentFiles.size()];
      rf = (String[]) recentFiles.toArray(rf);
      if (rf != null)
        config.writeEntry("Recent Files", rf);
   }



    /** read general Options again and initialize all variables like the recent file list
     */
   private void readOptions() {

      config.setGroup("General Options");

      // bar status settings
      boolean bViewToolbar = config.readBoolEntry("Show Toolbar", true);
      menuBar().setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
      if(!bViewToolbar) {
         toolBar("mainToolBar").hide();
      }

      boolean bViewStatusbar = config.readBoolEntry("Show Statusbar", true);
      menuBar().setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
      if(!bViewStatusbar) {
         toolBar("mainToolBar").hide();
      }

      // bar position settings
      int toolBarPos = TDEToolBar.Top;
      toolBarPos = config.readUnsignedNumEntry("ToolBarPos", TDEToolBar.Top);
      toolBar().setBarPos(toolBarPos);

     // initialize the recent file list
     //  commented out until fix is applied.
     recentFiles = config.readListEntry("Recent Files");
     Iterator it = recentFiles.iterator();
	 while (it.hasNext()) {
       pRecentFileMenu.insertItem((String) it.next());
	 }

      // Read the size information and resize from settings.
      TQSize size = new TQSize();
      config.readSizeEntry("Geometry",size);
      if(!size.isEmpty()) {
         resize(size);
      }
      else
         resize(400,350);

   }

    /** saves the window properties for each open window during session end to the session config file, including saving the currently
     * opened file by a temporary filename provided by TDEApplication.
     * @see KTMainWindow#saveProperties
     */
   protected void saveProperties(TDEConfig _cfg) {

   }


    /** reads the session config file and restores the application's state including the last opened files and documents by reading the
     * temporary files saved by saveProperties()
     * @see KTMainWindow#readProperties
     */
   protected void readProperties(TDEConfig _cfg) {
   }

    /** queryClose is called by KTMainWindow on each closeEvent of a window. Against the
     * default implementation (only returns true), this calles saveModified() on the document object to ask if the document shall
     * be saved if Modified; on cancel the closeEvent is rejected.
     * @see KTMainWindow#queryClose
     * @see KTMainWindow#closeEvent
     */
   protected boolean queryClose() {

      ArrayList saveFiles = new ArrayList();
      KScribbleDoc doc;

      if(pDocList.isEmpty())
         return true;

      Iterator it = pDocList.iterator();

      while (it.hasNext()) {
         doc = (KScribbleDoc)it.next();
         if(doc.isModified())
            saveFiles.add(doc.title());

      }

      if(saveFiles.isEmpty())
         return true;

      // lets load up a String array with the documents to save.
      String[] sf = new String[saveFiles.size()];
      for (int x = 0; x < saveFiles.size(); x++) {
         sf[x] = (String)saveFiles.get(x);
      }
      switch (KMessageBox.questionYesNoList(this,
         i18n("One or more documents have been modified.\nSave changes before exiting?"),sf))
      {
         case KMessageBox.Yes:

            Iterator itr = pDocList.iterator();

            while (itr.hasNext()) {
               doc = (KScribbleDoc)itr.next();
               if(doc.title().indexOf(i18n("Untitled")) > 0) {

                  slotFileSaveAs();
               }
               else {
                  if(!doc.saveDocument(doc.pathName())){
                     KMessageBox.error (this,i18n("Could not save the current document !"), i18n("I/O Error !"),KMessageBox.Notify);
                     return false;
                  }

               }


            }
            return true;
         case KMessageBox.No:
         default:
            return true;
      }

   }

    /** queryExit is called by KTMainWindow when the last window of the application is going to be closed during the closeEvent().
     * Against the default implementation that just returns true, this calls saveOptions() to save the settings of the last window's
     * properties.
     * @see KTMainWindow#queryExit
     * @see KTMainWindow#closeEvent
     */
   protected boolean queryExit() {
      saveOptions();
      return true;
   }

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////


   void slotFileNew() {
      slotStatusMsg(i18n("Creating new document..."));

      openDocumentFile();

      slotStatusMsg(i18n("Ready."));
   }

   void slotFileOpen() {
      slotStatusMsg(i18n("Opening file..."));

      String fileToOpen=KFileDialog.getOpenFileName(TQDir.currentDirPath(),
               KImageIO.pattern(KImageIO.Reading), this, i18n("Open File..."));
      if(fileToOpen != null && fileToOpen.length() > 0) {
         openDocumentFile(new KURL(fileToOpen));
      }

      slotStatusMsg(i18n("Ready."));
   }


   void slotFileSave() {
      slotStatusMsg(i18n("Saving file..."));
      KScribbleView m = (KScribbleView)pWorkspace.activeWindow();
      if( m != null)  {
         KScribbleDoc doc =  m.getDocument();
         if(doc.title().indexOf(i18n("Untitled")) > 0)
            slotFileSaveAs();
         else
            if(!doc.saveDocument(doc.pathName()))
               KMessageBox.error (this,i18n("Could not save the current document !"), i18n("I/O Error !"),KMessageBox.Notify);      }


      slotStatusMsg(i18n("Ready."));
   }

   void slotFileSaveAs() {
      slotStatusMsg(i18n("Saving file with a new filename..."));

      String newName=KFileDialog.getSaveFileName(TQDir.currentDirPath(),
                                  KImageIO.pattern(KImageIO.Writing), this, i18n("Save as..."));

	  if(newName != null) {
         KScribbleView m = (KScribbleView)pWorkspace.activeWindow();
         if( m != null ) {
            KScribbleDoc doc =  m.getDocument();

            String format=new TQFileInfo(newName).extension();
            format=format.toUpperCase();

			if(!doc.saveDocument(newName,format)) {
              KMessageBox.error (this,i18n("Could not save the current document !"), i18n("I/O Error !"),KMessageBox.Notify);
              return;
            }
            doc.changedViewList();
            setWndTitle(m);
         }

      }

      slotStatusMsg(i18n("Ready."));
   }

   void slotFileClose() {
      slotStatusMsg(i18n("Closing file..."));

      KScribbleView m = (KScribbleView)pWorkspace.activeWindow();
      if( m != null ) {
         KScribbleDoc doc=m.getDocument();
         doc.closeDocument();
      }


      slotStatusMsg(i18n("Ready."));
   }

   void slotFilePrint() {
      slotStatusMsg(i18n("Printing..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
      if ( m != null)
         m.print( printer );

      slotStatusMsg(i18n("Ready."));
   }

   void slotFileQuit() {
      slotStatusMsg(i18n("Exiting..."));
      saveOptions();
      // close the first window, the list makes the next one the first again.
      // This ensures that queryClose() is called on each window to ask for closing
      TDEMainWindow w;

      ArrayList memberlist = memberList();
      if(memberlist != null) {
         Iterator it = memberlist.iterator();
         while (it.hasNext()) {
            w = (TDEMainWindow)it.next();
            // only close the window if the closeEvent is accepted. If the user
            // presses Cancel on the saveModified() dialog,
            // the window and the application stay open.
            if(!w.close())
               break;
         }
      }
      slotStatusMsg(i18n("Ready."));
   }

   void slotFileOpenRecent(int id_) {
      slotStatusMsg(i18n("Opening file..."));

      KURL kurl = new KURL(pRecentFileMenu.text(id_));
//      openDocumentFile(pRecentFileMenu.text(id_));
      openDocumentFile(kurl);
      slotStatusMsg(i18n("Ready."));
   }

   void slotEditClearAll() {
     slotStatusMsg(i18n("Clearing the document contents..."));

     KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
     if ( m != null ){
       KScribbleDoc pDoc = m.getDocument();
       pDoc.editClearAll();
     }
     slotStatusMsg(i18n("Ready."));
   }

   void slotPenBrush() {
     slotStatusMsg(i18n("Setting brush width..."));

     // get one window with document for a current pen width
     ArrayList windows = pWorkspace.windowList();
     KScribbleView m = (KScribbleView)windows.get(0);
     KScribbleDoc pDoc = m.getDocument();
     int curr_width=pDoc.penWidth();

     // create the dialog, get the new width and set the pen width for all documents
     KPenBrushDlg dlg= new KPenBrushDlg(curr_width,this,"");
     if(dlg.exec() > 0){
       int width=dlg.getPenWidth();
       for ( int i = 0; i < windows.size(); ++i ) {
         m = (KScribbleView)windows.get(i);
         if ( m != null ) {
           pDoc = m.getDocument();
           pDoc.setPenWidth(width);
         }
       }
     }
     slotStatusMsg(i18n("Ready."));
   }

   void slotPenColor() {
     slotStatusMsg(i18n("Selecting pen color..."));

     TQColor myColor = new TQColor();
     int result = KColorDialog.getColor( myColor, this );
     if ( result == KColorDialog.Accepted )
     {
       ArrayList windows = pWorkspace.windowList();
       KScribbleDoc pDoc;
       KScribbleView m;
       for ( int i = 0; i < windows.size(); ++i )  {
         m = (KScribbleView)windows.get(i);
         if ( m != null) {
           pDoc = m.getDocument();
           pDoc.setPenColor(myColor);
         }
       }
     }
     slotStatusMsg(i18n("Ready."));
   }

   void slotEditUndo() {
      slotStatusMsg(i18n("Reverting last action..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
//      if ( m != null )
//          m.undo();

      slotStatusMsg(i18n("Ready."));
   }

    /** put the marked object into the clipboard and remove
     *	it from the document
     */
   void slotEditCut() {
      slotStatusMsg(i18n("Cutting selection..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
      if ( m != null )
         m.cutSelection();

      slotStatusMsg(i18n("Ready."));
   }

    /** put the marked text/object into the clipboard
     */
   public void slotEditCopy() {
      slotStatusMsg(i18n("Copying selection to clipboard..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
      if ( m != null)
         m.copySelection();

      slotStatusMsg(i18n("Ready."));
   }

    /** paste the clipboard into the document
     */
   public void slotEditPaste() {
      slotStatusMsg(i18n("Inserting clipboard contents..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
      if ( m != null ) {
         m.pasteSelection();
      }
      slotStatusMsg(i18n("Ready."));
   }

    /** toggles the toolbar
     */

   void slotViewToolBar() {
      slotStatusMsg(i18n("Toggle the toolbar..."));
      ///////////////////////////////////////////////////////////////////
      // turn Toolbar on or off
      if( menuBar().isItemChecked(ID_VIEW_TOOLBAR)) {
         menuBar().setItemChecked(ID_VIEW_TOOLBAR, false);
         toolBar("mainToolBar").hide();

      }
      else {
         menuBar().setItemChecked(ID_VIEW_TOOLBAR, true);
         toolBar("mainToolBar").show();
      }

      slotStatusMsg(i18n("Ready."));
   }

    /** toggles the statusbar
     */
   void slotViewStatusBar() {
      slotStatusMsg(i18n("Toggle the statusbar..."));
      ///////////////////////////////////////////////////////////////////
      //turn Statusbar on or off
      if( menuBar().isItemChecked(ID_VIEW_STATUSBAR)) {
         menuBar().setItemChecked(ID_VIEW_STATUSBAR, false);
         kstatusBar().hide();

      }
      else {
         menuBar().setItemChecked(ID_VIEW_STATUSBAR, true);
         kstatusBar().show();
      }

      slotStatusMsg(i18n("Ready."));
   }

   void slotWindowNewWindow() {
      slotStatusMsg(i18n("Opening a new application window..."));

      KScribbleView m = (KScribbleView) pWorkspace.activeWindow();
      if ( m != null ){
         KScribbleDoc doc = m.getDocument();
          createClient(doc);
      }

      slotStatusMsg(i18n("Ready."));
   }

    /** changes the statusbar contents for the standard label permanently, used to indicate current actions.
     * @param text the text that is displayed in the statusbar
     */
   public void slotStatusMsg(String text) {
      ///////////////////////////////////////////////////////////////////
      // change status message permanently
      kstatusBar().clear();
      kstatusBar().changeItem(text, ID_STATUS_MSG);
   }

   /** accepts drops and opens a new document
   for each drop */
   protected void dropEvent( TQDropEvent e){
      TQImage img = new TQImage();
      if ( TQImageDrag.decode(e, img) ) {
         KScribbleDoc doc = new KScribbleDoc();
         untitledCount+=1;
         String fileName= i18n("Untitled") + untitledCount;
         doc.setPathName(fileName);
         doc.setTitle(fileName);
         doc.newDocument();
         pDocList.add(doc);
         KPixmap tmp = new KPixmap();
         tmp.resize(img.size());
         tmp.convertFromImage(img);
         doc.setPixmap(tmp);
         doc.resizeDocument(tmp.size());
         doc.setModified();
         createClient(doc);
      }
   }
   /** accepts drag events for images */
   protected void dragEnterEvent( TQDragEnterEvent e){
     e.accept(TQImageDrag.canDecode(e));
   }


}
