/***************************************************************************
    smb4k_umount  -  This is the unmount utility of Smb4K.
                             -------------------
    begin                : Sa Sep 25 2004
    copyright            : (C) 2004-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,   *
 *   MA  02110-1301 USA                                                    *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if !defined(__FreeBSD__) && !defined(__sun)
#include <sys/statfs.h>
#elif defined(__sun)
#include <sys/types.h>
#include <sys/statvfs.h>
#elif defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/mount.h>
#endif

#ifdef __linux__
#include <sys/utsname.h>
#endif

#include <locale.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <iostream>
using namespace std;

#define SMB4K_UMOUNT_VERSION "0.14"


void info()
{
	cout << "This is smb4k_umount (version " << SMB4K_UMOUNT_VERSION << "), the unmount utility of Smb4K" << endl;
	cout << "Copyright (C) 2004-2007, Alexander Reinholdt" << endl;
	cout << endl;
	cout << "Usage:" << endl;
#ifndef __FreeBSD__
	cout << "  smb4k_umount {mode} {options} {mountpoint}" << endl;
#else
	cout << "  smb4k_umount {mountpoint}" << endl;
#endif
	cout << "  smb4k_umount --help" << endl;
	cout << "  smb4k_umount --version" << endl;
	cout << endl;
	cout << "Arguments:" << endl;
	cout << "  {mode}" << endl;
#ifndef __FreeBSD__
        cout << "    --no-suid\tsmb4k_umount is run in normal user mode, so smbumount or" << endl;
	cout << "\t\tumount.cifs is invoked." << endl;
	cout << "    --suid\tsmb4k_umount is run in super user mode, so umount is invoked." << endl;
	cout << "    -n\t\tThe same as the '--no-suid' argument." << endl;
	cout << "    -s\t\tThe same as the '--suid' argument." << endl;
	cout << endl;
	cout << "  {options}" << endl;
	cout << "    -t <fs>\tThe file system that should be used for unmounting. Only 'smbfs'" << endl;
        cout << "\t\tand 'cifs' are supported. All other file systems will result in" << endl;
	        cout << "\t\tan error. Please note, that this argument is mandatory." << endl;
#endif
#ifdef __linux__
	cout << "    -l\t\tPerform a lazy unmount. See the manual page of umount for" << endl;
	cout << "\t\tmore information. Please note, that this argument is only" << endl;
	cout << "\t\trecognized in super user mode, i.e. if '--suid' is supplied," << endl;
	cout << "\t\tand that you need kernel version 2.4.11 or later." << endl;
#endif
	cout << endl;
	cout << "  {mountpoint}\tThe path where the share is mounted to." << endl;
	cout << endl;
	cout << "  --help\tDisplay this help screen and exit." << endl;
	cout << "  --version\tDisplay the version information and exit." << endl;
	cout << endl;
}


void version()
{
	cout << "Version " << SMB4K_UMOUNT_VERSION << endl;
}


bool find_program( const char *name, char *path )
{
	const char *paths[] = { "/bin/", "/sbin/", "/usr/bin/", "/usr/sbin/", "/usr/local/bin/", "/usr/local/sbin/" };
	string file = "";

	for ( uint i = 0; i < sizeof( paths ) / sizeof( char * ); i++ )
	{
		string p( paths[i] );
		p.append( name );

		if ( access( p.c_str(), X_OK ) == 0 )
		{
			file.assign( p );
			break;
		}
	}

	if ( !strcmp( file.c_str(), "" ) )
	{
		cerr << "smb4k_umount: Could not find " << name << " binary" << endl;

		return false;
	}

	int len = strlen( file.c_str() ) + 1;
	strncpy( path, file.c_str(), len );
	path[len-1] = '\0';

	return true;
}


bool check_filesystem( const char *path, const char *fs )
{
	bool ok = false;

#if !defined(__sun)
	struct statfs filesystem;
#else
	struct statvfs filesystem;
#endif

#if !defined(__sun)
	if ( statfs( path, &filesystem ) == -1 )
#else
	if ( statvfs( path, &filesystem ) == -1 )
#endif
	{
		int err_code = errno;

		if ( err_code != EIO && err_code != EACCES )
		{
			// ok is still false
			cerr << "smb4k_umount: " << strerror( err_code ) << endl;
		}
		else
		{
			ok = true; // Bypass the check below, because it would yield ok == false
			           // and we want to be able to unmount broken shares as well.
		}

		return ok;
	}

#if !defined(__FreeBSD__) && !defined(__sun)
	// First entry is for CIFS, the second for SMBFS.
	if ( (uint)filesystem.f_type == 0xFF534D42 && !strncmp( fs, "cifs", strlen( fs )+1 ) )
	{
		ok = true;
	}
	else if ( (uint)filesystem.f_type == 0x517B && !strncmp( fs, "smbfs", strlen( fs )+1 ) )
	{
		ok = true;
	}
#elif defined(__FreeBSD__)
	if ( !strncmp( filesystem.f_fstypename, fs, strlen( fs ) ) )
	{
		ok = true;
	}
#elif defined(__sun)
	if ( (uint)filesystem.f_basetype == 0xFF534D42 && !strncmp( fs, "cifs", strlen( fs )+1 ) )
	{
		ok = true;
	}
	else if ( (uint)filesystem.f_basetype == 0x517B && !strncmp( fs, "smbfs", strlen( fs )+1 ) )
	{
		ok = true;
	}
#endif
	else
	{
		// ok is still false.
		cerr << "smb4k_umount: Wrong file system specified" << endl;
	}

	return ok;
}


int main( int argc, char *argv[], char *envp[] )
{
	// First of all, set the locale
	(void) setlocale( LC_ALL, "" );

	if ( argc < 2 )
	{
		info();
		exit( EXIT_FAILURE );
	}

	int new_argc = argc + 1;
	char *new_argv[new_argc];
	int index = 0;
	char path[255];
	path[0] = '\0';
	char *mountpoint = NULL;
#ifndef __FreeBSD__
	char *filesystem = NULL;
	bool normal_user_mode = true;
	bool have_user_mode = false;
#ifdef __linux__
	bool lazy_unmount = false;
#endif
#endif

	// Get the options that were passed:
	int c;

	while ( 1 )
	{
		int option_index = 0;

		static struct option long_options[] =
		{
			{ "help", 0, 0, 0 },
			{ "version", 0, 0, 0 },
#ifndef __FreeBSD__
			{ "suid", 0, 0, 0 },
			{ "no-suid", 0, 0, 0 },
#endif
			{ 0, 0, 0, 0 }
		};
#ifdef __linux__
		c = getopt_long( argc, argv, "t:nsl", long_options, &option_index );
#else
#ifndef __FreeBSD__
		c = getopt_long( argc, argv, "t:ns", long_options, &option_index );
#else
		c = getopt_long( argc, argv, "", long_options, &option_index );
#endif
#endif

		if ( c == -1 )
		{
			break;
		}

		switch ( c )
		{
			case 0:
			{
				int len = strlen( long_options[option_index].name ) + 1;
				char opt[len];
				opt[0] = '\0';
				(void) strncpy( opt, long_options[option_index].name, len );
				opt[len-1] = '\0';

				if ( !strncmp( opt, "help", len ) )
				{
					info();

					exit( EXIT_SUCCESS );
				}
				else if ( !strncmp( opt, "version", len ) )
				{
					version();

					exit( EXIT_SUCCESS );
				}
#ifndef __FreeBSD__
				else if ( !strncmp( opt, "suid", len ) )
				{
					// Enter super user mode

					normal_user_mode = false;
					have_user_mode = true;

					break;
				}
				else if ( !strncmp( opt, "no-suid", len ) )
				{
					// Enter normal user mode

					normal_user_mode = true;
					have_user_mode = true;

					break;
				}
#endif
				else
				{
					break;
				}

				break;
			}
#ifndef __FreeBSD__
			case 't':
			{
				// Get the length of the option argument:
				int len = strlen( optarg ) + 1;

				if ( strncmp( "smbfs", optarg, len) != 0 &&
				     strncmp( "cifs", optarg, len ) != 0 )
				{
					cerr << "smb4k_umount: File system " << optarg << " is not supported" << endl;
					exit( EXIT_FAILURE );
				}

				if ( !filesystem )
				{
					filesystem = new char[len];
					filesystem[0] = '\0';
					(void) strncpy( filesystem, optarg, len );
					filesystem[len-1] = '\0';
				}

				break;
			}
			case 's':
			{
				// Enter super user mode
				normal_user_mode = false;
				have_user_mode = true;

				break;
			}
			case 'n':
			{
				// Enter normal user mode
				normal_user_mode = true;
				have_user_mode = true;

				break;
			}
#endif
#ifdef __linux__
			case 'l':
			{
				// Initiate a lazy unmount. The umount binary
				// will complain, if '-l' is not supported.
				lazy_unmount = true;

				break;
			}
#endif
			case '?':
			{
				// Abort the program if an unknown option
				// is encountered:
				exit( EXIT_FAILURE );
			}
			default:
			{
				break;
			}
		};
	}

#ifndef __FreeBSD__
        // Error out if no user mode was specified.
	if ( !have_user_mode )
	{
		cerr << "smb4k_umount: No mode was specified" << endl;
		exit( EXIT_FAILURE );
	}

        // Error out if the user did not specify any file system.
	if ( !filesystem )
	{
		cerr << "smb4k_umount: No file system was specified" << endl;
		exit( EXIT_FAILURE );
	}

	if ( normal_user_mode )
	{
		int len = strlen( filesystem ) + 1;

		if ( !strncmp( "smbfs", filesystem, len ) )
		{
			if ( !find_program( "smbumount", path ) )
			{
				exit( EXIT_FAILURE );
			}
		}
		else
		{
			if ( !find_program( "umount.cifs", path ) )
			{
				exit( EXIT_FAILURE );
			}
		}

		len = strlen( path ) + 1;

		new_argv[index] = new char[len];
		new_argv[index][0] = '\0';
		(void) strncpy( new_argv[index], path, len );
		new_argv[index][len-1] = '\0';

		index++;
	}
	else
	{
		if ( !find_program( "umount", path ) )
		{
			exit( EXIT_FAILURE );
		}

		int len = strlen( path ) + 1;

		new_argv[index] = new char[len];
		new_argv[index][0] = '\0';
		(void) strncpy( new_argv[index], path, len );
		new_argv[index][len-1] = '\0';

		index++;

		len = strlen( "-t" ) + 1;

		new_argv[index] = new char[len];
		new_argv[index][0] = '\0';
		(void) strncpy( new_argv[index], "-t", len );
		new_argv[index][len-1] = '\0';

		index++;

		len = strlen( filesystem ) + 1;

		new_argv[index] = new char[len];
		new_argv[index][0] = '\0';
		(void) strncpy( new_argv[index], filesystem, len );
		new_argv[index][len-1] = '\0';

		index++;

#ifdef __linux__
		// Lazy unmount?
		if ( lazy_unmount )
		{
			len = strlen( "-l" ) + 1;

			new_argv[index] = new char[len];
			new_argv[index][0] = '\0';
			(void) strncpy( new_argv[index], "-l", len );
			new_argv[index][len-1] = '\0';

			index++;
		}
#endif
	}
#else
	// We do not need to care about the user mode and
	// we also need not to check for the file system,
	// since there is only one.
	if ( !find_program( "umount", path ) )
	{
		exit( EXIT_FAILURE );
	}

	int length = strlen( path ) + 1;
	new_argv[index] = new char[length];
	new_argv[index][0] = '\0';
	(void) strncpy( new_argv[index], path, length );
	new_argv[index][length-1] = '\0';

	index++;
#endif

  // Add the mount point:
	if ( optind < argc )
	{
		while ( optind < argc )
		{
			if ( !mountpoint )
			{
				if ( argv[optind][0] != '\057' )
				{
					cerr << "smb4k_umount: Argument " << optind << " is not a mount point" << endl;
					exit( EXIT_FAILURE );
				}
#ifndef __FreeBSD__
				if ( !check_filesystem( argv[optind], filesystem  ) )
#else
				if ( !check_filesystem( argv[optind], "smbfs" ) )
#endif
				{
					// Error message is given by check_filesystem()
					exit( EXIT_FAILURE );
				}

	 			int len = strlen( argv[optind] ) + 1;

				mountpoint = new char[len];
				mountpoint[0] = '\0';
				(void) strncpy( mountpoint, argv[optind], len );
				mountpoint[len-1] = '\0';

				optind++;
			}
			else
			{
				break;
			}
		}
	}

	if ( !mountpoint )
	{
		cerr << "smb4k_umount: No mount point was specified" << endl;
		exit( EXIT_FAILURE );
	}
	else
	{
		int len = strlen( mountpoint ) + 1;
		new_argv[index] = new char[len];
		new_argv[index][0] = '\0';
		(void) strncpy( new_argv[index], mountpoint, len );
		new_argv[index][len-1] = '\0';

		index++;
	}


	if ( index >= new_argc )
	{
		cerr << "smb4k_umount: There are too many arguments" << endl;
		exit( EXIT_FAILURE );
	}

	// Terminate new_argv:
	new_argv[index] = NULL;

	// Execute command:
	if ( execve( new_argv[0], new_argv, envp ) == -1 )
	{
		int err = errno;
		cerr << "smb4k_umount: " <<  strerror( err ) << endl;

		exit( EXIT_FAILURE );
	}

	return EXIT_SUCCESS;
}

