/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2004-05-16
 * Description : a plugin to set time stamp of picture files.
 *
 * Copyright (C) 2003-2005 by Jesper Pedersen <blackie@kde.org>
 * Copyright (C) 2006-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// C Ansi includes.

extern "C"
{
#include <sys/types.h>
#include <utime.h>
}

// TQt includes.

#include <tqtooltip.h>
#include <tqlayout.h>
#include <tqlabel.h>
#include <tqvbuttongroup.h>
#include <tqvgroupbox.h>
#include <tqgroupbox.h>
#include <tqhbox.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <tqfile.h>
#include <tqspinbox.h>
#include <tqgrid.h>
#include <tqpushbutton.h>
#include <tqframe.h>
#include <tqtoolbutton.h>

// KDE includes.

#include <kiconloader.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <tdeapplication.h>
#include <tdeaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <tdepopupmenu.h>
#include <kstandarddirs.h>
#include <kdatetimewidget.h>
#include <tdeconfig.h>
#include <tdemessagebox.h>

// LibKipi includes.

#include <libkipi/interface.h>
#include <libkipi/imageinfo.h>

// LibKExiv2 includes. 

#include <libkexiv2/kexiv2.h>

// Local includes.

#include "kpaboutdata.h"
#include "pluginsversion.h"
#include "timeadjustdialog.h"
#include "timeadjustdialog.moc"

namespace KIPITimeAdjustPlugin
{

class TimeAdjustDialogPrivate
{

public:

    TimeAdjustDialogPrivate()
    {
        add               = 0;
        subtract          = 0;
        exif              = 0;
        custom            = 0;
        syncEXIFDateCheck = 0;
        syncIPTCDateCheck = 0;
        exampleBox        = 0;
        adjustValGrp      = 0;
        adjustTypeGrp     = 0;
        infoLabel         = 0;
        exampleAdj        = 0;
        secs              = 0;
        minutes           = 0;
        hours             = 0;
        days              = 0;
        months            = 0;
        years             = 0;
        dateCreatedSel    = 0;
        interface         = 0;
        about             = 0;
        todayBtn          = 0;
    }

    TQRadioButton             *add;
    TQRadioButton             *subtract;
    TQRadioButton             *exif;
    TQRadioButton             *custom;

    TQToolButton              *todayBtn;

    TQCheckBox                *syncEXIFDateCheck;
    TQCheckBox                *syncIPTCDateCheck;

    TQVGroupBox               *exampleBox;
    TQVButtonGroup            *adjustValGrp;
    TQButtonGroup             *adjustTypeGrp;
    
    TQLabel                   *infoLabel;
    TQLabel                   *exampleAdj;
    
    TQSpinBox                 *secs;
    TQSpinBox                 *minutes;
    TQSpinBox                 *hours;
    TQSpinBox                 *days;
    TQSpinBox                 *months;
    TQSpinBox                 *years;
    
    TQDateTime                 exampleDate;

    KDateTimeWidget          *dateCreatedSel;

    KURL::List                images;

    KIPI::Interface          *interface;

    KIPIPlugins::KPAboutData *about;
};

TimeAdjustDialog::TimeAdjustDialog(KIPI::Interface* interface, TQWidget* parent)
                : KDialogBase(Plain, i18n("Adjust Time & Date"), Help|Ok|Cancel, 
                              Ok, parent, 0, true, true)
{
    d = new TimeAdjustDialogPrivate;
    d->interface = interface;

    TQVBoxLayout *vlay = new TQVBoxLayout(plainPage());

    // -- About data and help button ----------------------------------------

    d->about = new KIPIPlugins::KPAboutData(I18N_NOOP("Time Adjust"),
                                           0,
                                           TDEAboutData::License_GPL,
                                           I18N_NOOP("A Kipi plugin for adjusting time stamp of picture files"),
                                           "(c) 2003-2005, Jesper K. Pedersen\n"
                                           "(c) 2006-2008, Gilles Caulier");

    d->about->addAuthor("Jesper K. Pedersen", I18N_NOOP("Author"),
                        "blackie@kde.org");

    d->about->addAuthor("Gilles Caulier", I18N_NOOP("Developper and maintainer"),
                        "caulier dot gilles at gmail dot com");

    TQPushButton *helpButton = actionButton(Help);
    KHelpMenu* helpMenu     = new KHelpMenu(this, d->about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Plugin Handbook"), this, TQ_SLOT(slotHelp()), 0, -1, 0);
    helpButton->setPopup(helpMenu->menu());

    // -- Adjustment type ------------------------------------------------------------

    TQVGroupBox *adjGB = new TQVGroupBox(i18n("Adjustment Type"), plainPage());
    d->adjustTypeGrp  = new TQButtonGroup(1, TQt::Horizontal, adjGB);
    d->add            = new TQRadioButton(i18n("Add"), d->adjustTypeGrp);
    d->subtract       = new TQRadioButton(i18n("Subtract"), d->adjustTypeGrp);
    d->exif           = new TQRadioButton(i18n("Set file date to EXIF/IPTC creation date"), d->adjustTypeGrp);
    d->custom         = new TQRadioButton(i18n("Custom date"), d->adjustTypeGrp);

    d->adjustTypeGrp->setFrameStyle(TQFrame::NoFrame);
    d->adjustTypeGrp->setInsideMargin(0); 
    d->adjustTypeGrp->setRadioButtonExclusive(true);

    TQHBox *hbox       = new TQHBox(d->adjustTypeGrp);
    TQLabel *space1    = new TQLabel(hbox);
    space1->setFixedWidth(15);
    d->dateCreatedSel = new KDateTimeWidget(hbox);
    TQLabel *space2    = new TQLabel(hbox);
    space2->setFixedWidth(15);
    d->todayBtn       = new TQToolButton(hbox);   
    d->todayBtn->setIconSet(SmallIcon("today"));
    TQToolTip::add(d->todayBtn, i18n("Reset to current date"));
    new TQLabel(hbox);
    
    d->syncEXIFDateCheck = new TQCheckBox(i18n("Update EXIF creation date"), d->adjustTypeGrp);
    d->syncIPTCDateCheck = new TQCheckBox(i18n("Update IPTC creation date"), d->adjustTypeGrp);

    vlay->addWidget(adjGB);

    // -- Adjustments ------------------------------------------------------------

    d->adjustValGrp = new TQVButtonGroup(i18n("Adjustments"), plainPage());
    vlay->addWidget(d->adjustValGrp);

    TQWidget* grid        = new TQWidget(d->adjustValGrp);
    TQGridLayout* gridLay = new TQGridLayout(grid, 1, 7, spacingHint());
    gridLay->setColStretch( 2, 1 );
    gridLay->setColStretch( 5, 1 );

    TQLabel* label = new TQLabel( i18n("Hours:"), grid );
    d->hours      = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 0, 0 );
    gridLay->addWidget( d->hours, 0, 1 );

    label      = new TQLabel( i18n("Minutes:"), grid );
    d->minutes = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 0, 3 );
    gridLay->addWidget( d->minutes, 0, 4 );

    label   = new TQLabel( i18n("Seconds:"), grid );
    d->secs = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 0, 6 );
    gridLay->addWidget( d->secs, 0, 7 );

    label   = new TQLabel( i18n("Days:"), grid );
    d->days = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 1, 0 );
    gridLay->addWidget( d->days, 1, 1 );

    label     = new TQLabel( i18n("Months:"), grid );
    d->months = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 1, 3 );
    gridLay->addWidget( d->months, 1, 4 );

    label    = new TQLabel( i18n("Years:"), grid );
    d->years = new TQSpinBox( 0, 1000, 1, grid );
    gridLay->addWidget( label, 1, 6 );
    gridLay->addWidget( d->years, 1, 7 );

    // -- Example ------------------------------------------------------------

    d->exampleBox = new TQVGroupBox(i18n("Example"), plainPage());
    vlay->addWidget(d->exampleBox);

    d->infoLabel  = new TQLabel(d->exampleBox);
    d->exampleAdj = new TQLabel(d->exampleBox);
    d->exampleAdj->setAlignment(TQt::AlignCenter);

    vlay->addStretch();

    // -- Slots/Signals ------------------------------------------------------

    connect(d->adjustTypeGrp, TQ_SIGNAL( clicked(int) ),
            this, TQ_SLOT( slotAdjustmentTypeChanged() ));

    connect(d->secs, TQ_SIGNAL( valueChanged( int ) ), 
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->minutes, TQ_SIGNAL( valueChanged( int ) ),
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->hours, TQ_SIGNAL( valueChanged( int ) ),
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->days, TQ_SIGNAL( valueChanged( int ) ),
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->months, TQ_SIGNAL( valueChanged( int ) ),
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->years, TQ_SIGNAL( valueChanged( int ) ),
            this, TQ_SLOT( slotUpdateExample() ));

    connect(d->todayBtn, TQ_SIGNAL(clicked()),
            this, TQ_SLOT(slotResetDateToCurrent()));

    // -----------------------------------------------------------------------

    readSettings();
    slotAdjustmentTypeChanged();
}

TimeAdjustDialog::~TimeAdjustDialog()
{
    delete d->about;
    delete d;
}

void TimeAdjustDialog::slotHelp()
{
    TDEApplication::kApplication()->invokeHelp("timeadjust", "kipi-plugins");
}

void TimeAdjustDialog::slotResetDateToCurrent()
{
    d->dateCreatedSel->setDateTime(TQDateTime::currentDateTime());
}

void TimeAdjustDialog::closeEvent(TQCloseEvent *e)
{
    if (!e) return;
    saveSettings();
    e->accept();
}

void TimeAdjustDialog::slotCancel()
{
    saveSettings();
    KDialogBase::slotCancel();
}

void TimeAdjustDialog::readSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("Time Adjust Settings");

    int adjType = config.readNumEntry("Adjustment Type", 0);   // add by default.
    if (adjType == 0) d->add->setChecked(true);
    if (adjType == 1) d->subtract->setChecked(true);
    if (adjType == 2) d->exif->setChecked(true);
    if (adjType == 3) d->custom->setChecked(true);

    TQDateTime current = TQDateTime::currentDateTime();
    d->dateCreatedSel->setDateTime(config.readDateTimeEntry("Custom Date", &current));

    d->syncEXIFDateCheck->setChecked(config.readBoolEntry("Sync EXIF Date", true));
    d->syncIPTCDateCheck->setChecked(config.readBoolEntry("Sync IPTC Date", true));
    resize(configDialogSize(config, TQString("Time Adjust Dialog")));
}

void TimeAdjustDialog::saveSettings()
{
    TDEConfig config("kipirc");
    config.setGroup("Time Adjust Settings");

    int adjType = 0;              // add
    if (d->subtract->isChecked()) adjType = 1;
    if (d->exif->isChecked())     adjType = 2;
    if (d->custom->isChecked())   adjType = 3;
    config.writeEntry("Adjustment Type", adjType);

    config.writeEntry("Custom Date", d->dateCreatedSel->dateTime());

    config.writeEntry("Sync EXIF Date", d->syncEXIFDateCheck->isChecked());
    config.writeEntry("Sync IPTC Date", d->syncIPTCDateCheck->isChecked());
    saveDialogSize(config, TQString("Time Adjust Dialog"));
    config.sync();
}

void TimeAdjustDialog::setImages(const KURL::List& images)
{
    d->images.clear();
    int exactCount   = 0;
    int inexactCount = 0;

    for( KURL::List::ConstIterator it = images.begin(); it != images.end(); ++it ) 
    {
        KIPI::ImageInfo info = d->interface->info( *it );
        if (info.isTimeExact()) 
        {
            exactCount++;
            d->exampleDate = info.time();
            d->images.append(*it);
        }
        else
            inexactCount++;
    }

    if ( inexactCount > 0 ) 
    {
        TQString tmpLabel = i18n("1 image will be changed; ",
                                "%n images will be changed; ",
                                exactCount)
                         + i18n("1 image will be skipped due to an inexact date.",
                                "%n images will be skipped due to inexact dates.",
                                inexactCount );

        d->infoLabel->setText(tmpLabel);
    }
    else 
    {
        d->infoLabel->setText(i18n("1 image will be changed",
                                  "%n images will be changed",
                                  d->images.count()));
    }
    // PENDING(blackie) handle all images being inexact.

    slotUpdateExample();
}

void TimeAdjustDialog::slotUpdateExample()
{
    TQString oldDate = d->exampleDate.toString(TQt::LocalDate);
    TQDateTime date  = updateTime(KURL(), d->exampleDate);
    TQString newDate = date.toString(TQt::LocalDate);
    d->exampleAdj->setText(i18n("<b>%1</b><br>would, for example, "
                               "change into<br><b>%2</b>")
                           .arg(oldDate).arg(newDate));
}

void TimeAdjustDialog::slotAdjustmentTypeChanged()
{
    d->exampleBox->setEnabled(false);
    d->adjustValGrp->setEnabled(false);
    d->dateCreatedSel->setEnabled(false);
    d->todayBtn->setEnabled(false);
    d->syncEXIFDateCheck->setEnabled(false);
    d->syncIPTCDateCheck->setEnabled(false);

    if (d->add->isChecked() || d->subtract->isChecked())
    {
        d->exampleBox->setEnabled(true);
        d->adjustValGrp->setEnabled(true);
        d->syncEXIFDateCheck->setEnabled(true);
        d->syncIPTCDateCheck->setEnabled(true);
    }
    else if (d->custom->isChecked())
    {
        d->dateCreatedSel->setEnabled(true);
        d->todayBtn->setEnabled(true);
        d->syncEXIFDateCheck->setEnabled(true);
        d->syncIPTCDateCheck->setEnabled(true);
    }
}

void TimeAdjustDialog::slotOk()
{
    KURL::List  updatedURLs;
    TQStringList errorFiles;

    for( KURL::List::ConstIterator it = d->images.begin(); it != d->images.end(); ++it )
    {
        KURL url             = *it;
        KIPI::ImageInfo info = d->interface->info(url);
        TQDateTime dateTime   = info.time();
        dateTime             = updateTime(info.path(), info.time());
        info.setTime(dateTime);

        if (d->syncEXIFDateCheck->isChecked() || d->syncIPTCDateCheck->isChecked())
        {
            bool ret = true;
            if (!KExiv2Iface::KExiv2::isReadOnly(url.path()))
            {
                KExiv2Iface::KExiv2 exiv2Iface;

                ret &= exiv2Iface.load(url.path());
                if (ret)
                {
                    if (d->syncEXIFDateCheck->isChecked())
                    {
                        ret &= exiv2Iface.setExifTagString("Exif.Image.DateTime",
                            dateTime.toString(TQString("yyyy:MM:dd hh:mm:ss")).ascii());
                        ret &= exiv2Iface.setExifTagString("Exif.Photo.DateTimeOriginal",
                            dateTime.toString(TQString("yyyy:MM:dd hh:mm:ss")).ascii());
                    }
        
                    if (d->syncIPTCDateCheck->isChecked())
                    {
                        ret &= exiv2Iface.setIptcTagString("Iptc.Application2.DateCreated",
                            dateTime.date().toString(TQt::ISODate));
                        ret &= exiv2Iface.setIptcTagString("Iptc.Application2.TimeCreated",
                            dateTime.time().toString(TQt::ISODate));
                    }
    
                    ret &= exiv2Iface.save(url.path());

                    if (!ret)
                    {
                        kdDebug() << "Failed to save metadata to file " << url.fileName() << endl;
                    }
                }
                else
                {
                    kdDebug() << "Failed to load metadata from file " << url.fileName() << endl;
                }
            }
    
            if (!ret)
                errorFiles.append(url.fileName());
            else 
                updatedURLs.append(url);
        }

        // See B.K.O #138880: set the file acess and modification time (after than Exiv2 play with metadata).
        struct utimbuf ut;
        ut.modtime = dateTime.toTime_t();
        ut.actime  = dateTime.toTime_t();
        ::utime(TQFile::encodeName(url.path()), &ut);
    }

    // We use kipi interface refreshImages() method to tell to host than 
    // metadata from pictures have changed and need to be re-read.
    d->interface->refreshImages(d->images);

    if (!errorFiles.isEmpty() && (d->syncEXIFDateCheck->isChecked() || d->syncIPTCDateCheck->isChecked()))
    {
        KMessageBox::informationList(
                     kapp->activeWindow(),
                     i18n("Unable to set date and time like picture metadata from:"),
                     errorFiles,
                     i18n("Adjust Time & Date"));  
    }

    saveSettings();
    accept();
}

TQDateTime TimeAdjustDialog::updateTime(const KURL& url, const TQDateTime& time) const
{
    if (d->custom->isChecked())
    {
        return d->dateCreatedSel->dateTime();
    }
    else if (d->exif->isChecked())
    {
        KExiv2Iface::KExiv2 exiv2Iface;
        if ( !exiv2Iface.load(url.path()) )
            return time;

        TQDateTime newTime = exiv2Iface.getImageDateTime();
        if (newTime.isValid())
            return newTime;
        else
            return time;
    }
    else
    {
        int sign = -1;
        if (d->add->isChecked())
            sign = 1;

        TQDateTime newTime = time.addSecs( sign * ( d->secs->value()
                                                   + 60*d->minutes->value()
                                                   + 60*60*d->hours->value()
                                                   + 24*60*60*d->days->value() ) );
        newTime = newTime.addMonths( sign * d->months->value() );
        newTime = newTime.addYears( sign * d->years->value() );
        return newTime;
    }
}

}  // NameSpace KIPITimeAdjustPlugin
