/****************************************************************************
**
** Copyright (C) 2017 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt Quick Templates 2 module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qquickdeferredexecute_p_p.h"

#include <QtCore/qhash.h>
#include <QtQml/qqmlengine.h>
#include <QtQml/private/qqmldata_p.h>
#include <QtQml/private/qqmlcomponent_p.h>
#include <QtQml/private/qqmlobjectcreator_p.h>

#include <deque>

QT_BEGIN_NAMESPACE

namespace QtQuickPrivate {

static void cancelDeferred(QQmlData *ddata, int propertyIndex)
{
    auto dit = ddata->deferredData.rbegin();
    while (dit != ddata->deferredData.rend()) {
        (*dit)->bindings.remove(propertyIndex);
        ++dit;
    }
}

static bool beginDeferred(QQmlEnginePrivate *enginePriv, const QQmlProperty &property, QQmlComponentPrivate::DeferredState *deferredState)
{
    QObject *object = property.object();
    QQmlData *ddata = QQmlData::get(object);
    Q_ASSERT(!ddata->deferredData.isEmpty());

    int propertyIndex = property.index();
    int wasInProgress = enginePriv->inProgressCreations;

    for (auto dit = ddata->deferredData.rbegin(); dit != ddata->deferredData.rend(); ++dit) {
        QQmlData::DeferredData *deferData = *dit;

        auto bindings = deferData->bindings;
        auto range = bindings.equal_range(propertyIndex);
        if (range.first == bindings.end())
            continue;

        QQmlComponentPrivate::ConstructionState state;
        state.completePending = true;

        QQmlContextData *creationContext = nullptr;
        state.creator.reset(new QQmlObjectCreator(deferData->context->parent(), deferData->compilationUnit, creationContext));

        enginePriv->inProgressCreations++;

        std::deque<const QV4::CompiledData::Binding *> reversedBindings;
        std::copy(range.first, range.second, std::front_inserter(reversedBindings));
        state.creator->beginPopulateDeferred(deferData->context);
        for (const QV4::CompiledData::Binding *binding : reversedBindings)
            state.creator->populateDeferredBinding(property, deferData->deferredIdx, binding);
        state.creator->finalizePopulateDeferred();
        state.errors << state.creator->errors;

        deferredState->push_back(std::move(state));

        // Cleanup any remaining deferred bindings for this property, also in inner contexts,
        // to avoid executing them later and overriding the property that was just populated.
        cancelDeferred(ddata, propertyIndex);
        break;
    }

    return enginePriv->inProgressCreations > wasInProgress;
}

void beginDeferred(QObject *object, const QString &property,
                   QQuickUntypedDeferredPointer *delegate, bool isOwnState)
{
    QQmlData *data = QQmlData::get(object);
    if (data && !data->deferredData.isEmpty() && !data->wasDeleted(object) && data->context) {
        QQmlEnginePrivate *ep = QQmlEnginePrivate::get(data->context->engine());

        QQmlComponentPrivate::DeferredState state;
        if (beginDeferred(ep, QQmlProperty(object, property), &state)) {
            if (QQmlComponentPrivate::DeferredState *delegateState = delegate->deferredState())
                delegateState->swap(state);
        } else if (isOwnState) {
            delegate->clearDeferredState();
        }

        // Release deferred data for those compilation units that no longer have deferred bindings
        data->releaseDeferredData();
    } else if (isOwnState) {
        delegate->clearDeferredState();
    }
}

void cancelDeferred(QObject *object, const QString &property)
{
    QQmlData *data = QQmlData::get(object);
    if (data)
        cancelDeferred(data, QQmlProperty(object, property).index());
}

void completeDeferred(QObject *object, const QString &property, QQuickUntypedDeferredPointer *delegate)
{
    Q_UNUSED(property);
    QQmlComponentPrivate::DeferredState *state = delegate->deferredState();
    if (!state)
        return;

    QQmlData *data = QQmlData::get(object);
    if (data && !data->wasDeleted(object)) {
        QQmlComponentPrivate::DeferredState localState = std::move(*state);
        delegate->clearDeferredState();
        QQmlEnginePrivate *ep = QQmlEnginePrivate::get(data->context->engine());
        QQmlComponentPrivate::completeDeferred(ep, &localState);
    } else {
        delegate->clearDeferredState();
    }
}

} // QtQuickPrivate

QT_END_NAMESPACE
