/*
 * ipv6_addrconf.c  --  Address configuration routines, including duplicate
 *                      address detection (DAD), processing router
 *                      advertisements into addresses, and kicking user-level
 *                      DHCP.
 *
 * Copyright 1995 by Dan McDonald, Bao Phan, and Randall Atkinson,
 *	All Rights Reserved.  
 *      All Rights under this copyright have been assigned to NRL.
 */

/*----------------------------------------------------------------------
#       @(#)COPYRIGHT   1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

        This product includes software developed at the Information
        Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/malloc.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/time.h>
#include <sys/kernel.h>
#include <sys/mbuf.h>

#include <net/if.h>
#include <net/if_types.h>
#include <net/if_dl.h>
#include <net/route.h>

#include <netinet6/in6.h>
#include <netinet6/in6_var.h>
#include <netinet6/ipv6.h>
#include <netinet6/ipv6_var.h>
#include <netinet6/ipv6_icmp.h>
#include <netinet6/ipv6_addrconf.h>
#include <netinet6/in6_debug.h>

/*
 * External Globals
 */

extern struct in6_ifaddr *in6_ifaddr;
extern u_long v6d_retranstime;
extern int ipv6forwarding;

void send_nsolicit __P((struct rtentry *, struct ifnet *, struct in_addr6 *, int));
struct mbuf *get_discov_cluster __P((void));

static void send_rsolicit __P((struct ifnet *));

/*----------------------------------------------------------------------
 * Initialize addrconf.
 ----------------------------------------------------------------------*/

void
addrconf_init()
{
  timeout(addrconf_timer,NULL,v6d_retranstime * hz);
}

/*----------------------------------------------------------------------
 * Send a router solicitation out a certain interface.
 ----------------------------------------------------------------------*/

static void
send_rsolicit(ifp)
     struct ifnet *ifp;
{
  struct mbuf *solicit = NULL;
  struct ipv6 *header;
  struct ipv6_icmp *icmp;
  struct ipv6_moptions i6mo,*i6mop = NULL;
  struct in6_ifaddr *i6a;

  if ((solicit = get_discov_cluster()) == NULL)
    {
      DPRINTF(IDL_ERROR, ("Can't allocate mbuf in send_gsolicit().\n"));
      return;
    }
  header = mtod(solicit,struct ipv6 *);
  icmp = (struct ipv6_icmp *)(header + 1);/* I want the bytes after the hdr. */

  bzero(&i6mo,sizeof(struct ipv6_moptions));
  i6mo.i6mo_multicast_ifp = ifp;
  i6mo.i6mo_multicast_ttl = 255;
  i6mop = &i6mo;
  /* Find source link-local or use unspec. */
  for (i6a = in6_ifaddr; i6a; i6a = i6a->i6a_next)
    if (i6a->i6a_ifp == ifp && (i6a->i6a_addrflags & I6AF_LINKLOC) &&
	!(i6a->i6a_addrflags & I6AF_NOTSURE))
      break;
  if (i6a == NULL)
    {
      IN6_ADDR_ASSIGN(header->ipv6_src,0,0,0,0);
    }
  else header->ipv6_src = i6a->i6a_addr.sin6_addr;
  IN6_ADDR_ASSIGN(header->ipv6_dst,htonl(0xff020000),0,0,htonl(2));
  header->ipv6_versfl = htonl(0x6f000000);
  header->ipv6_hoplimit = 255;  /* Guaranteed to be intra-link if arrives with
				   255. */
  header->ipv6_nexthdr = NEXTHDR_ICMP;
  header->ipv6_length = ICMPV6_RSOLMINLEN;  /* For now. */
  icmp->icmp_type = ICMPV6_ROUTERSOL;
  icmp->icmp_code = 0;
  icmp->icmp_unused = 0;
  icmp->icmp_cksum = 0;

  /*
   * For now, just let ND run its normal course and don't include the link
   * extension.
   */

  solicit->m_len = solicit->m_pkthdr.len = 
                       header->ipv6_length + sizeof(*header);
  icmp->icmp_cksum = in6_cksum(solicit,NEXTHDR_ICMP,header->ipv6_length,
			       sizeof(struct ipv6));
  /*
   * NOTE: The solicit mbuf chain will have a NULL instead of a valid
   * socket ptr.  When ipv6_output() calls ipsec_output_policy(),
   * this socket ptr will STILL be NULL.  Sooo, the security
   * policy on outbound packets from here will == system security
   * level (set in ipsec_init() of netinet6/ipsec.c).  If your
   * system security level is paranoid, then you won't move packets
   * unless you have _preloaded_ keys for at least the ND addresses. 
   *  - danmcd rja
   */
  ipv6_output(solicit, NULL, IPV6_RAWOUTPUT, i6mop);
}

/*----------------------------------------------------------------------
 * Scan list if in6_ifaddrs and see if any are expired (or can go to
 * being unique).
 ----------------------------------------------------------------------*/

void
addrconf_timer(whocares)
     void *whocares;
{
  struct in6_ifaddr *i6a = in6_ifaddr;
  int s = splnet();

  while (i6a != NULL)
    {
      /*
       * Scan address list for all sorts of neat stuff.  Also, in6_ifaddr
       * may be a "prefix list" as well.  This will be difficult when
       * an router advert. advertises an on-link prefix, but I don't have
       * (for whatever reason) an address on that link.
       */
      if (i6a->i6a_preferred && i6a->i6a_preferred <= time.tv_sec)
	{
	  /*
	   * Do deprecation code.
	   */
	}
      if (i6a->i6a_expire && i6a->i6a_expire <= time.tv_sec)
	if (i6a->i6a_addrflags & I6AF_NOTSURE)
	  {
	    DPRINTF(IDL_MAJOR_EVENT,("Address appears to be unique.\n"));
	    i6a->i6a_addrflags &= ~I6AF_NOTSURE;
	    /*
	     * From what I can tell, addrs that survive DAD are
	     * permanent.  I won't mark as permanent, but I will zero
	     * expiration for now.
	     *
	     * If this is a link-local address, it may be a good idea to
	     * send a router solicit.  (But only if I'm a host.)
	     */
	    if (ipv6forwarding == 0 && IS_IN6_LINKLOC(i6a->i6a_addr.sin6_addr))
	      send_rsolicit(i6a->i6a_ifp);

	    i6a->i6a_expire = 0;
	    i6a->i6a_preferred = 0;
	  }
	else
	  {
	    /*
	     * Do address deletion, and nuke any routes, pcb's, etc.
	     * that use this address.
	     *
	     * As an implementation note, it's probably more likely than
	     * not that addresses that get deprecated (see above) will be
	     * moved off the master list, as that keeps them away from
	     * some things.  This is something we couldn't implement in time,
	     * however.
	     */
	  }
      i6a = i6a->i6a_next;
    }
  timeout(addrconf_timer,NULL,v6d_retranstime * hz);
  splx(s);
}

/*----------------------------------------------------------------------
 * Send multicast solicit for this address from all 0's.  Set timer such
 * that if address is still in in6_ifaddr list, it's good.
 ----------------------------------------------------------------------*/

void
addrconf_dad(i6a)
     struct in6_ifaddr *i6a;
{
  int s;
  struct rtentry dummy;
  struct in_addr6 allzero;

  DPRINTF(IDL_MAJOR_EVENT,("Sending DAD solicit!\n"));
  s = splnet();
  IN6_ADDR_ASSIGN(allzero,0,0,0,0);
  rt_key(&dummy) = i6a->i6a_ifa.ifa_addr;
  dummy.rt_gateway = NULL;
  /*
   * Set i6a flags and expirations such that it is NOT SURE about uniqueness.
   *
   * What about random delay?
   */
  i6a->i6a_addrflags |= I6AF_NOTSURE;  /* Might be done already. */
  i6a->i6a_preferred = 0;
  i6a->i6a_expire = time.tv_sec + v6d_retranstime;
  splx(s);
  /*
   * It would be nice if I delayed a random amount of time here.
   */
  send_nsolicit(&dummy,i6a->i6a_ifp,&allzero,1);
}
