// Persistence of Vision Ray Tracer Scene Description File
// File: mannatur.pov
// Vers: 3.61
// Desc: Bruno Cabasson's entry for Sept/Oct 2005 IRTC topic: The Opposites.
// Date: Oct 2005
// Auth: Bruno Cabasson
// This file is the main part of my entry to the current IRTC topic "The Opposites". My idea: wild nature 
// as a paradisiac island, opposed so-called civilized man and an ugly factory. The opposition is also 
// concerning the amount of code. The beautiful island is described by few lines, the factory required much more.
//
// Philosiopher's word: This could mean that too complicated things, even though they may be in man's capabilities
// and in the scope of his mind, are not natural and thus dammageable for himself. Happiness is simple. But simplicity 
// sometimes requires one (or more ..) life to reach.
//
// 
// I wanted:
//      few code and simple code, no endless marcos: don't need to be a pov-guru (unless some non-trivial textures,, but it was rather an exercice for myself...
//      stand-alone code: one single file
//      much effect (my opinion)
//      fast render
//      all my own: all non-POV-standard stuff is mine, and all made from scratch.
//      no use of third party complicated macros, tools, or environment. 
//      basic lighting: one single light_source (point_light), no radiosity, no photons.
//      standard colors.inc colors
//
// One can play with render parameters at the beginning of the file, which I hope to be easily understandable.
// The accompanying partial renders were obtained simply by turning some variables on/off or by copy/paste of a single indentifier.

#version 3.5;

#include "colors.inc"
#include "shapes.inc"
#include "stones.inc"
#include "metals.inc"
#include "textures.inc"
#include "finish.inc"

global_settings {assumed_gamma 1.0 max_trace_level 10}

// --------- G L O B A L    D E F I N I T I O N S  &  C O N T R O L S ---
// NOTA: In the gloabl scene, the origin <0,0,0> is in the center of the crater at level 0 (= sea bottom).

// ------------------------------------------------------- User definiable scene switches
// -------> Choose the items you want to render.
#declare _show_island = on;
#declare _use_isosurface = on; // _show_island must be on
#declare _show_lake = on;
#declare _show_water = on;
#declare _show_factory = on;
#declare _show_reactor = on; // _show_factory must be on
#declare _show_smoke = on; // _show_factory must be on
#declare _show_cage_mesh = on; // _show_reactor must be on

// Camera choices
#declare FINAL_CAMERA = 1;
#declare ABOVE_CRATER_CAMERA = 2;
#declare ABOVE_FACTORY_CAMERA = 3;
#declare FOCUS_CAMERA = 4;

// Focus choces: delta rotations of the camera & angle of view wrt final scene values
//  x = tilt = x-rotation, 
//  y = swing = y-rotation
//  z = angle of view
//
// I found rotations more convienient than cartesian coordinates to control the camera(s).
// here are some points of interest. You can define your own.
#declare SOUTH_SHORE = <-1.0, 0, 5>; // use _show_island = on;
#declare CRATER = <-5.8, -1.1, 4.8>;   // use _show_island = on;
#declare CHEMINEY = <3, -8.5, 8.3>;
#declare FENCE_POST = <CHEMINEY.x+3.075, CHEMINEY.y+0.34, 0.32>; 
#declare PIPES = <CHEMINEY.x+2.3, CHEMINEY.y+7, 2>; 
#declare BUILDING = <PIPES.x-1.2, PIPES.y+3.8, 2> ; 
#declare REACTOR = <BUILDING.x-2.8, BUILDING.y-2.9, 4.0>;
#declare GATE = <REACTOR.x+1.27, REACTOR.y+0.17, 0.45>;
#declare ALU_PIPE = <REACTOR.x+1.65, REACTOR.y+1.4, 0.5>;
#declare FACTORY = <CHEMINEY.x+0.5, CHEMINEY.y+4.2, 25>;
#declare PILLAR = <BUILDING.x+1.0, BUILDING.y+2.5, 3>;
#declare WALL = <PILLAR.x+1.1, PILLAR.y-4.2, 0.7>;

#declare CAMERA = FINAL_CAMERA; // -------> Choose your camera here
#declare FOCUS  = REACTOR;      // -------> Choose your focus here for FOCUS_CAMERA


// ------------------------------------------------------- constants
// Scene constants
#declare GLOBAL_SCALE = 100; // to avoid artifacts due to small numbers...
#declare GLOBAL_ORIENT = 60; // wrt y
#declare ISLAND_SCALE = 3;              // in combination with camera parameters (could keep size 1 and play with distance instead ...)
#declare ISLAND_RELATIVE_HEIGHT = 0.4;  // good aspect according to me.
#declare ISLAND_HEIGHT = ISLAND_RELATIVE_HEIGHT*ISLAND_SCALE;
#declare FACTORY_SCALE = 2;              // in combination with camera parameters (could keep size 1 and play with distance instead ...)
#declare FACTORY_POSITION = 5*x -5*z; // in x-z plane
#declare PLATFORM_ORIENT = 110; // wrt y
#declare HF_POINTS = 500;
#declare LAKE_ALTITUDE = ISLAND_HEIGHT - 0.2;
#declare SURFACE_LEVEL = 0.1;
// NB: the island will be about ISLAND_SCALE wide and exactly ISLAND_RELATIVE_HEIGHT*ISLAND_SCALE high at its highest point where
// the pigment P_SphericalTurb below evaluates to 1.


#declare ABOVE_ZOOM = 0.5;
declare  FINAL_LOCATION = 2.5*<0, 5, -9.0>;
#declare FINAL_ORIGIN_ROTATIONS = <abs(degrees(atan(FINAL_LOCATION.y/FINAL_LOCATION.z))), 2, 22>;
#declare FINAL_ROTATIONS = FINAL_ORIGIN_ROTATIONS + <3.8, -1, 0>;

#declare rnd1 = seed (1961);
#declare rnd2 = seed (1992+1996);
#declare rnd3 = seed (545);

// ------------------------------------------------------- pigment for the island
// Define a pigment that will be an height_field (or an isofurface, but render much slower).
// NB: it was defined in the x-y plane (with the K_Pigment camera, see at bottom of this source file).
//     The origin is at the center of the crater.
#declare P_SphericalTurb = pigment
{
    spherical                   // let's start from a simple spherical pattern
    scale 1.5                   // expand so that the turbulence next line acts in an interresting way
    scale -x-y                  // flip in x and y because I prefer the way it looks so ...
    turbulence 0.65 omega 0.55  // first tubulence to get the global shape of the island
    scale 2                     // expand space for turbulence scaling
    warp {turbulence 0.6 lambda 2.7 omega 0.5} // second turbulence to get a more realistic shape
    scale 1/2                   // un-expand space
    translate 0.62*x+0.73*y     // crater's position
    warp {black_hole 0, 2.5 strength 1.4 falloff 4 inverse} // the crater (will be a perfect circle). Center of the pigment.
    scale 5                     // expand space for turbulence scaling
    warp {turbulence 0.35 omega 0.5} // third turbulence for the crater (the whole island is also affected, but it makes it even more realistic ...)
    scale 1/5                   // un-expand space
    scale 1/1.5                 // retrieve original size (ie 1)
    color_map                   // after a few tries (only 15462228) , I got this color-map (only gray scale is interresting here) to combine with the spherical pattern and turbulences parameters
    {
        [0 Black]
        [0.65 0.65*White]
        [0.8 White]
        [1 0* White]
    }
}

// make a function with this pigment.
#declare fn_SphericalTurb = function {pigment {P_SphericalTurb}}

// derive a pigment with the same shape but with a stronger 3rd turbulence for the shore, and a transparency
// color_map.
#declare P_SphericalTurb_shore = pigment
{
    #local T = 0.1;
    #local dT = 0.06;
    spherical 
    scale 1.5
    scale -x-y
    turbulence 0.65 omega 0.55
    scale 2
    warp {turbulence 0.6 lambda 2.7 omega 0.5}
    scale 1/2
    translate 0.62*x+0.73*y
    warp {black_hole 0, 2.5 strength 1.4 falloff 4 inverse}
    scale 10
    warp {turbulence 0.75 omega 0.9}
    scale 1/10
    scale 1/1.5
    color_map
    {
        [T-dT White transmit 1]
        [T    White transmit 0]
    }
}

// --------------- T E X T U R E S ----------------------------------
#declare N_AluPipe = normal 
{
    radial rotate 90*x frequency 40
    scallop_wave
    slope_map
    {
        [0   <0, 0.1>]
        [0.5 <5, 1.5>]
        [0.5 <5, -1.5>]
        [1   <0, -0.1>]
    }
}
#declare N_AluRough = normal {agate 1.5 scale 0.005 scale 2*x}
                 
#declare T_AluTex = 
texture 
{
    Aluminum
    normal
    {
        average
        normal_map
        {  
            [1 N_AluPipe ]
            [1 N_AluRough]
        }
    }
}

#declare T_ReactorCagePilar =
texture {Rusty_Iron scale 1/15}
texture
{
    pigment {Gray05 transmit 0.45}
    normal {granite 0.2 scale 0.15}
    finish {specular 0.2 phong 0.5 phong_size 100}
}
texture {pigment {Firebrick transmit 0.7}}


#declare N_ReactorBody = 
normal {bumps 0.7 scale 0.3}

#macro T_ReactorBodyClean (p_ambient)
bozo scale 0.05
turbulence 0.6
texture_map
{
    [0.30 T_Copper_4A normal {N_ReactorBody} finish {ambient p_ambient}]
    [0.60 T_Chrome_1B normal {N_ReactorBody} finish {ambient p_ambient}]
    [0.75 T_Brass_3A normal {N_ReactorBody} finish {ambient p_ambient}]
}
#end

#macro T_VerticalDirt (p_coloration)
pigment 
{
    gradient y scale 2.2 translate 0.2*y 
    turbulence 0.2 
    scale 0.05*(x+z)
    color_map
    {
        [0.1 p_coloration transmit 0.2]
        [0.5 p_coloration transmit 0.8]
        [0.9 p_coloration transmit 0.1]
    }
}
#end

#declare T_DirtyGround = texture
{
     pigment
     {
        wrinkles scale 0.05
        color_map
        {
            [0 color (4*Gray15+2*ForestGreen)/6 transmit 0]
            [0.15 color (4*Gray15+2*ForestGreen)/6 transmit 0]
            [0.3 Gray25 transmit 0.8]
            [0.7 Gray05 transmit 0.1]
            [0.8 Gray05 transmit 0.0]
        }
    }
    normal {bumps 0.5 scale 0.003}
    finish {specular 0.05}
}

#declare T_Wall =
texture {T_Stone44}
texture {pigment{Firebrick transmit 0.3}}
texture {pigment {granite scale 0.3 scale 30*y  color_map {[0.1 transmit 0.05][0.3 transmit 0.7]}}}

#declare T_Building = 
texture
{
    cells scale 0.1
    texture_map
    {
        [0.49 T_Wall]
        [0.5 T_Chrome_4D]
        [0.6 T_Chrome_4D]
        [0.61 T_Wall]
    }
}

#local T_Pipe = 
texture
{
    pigment
    {
        gradient y
        turbulence 0.4
        lambda 2.4
        omega 0.8
        color_map
        {
            [0 Gray25]
            [0.15 Gray30]
            [0.28 VeryDarkBrown]
            [0.35 Firebrick]
            [0.55 Khaki]
            [0.85 Gray45]
            [1 Gray25]
        }
    }
    normal {bumps 0.1 scale 1/40}
    finish {brilliance 2 specular 0.3}
}
texture
{
    pigment
    {
        gradient y
        scale 2 warp {turbulence 1.2 omega 0.9} scale 1/2
        scale 0.1*(x+z)
        color_map
        {
            [ 0 Gray25 transmit 0.6]
            [ 0.25 color rgbf 3*<0.30, 0.20, 0.10, 0.5>]
            [ 0.75 color rgbf 3*<0.30, 0.20, 0.10, 0.5>]
            [ 1 Gray30 transmit 0.6]
        }
    }
    finish 
    {
        brilliance 3
        metallic
        specular 0.40
        roughness 1/20
    }
}

#declare T_Post =
texture {Rust scale 0.05}
texture 
{
    pigment{agate scale 0.08 color_map {[0.9 DarkSlateBlue transmit 0.7][0.95 MediumSlateBlue transmit 0.8]}} 
    finish {specular 0.4 roughness 0.03} 
    normal {bumps 0.3 scale 0.007}
}
texture {T_DirtyGround}

#declare T_Platform =
texture {pigment {cells turbulence 0.003 lambda 1.8 omega 0.8 scale 0.05 color_map {[0 Gray25][0.8 NewTan]}}}
texture 
{
    pigment
    {
        crackle scale 0.15
        turbulence 0.2 lambda 2.5 omega 0.5
        color_map
        {
            [0 Black]
            [0.005 Black]
            [0.005 transmit 0.9]
        }
    }
}
texture {T_DirtyGround}

#declare T_Cheminey = 
texture {pigment{brick color Gray45, color Gray60 scale 0.02 warp{cylindrical}}}
texture {pigment{wrinkles scale 0.25 color_map {[0 transmit 0.5]}}}
texture
{
    pigment
    {
        gradient -y
        scale 4
        warp {turbulence 0.4 lambda 2.5 omega 0.5}
        scale 1/4
        warp {turbulence 0.1 omega 1}
        scale 1/8*(x+z)
        color_map
        {
            [0.0 Black]
            [0.2 Gray55]
            [0.6 White transmit 1]
            [0.85 MediumSeaGreen transmit 0.5]
            [0.88 DarkTan transmit 0.40]
            [0.92 DarkBrown transmit 0.3]
            [0.94 Gray25 transmit 0]
        }
    }
}

#declare T_Pedestal = // designed for vertical posts.
texture {T_Brass_1A} // background metal
texture
{
    
    pigment
    {
        granite 
        turbulence 0.5 omega 0.9
        scale 1/15*(x+z)
        scale 5*y
        scale 6
        warp {turbulence 0.2}
        scale 1/6
        color_map
        {
            [0.10 Salmon transmit 0.05]
            [0.15 MandarinOrange]
            //[0.25 MandarinOrange]
            [0.20 Scarlet]
            [0.37 color (3*Gray15+YellowGreen)/4]
            [0.45 Gray15 transmit 0.05]
            [0.85 Gray15 transmit 0.2]
            [0.96 DarkSlateBlue transmit 0.1]
            //[0.96 Black]
        }
    }
    normal {agate 0.1 scale 0.004 scale 5*y}
    scale 5
}
//texture {T_DirtyGround scale 4 scale 6*y}


#declare M_Ocean = material
{
    texture
    {
        pigment {color rgb 1 transmit 0.95}
        normal {bozo 1.5/2 scale 0.01 scale 3*x rotate -10*y}
        finish {reflection{0.05, 0.2 fresnel}}
    }
    texture {pigment {P_SphericalTurb_shore rotate 90*x  scale ISLAND_SCALE*(x+z)}}
    //texture {pigment{agate scale 0.3 scale 4*x rotate -10*y color_map {[0.9995 White transmit 1][1 White transmit 0.4]}}}
    texture {pigment{wrinkles warp{turbulence 0.6 omega 0.8} scale 3*x scale 1/12 rotate -10*y color_map {[0.75 White transmit 1][1 White transmit 0]}}}
    interior
    {
        ior 1.3
        fade_distance 0.06
        fade_power 2
        media {absorption 25*(1-(NavyBlue+3*MediumAquamarine)/4)}
    }
}

#declare M_Lake = material
{
    texture
    {
        pigment {color rgb 1 transmit 0.95}
        normal {bozo 0.8 scale 0.001 turbulence 0.8}
        finish {specular 0.15 reflection{0.2, 0.65 fresnel on}}
    }
    interior
    {
        ior 1.3
        fade_distance 0.015
        fade_power 1
        media
        {
            absorption 15/GLOBAL_SCALE*(1-SeaGreen)
            emission color 3/GLOBAL_SCALE*SeaGreen
        }
    }
}

#declare T_Ocean_Bottom = texture
{
    pigment {granite scale 4 turbulence 0.3 color_map {[0 White] [0.8 Tan] [0.9 Green]}}
}

#declare T_Forest = texture
{
    pigment 
    {
        wrinkles scale 0.2 turbulence 0.5 omega 0.8
        color_map
        {
            [0 color Gray40]
            [0.05 color HuntersGreen]
            [0.35 color HuntersGreen]
            [0.45 color MediumForestGreen]
            [0.55 color ForestGreen]
            [0.65 color HuntersGreen]
            [0.75 color IndianRed]
            [0.85 color Tan]
        }
    }
    normal
    {
        average
        normal_map
        {
            [ 1 bozo 5 sine_wave scale 0.15]
            [ 1 bozo 40  scale 0.03]
        }
    }
    scale 0.1
}

#declare T = 0.85;
#declare T_Rock1 = texture
{
    pigment
    {
        wrinkles scale 0.05
        turbulence 0.3
        pigment_map
        {
            [0   color 0.8*White+Tan/2]
            [0.85 color 0.8*White+Tan/2]
            [0.9 Very_Light_Purple]
        }
    }
    normal {agate 0.4 scale 0.15 scale 7*(x+z)}
    scale 0.1
}

#declare T_Rock2 = texture
{
    pigment
    {
        gradient y/5+(x+z) scale 0.005 turbulence 0.4 lambda 2 omega 0.7
        color_map
        {
            [0 color Khaki]
            [0.15 color Firebrick]
            [0.2 color MediumWood]
            [0.3 color VeryDarkBrown]
            [0.33 color VeryDarkBrown]
            [0.33 color Gray15]
            [0.35 color Gray15]
            [0.38 color Gray05]
            [0.40 color DarkTurquoise]
            [0.42 color MediumWood]
            [0.80 color VeryDarkBrown]
            [0.95 color VeryDarkBrown]
            [0.96 color Firebrick]
            [0.98 color Firebrick]
        }
    }
    
    scale 0.2
    scale 6*y
}

#declare TM_Island = texture_map
{
    [0.15 T_Ocean_Bottom]
    [0.25 T_Forest ]
    [0.82 T_Forest ]
    [0.85 T_Rock1]
    [0.90 T_Rock1 ]
    [0.95 T_Rock2 ]
}

#declare T_Slope = texture
{
    slope {-y, 0, 0.5}
    texture_map {TM_Island}
}

#declare T_Slope_Simple = texture
{
    pigment
    {
        slope {-y, 0, 0.5}
        pigment_map
        {
            [0.15 Tan]
            [0.23 ForestGreen ]
            [0.72 ForestGreen ]
            [0.75 White]
            [0.90 White ]
            [0.95 VeryDarkBrown ]
        }
    }
}

#macro T_Slate (p_use_normal, p_use_finish) 
//texture
//{
    pigment {color (14*Gray05+1*DarkSlateBlue+DarkBrown)/16}
    #if (p_use_normal) normal {granite 0.03 scale 0.5} #end
    #if (p_use_finish) finish {specular 0.05 roughness 0.04 reflection {0.03}} #end
//}
#end

#macro T_Rust1 (p_use_normal, p_use_finish) //, p_seed, p_nb_spots)
//texture
//{
    pigment
    {
        granite 
        turbulence 0.4 omega 0.8
        scale 20
        warp {turbulence 0.8 omega 1.2}
        scale 1/20
        color_map
        {
            [0.10 Salmon transmit 0.05]
            [0.11 MandarinOrange]
            [0.15 MandarinOrange]
            //[0.25 MandarinOrange]
            [0.35 Scarlet]
            [0.39 Scarlet]
            [0.40 color (3*Gray15+YellowGreen)/4]
            [0.55 Gray15 transmit 0.1]
            [0.75 Gray15 transmit 0.3]
            [0.80 DarkSlateBlue transmit 0.2]
        }
    }
    #if (p_use_normal) normal {granite -0.2 scale 0.2} #end
    #if (p_use_finish) finish {specular 0.5 roughness 0.05} #end
    scale 8
//}
#end

#declare T_Door = texture
{
    wrinkles
    scale 0.7
    texture_map
    {
        [0.2 T_Slate(true, true)]
        [0.9 T_Rust1(false, false) scale 1.5]
    }
}

#macro T_DoorFrame (p_use_normal, p_use_finish) 
//texture
//{
    pigment
    {
        agate scale 0.05 
        warp {turbulence 0.4 omega 0.9}
        scale 4*z 
        color_map
        {
            [0 0.3*Firebrick]
            [0.25 0.2*Brown]
            [0.55 0.2*MediumSeaGreen]
            [0.85 Gray05]
        }
    } 
    #if (p_use_normal) normal {bozo 0.8 scale 0.003 scale 8*z} #end
    #if (p_use_finish) finish {specular 0.05} #end
//}
#end


#declare T_Hinge = texture
{
    #local _ratio = 0.35; // [0..1]
    gradient y
    texture_map
    {
        [_ratio/2 T_Door]
        [_ratio/2 T_Post scale 5 scale 10*y]
        [1-_ratio/2 T_Post scale 5 scale 10*y]
        [1-_ratio/2 T_Door]
    }
}

#declare T_Brick = texture
{
    pigment
    {
        granite scale 0.2
        turbulence 0.3 omega 0.8
        color_map
        {
            [0.2 0.1*White]
            [0.25 Coral]
            [0.25 color (6*Firebrick+Red)/7]
            [0.75 color (6*Firebrick+Red)/7]
            [0.9 Gray20]
        }
    }
    //normal {agate 0.6 scale 0.2 scale 10*(x.z)}
}

#declare T_Mortar = texture
{
    pigment {Gray20}
    normal {agate 0.6 scale 0.08}
}

#macro P_BrickPattern (_roundness)
//pigment/normal
//{
    Irregular_Bricks_Ptrn (0.1, 2, 0.15, _roundness) //scale 0.15
    scale 8 warp {turbulence 0.2} scale 1/8
//}
#end

#declare P1 = pigment{P_BrickPattern (0.3)}
#declare N1 = normal{P_BrickPattern (0.3)}
#declare N2 = normal
{
    P_BrickPattern (0.3) 
    slope_map 
    {
        [0   <-4, 0>]
        [0.1 <-4, 1>]
        [1   <0, 5>]
    }
}

#declare N3 = normal{average normal_map{[1 N2] [0.7 agate 0.6 scale 0.07 scale 6*(x+z)]}}

#declare T_WallBase = texture
{
    pigment_pattern {P1}
    texture_map
    {
        [0 T_Mortar]
        [0.5 T_Brick normal {N3}]
    }
    scale 0.2
}

#declare P_WallCrack = pigment
{
    wrinkles scale 0.03
    color_map
    {
        [0.2 DarkOliveGreen]
        [0.4 0.02*White]
        [0.6 Gray05]
        [0.8 DarkOliveGreen]
    }
}

#declare T_WallConcrete =
texture 
{
    pigment 
    {
        granite
        color_map 
        {
            [0.0 0.6*Plum ]
            [1.0 0.5*(3*Plum+PaleGreen)/4 ]
        }
        scale 0.1
    }
    normal {granite 0.7 scale 0.1 } // normal follows pigment
}
texture 
{
    pigment 
    {
        wrinkles
        scale 1/3
        warp {turbulence 0.4 octaves 4} 
        color_map 
        {
            [0.0 White transmit 0.8]
            [1.0 color 0.5*Flesh transmit 0.3]
        }
    }
}
texture 
{
    pigment
    {
        pigment_pattern {crackle scale 10 scale 1/2 warp {turbulence 0.3} scale 2}
        pigment_map
        {
            [0.015/6 P_WallCrack]
            [0.02/6 transmit 1] 
        }
    }
}

#declare T_WallConcreteThickness = texture
{
    pigment
    {
        bozo scale 0.002
        color_map
        {
            [0.55 DustyRose]
            [0.6 0.5*DarkGreen]
            [0.7 0.5*ForestGreen]
            [0.8 Gray05]
        }
    }
    normal {bozo 2 scale 0.005}
    finish {diffuse 0.4}
}

#macro PP_Crack (_solid) 
pigment_pattern 
{
    crackle 
    #if (_solid) solid #end 
    scale 1/2 warp {turbulence 0.15 octaves 7 omega 0.7}
    scale 2
}
#end

#declare T_WallCell = texture
{
    pigment_pattern {PP_Crack(false)}
    texture_map
    {
        [0.05 T_WallConcreteThickness]
        [0.07 T_WallBase]
    }
}

#macro T_FactoryWall(_density, _accuracy) 
// Only cells between _threshold and _threshold+range are 'transparent' and show the wall behind.
// Nota: _range should be chosen small enough to avoid contiguous cells that are close in value to be 'transparent'. 0.005 is OK.
//texture
//{
    #local _start = 0.25;
    #local _end = 0.75;
    #local _f = _end - _start;
    pigment_pattern {PP_Crack(true)}
    texture_map
    {
        [0 T_WallConcrete]
        #local i=4;
        #while (i<256*_density)   
            [_start+_f*i/(256*_density) T_WallConcrete]
            [_start+_f*i/(256*_density) T_WallCell]
            [_start+_f*i/(256*_density)+_accuracy T_WallCell]
            [_start+_f*i/(256*_density)+_accuracy T_WallConcrete]
        #local i=i+4;
        #end
    }
//}
#end

#declare T_CrackedWall = texture {T_FactoryWall(0.5, 1/1000)}

#declare T_WallHat = texture
{
    T_Grnt18
    normal
    {
        average
        normal_map
        {
            [1 granite 1.2 turbulence 0.0 scale 1/2]
            [2 quilted 0.8 control0 1 control1 1 turbulence 0.2 scale 1/3]
        }
    }
    
    scale 1/5
}
texture {T_DirtyGround scale 4 scale 4*y}


// --------------- S H A P E S & O B J E C T S ------------------
// ------------------------------------------------------- island
#if (_show_island)
#if (!_use_isosurface)
#declare O_Island_hf = union
{

    height_field
    {
        function HF_POINTS HF_POINTS {fn_SphericalTurb ((x-0.5)*6,-(y-0.5)*6,0).red} // -0.5 to center the pigment, *6 to shrink and have all the pigment within the height_field
        smooth
        water_level 0.001
        translate <-0.5, 0, -0.5>       // the bottom of height_fields are 
        scale 6*(x+z)                   // *6 in x and z to compensate the *6 in the variable substitution in the function above
        scale ISLAND_RELATIVE_HEIGHT*y  // adjust the height of the island to desired value.
    }
    plane {y, 0.0015} // sea bottom
    texture {T_Slope}
}
#else
    // The isosurface version of the island. It works, though it renders significantly longer than the height_field 
    // version above. 
    // For reasons concerning high gradient values around the crater and for render time, I use 2 isosurfaces.
    // The first is the whole island     // for which a moderate gradient is enough but generates holes in the
    // region of the crater, and a second one, contained by a sphere around the crater, but with higher value of
    // the gradient to avoid holes, and translated a very little bit vercically so that rays hit this one.
    #declare FoundMaxGradient = 5;
    #declare MinFactor = 0.6;
    #declare limit = <-10, -0.5, -10>;
    #declare O_IslandLand_iso = isosurface
    {
        function {y - ISLAND_RELATIVE_HEIGHT*fn_SphericalTurb (x,z,0).red}
        evaluate FoundMaxGradient*MinFactor,  sqrt(FoundMaxGradient/(FoundMaxGradient*MinFactor)),  0.7
        accuracy 0.001
        contained_by {box {-limit, limit}}
        texture {T_Slope}
    }
    #declare FoundMaxGradient = 50;
    #declare O_IslandCraterDetail_iso = isosurface
    {
        function {y - ISLAND_RELATIVE_HEIGHT*fn_SphericalTurb (x,z,0).red}
        evaluate FoundMaxGradient*MinFactor,  sqrt(FoundMaxGradient/(FoundMaxGradient*MinFactor)),  0.7
        accuracy 0.00005
        contained_by {sphere {ISLAND_RELATIVE_HEIGHT*y, 0.36}}
        texture {T_Slope}
    }
    
    #declare O_Island_iso = union
    {
        object {O_IslandLand_iso}
        object {O_IslandCraterDetail_iso translate 0.001*y}
    }
    
    #end // _use_isosurface
#end // _show_island

// ------------------------------------------------------- lake
// The lake is a prism with a cubic-spline that follows the contour of the crater. I used the orthographic K_Above
// camera placed above the crater and the 'spilin' utility. I resized their respective windows so they have same dimensions
// and switched alternatively from both applications (ALT + TAB on Windows) and kept the cursor in place while followin the crater.
#if (_show_lake)
#declare O_IslandLake = prism // normalized
{
	cubic_spline
	0,
	0.250000,
	188,
	<0.734450,0.998912>, <0.743517,0.986822>, <0.751073,0.974732>, <0.755606,0.962643>,	<0.766185,0.955087>,
	<0.775252,0.942997>, <0.781297,0.932418>, <0.787342,0.915795>, <0.787342,0.906728>, <0.769207,0.900683>,
	<0.754095,0.896149>, <0.743517,0.891616>, <0.731427,0.885571>, <0.723871,0.882548>, <0.713293,0.881037>,
	<0.698180,0.882548>, <0.683068,0.882548>, <0.670979,0.882548>, <0.657378,0.887082>, <0.643777,0.887082>,
	<0.636221,0.885571>, <0.639243,0.874992>, <0.643777,0.867436>, <0.637732,0.858369>, <0.637732,0.847790>,
	<0.633198,0.834190>, <0.631687,0.825122>, <0.639243,0.820589>, <0.652844,0.816055>, <0.670979,0.811521>,
	<0.686091,0.806988>, <0.696669,0.800943>, <0.704225,0.797920>, <0.717826,0.782808>, <0.726893,0.776763>,
	<0.735961,0.763163>, <0.735961,0.755606>, <0.725382,0.745028>, <0.713293,0.742006>, <0.704225,0.738983>,
	<0.702714,0.731427>, <0.705736,0.726893>, <0.722360,0.722360>, <0.737472,0.722360>, <0.745028,0.716315>,
	<0.758629,0.710270>, <0.778275,0.708759>, <0.797920,0.699692>, <0.808499,0.689113>, <0.816055,0.669467>,
	<0.816055,0.657378>, <0.820589,0.636221>, <0.817566,0.612041>, <0.817566,0.593907>, <0.810010,0.581817>,
	<0.805477,0.572750>, <0.816055,0.563682>, <0.831167,0.550082>, <0.840234,0.537992>, <0.847790,0.528925>,
	<0.849302,0.507768>, <0.847790,0.474521>, <0.843257,0.441274>, <0.826634,0.392915>, <0.817566,0.374781>,
	<0.811521,0.361180>, <0.814544,0.346068>, <0.811521,0.330956>, <0.803965,0.318866>, <0.796409,0.306776>,
	<0.776763,0.296198>, <0.742006,0.293175>, <0.702714,0.278063>, <0.667956,0.270507>, <0.642266,0.258417>,
	<0.627153,0.229704>, <0.655866,0.207036>, <0.675512,0.185879>, <0.680046,0.160189>, <0.707248,0.157166>,
	<0.760140,0.164722>, <0.800943,0.175301>, <0.840234,0.181346>, <0.861391,0.176812>, <0.874992,0.155655>,
	<0.888593,0.145076>, <0.905217,0.123919>, <0.912773,0.114852>, <0.908239,0.104274>, <0.900683,0.093695>,
	<0.896149,0.083117>, <0.885571,0.083117>, <0.874992,0.080094>, <0.864414,0.071027>, <0.849302,0.058937>,
	<0.843257,0.045336>, <0.838723,0.039292>, <0.823611,0.024179>, <0.803965,0.010578>, <0.790364,0.006045>,
	<0.773741,0.000000>, <0.754095,0.009067>, <0.740494,0.012090>, <0.726893,0.015112>, <0.717826,0.021157>,
	<0.705736,0.024179>, <0.696669,0.022668>, <0.678535,0.021157>, <0.658889,0.025691>, <0.628665,0.034758>,
	<0.613552,0.042314>, <0.598440,0.048359>, <0.586351,0.061960>, <0.584839,0.072538>, <0.574261,0.083117>,
	<0.556126,0.080094>, <0.533458,0.086139>, <0.518346,0.093695>, <0.495678,0.116363>, <0.480566,0.120897>,
	<0.460920,0.125431>, <0.445808,0.137520>, <0.430696,0.145076>, <0.418606,0.151121>, <0.398960,0.163211>,
	<0.380826,0.173789>, <0.359669,0.185879>, <0.344557,0.211570>, <0.341534,0.234238>, <0.323400,0.272018>,
	<0.302243,0.302243>, <0.288642,0.308287>, <0.259929,0.314332>, <0.241794,0.321888>, <0.232727,0.332467>,
	<0.225171,0.332467>, <0.217615,0.353624>, <0.211570,0.377803>, <0.200991,0.409539>, <0.191924,0.444297>,
	<0.188902,0.468476>, <0.191924,0.485099>, <0.197969,0.495678>, <0.196458,0.510790>, <0.196458,0.516835>,
	<0.208547,0.522880>, <0.219126,0.527413>, <0.225171,0.530436>, <0.237260,0.528925>, <0.249350,0.534969>,
	<0.256906,0.545548>, <0.259929,0.559149>, <0.270507,0.575772>, <0.284108,0.580306>, <0.293175,0.580306>,
	<0.303754,0.590884>, <0.323400,0.604485>, <0.332467,0.618086>, <0.338512,0.651333>, <0.340023,0.698180>,
	<0.344557,0.726893>, <0.352113,0.760140>, <0.364202,0.788853>, <0.374781,0.800943>, <0.397449,0.816055>,
	<0.415584,0.840234>, <0.436741,0.862903>, <0.456386,0.874992>, <0.477543,0.879526>, <0.506256,0.881037>,
	<0.534969,0.881037>, <0.553104,0.884060>, <0.557638,0.890104>, <0.568216,0.902194>, <0.575772,0.909750>,
	<0.596929,0.915795>, <0.616575,0.920329>, <0.631687,0.929396>, <0.639243,0.941486>, <0.637732,0.958109>,
	<0.639243,0.979266>, <0.649822,0.991356>, <0.674001,0.994378>, <0.701203,0.994378>, <0.716315,0.998912>,
	<0.734450,0.998912>, <0.743517,0.986822>, <0.751073,0.974732>
	hollow 
	material {M_Lake}
}

#declare lakeBBox = max_extent(O_IslandLake)-min_extent(O_IslandLake);
#declare lakeHeight = lakeBBox.y;
#end

// --------------------------------------------------- factory
#if (_show_factory)

//#declare O_Smoke = 

#declare O_Cheminey = lathe // normalized: radius is approx 0.5.
{
	cubic_spline
	18,
	<0.508825,0.019788>, <0.480557,0.189396>, <0.452289,0.330736>, <0.424021,0.556881>, <0.424021,0.783026>,
	<0.452289,0.980902>, <0.407060,0.979489>, <0.399993,0.879137>, <0.395753,0.783026>, <0.395753,0.556881>,
	<0.424021,0.330736>, <0.452289,0.189396>, <0.480557,0.019788>, <0.508825,0.019788>, <0.480557,0.189396>,
	<0.508825,0.019788>, <0.480557,0.189396>, <0.452289,0.330736>
	texture {T_Cheminey}
}
#declare CHEM_RADIUS = max_extent(O_Cheminey).x;
#declare FENCE_RADIUS = CHEM_RADIUS+0.12;

// Simple light
#declare O_Light = union
{
    intersection
    {
        object {Round_Cylinder (-x, x, 0.5, 0.2, false)}
        plane {-z, 0}
        pigment {color 0.1*White} normal {granite scale 0.5}
    }
    box {-0.8, 0.8 scale y/2 + 0.02*z texture {T_Chrome_1C}}
    translate -0.5*z
}

// Array of tubes
#macro O_TubesArray (_width, _height, _lines, _cols, _r)
union
{
    //Lines
    #local i=0;
    #while (i<_lines)
        cylinder {-_width/2*x, _width/2*x, _r translate (-_height/2 + i*_height/(_lines-1))*y}
        #local i=i+1;
    #end
    
    // Columns
    #local i=0;
    #while (i<_cols)
        cylinder {-_height/2*y, _height/2*y, _r translate (-_width/2 + i*_width/(_cols-1))*x}
        #local i=i+1;
    #end
    
    // Corners
    sphere {0,_r translate _width/2*x+_height/2*y}
    sphere {0,_r translate _width/2*x-_height/2*y}
    sphere {0,_r translate -_width/2*x+_height/2*y}
    sphere {0,_r translate -_width/2*x-_height/2*y}
}
#end

#macro O_LightsArray (_width, _height, _m, _n, _scale)
union
{
    #local i=0;
    #while (i<_n)
        #local j=0;
        #while (j<_m)
            object {O_Light scale _scale translate (-_width/2+i*_width/(_n-1))*x + (-_height/2+j*_height/(_m-1))*y}
            #local j=j+1;
        #end
        #local i=i+1;
    #end
}
#end
 
// Legs
#macro O_Legs (_width, _r, _H, _angle)   
union
{
    union
    {
        cylinder {0, -_H*y, 2*_r}
        sphere {0, 2*_r}
        translate -_width/2*x
    }
    union
    {
        cylinder {0, -_H*(y-sin(radians(_angle))*z), 2*_r}
        sphere {0, 2*_r}
    }
    union
    {
        cylinder {0, -_H*y, 2*_r}
        sphere {0, 2*_r}
        translate _width/2*x
    }
}
#end

#declare O_LightStand = union
{
    // Stand
    #local w1=3;
    #local h1=2;
    #local n1=10;
    #local m1=10;
    #local r=0.02;
    // Lights
    #local w2=2.5;
    #local h2=1.8;
    #local n2=4;
    #local m2=5;
    #local s=0.25;
    union
    {
        object {O_TubesArray(w1,h1,n1,m1,r) translate h1/2*y }
        object {O_LightsArray (w2, h2, n2, m2, s) translate (h2/2+abs(h1-h2)/2)*y}
        rotate -20*x
    }
    object {O_Legs (0.6, 0.04, 2, 15)}
    texture {Aluminum} texture {T_DirtyGround scale 1/5}
    translate 2*y
}

#declare O_Pylone = union
{
    // Pylone
    isosurface
    {
        function {f_helix1(x,y,z,4,2,0.05,max(0.9-abs(y+2)/8, 0.05),1,0,0)*f_helix1(-x,y,z,4,2,0.05,max(0.9-abs(y+2)/8, 0.05),1,0,0)-0.005}
        max_gradient 2
        contained_by {box {-1-3*y,1+3*y }}
        texture {T_Rust1(false, false)}
        translate 4*y
    }
    
    // A little platform
    cylinder {-0.05*y, 0.05*y 0.7 texture {T_Rust1(true, true) scale 0.1} translate 8*y}
    
    // Lights stand
   object {O_LightStand translate 8*y}
}

#declare fn_Cylinder = function (x,y,z, _radius) {x*x+z*z-_radius}
#declare fn_BoundedCylinder = function (x,y,z, _radius, _height) {max(fn_Cylinder (x,y,z, _radius), abs(y)-_height/2)}
#declare fn_DoubleHelix = function (x,y,z, _radius) {f_helix1(x,y,z,2,3,0.03*_radius,1.05*_radius,1,1,0)*f_helix1(-x,y,z,2,3,0.03*_radius,1.05*_radius,1,1,0)}
#declare fn_BoundedDoubleHelix = function (x,y,z, _radius, _height) {max(fn_DoubleHelix (x,y,z, _radius), abs(y)-_height/2)}
#declare fn_PillarBlob = function (x,y,z, _radius, _height, _threshold) 
                            {
                                // Beware of second-order terms and unexpected side-effects.
                                (fn_BoundedCylinder(x,y,z, _radius, _height) -0.12*f_granite(x,y,z))
                              * (fn_BoundedDoubleHelix(x,y,z, _radius, _height)-0.03*f_granite(x,y,z))
                              - _threshold
                            }

#macro O_Pillar (_radius, _height, _orient)
union
{
    // Pillar
    isosurface
    {
        function {fn_PillarBlob (x,y,z, _radius, _height, 0.001)}
        contained_by {box {-1.3*<_radius, _height, _radius>, 1.3*<_radius, _height, _radius>}}
        evaluate 7 1/sqrt(0.6) 0.7max_gradient 5 //
        texture{T_Stone44 scale 2}
    }
    
    // Platform sustainer
    union
    {
        box {-2.5*_radius, 2.5*_radius scale 0.05*y texture {T_Door translate 2*rand(rnd3)}}
        #local i=0;
        #local _nmax = 10;
        #while (i<_nmax)
            box {-0.5, 0.5 scale 0.2*x rotate 45*x translate _radius*z rotate i*360/_nmax*y texture {T_Pedestal}}
            #local i = i+1;
        #end
        translate 2.5*y 
    }
    
    // Footer
    object {Round_Cylinder(-_height/2*y, -3/5*_height/2*y, 1.7*_radius, 0.45*_radius, false) texture {T_Stone44}}
    
    // Pylone
    object {O_Pylone rotate _orient*y translate _height/2*y}
    
    scale 1/_height
}
#end

#declare O_Platform = union
{
    object
    {
        Round_Box(<-0.75, -0.05, -0.35>, <0.35, 0, 0.4>, 0.001, false)
        texture {T_Platform}
    }
    object
    {
        Round_Box(<-0.755, -0.051, -0.355>, <0.355, -0.005, 0.405>, 0.001, false)
        texture {T_Stone44 scale 0.5}
    }
    object {O_Pillar(1, 10, -135) scale 0.2 translate -0.77*x -0.08*y - 0.37*z}
    object {O_Pillar(1, 10, 180) scale 0.2 translate -(0.77-0.37)/2*x -0.08*y - 0.37*z}
    object {O_Pillar(1, 10, 135) scale 0.2 translate + 0.37*x -0.08*y - 0.37*z}
    object {O_Pillar(1, 10, 45) scale 0.2 translate +0.37*x -0.08*y + 0.43*z}
    object {O_Pillar(1, 10, 0) scale 0.2 translate -(0.77-0.37)/2*x -0.08*y + 0.43*z}
    object {O_Pillar(1, 10, -45) scale 0.2 translate -0.77*x -0.08*y + 0.43*z}
    scale 2
}


#declare O_ChemineyFencePost = union
{
    cylinder {0, y, 0.05}
    sphere {0.6*y, 0.08}
    sphere {y, 0.08}
    texture {T_Post}
}

#declare _nb_posts = 40;
#declare O_ChemineyFence = union
{
    #declare i=0;
    #while (i<_nb_posts)
        object {O_ChemineyFencePost scale 0.04 translate FENCE_RADIUS*x rotate 360*i/_nb_posts*y}
        #declare i=i+1;
    #end
    torus {FENCE_RADIUS, FENCE_RADIUS/500 texture {Rusty_Iron scale 0.005} texture {T_DirtyGround scale 0.05} translate 0.6*0.04*y}
    torus {FENCE_RADIUS, FENCE_RADIUS/500 texture {Rusty_Iron scale 0.005} texture {T_DirtyGround scale 0.05} translate 0.04*y}
}

// ---------- Pipes ---------
#declare O_PipePost = union
{
    object{Round_Cylinder(0, y, 1, 0.2, false) translate -3*x}
    object{Round_Cylinder(0, y, 1, 0.2, false) translate 3*x}
}
    
#declare O_Pipe = sphere_sweep
{
	cubic_spline // Normalized in x
	9,
	<0.000000,-0.176330, 0>, 0.1,
	<0.000000,0.000000, 0>, 0.1,
	<0.033796,0.210126, 0>, 0.1,
	<0.293883,0.235106, 0>, 0.1,
	<0.499600,0.235106, 0>, 0.1,
	<0.705318,0.235106, 0>, 0.1,
	<0.968343,0.210126, 0>, 0.1,
	<0.999201,0.000000, 0>, 0.1,
	<0.999201,-0.176330, 0>, 0.1
	translate -x/2
	scale 6
}


#declare O_Pipes = union
{
    union
    {
        object {O_Pipe translate -12*z}
        object {O_Pipe translate -10*z}
        object {O_Pipe translate -8*z}
        object {O_Pipe translate -5*z}
        object {O_Pipe translate -3*z}
        object {O_Pipe}
        object {O_Pipe translate 3*z}
        object {O_Pipe translate 5*z}
        object {O_Pipe translate 8*z}
        object {O_Pipe translate 10*z}
        object {O_Pipe translate 12*z}
        texture {T_Pipe scale 1.7}
	    translate y
    }
    union
    {
        object {O_PipePost translate -12*z}
        object {O_PipePost translate -10*z}
        object {O_PipePost translate -8*z}
        object {O_PipePost translate -5*z}
        object {O_PipePost translate -3*z}
        object {O_PipePost}
        object {O_PipePost translate 3*z}
        object {O_PipePost translate 5*z}
        object {O_PipePost translate 8*z}
        object {O_PipePost translate 10*z}
        object {O_PipePost translate 12*z}
        texture {Brass_Valley finish {reflection 0.1} scale 0.2}
    }
    scale 0.03
}

// ---------- Buildings ---------
#declare O_Building1 = union
{
    box
    {
        -x, x+y-z
        texture {T_Building}
    }
    box
    {
        -x, x+0.05*y-z
        scale 1.05
        translate y + 0.025*z
        pigment {gradient x  scale 0.05 sine_wave color_map{[0 0.05*White][1 0.15*White]}}
    }
    box
    {
        -x, x+0.05*y-z
        scale 1.03
        translate 0.55*y + 0.015*z
        pigment {gradient y scale 0.15 scale 0.1*(x+z) turbulence 0.5 color_map {[ 0.2 Scarlet] [0.5 Black]} phase 0.5}
    }
    box
    {
        -x, x-0.05*y-z
        scale 1.4
        translate 0.005*y + 0.3*z
        texture {pigment {HuntersGreen}}
        texture {T_DirtyGround scale 5} // scale 5 to compensate the scale 0.2 below
        normal {bumps 1.5 scale 0.02}
    }
    scale 0.2
}


#declare O_Building2 = union
{
    box {-x, x+4*y-2*z scale 0.5 texture {T_Building}}
    box {-x, x+0.20*y-2*z scale 0.55 translate 2*y texture {Brushed_Aluminum scale 3}texture {T_DirtyGround scale 4}}
    box {-x, x+0.02*y-2*z translate 0.025*z+0.015*y scale 0.55 scale 1.05 translate 2*y pigment{Gray10}}
    box {-x, x+0.02*y-2*z translate 0.025*z+0.085*y scale 0.55 scale 1.05 translate 2*y pigment{Gray10}}
    #local i=0;
    #while (i<20)
        #local _r1 = rand(rnd2);
        #local _r2 = rand(rnd2);
        #local _r3 = rand(rnd2);
        box {-x-z, x+y+z scale 0.1*_r1 translate <_r2-0.5, 2.1, _r3-1> texture {T_Stone21 scale 0.05}}
        box {-x-z, x+0.1*y+z translate y scale 0.1*_r1 scale 1.1 translate <_r2-0.5, 2.1, _r3-1> pigment {Gray05}}
      #declare i=i+1;   // increment our counter
    #end

    scale 0.2
}

// ---------- Reactor ---------
#declare NB_SATELLITES = 5;
#declare SATELLITE_OFFSET = 1.6;
#declare SATELLITE_RATIO = 0.5;

#macro O_ReactorCore (p_glow_color, p_glow_color_transmit, p_dirt_color, p_ambient)
union
{
    superellipsoid {<1, 0.4> rotate 90*x translate y texture {T_ReactorBodyClean(p_ambient)}}
    // Trick: to layer one or more texture (wich must be pratially tranparent, of course ...) 
    // ver a patterned one(ie with texture_map), you can define an identical shape a little larger 
    // and apply the layes(s) on that shape. Thus:
    superellipsoid {<1, 0.4> rotate 90*x scale 1.01 translate y texture {pigment {color p_glow_color transmit p_glow_color_transmit}} texture {T_VerticalDirt(p_dirt_color)}} 
    union
    {
        #local i=0;
        #local _nmax = 10;
        #while (i<_nmax)
            box {<-0.03, -1.05, -1.05>, -<-0.03, -1.05, -1.05> scale 1.05*y translate y rotate 180*i/_nmax*y}
            #declare i=i+1;
        #end
        object {Round_Cylinder(-0.05*y, 0.05*y, 1.07, 0.02, false) translate 1.3*y}
        object {Round_Cylinder(-0.05*y, 0.05*y, 1.07, 0.02, false) translate 0.7*y}
        texture {pigment {Goldenrod}} // background color
        texture {T_VerticalDirt(p_dirt_color)}
    }
    cylinder {0, 0.2*y, 0.2 translate 2.1*y texture{T_ReactorCagePilar}}
    sphere {0, 1 texture {T_Brass_1A normal {bumps 0.4 scale 0.05}}  texture {T_VerticalDirt(p_dirt_color) translate 0.8*y} scale 0.4 translate 2.6*y}
}
#end

#declare O_ReactorTubes = union
{
    #local _use_f_helix=on;
    #macro _tubes_transform (n) transform
    {
        rotate -180/NB_SATELLITES*y
        translate SATELLITE_OFFSET*x
        rotate n*360/NB_SATELLITES*y
        translate 1.3*y
    }
    #end
    
    // Tubes connecting the satellite reactors one another
    #local _tube_length = 2*SATELLITE_OFFSET*sin(pi/NB_SATELLITES);
    #local i=0;
    #while (i<NB_SATELLITES)
        #if (_use_f_helix)
        // Parameters for helical tubes connecting satellite reactors. I had to find a suitable combination
        // because major and minor radii are of the same order and thus are not in the normal scope of
        // the built-in f_helix1() function.
        #local _nb_helixes = 3;
        #local _nb_turns = 3;
        #local _minor = 0.05;
        #local _major = 0.05;
        #local _shape = 5;
        #local _cross_type = 1; // 1=circle
        #local _cross_rot = 0;
        isosurface
        {
            function {f_helix1(x,y,z,_nb_helixes,_nb_turns,_minor,_major,_shape,_cross_type,_cross_rot)}
            evaluate 1, sqrt(1/0.6), 0.7
            contained_by {box {1.2*(_major+_minor)*(x+z), -1.2*(_major+_minor)*(x+z)+_tube_length*y}}
            rotate 90*x
            transform {_tubes_transform(i)}
        }
        #else
        cylinder 
        {
            0, _tube_length*z, 0.07 
            transform {_tubes_transform(i)}
        }
        #end
        #local i=i+1;
    #end
    
    // Tubes connecting the main reactor to the others (separate loop to make the 2 sets of tubes independant)
    #local i=0;
    #while (i<NB_SATELLITES)
        #local _l1 = 0.7*SATELLITE_OFFSET;
        #local _angle = 70;
        #local _l2 = 1.3*0.3*SATELLITE_OFFSET/cos(radians(_angle));
        #local _r = 0.05;
        union
        {
            cylinder {0, _l1*x, _r}
            sphere {0, 1.5*_r translate _l1*x}
            cylinder {0, _l2*x, _r rotate -_angle*z translate _l1*x}
            rotate i*360/NB_SATELLITES*y
            translate 2.6*y
        }
        #local i=i+1;
    #end
    
    // Tubes on the ground
    #local _r = SATELLITE_OFFSET/50;
    #local _f = 2.5;
    union
    {
        torus {SATELLITE_OFFSET-_f*_r, _r}
        torus {SATELLITE_OFFSET, _r}
        torus {SATELLITE_OFFSET+_f*_r, _r}
        translate _r*y
    }
    union
    {
        torus {SATELLITE_OFFSET-_f*_r, _r}
        torus {SATELLITE_OFFSET, _r}
        torus {SATELLITE_OFFSET+_f*_r, _r}
        translate (_r+0.3)*y
    }
    union
    {
        torus {SATELLITE_OFFSET-_f*_r, _r}
        torus {SATELLITE_OFFSET, _r}
        torus {SATELLITE_OFFSET+_f*_r, _r}
        translate (_r+0.6)*y
    }
    
    // The T_Pipe texture is OK here for the whole stuff
    texture {T_Pipe scale 0.3}
}

#declare O_Handle = blob
{
    cylinder {0, y/3, 0.07, 1}
    sphere {y/3, 0.07, -1}
    cylinder {-0.2*y, y, 0.055, 1 rotate -90*z translate y/3}
    sphere {y/3, 0.07, -0.4 scale 0.7*x translate x*0.7}
    sphere {y/3, 0.07, -0.4 scale 0.7*x translate x*0.8}
    sphere {y/3, 0.07, -0.4 scale 0.7*x translate x*0.9}
    sphere {y/3, 0.085, 1  translate -0.2*x}
    sphere {y/3, 0.085, 0.1  translate x}
    threshold 0.01
    sturm
    texture {T_Post scale 10}
}

#declare O_ReactorGate = union
{
    #local _phi = (1+sqrt(5))/2;
    
    // Frame
    object {Supertorus(1, 0.1, 0.15, 0.1, 0.0001, 1) scale _phi*z texture {T_DoorFrame(true, false)}}
    // Door + handle
    box {-x-z-0.05*y, x+z+0.05*y scale _phi*z texture {T_Door}}
    union
    {
        object {Round_Cylinder (0, 0.08*y, 0.1, 0.01, false) texture {T_Pedestal scale 0.3}}
        object {O_Handle translate 0.05*y scale 0.6}
        translate -0.5*x - 0.1*z
    }
    
    // Hinges
    union
    {
        object {Round_Cylinder(-z/2, z/2, 0.3, 0.1, false) texture {T_Hinge scale 1.25*1/2 rotate 90*x} translate 1.8*z }
        object {Round_Cylinder(-z/2, z/2, 0.3, 0.1, false) texture {T_Hinge scale 1.25*1/2 rotate 90*x} translate -1.5*z}
         
        scale 0.5
        translate 0.9*x
    }
    
    rotate -90*x
    translate _phi*y
}


#declare O_ReactorCage = union
{
    // A little surrounding wall
    object
    {
        Supertorus(2.4, 0.07, 1, 0.1, 0.001, 1.5) 
        scale 2*y
        texture {T_Stone44 scale 2.5} // Standard wall texture
        // A layer of paint: compensate color fading due to partial tranparency by a multiplying factor to the color and a lower ambient
        texture {pigment {color 3*Scarlet transmit 0.7} finish{specular 0.8 ambient 0.8}} 
        texture {T_VerticalDirt(0.05*White) translate 0.25*y} // A layer of dirt
        translate 0.14*y
    }
    
    // Grating
    #if (_show_cage_mesh)
    isosurface
    {
        function
        {
            f_mesh1 
            (
                f_th(x,y,z), f_r(x,0,z)-1, f_ph(x,y,z),     // cartesian -> pseudo-cylindrical (the height of the cells increase with y... my maths are far behind... suggestions welcome.)
                0.015, // distance between neighboring threads in the x direction,
                0.015, // distance between neighboring threads in the z direction,
                1, // relative thickness in the x and z directions,
                0.001, // amplitude of the weaving effect,
                0.8  // relative thickness in the y direction
            ) -0.001
        }
        accuracy 0.0001
        contained_by {box {-1.1*(x+z), 1.1}}
        max_gradient 1.0
        scale 2.4
        pigment {Gray85}
        texture {T_VerticalDirt(0.05*White) translate 0.25*y} // A layer of dirt
    }
    #end
    
    // Some pilars to sustain the grating
    #local i=0;
    #local _nmax = 10;
    #while (i<_nmax)
        text 
        {
            ttf "arial.ttf" "H" 1, 0
            translate -0.3*x  
            rotate -90*x scale 0.15 scale 2.4/0.15*y 
            translate 2.4*x rotate i*360/_nmax*y 
            translate 0.28*y
            texture {T_ReactorCagePilar scale 5*y}}
        
        #local i=i+1;
    #end
    
    // Gate(s). Didn't want to make a difference {} with the cage --> The cage is actually visible from behind... Please, don't tell.
    object {O_ReactorGate scale 0.15 translate 0.28*y-2.404 *z rotate 180*y}
    object {O_ReactorGate scale 0.15 translate 0.28*y-2.404*z rotate -90*y}
}


#declare O_ReactorPlatform = 
lathe{
	cubic_spline
	10,
	<0.005655,0.285598>,
	<0.011311,0.285598>,
	<0.311048,0.285598>,
	<0.630579,0.285598>,
	<0.851140,0.282771>,
	<0.916177,0.237527>,
	<0.944454,0.158352>,
	<0.972731,0.076348>,
	<1.029285,0.000000>,
	<1.037768,-0.042416>

    scale 2.7
    texture { T_Pedestal}
}

#declare O_ReactorFeedPipes = union
{
    #local i=0;
    #local _nmax = 20;
    #while (i<_nmax)
        object {Supertorus(1, 0.1, 0.2, 0.8, 0.0001, 1) scale 0.3 rotate 90*x texture{T_AluTex} translate 2.8*x rotate i*360/_nmax*y}
        #local i=i+1;
    #end
    torus {3.1, 0.07 texture {T_Pipe} translate 0.03*y}
}

#macro O_Steps (_H, _h, p_thick, p_depth, p_width, p_slope, p_backoff)
#local _nbSteps = _H/_h;
union
{
    // Sides
    union
    {
        box {-1/2, 1/2 scale p_thick*x+p_depth*z translate -p_width/2*x}
        box {-1/2, 1/2 scale p_thick*x+p_depth*z translate p_width/2*x}
        translate y/2
        #if (p_slope>89.9) #declare _shear = 0; #else #declare _shear = 1/tan(radians(p_slope)); #end
        matrix <1, 0, 0,
               0, 1, _shear,
               0, 0, 1,
               0, 0, 0>
        scale _H
    }
    
    // Steps
    union
    {
        #local i=0;
        #while (i<_nbSteps)
            box {-1/2, 1/2 scale p_width*x + _h*y + p_depth*z translate (i+1/2)*_h*y + ((i+1)*_h/tan(radians(p_slope))+p_backoff)*z}
            #local i=i+1;
        #end
    }
}
#end


#macro O_Reactor (p_glow_color, p_glow_color_transmit, p_dirt_color, p_ambient)
union
{
    #if (_show_reactor)
    union
    {
        // Main reactor
        object {O_ReactorCore(p_glow_color, p_glow_color_transmit, p_dirt_color, p_ambient)}
        
        //Smaller satellite reactors around
        #local i = 0;
        #while (i<NB_SATELLITES)
            object 
            {
                O_ReactorCore(rgb <rand(rnd1), rand(rnd1), rand(rnd1)>, p_glow_color_transmit, p_dirt_color, p_ambient + 0.5*(rand(rnd1)-0.5)) 
                scale SATELLITE_RATIO
                translate SATELLITE_OFFSET*x 
                rotate 360*i/NB_SATELLITES*y
            }
            
            #local i=i+1;
        #end
        
        object {O_ReactorCage}
        
        object {O_ReactorTubes}
        
        translate (max_extent(O_ReactorPlatform) - min_extent(O_ReactorPlatform)).y*y
    }
    #end
    
    object {O_ReactorPlatform}
    
    object {O_ReactorFeedPipes}
    
    object {O_Steps (1, 0.025, 0.03, 0.3, 0.4, 30, 0.05) texture {T_Stone44 scale 0.2} texture {T_VerticalDirt (0.05*White)} scale 1.06 translate -4.3*z rotate -180*y}
    object {O_Steps (1, 0.025, 0.03, 0.3, 0.4, 30, 0.05) texture {T_Stone44 scale 0.2} texture {T_VerticalDirt (0.05*White)} scale 1.06 translate -4.3*z rotate -90*y}
    
    scale 0.11
}
#end


// ---------- Wall around the factory ---------
#macro MakeWall (_x, _y, _z, _rnd) 
union
{
    // Textures scaled for height=1.2
    #local _f = 1.2/_y; // scaling factor
    #local _v = 1/2*_f*<_x, _y, _z>;
    box {-_v, _v texture {T_CrackedWall translate 10*rand(_rnd) scale 0.4}}
    box {-_v, _v scale 1.005 texture {T_VerticalDirt (color 0.5*(2*DarkBrown + Firebrick)/3) translate 10*rand(_rnd)*x scale <0.75,2*_f*_y,0.75> translate 1.5*_f*_y*y }}
    box {-<_v.x, 1/12*_v.y, 1.1*_v.z>, <_v.x, 1/12*_v.y, 1.1*_v.z> texture {T_WallHat  translate 10*rand(_rnd)*x} translate _v.y*y }
    translate _v.y*y
    scale 1/_f
}
#end

#macro MakeTower (_h, _r, _rnd) 
union
{
    // Textures scaled for height=1.2
    #local _f = 1.2/_h; // scaling factor
    cylinder {0, _f*_h*y, _f*_r texture { T_CrackedWall translate 10*rand(_rnd) scale 0.4}}
    cylinder {0, _f*_h*y,_f* _r scale 1.005 texture {T_VerticalDirt (color 0.5*(2*DarkBrown + Firebrick)/3)translate 10*rand(_rnd)*x scale <0.75,2*_f*_h,0.75> translate 2.3*_f*_h*y}}
    scale 1/_f
}
#end

#declare rnd3 = seed(159);
#declare O_FactoryWall = union
{
    object {MakeWall(1.1, 0.05, 0.01, rnd3) translate -0.35*z}
    object {MakeWall(1.1, 0.05, 0.01, rnd3) translate 0.4*z}
    object {MakeWall(0.75, 0.05, 0.01, rnd3) rotate -90*y translate -0.55*x + 0.025*z}
    object {MakeWall(0.75, 0.05, 0.01, rnd3) rotate 90*y translate 0.55*x + 0.025*z}
    object {MakeTower(0.06, 0.015, rnd3) translate 0.55*x + 0.4*z }
    object {MakeTower(0.06, 0.015, rnd3) translate 0.55*x - 0.35*z }
    object {MakeTower(0.06, 0.015, rnd3) translate -0.55*x + 0.4*z }
    object {MakeTower(0.06, 0.015, rnd3) translate -0.55*x - 0.35*z }
    scale 2
}

// ---------- Disgusting smoke ---------
#declare D_ColoredCylindrical = density
{
    #local _s1 = 4; #local _a1 = 0.5;
    #local _s2 = 1/2; #local _a2 = 0.1;
    cylindrical 
    scale 0.8
    scale _s1 warp {turbulence _a1 omega 0.5 octaves 6} scale 1/_s1 scale 0.7*y
    //translate -10*y scale _s2 warp {turbulence _a2 omega 0.5 octaves 3} scale 1/_s2 translate 10*y
    
    color_map
    {
        [0.05 transmit 1]
        [0.05 color (1-OrangeRed) transmit 0 filter 0]
        [0.2 2*ForestGreen]
        [0.5 2*ForestGreen]
        [0.8 1.5*Scarlet]
        [0.95 color 0.5*(1-NeonBlue) transmit 0 filter 0]
    }
}

#declare D_Planar = density
{
    planar 
    density_map
    {
        [0.03 function {0.01}]
        [0.06 function {0.02}]
        [0.08 function {0.01}]
        [0.1 function {0.02}]
        [0.15 function {0.05}]
        [0.27 function {0.2}]
        //[0.4 function {0.1}]
        [0.6 function {0.8}]
        [1 function {1}]
    }
    
}
#declare O_Smoke = union
{
    cylinder
    {
        0, 4*y, 1
        //sphere {3*y, 0.64 }
        hollow
        pigment {color rgbt 1}
        interior
        {
            media
            {
                scattering {1, 8*White/GLOBAL_SCALE extinction 1}
                absorption 8*White/GLOBAL_SCALE
                emission 1/GLOBAL_SCALE
                density {D_ColoredCylindrical}
                density {D_Planar scale 4}
                intervals 4 samples 25, 25
                //intervals 2 samples 5, 5
            }
        }
    }
    matrix     <1,0,0,     0.2,1,0,     0,0,1,     0,0,0>
    scale 0.55
}

// ---------- Here it is! ---------
#declare O_Factory = union
{
    object {O_Cheminey}
    object {O_ChemineyFence translate 0.012*y}
    object {Round_Cylinder(-0.01*y, 0.012*y, FENCE_RADIUS+0.01, 0.01, false) texture{T_Stone44 scale 0.05} texture {T_DirtyGround}}
    object {O_Pipes translate -0.80*x + 0.35*z}
    object {O_Building1 rotate -90*y translate -1.4*x + 0.5*z}
    object {O_Building2  translate 0.55*x + 0.7*z}
    object {O_Reactor(Green, 0.3, 0.0*White, 3) translate -1.12*x -0.33*z}
    object {O_Platform}
    object {O_FactoryWall translate -0.4*x}
    object {O_Smoke translate 0.95*y}
    rotate PLATFORM_ORIENT*y
    //rotate -16*y rotate -60*z rotate 20*y  scale 0.9 // To see from above. _show_island and _show_water should be off. Can use the ABOVE_FACTORY camera.
}

#end

// --------------- S C E N E ---------------------------------
union
{
    #if (_show_island)
        object 
        {
            #if (_use_isosurface) O_Island_iso #else O_Island_hf #end 
            scale ISLAND_SCALE
        }
    #end
    
    #if (_show_water) plane {y, 0 hollow material {M_Ocean} translate SURFACE_LEVEL*y} #end
    
    #if (_show_lake)
        object 
        {
            O_IslandLake 
            translate -lakeBBox/2 - lakeHeight/2*y // Recenter the origin and put it on the surface.
            scale 2.15 // rescale to fit the crater
            translate -0.55*x +0.07*z // Place it correctly (with the orthographic "ABOVE_CAM" camera)
            translate LAKE_ALTITUDE*y
        }
    #end
    
    #if (_show_factory)
        object
        {
            O_Factory
            scale 2 
            translate FACTORY_POSITION  + 0.7*y
         }
    #end
    scale GLOBAL_SCALE
    rotate GLOBAL_ORIENT*y 
}

// ------------------------------------------------------- cameras
// Camera used for the final render
#declare K_Final = camera
{
  location  0
  angle     FINAL_ROTATIONS.z
  right     x*image_width/image_height
  look_at   z
  rotate    FINAL_ROTATIONS.x*x
  rotate    FINAL_ROTATIONS.y*y
  translate FINAL_LOCATION
  scale GLOBAL_SCALE
}

// Camera used for viewing the scene from above the crater during the creation process of the island (for the lake, mainly)
#declare K_AboveCrater = camera 
{
  orthographic
  location 3*y 
  look_at  0
  right ISLAND_SCALE/ABOVE_ZOOM*x*image_width/image_height
  up ISLAND_SCALE/ABOVE_ZOOM*z
  scale -x               // why do I need this?
  scale GLOBAL_SCALE
}

// Camera used for viewing the scene from above the factory during the creation process of the factory.
#declare K_AboveFactory = camera 
{
  orthographic
  location 5*y 
  look_at  0
  right FACTORY_SCALE/ABOVE_ZOOM*x*image_width/image_height
  up FACTORY_SCALE/ABOVE_ZOOM*z
  scale -x               // why do I need this?
  rotate (PLATFORM_ORIENT-180)*y
  translate FACTORY_POSITION  + 0.4*x + 0.7*z // Origin was in the center of the cheminey.
  rotate GLOBAL_ORIENT*y
  scale GLOBAL_SCALE
}

// Camera used for focussing to a particular point from 
#declare K_Focus = camera
{
  location  0
  angle     FOCUS.z
  right     x*image_width/image_height
  look_at   z
  rotate    (FINAL_ROTATIONS+FOCUS).x*x
  rotate    (FINAL_ROTATIONS+FOCUS).y*y
  translate FINAL_LOCATION
  scale GLOBAL_SCALE
}

// ------------------------------------------------------- environment
sky_sphere { // from standard insert-menu.
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

#declare LS_Scene = light_source 
{
  <0, 0, 0>            // light's position (translated below)
  color rgb <1, 1, 1>  // light's color
  translate <30, 60, -20>
  scale GLOBAL_SCALE
}

#declare LS_Above = light_source
{
  <0, 0, 0>            // light's position (translated below)
  color rgb <1, 1, 1>  // light's color
  parallel
  point_at -y
  scale GLOBAL_SCALE
}

#if (CAMERA != ABOVE_CRATER_CAMERA & CAMERA != ABOVE_FACTORY_CAMERA)
    light_source {LS_Scene}
#else
    light_source {LS_Above}
#end

#switch (CAMERA)
    #case (FINAL_CAMERA) camera {K_Final} #break
    #case (ABOVE_CRATER_CAMERA) camera {K_AboveCrater} #break
    #case (ABOVE_FACTORY_CAMERA) camera {K_AboveFactory} #break
    #case (FOCUS_CAMERA) camera {K_Focus} #break
    #else camera {K_Final}
#end
