# SPDX-License-Identifier: GPL-2.0-only
# This file is part of Scapy
# See https://scapy.net/ for more information
# Copyright (C) Philippe Biondi <phil@secdev.org>

"""
All MSRPCE layers
"""

import uuid

from scapy.error import log_loading
from scapy.main import load_layer

from scapy.layers.dcerpc import (
    DCE_RPC_INTERFACES_NAMES,
    DCE_RPC_INTERFACES_NAMES_rev,
)

__all__ = []


# Load all layers bundled with Scapy
_LAYERS = [
    # High-level classes
    "msrpce.msdcom",
    "msrpce.msnrpc",
    "msrpce.mspac",
    # Client / Server
    "msrpce.rpcclient",
    "msrpce.rpcserver",
    # Low-level RPC definitions
    "msrpce.raw.ept",
    "msrpce.raw.ms_dcom",
    "msrpce.raw.ms_drsr",
    "msrpce.raw.ms_nrpc",
    "msrpce.raw.ms_samr",
    "msrpce.raw.ms_srvs",
    "msrpce.raw.ms_wkst",
]

for _l in _LAYERS:
    log_loading.debug("Loading MSRPCE layer %s", _l)
    try:
        load_layer(_l, globals_dict=globals(), symb_list=__all__)
    except Exception as e:
        log_loading.warning("can't import layer %s: %s", _l, e)


# Populate DCE_RPC_INTERFACES_NAMES for some well-known interfaces

# Well-Known = from MSDN
_DCE_RPC_WELL_KNOWN_UUIDS = [
    (uuid.UUID("00000000-0000-0000-c000-000000000046"), "IUnknown"),
    (uuid.UUID("00000131-0000-0000-c000-000000000046"), "IRemUnknown"),
    (uuid.UUID("00000143-0000-0000-c000-000000000046"), "IRemUnknown2"),
    (uuid.UUID("000001a0-0000-0000-c000-000000000046"), "IRemoteSCMActivator"),
    (uuid.UUID("00020400-0000-0000-c000-000000000046"), "IDispatch"),
    (uuid.UUID("00020401-0000-0000-c000-000000000046"), "ITypeInfo"),
    (uuid.UUID("00020402-0000-0000-c000-000000000046"), "ITypeLib"),
    (uuid.UUID("00020403-0000-0000-c000-000000000046"), "ITypeComp"),
    (uuid.UUID("00020404-0000-0000-c000-000000000046"), "IEnumVARIANT"),
    (uuid.UUID("00020411-0000-0000-c000-000000000046"), "ITypeLib2"),
    (uuid.UUID("00020412-0000-0000-c000-000000000046"), "ITypeInfo2"),
    (uuid.UUID("004c6a2b-0c19-4c69-9f5c-a269b2560db9"), "IWindowsDriverUpdate4"),
    (uuid.UUID("0191775e-bcff-445a-b4f4-3bdda54e2816"), "IAppHostPropertyCollection"),
    (uuid.UUID("01954e6b-9254-4e6e-808c-c9e05d007696"), "IVssEnumMgmtObject"),
    (uuid.UUID("027947e1-d731-11ce-a357-000000000001"), "IEnumWbemClassObject"),
    (uuid.UUID("0316560b-5db4-4ed9-bbb5-213436ddc0d9"), "IVdsRemovable"),
    (
        uuid.UUID("0344cdda-151e-4cbf-82da-66ae61e97754"),
        "IAppHostElementSchemaCollection",
    ),
    (uuid.UUID("034634fd-ba3f-11d1-856a-00a0c944138c"), "IManageTelnetSessions"),
    (uuid.UUID("038374ff-098b-11d8-9414-505054503030"), "IDataCollector"),
    (uuid.UUID("03837502-098b-11d8-9414-505054503030"), "IDataCollectorCollection"),
    (
        uuid.UUID("03837506-098b-11d8-9414-505054503030"),
        "IPerformanceCounterDataCollector",
    ),
    (uuid.UUID("0383750b-098b-11d8-9414-505054503030"), "ITraceDataCollector"),
    (uuid.UUID("03837510-098b-11d8-9414-505054503030"), "ITraceDataProviderCollection"),
    (uuid.UUID("03837512-098b-11d8-9414-505054503030"), "ITraceDataProvider"),
    (uuid.UUID("03837514-098b-11d8-9414-505054503030"), "IConfigurationDataCollector"),
    (uuid.UUID("03837516-098b-11d8-9414-505054503030"), "IAlertDataCollector"),
    (uuid.UUID("0383751a-098b-11d8-9414-505054503030"), "IApiTracingDataCollector"),
    (uuid.UUID("03837520-098b-11d8-9414-505054503030"), "IDataCollectorSet"),
    (uuid.UUID("03837524-098b-11d8-9414-505054503030"), "IDataCollectorSetCollection"),
    (uuid.UUID("03837533-098b-11d8-9414-505054503030"), "IValueMapItem"),
    (uuid.UUID("03837534-098b-11d8-9414-505054503030"), "IValueMap"),
    (uuid.UUID("0383753a-098b-11d8-9414-505054503030"), "ISchedule"),
    (uuid.UUID("0383753d-098b-11d8-9414-505054503030"), "IScheduleCollection"),
    (uuid.UUID("03837541-098b-11d8-9414-505054503030"), "IDataManager"),
    (uuid.UUID("03837543-098b-11d8-9414-505054503030"), "IFolderAction"),
    (uuid.UUID("03837544-098b-11d8-9414-505054503030"), "IFolderActionCollection"),
    (uuid.UUID("04c6895d-eaf2-4034-97f3-311de9be413a"), "IUpdateSearcher3"),
    (uuid.UUID("070669eb-b52f-11d1-9270-00c04fbbbfb3"), "IDataFactory2"),
    (uuid.UUID("0716caf8-7d05-4a46-8099-77594be91394"), "IAppHostConstantValue"),
    (uuid.UUID("0770687e-9f36-4d6f-8778-599d188461c9"), "IFsrmFileManagementJob"),
    (uuid.UUID("07e5c822-f00c-47a1-8fce-b244da56fd06"), "IVdsDisk"),
    (uuid.UUID("07f7438c-7709-4ca5-b518-91279288134e"), "IUpdateCollection"),
    (uuid.UUID("0818a8ef-9ba9-40d8-a6f9-e22833cc771e"), "IVdsService"),
    (uuid.UUID("081e7188-c080-4ff3-9238-29f66d6cabfd"), "IMessenger"),
    (
        uuid.UUID("08a90f5f-0702-48d6-b45f-02a9885a9768"),
        "IAppHostChildElementCollection",
    ),
    (uuid.UUID("09829352-87c2-418d-8d79-4133969a489d"), "IAppHostChangeHandler"),
    (uuid.UUID("0ac13689-3134-47c6-a17c-4669216801be"), "IVdsServiceHba"),
    (uuid.UUID("0b1c2170-5732-4e0e-8cd3-d9b16f3b84d7"), "authzr"),
    (uuid.UUID("0bb8531d-7e8d-424f-986c-a0b8f60a3e7b"), "IUpdateServiceManager2"),
    (
        uuid.UUID("0d521700-a372-4bef-828b-3d00c10adebd"),
        "IWindowsDriverUpdateEntryCollection",
    ),
    (uuid.UUID("0dd8a158-ebe6-4008-a1d9-b7ecc8f1104b"), "IAppHostSectionGroup"),
    (uuid.UUID("0e3d6630-b46b-11d1-9d2d-006008b0e5ca"), "ICatalogTableRead"),
    (uuid.UUID("0e3d6631-b46b-11d1-9d2d-006008b0e5ca"), "ICatalogTableWrite"),
    (uuid.UUID("0eac4842-8763-11cf-a743-00aa00a3f00d"), "IDataFactory"),
    (uuid.UUID("0fb15084-af41-11ce-bd2b-204c4f4f5020"), "ITransaction"),
    (uuid.UUID("1088a980-eae5-11d0-8d9b-00a02453c337"), "qm2qm"),
    (uuid.UUID("10c5e575-7984-4e81-a56b-431f5f92ae42"), "IVdsProvider"),
    (uuid.UUID("112eda6b-95b3-476f-9d90-aee82c6b8181"), "IUpdate3"),
    (uuid.UUID("118610b7-8d94-4030-b5b8-500889788e4e"), "IEnumVdsObject"),
    (uuid.UUID("11899a43-2b68-4a76-92e3-a3d6ad8c26ce"), "TermSrvNotification"),
    (uuid.UUID("11942d87-a1de-4e7f-83fb-a840d9c5928d"), "IClusterStorage3"),
    (uuid.UUID("12345678-1234-abcd-ef00-0123456789ab"), "winspool"),
    (uuid.UUID("12345678-1234-abcd-ef00-01234567cffb"), "logon"),
    (uuid.UUID("12345778-1234-abcd-ef00-0123456789ab"), "lsarpc"),
    (uuid.UUID("12345778-1234-abcd-ef00-0123456789ac"), "samr"),
    (uuid.UUID("1257b580-ce2f-4109-82d6-a9459d0bf6bc"), "SessEnvPublicRpc"),
    (uuid.UUID("12937789-e247-4917-9c20-f3ee9c7ee783"), "IFsrmActionCommand"),
    (uuid.UUID("135698d2-3a37-4d26-99df-e2bb6ae3ac61"), "IVolumeClient3"),
    (uuid.UUID("13b50bff-290a-47dd-8558-b7c58db1a71a"), "IVdsPack2"),
    (uuid.UUID("144fe9b0-d23d-4a8b-8634-fb4457533b7a"), "IUpdate2"),
    (uuid.UUID("14a8831c-bc82-11d2-8a64-0008c7457e5d"), "ExtendedError"),
    (uuid.UUID("14fbe036-3ed7-4e10-90e9-a5ff991aff01"), "IVdsServiceIscsi"),
    (uuid.UUID("1518b460-6518-4172-940f-c75883b24ceb"), "IUpdateService2"),
    (uuid.UUID("1544f5e0-613c-11d1-93df-00c04fd7bd09"), "rfri"),
    (uuid.UUID("1568a795-3924-4118-b74b-68d8f0fa5daf"), "IFsrmQuotaBase"),
    (uuid.UUID("15a81350-497d-4aba-80e9-d4dbcc5521fe"), "IFsrmStorageModuleDefinition"),
    (uuid.UUID("15fc031c-0652-4306-b2c3-f558b8f837e2"), "IVdsServiceSw"),
    (uuid.UUID("17fdd703-1827-4e34-79d4-24a55c53bb37"), "msgsvc"),
    (uuid.UUID("182c40fa-32e4-11d0-818b-00a0c9231c29"), "ICatalogSession"),
    (uuid.UUID("1a9134dd-7b39-45ba-ad88-44d01ca47f28"), "RemoteRead"),
    (uuid.UUID("1a927394-352e-4553-ae3f-7cf4aafca620"), "WdsRpcInterface"),
    (uuid.UUID("1bb617b8-3886-49dc-af82-a6c90fa35dda"), "IFsrmMutableCollection"),
    (uuid.UUID("1be2275a-b315-4f70-9e44-879b3a2a53f2"), "IVdsVolumeOnline"),
    (uuid.UUID("1c1c45ee-4395-11d2-b60b-00104b703efd"), "IWbemFetchSmartEnum"),
    (uuid.UUID("1d118904-94b3-4a64-9fa6-ed432666a7b9"), "ICatalog64BitSupport"),
    (uuid.UUID("1e062b84-e5e6-4b4b-8a25-67b81e8f13e8"), "IVdsVDisk"),
    (uuid.UUID("1f7b1697-ecb2-4cbb-8a0e-75c427f4a6f0"), "IImport2"),
    (uuid.UUID("1ff70682-0a51-30e8-076d-740be8cee98b"), "atsvc"),
    (uuid.UUID("205bebf8-dd93-452a-95a6-32b566b35828"), "IFsrmFileScreenTemplate"),
    (uuid.UUID("20610036-fa22-11cf-9823-00a0c911e5df"), "rasrpc"),
    (uuid.UUID("20d15747-6c48-4254-a358-65039fd8c63c"), "IServerHealthReport2"),
    (
        uuid.UUID("214a0f28-b737-4026-b847-4f9e37d79529"),
        "IVssDifferentialSoftwareSnapshotMgmt",
    ),
    (uuid.UUID("21546ae8-4da5-445e-987f-627fea39c5e8"), "IWRMConfig"),
    (uuid.UUID("22bcef93-4a3f-4183-89f9-2f8b8a628aee"), "IFsrmObject"),
    (uuid.UUID("22e5386d-8b12-4bf0-b0ec-6a1ea419e366"), "NetEventForwarder"),
    (uuid.UUID("23857e3c-02ba-44a3-9423-b1c900805f37"), "IUpdateServiceManager"),
    (uuid.UUID("23c9dd26-2355-4fe2-84de-f779a238adbd"), "IProcessDump"),
    (uuid.UUID("27b899fe-6ffa-4481-a184-d3daade8a02b"), "IFsrmReportManager"),
    (uuid.UUID("27e94b0d-5139-49a2-9a61-93522dc54652"), "IUpdate4"),
    (uuid.UUID("29822ab7-f302-11d0-9953-00c04fd919c1"), "IWamAdmin"),
    (uuid.UUID("29822ab8-f302-11d0-9953-00c04fd919c1"), "IWamAdmin2"),
    (uuid.UUID("2a3eb639-d134-422d-90d8-aaa1b5216202"), "IResourceManager2"),
    (uuid.UUID("2abd757f-2851-4997-9a13-47d2a885d6ca"), "IVdsHbaPort"),
    (uuid.UUID("2c9273e0-1dc3-11d3-b364-00105a1f8177"), "IWbemRefreshingServices"),
    (uuid.UUID("2d9915fb-9d42-4328-b782-1b46819fab9e"), "IAppHostMethodSchema"),
    (uuid.UUID("2dbe63c4-b340-48a0-a5b0-158e07fc567e"), "IFsrmActionReport"),
    (uuid.UUID("300f3532-38cc-11d0-a3f0-0020af6b0add"), "trkwks"),
    (uuid.UUID("31a83ea0-c0e4-4a2c-8a01-353cc2a4c60a"), "IAppHostMappingExtension"),
    (uuid.UUID("326af66f-2ac0-4f68-bf8c-4759f054fa29"), "IFsrmPropertyCondition"),
    (uuid.UUID("338cd001-2244-31f1-aaaa-900038001003"), "winreg"),
    (uuid.UUID("367abb81-9844-35f1-ad32-98f038001003"), "svcctl"),
    (uuid.UUID("370af178-7758-4dad-8146-7391f6e18585"), "IAppHostConfigLocation"),
    (uuid.UUID("377f739d-9647-4b8e-97d2-5ffce6d759cd"), "IFsrmQuota"),
    (uuid.UUID("378e52b0-c0a9-11cf-822d-00aa0051e40f"), "sasec"),
    (uuid.UUID("3858c0d5-0f35-4bf5-9714-69874963bc36"), "IVdsAdvancedDisk3"),
    (uuid.UUID("38a0a9ab-7cc8-4693-ac07-1f28bd03c3da"), "IVdsIscsiInitiatorPortal"),
    (uuid.UUID("38e87280-715c-4c7d-a280-ea1651a19fef"), "IFsrmReportJob"),
    (uuid.UUID("3919286a-b10c-11d0-9ba8-00c04fd92ef5"), "dssetup"),
    (uuid.UUID("39322a2d-38ee-4d0d-8095-421a80849a82"), "IFsrmDerivedObjectsResult"),
    (uuid.UUID("3a410f21-553f-11d1-8e5e-00a0c92c9d5d"), "IDMRemoteServer"),
    (uuid.UUID("3a56bfb8-576c-43f7-9335-fe4838fd7e37"), "ICategoryCollection"),
    (uuid.UUID("3b69d7f5-9d94-4648-91ca-79939ba263bf"), "IVdsPack"),
    (uuid.UUID("3bbed8d9-2c9a-4b21-8936-acb2f995be6c"), "INtmsObjectManagement3"),
    (uuid.UUID("3dde7c30-165d-11d1-ab8f-00805f14db40"), "BackupKey"),
    (uuid.UUID("3f3b1b86-dbbe-11d1-9da6-00805f85cfe3"), "IContainerControl"),
    (uuid.UUID("40f73c8b-687d-4a13-8d96-3d7f2e683936"), "IVdsDisk2"),
    (uuid.UUID("41208ee0-e970-11d1-9b9e-00e02c064c39"), "qmmgmt"),
    (uuid.UUID("4173ac41-172d-4d52-963c-fdc7e415f717"), "IFsrmQuotaTemplateManager"),
    (uuid.UUID("423ec01e-2e35-11d2-b604-00104b703efd"), "IWbemWCOSmartEnum"),
    (uuid.UUID("426677d5-018c-485c-8a51-20b86d00bdc4"), "IFsrmFileGroupManager"),
    (uuid.UUID("42dc3511-61d5-48ae-b6dc-59fc00c0a8d6"), "IFsrmQuotaObject"),
    (uuid.UUID("44aca674-e8fc-11d0-a07c-00c04fb68820"), "IWbemContext"),
    (uuid.UUID("44aca675-e8fc-11d0-a07c-00c04fb68820"), "IWbemCallResult"),
    (uuid.UUID("44e265dd-7daf-42cd-8560-3cdb6e7a2729"), "TsProxyRpcInterface"),
    (uuid.UUID("450386db-7409-4667-935e-384dbbee2a9e"), "IAppHostPropertySchema"),
    (uuid.UUID("456129e2-1078-11d2-b0f9-00805fc73204"), "ICatalogUtils"),
    (uuid.UUID("45f52c28-7f9f-101a-b52b-08002b2efabe"), "winsif"),
    (uuid.UUID("46297823-9940-4c09-aed9-cd3ea6d05968"), "IUpdateIdentity"),
    (uuid.UUID("4639db2a-bfc5-11d2-9318-00c04fbbbfb3"), "IDataFactory3"),
    (uuid.UUID("47782152-d16c-4229-b4e1-0ddfe308b9f6"), "IFsrmPropertyDefinition2"),
    (uuid.UUID("47cde9a1-0bf6-11d2-8016-00c04fb9988e"), "ICapabilitySupport"),
    (uuid.UUID("481e06cf-ab04-4498-8ffe-124a0a34296d"), "IWRMCalendar"),
    (uuid.UUID("4846cb01-d430-494f-abb4-b1054999fb09"), "IFsrmQuotaManagerEx"),
    (uuid.UUID("484809d6-4239-471b-b5bc-61df8c23ac48"), "TermSrvSession"),
    (uuid.UUID("497d95a6-2d27-4bf5-9bbd-a6046957133c"), "RCMListener"),
    (uuid.UUID("49ebd502-4a96-41bd-9e3e-4c5057f4250c"), "IWindowsDriverUpdate3"),
    (uuid.UUID("4a2f5c31-cfd9-410e-b7fb-29a653973a0f"), "IAutomaticUpdates2"),
    (uuid.UUID("4a6b0e15-2e38-11d1-9965-00c04fbbb345"), "IEventSubscription"),
    (uuid.UUID("4a6b0e16-2e38-11d1-9965-00c04fbbb345"), "IEventSubscription2"),
    (uuid.UUID("4a73fee4-4102-4fcc-9ffb-38614f9ee768"), "IFsrmProperty"),
    (uuid.UUID("4afc3636-db01-4052-80c3-03bbcb8d3c69"), "IVdsServiceInitialization"),
    (uuid.UUID("4b324fc8-1670-01d3-1278-5a47bf6ee188"), "srvsvc"),
    (uuid.UUID("4bb8ab1d-9ef9-4100-8eb6-dd4b4e418b72"), "IADProxy"),
    (uuid.UUID("4bdafc52-fe6a-11d2-93f8-00105a11164a"), "IVolumeClient2"),
    (uuid.UUID("4c8f96c3-5d94-4f37-a4f4-f56ab463546f"), "IFsrmActionEventLog"),
    (uuid.UUID("4cbdcb2d-1589-4beb-bd1c-3e582ff0add0"), "IUpdateSearcher2"),
    (uuid.UUID("4d9f4ab8-7d1c-11cf-861e-0020af6e7c57"), "IActivation"),
    (uuid.UUID("4da1c422-943d-11d1-acae-00c04fc2aa3f"), "trksvr"),
    (uuid.UUID("4daa0135-e1d1-40f1-aaa5-3cc1e53221c3"), "IVdsVolumePlex"),
    (uuid.UUID("4dbcee9a-6343-4651-b85f-5e75d74d983c"), "IVdsVolumeMF2"),
    (uuid.UUID("4dfa1df3-8900-4bc7-bbb5-d1a458c52410"), "IAppHostConfigException"),
    (uuid.UUID("4e14fb9f-2e22-11d1-9964-00c04fbbb345"), "IEventSystem"),
    (uuid.UUID("4e6cdcc9-fb25-4fd5-9cc5-c9f4b6559cec"), "IComTrackingInfoEvents"),
    (uuid.UUID("4e934f30-341a-11d1-8fb1-00a024cb6019"), "INtmsLibraryControl1"),
    (uuid.UUID("4f7ca01c-a9e5-45b6-b142-2332a1339c1d"), "IWRMAccounting"),
    (uuid.UUID("4fc742e0-4a10-11cf-8273-00aa004ae673"), "netdfs"),
    (uuid.UUID("503626a3-8e14-4729-9355-0fe664bd2321"), "IUpdateExceptionCollection"),
    (uuid.UUID("50abc2a4-574d-40b3-9d66-ee4fd5fba076"), "DnsServer"),
    (
        uuid.UUID("515c1277-2c81-440e-8fcf-367921ed4f59"),
        "IFsrmPipelineModuleDefinition",
    ),
    (uuid.UUID("5261574a-4572-206e-b268-6b199213b4e4"), "asyncemsmdb"),
    (uuid.UUID("52c80b95-c1ad-4240-8d89-72e9fa84025e"), "IClusCfgAsyncEvictCleanup"),
    (uuid.UUID("538684e0-ba3d-4bc0-aca9-164aff85c2a9"), "IVdsDiskPartitionMF"),
    (uuid.UUID("53b46b02-c73b-4a3e-8dee-b16b80672fc0"), "TSVIPPublic"),
    (uuid.UUID("541679ab-2e5f-11d3-b34e-00104bcc4b4a"), "IWbemLoginHelper"),
    (uuid.UUID("5422fd3a-d4b8-4cef-a12e-e87d4ca22e90"), "ICertRequestD2"),
    (uuid.UUID("54a2cb2d-9a0c-48b6-8a50-9abb69ee2d02"), "IUpdateDownloadContent"),
    (uuid.UUID("59602eb6-57b0-4fd8-aa4b-ebf06971fe15"), "IWRMPolicy"),
    (uuid.UUID("5a7b91f8-ff00-11d0-a9b2-00c04fb6e6fc"), "msgsvcsend"),
    (
        uuid.UUID("5b5a68e6-8b9f-45e1-8199-a95ffccdffff"),
        "IAppHostConstantValueCollection",
    ),
    (uuid.UUID("5b821720-f63b-11d0-aad2-00c04fc324db"), "dhcpsrv2"),
    (uuid.UUID("5ca4a760-ebb1-11cf-8611-00a0245420ed"), "IcaApi"),
    (uuid.UUID("5f6325d3-ce88-4733-84c1-2d6aefc5ea07"), "IFsrmFileScreen"),
    (uuid.UUID("5ff9bdf6-bd91-4d8b-a614-d6317acc8dd8"), "IRemoteSstpCertCheck"),
    (uuid.UUID("6099fc12-3eff-11d0-abd0-00c04fd91a4e"), "faxclient"),
    (uuid.UUID("6139d8a4-e508-4ebb-bac7-d7f275145897"), "IRemoteIPV6Config"),
    (uuid.UUID("615c4269-7a48-43bd-96b7-bf6ca27d6c3e"), "IWindowsDriverUpdate2"),
    (uuid.UUID("64ff8ccc-b287-4dae-b08a-a72cbf45f453"), "IAppHostElement"),
    (uuid.UUID("6619a740-8154-43be-a186-0319578e02db"), "IRemoteDispatch"),
    (uuid.UUID("66a2db1b-d706-11d0-a37b-00c04fc9da04"), "IRemoteNetworkConfig"),
    (uuid.UUID("66a2db20-d706-11d0-a37b-00c04fc9da04"), "IRemoteRouterRestart"),
    (uuid.UUID("66a2db21-d706-11d0-a37b-00c04fc9da04"), "IRemoteSetDnsConfig"),
    (uuid.UUID("66a2db22-d706-11d0-a37b-00c04fc9da04"), "IRemoteICFICSConfig"),
    (uuid.UUID("673425bf-c082-4c7c-bdfd-569464b8e0ce"), "IAutomaticUpdates"),
    (uuid.UUID("6788faf9-214e-4b85-ba59-266953616e09"), "IVdsVolumeMF3"),
    (uuid.UUID("67e08fc2-2984-4b62-b92e-fc1aae64bbbb"), "IRemoteStringIdConfig"),
    (uuid.UUID("6879caf9-6617-4484-8719-71c3d8645f94"), "IFsrmReportScheduler"),
    (uuid.UUID("69ab7050-3059-11d1-8faf-00a024cb6019"), "INtmsObjectInfo1"),
    (uuid.UUID("6a92b07a-d821-4682-b423-5c805022cc4d"), "IUpdate"),
    (uuid.UUID("6b5bdd1e-528c-422c-af8c-a4079be4fe48"), "RemoteFW"),
    (uuid.UUID("6bffd098-a112-3610-9833-012892020162"), "browser"),
    (uuid.UUID("6bffd098-a112-3610-9833-46c3f874532d"), "dhcpsrv"),
    (uuid.UUID("6bffd098-a112-3610-9833-46c3f87e345a"), "wkssvc"),
    (uuid.UUID("6c935649-30a6-4211-8687-c4c83e5fe1c7"), "IContainerControl2"),
    (uuid.UUID("6cd6408a-ae60-463b-9ef1-e117534d69dc"), "IFsrmAction"),
    (uuid.UUID("6e6f6b40-977c-4069-bddd-ac710059f8c0"), "IVdsAdvancedDisk"),
    (uuid.UUID("6f4dbfff-6920-4821-a6c3-b7e94c1fd60c"), "IFsrmPathMapper"),
    (uuid.UUID("708cca10-9569-11d1-b2a5-0060977d8118"), "dscomm2"),
    (uuid.UUID("70b51430-b6ca-11d0-b9b9-00a0c922e750"), "IMSAdminBaseW"),
    (uuid.UUID("70cf5c82-8642-42bb-9dbc-0cfd263c6c4f"), "IWindowsDriverUpdate5"),
    (uuid.UUID("72ae6713-dcbb-4a03-b36b-371f6ac6b53d"), "IVdsVolume2"),
    (uuid.UUID("75c8f324-f715-4fe3-a28e-f9011b61a4a1"), "IVdsOpenVDisk"),
    (uuid.UUID("76b3b17e-aed6-4da5-85f0-83587f81abe3"), "IUpdateService"),
    (uuid.UUID("76d12b80-3467-11d3-91ff-0090272f9ea3"), "qmcomm2"),
    (uuid.UUID("76f03f96-cdfd-44fc-a22c-64950a001209"), "IRemoteWinspool"),
    (uuid.UUID("77df7a80-f298-11d0-8358-00a024c480a8"), "dscomm"),
    (uuid.UUID("784b693d-95f3-420b-8126-365c098659f2"), "IOCSPAdminD"),
    (uuid.UUID("7883ca1c-1112-4447-84c3-52fbeb38069d"), "IAppHostMethod"),
    (uuid.UUID("7c44d7d4-31d5-424c-bd5e-2b3e1f323d22"), "dsaop"),
    (uuid.UUID("7c4e1804-e342-483d-a43e-a850cfcc8d18"), "IIISApplicationAdmin"),
    (uuid.UUID("7c857801-7381-11cf-884d-00aa004b2e24"), "IWbemObjectSink"),
    (uuid.UUID("7c907864-346c-4aeb-8f3f-57da289f969f"), "IImageInformation"),
    (uuid.UUID("7d07f313-a53f-459a-bb12-012c15b1846e"), "IRobustNtmsMediaServices1"),
    (uuid.UUID("7f43b400-1a0e-4d57-bbc9-6b0c65f7a889"), "IAlternateLaunch"),
    (uuid.UUID("7fb7ea43-2d76-4ea8-8cd9-3decc270295e"), "IEventClass3"),
    (uuid.UUID("7fe0d935-dda6-443f-85d0-1cfb58fe41dd"), "ICertAdminD2"),
    (uuid.UUID("811109bf-a4e1-11d1-ab54-00a0c91e9b45"), "winsi2"),
    (uuid.UUID("8165b19e-8d3a-4d0b-80c8-97de310db583"), "IServicedComponentInfo"),
    (uuid.UUID("816858a4-260d-4260-933a-2585f1abc76b"), "IUpdateSession"),
    (uuid.UUID("81ddc1b8-9d35-47a6-b471-5b80f519223b"), "ICategory"),
    (uuid.UUID("82273fdc-e32a-18c3-3f78-827929dc23ea"), "eventlog"),
    (uuid.UUID("8276702f-2532-4839-89bf-4872609a2ea4"), "IFsrmActionEmail2"),
    (uuid.UUID("8298d101-f992-43b7-8eca-5052d885b995"), "IMSAdminBase2W"),
    (uuid.UUID("82ad4280-036b-11cf-972c-00aa006887b0"), "inetinfo"),
    (uuid.UUID("8326cd1d-cf59-4936-b786-5efc08798e25"), "IVdsAdviseSink"),
    (
        uuid.UUID("832a32f7-b3ea-4b8c-b260-9a2923001184"),
        "IAppHostConfigLocationCollection",
    ),
    (uuid.UUID("833e4100-aff7-4ac3-aac2-9f24c1457bce"), "IPCHCollection"),
    (uuid.UUID("833e41aa-aff7-4ac3-aac2-9f24c1457bce"), "ISAFSession"),
    (uuid.UUID("83bfb87f-43fb-4903-baa6-127f01029eec"), "IVdsSubSystemImportTarget"),
    (uuid.UUID("85713fa1-7796-4fa2-be3b-e2d6124dd373"), "IWindowsUpdateAgentInfo"),
    (uuid.UUID("86d35949-83c9-4044-b424-db363231fd0c"), "ITaskSchedulerService"),
    (uuid.UUID("879c8bbe-41b0-11d1-be11-00c04fb6bf70"), "IClientSink"),
    (uuid.UUID("88143fd0-c28d-4b2b-8fef-8d882f6a9390"), "TermSrvEnumeration"),
    (uuid.UUID("88306bb2-e71f-478c-86a2-79da200a0f11"), "IVdsVolume"),
    (uuid.UUID("894de0c0-0d55-11d3-a322-00c04fa321a1"), "InitShutdown"),
    (uuid.UUID("895a2c86-270d-489d-a6c0-dc2a9b35280e"), "INtmsObjectManagement2"),
    (uuid.UUID("897e2e5f-93f3-4376-9c9c-fd2277495c27"), "FrsTransport"),
    (uuid.UUID("8bb68c7d-19d8-4ffb-809e-be4fc1734014"), "IFsrmQuotaManager"),
    (
        uuid.UUID("8bed2c68-a5fb-4b28-8581-a0dc5267419f"),
        "IAppHostPropertySchemaCollection",
    ),
    (uuid.UUID("8db2180e-bd29-11d1-8b7e-00c04fd7a924"), "IRegister"),
    (uuid.UUID("8dd04909-0e34-4d55-afaa-89e1f1a1bbb9"), "IFsrmFileGroup"),
    (uuid.UUID("8f09f000-b7ed-11ce-bbd2-00001a181cad"), "dimsvc"),
    (uuid.UUID("8f45abf1-f9ae-4b95-a933-f0f66e5056ea"), "IUpdateSearcher"),
    (uuid.UUID("8f4b2f5d-ec15-4357-992f-473ef10975b9"), "IVdsDisk3"),
    (uuid.UUID("8f6d760f-f0cb-4d69-b5f6-848b33e9bdc6"), "IAppHostConfigManager"),
    (uuid.UUID("8fb6d884-2388-11d0-8c35-00c04fda2795"), "W32Time"),
    (uuid.UUID("90681b1d-6a7f-48e8-9061-31b7aa125322"), "IVdsDiskOnline"),
    (uuid.UUID("906b0ce0-c70b-1067-b317-00dd010662da"), "IXnRemote"),
    (uuid.UUID("918efd1e-b5d8-4c90-8540-aeb9bdc56f9d"), "IUpdateSession3"),
    (uuid.UUID("91ae6020-9e3c-11cf-8d7c-00aa00c091be"), "ICertPassage"),
    (uuid.UUID("91caf7b0-eb23-49ed-9937-c52d817f46f7"), "IUpdateSession2"),
    (uuid.UUID("943991a5-b3fe-41fa-9696-7f7b656ee34b"), "IWRMMachineGroup"),
    (uuid.UUID("9556dc99-828c-11cf-a37e-00aa003240c7"), "IWbemServices"),
    (uuid.UUID("96deb3b5-8b91-4a2a-9d93-80a35d8aa847"), "IFsrmCommittableCollection"),
    (uuid.UUID("971668dc-c3fe-4ea1-9643-0c7230f494a1"), "IRegister2"),
    (uuid.UUID("97199110-db2e-11d1-a251-0000f805ca53"), "ITransactionStream"),
    (uuid.UUID("9723f420-9355-42de-ab66-e31bb15beeac"), "IVdsAdvancedDisk2"),
    (uuid.UUID("98315903-7be5-11d2-adc1-00a02463d6e7"), "IReplicationUtil"),
    (uuid.UUID("9882f547-cfc3-420b-9750-00dfbec50662"), "IVdsCreatePartitionEx"),
    (uuid.UUID("99cc098f-a48a-4e9c-8e58-965c0afc19d5"), "IEventSystem2"),
    (uuid.UUID("99fcfec4-5260-101b-bbcb-00aa0021347a"), "IObjectExporter"),
    (uuid.UUID("9a2bf113-a329-44cc-809a-5c00fce8da40"), "IFsrmQuotaTemplateImported"),
    (uuid.UUID("9aa58360-ce33-4f92-b658-ed24b14425b8"), "IVdsSwProvider"),
    (uuid.UUID("9b0353aa-0e52-44ff-b8b0-1f7fa0437f88"), "IUpdateServiceCollection"),
    (uuid.UUID("9be77978-73ed-4a9a-87fd-13f09fec1b13"), "IAppHostAdminManager"),
    (uuid.UUID("9cbe50ca-f2d2-4bf4-ace1-96896b729625"), "IVdsDiskPartitionMF2"),
    (
        uuid.UUID("9d07ca0d-8f02-4ed5-b727-acf37fea5bbc"),
        "ISingleSignonRemoteMasterSecret",
    ),
    (uuid.UUID("a0e8f27a-888c-11d1-b763-00c04fb926af"), "IEventSystemInitialize"),
    (uuid.UUID("a2efab31-295e-46bb-b976-e86d58b52e8b"), "IFsrmQuotaTemplate"),
    (uuid.UUID("a359dec5-e813-4834-8a2a-ba7f1d777d76"), "IWbemBackupRestoreEx"),
    (uuid.UUID("a35af600-9cf4-11cd-a076-08002b2bd711"), "type_scard_pack"),
    (uuid.UUID("a376dd5e-09d4-427f-af7c-fed5b6e1c1d6"), "IUpdateException"),
    (uuid.UUID("a4f1db00-ca47-1067-b31f-00dd010662da"), "emsmdb"),
    (
        uuid.UUID("a7f04f3c-a290-435b-aadf-a116c3357a5c"),
        "IUpdateHistoryEntryCollection",
    ),
    (uuid.UUID("a8927a41-d3ce-11d1-8472-006008b0e5ca"), "ICatalogTableInfo"),
    (uuid.UUID("a8e0653c-2744-4389-a61d-7373df8b2292"), "FileServerVssAgent"),
    (uuid.UUID("ad55f10b-5f11-4be7-94ef-d9ee2e470ded"), "IFsrmFileGroupImported"),
    (uuid.UUID("ada4e6fb-e025-401e-a5d0-c3134a281f07"), "IAppHostConfigFile"),
    (uuid.UUID("ae1c7110-2f60-11d3-8a39-00c04f72d8e3"), "IVssEnumObject"),
    (uuid.UUID("afa8bd80-7d8a-11c9-bef4-08002b102989"), "mgmt"),
    (uuid.UUID("afc052c2-5315-45ab-841b-c6db0e120148"), "IFsrmClassificationRule"),
    (uuid.UUID("afc07e2e-311c-4435-808c-c483ffeec7c9"), "lsacap"),
    (uuid.UUID("b057dc50-3059-11d1-8faf-00a024cb6019"), "INtmsObjectManagement1"),
    (uuid.UUID("b07fedd4-1682-4440-9189-a39b55194dc5"), "IVdsIscsiInitiatorAdapter"),
    (uuid.UUID("b196b284-bab4-101a-b69c-00aa00341d07"), "IConnectionPointContainer"),
    (uuid.UUID("b196b285-bab4-101a-b69c-00aa00341d07"), "IEnumConnectionPoints"),
    (uuid.UUID("b196b286-bab4-101a-b69c-00aa00341d07"), "IConnectionPoint"),
    (uuid.UUID("b196b287-bab4-101a-b69c-00aa00341d07"), "IEnumConnections"),
    (uuid.UUID("b383cd1a-5ce9-4504-9f63-764b1236f191"), "IWindowsDriverUpdate"),
    (uuid.UUID("b481498c-8354-45f9-84a0-0bdd2832a91f"), "IVdsVdProvider"),
    (uuid.UUID("b60040e0-bcf3-11d1-861d-0080c729264d"), "IGetTrackingData"),
    (uuid.UUID("b6b22da8-f903-4be7-b492-c09d875ac9da"), "IVdsServiceUninstallDisk"),
    (
        uuid.UUID("b7d381ee-8860-47a1-8af4-1f33b2b1f325"),
        "IAppHostSectionDefinitionCollection",
    ),
    (uuid.UUID("b80f3c42-60e0-4ae0-9007-f52852d3dbed"), "IAppHostMethodInstance"),
    (uuid.UUID("b9785960-524f-11df-8b6d-83dcded72085"), "ISDKey"),
    (uuid.UUID("b97db8b2-4c63-11cf-bff6-08002be23f2f"), "clusapi"),
    (uuid.UUID("b97db8b2-4c63-11cf-bff6-08002be23f2f"), "clusapi"),
    (
        uuid.UUID("bb36ea26-6318-4b8c-8592-f72dd602e7a5"),
        "IFsrmClassifierModuleDefinition",
    ),
    (uuid.UUID("bb39332c-bfee-4380-ad8a-badc8aff5bb6"), "INtmsNotifySink"),
    (uuid.UUID("bba9cb76-eb0c-462c-aa1b-5d8c34415701"), "Claims"),
    (
        uuid.UUID("bc5513c8-b3b8-4bf7-a4d4-361c0d8c88ba"),
        "IUpdateDownloadContentCollection",
    ),
    (uuid.UUID("bc681469-9dd9-4bf4-9b3d-709f69efe431"), "IWRMResourceGroup"),
    (uuid.UUID("bd0c73bc-805b-4043-9c30-9a28d64dd7d2"), "IIISCertObj"),
    (uuid.UUID("bd7c23c2-c805-457c-8f86-d17fe6b9d19f"), "IClusterLogEx"),
    (uuid.UUID("bde95fdf-eee0-45de-9e12-e5a61cd0d4fe"), "RCMPublic"),
    (uuid.UUID("be56a644-af0e-4e0e-a311-c1d8e695cbff"), "IUpdateHistoryEntry"),
    (uuid.UUID("bee7ce02-df77-4515-9389-78f01c5afc1a"), "IFsrmFileScreenException"),
    (uuid.UUID("c1c2f21a-d2f4-4902-b5c6-8a081c19a890"), "IUpdate5"),
    (uuid.UUID("c2be6970-df9e-11d1-8b87-00c04fd7a924"), "IImport"),
    (uuid.UUID("c2bfb780-4539-4132-ab8c-0a8772013ab6"), "IUpdateHistoryEntry2"),
    (uuid.UUID("c3fcc19e-a970-11d2-8b5a-00a0c9b7c9c4"), "IManagedObject"),
    (uuid.UUID("c49e32c7-bc8b-11d2-85d4-00105a1f8304"), "IWbemBackupRestore"),
    (uuid.UUID("c4b0c7d9-abe0-4733-a1e1-9fdedf260c7a"), "IADProxy2"),
    (uuid.UUID("c5c04795-321c-4014-8fd6-d44658799393"), "IAppHostSectionDefinition"),
    (uuid.UUID("c5cebee2-9df5-4cdd-a08c-c2471bc144b4"), "IResourceManager"),
    (uuid.UUID("c681d488-d850-11d0-8c52-00c04fd90f7e"), "efsrpc"),
    (uuid.UUID("c726744e-5735-4f08-8286-c510ee638fb6"), "ICatalogUtils2"),
    (uuid.UUID("c8550bff-5281-4b1e-ac34-99b6fa38464d"), "IAppHostElementCollection"),
    (uuid.UUID("c97ad11b-f257-420b-9d9f-377f733f6f68"), "IUpdateDownloadContent2"),
    (uuid.UUID("cb0df960-16f5-4495-9079-3f9360d831df"), "IFsrmRule"),
    (uuid.UUID("ccd8c074-d0e5-4a40-92b4-d074faa6ba28"), "Witness"),
    (uuid.UUID("cfadac84-e12c-11d1-b34c-00c04f990d54"), "IExport"),
    (
        uuid.UUID("cfe36cba-1949-4e74-a14f-f1d580ceaf13"),
        "IFsrmFileScreenTemplateManager",
    ),
    (uuid.UUID("d02e4be0-3419-11d1-8fb1-00a024cb6019"), "INtmsMediaServices1"),
    (uuid.UUID("d049b186-814f-11d1-9a3c-00c04fc9b232"), "NtFrsApi"),
    (uuid.UUID("d2d79df5-3400-11d0-b40b-00aa005ff586"), "IVolumeClient"),
    (uuid.UUID("d2d79df7-3400-11d0-b40b-00aa005ff586"), "IDMNotify"),
    (uuid.UUID("d2dc89da-ee91-48a0-85d8-cc72a56f7d04"), "IFsrmClassificationManager"),
    (uuid.UUID("d40cff62-e08c-4498-941a-01e25f0fd33c"), "ISearchResult"),
    (uuid.UUID("d4781cd6-e5d3-44df-ad94-930efe48a887"), "IWbemLoginClientID"),
    (uuid.UUID("d5d23b6d-5a55-4492-9889-397a3c2d2dbc"), "IVdsAsync"),
    (uuid.UUID("d646567d-26ae-4caa-9f84-4e0aad207fca"), "IFsrmActionEmail"),
    (uuid.UUID("d68168c9-82a2-4f85-b6e9-74707c49a58f"), "IVdsVolumeShrink"),
    (uuid.UUID("d6c7cd8f-bb8d-4f96-b591-d3a5f1320269"), "IAppHostMethodCollection"),
    (uuid.UUID("d8cc81d9-46b8-4fa4-bfa5-4aa9dec9b638"), "IFsrmReport"),
    (uuid.UUID("d95afe70-a6d5-4259-822e-2c84da1ddb0d"), "WindowsShutdown"),
    (uuid.UUID("d99bdaae-b13a-4178-9fdb-e27f16b4603e"), "IVdsHwProvider"),
    (uuid.UUID("d99e6e70-fc88-11d0-b498-00a0c90312f3"), "ICertRequestD"),
    (uuid.UUID("d99e6e71-fc88-11d0-b498-00a0c90312f3"), "ICertAdminD"),
    (uuid.UUID("d9a59339-e245-4dbd-9686-4d5763e39624"), "IInstallationBehavior"),
    (uuid.UUID("da5a86c5-12c2-4943-ab30-7f74a813d853"), "PerflibV2"),
    (uuid.UUID("db90832f-6910-4d46-9f5e-9fd6bfa73903"), "INtmsLibraryControl2"),
    (uuid.UUID("dc12a681-737f-11cf-884d-00aa004b2e24"), "IWbemClassObject"),
    (uuid.UUID("dde02280-12b3-4e0b-937b-6747f6acb286"), "IUpdateServiceRegistration"),
    (uuid.UUID("de095db1-5368-4d11-81f6-efef619b7bcf"), "IAppHostCollectionSchema"),
    (uuid.UUID("deb01010-3a37-4d26-99df-e2bb6ae3ac61"), "IVolumeClient4"),
    (uuid.UUID("e0393303-90d4-4a97-ab71-e9b671ee2729"), "IVdsServiceLoader"),
    (
        uuid.UUID("e1010359-3e5d-4ecd-9fe4-ef48622fdf30"),
        "IFsrmFileScreenTemplateImported",
    ),
    (uuid.UUID("e1af8308-5d1f-11c9-91a4-08002b14a0fa"), "ept"),
    (uuid.UUID("e33c0cc4-0482-101a-bc0c-02608c6ba218"), "LocToLoc"),
    (uuid.UUID("e3514235-4b06-11d1-ab04-00c04fc2dcd2"), "drsuapi"),
    (uuid.UUID("e3d0d746-d2af-40fd-8a7a-0d7078bb7092"), "BitsPeerAuth"),
    (uuid.UUID("e65e8028-83e8-491b-9af7-aaf6bd51a0ce"), "IServerHealthReport"),
    (uuid.UUID("e7927575-5cc3-403b-822e-328a6b904bee"), "IAppHostPathMapper"),
    (uuid.UUID("e7a4d634-7942-4dd9-a111-82228ba33901"), "IAutomaticUpdatesResults"),
    (uuid.UUID("e8fb8620-588f-11d2-9d61-00c04f79c5fe"), "IIisServiceControl"),
    (uuid.UUID("e946d148-bd67-4178-8e22-1c44925ed710"), "IFsrmPropertyDefinitionValue"),
    (uuid.UUID("ea0a3165-4834-11d2-a6f8-00c04fa346cc"), "fax"),
    (uuid.UUID("eafe4895-a929-41ea-b14d-613e23f62b71"), "IAppHostPropertyException"),
    (uuid.UUID("ed35f7a1-5024-4e7b-a44d-07ddaf4b524d"), "IAppHostProperty"),
    (uuid.UUID("ed8bfe40-a60b-42ea-9652-817dfcfa23ec"), "IWindowsDriverUpdateEntry"),
    (uuid.UUID("ede0150f-e9a3-419c-877c-01fe5d24c5d3"), "IFsrmPropertyDefinition"),
    (uuid.UUID("ee2d5ded-6236-4169-931d-b9778ce03dc6"), "IVdsVolumeMF"),
    (
        uuid.UUID("ee321ecb-d95e-48e9-907c-c7685a013235"),
        "IFsrmFileManagementJobManager",
    ),
    (uuid.UUID("ef13d885-642c-4709-99ec-b89561c6bc69"), "IAppHostElementSchema"),
    (uuid.UUID("eff90582-2ddc-480f-a06d-60f3fbc362c3"), "IStringCollection"),
    (uuid.UUID("f131ea3e-b7be-480e-a60d-51cb2785779e"), "IExport2"),
    (uuid.UUID("f1e9c5b2-f59b-11d2-b362-00105a1f8177"), "IWbemRemoteRefresher"),
    (uuid.UUID("f309ad18-d86a-11d0-a075-00c04fb68820"), "IWbemLevel1Login"),
    (uuid.UUID("f31931a9-832d-481c-9503-887a0e6a79f0"), "IWRMProtocol"),
    (uuid.UUID("f3637e80-5b22-4a2b-a637-bbb642b41cfc"), "IFsrmFileScreenBase"),
    (uuid.UUID("f411d4fd-14be-4260-8c40-03b7c95e608a"), "IFsrmSetting"),
    (uuid.UUID("f4a07d63-2e25-11d1-9964-00c04fbbb345"), "IEnumEventObject"),
    (uuid.UUID("f5cc59b4-4264-101a-8c59-08002b2f8426"), "frsrpc"),
    (uuid.UUID("f5cc5a18-4264-101a-8c59-08002b2f8426"), "nspi"),
    (uuid.UUID("f612954d-3b0b-4c56-9563-227b7be624b4"), "IMSAdminBase3W"),
    (uuid.UUID("f6beaff7-1e19-4fbb-9f8f-b89e2018337c"), "IEventService"),
    (uuid.UUID("f76fbf3b-8ddd-4b42-b05a-cb1c3ff1fee8"), "IFsrmCollection"),
    (uuid.UUID("f82e5729-6aba-4740-bfc7-c7f58f75fb7b"), "IFsrmAutoApplyQuota"),
    (uuid.UUID("f89ac270-d4eb-11d1-b682-00805fc79216"), "IEventObjectCollection"),
    (uuid.UUID("fa7660f6-7b3f-4237-a8bf-ed0ad0dcbbd9"), "IAppHostWritableAdminManager"),
    (uuid.UUID("fa7df749-66e7-4986-a27f-e2f04ae53772"), "IVssSnapshotMgmt"),
    (uuid.UUID("fb2b72a0-7a68-11d1-88f9-0080c7d771bf"), "IEventClass"),
    (uuid.UUID("fb2b72a1-7a68-11d1-88f9-0080c7d771bf"), "IEventClass2"),
    (uuid.UUID("fbc1d17d-c498-43a0-81af-423ddd530af6"), "IEventSubscription3"),
    (uuid.UUID("fc5d23e8-a88b-41a5-8de0-2d2f73c5a630"), "IVdsServiceSAN"),
    (uuid.UUID("fc910418-55ca-45ef-b264-83d4ce7d30e0"), "IWRMRemoteSessionMgmt"),
    (uuid.UUID("fdb3a030-065f-11d1-bb9b-00a024ea5525"), "qmcomm"),
    (uuid.UUID("ff4fa04e-5a94-4bda-a3a0-d5b4d3c52eba"), "IFsrmFileScreenManager"),
]

for uid, name in _DCE_RPC_WELL_KNOWN_UUIDS:
    DCE_RPC_INTERFACES_NAMES[uid] = name
    DCE_RPC_INTERFACES_NAMES_rev[name.lower()] = uid
