"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _next_url = require("../../../utils/next_url");
var _routes = require("./routes");
var _types = require("../../types");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "authTypes", void 0);
    _defineProperty(this, "authHandlers", void 0);
    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }
  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();
    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }
        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }
        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }
        case _common.AuthType.PROXY:
          {
            const ProxyAuth = new _types.ProxyAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await ProxyAuth.init();
            this.authHandlers.set(_common.AuthType.PROXY, ProxyAuth);
            break;
          }
        case _common.AuthType.JWT:
          {
            const JwtAuth = new _types.JwtAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await JwtAuth.init();
            this.authHandlers.set(_common.AuthType.JWT, JwtAuth);
            break;
          }
        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  }

  // override functions inherited from AuthenticationType
  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    var _cookie$authType;
    // To Do: refactor this method to improve the efficiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return await this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      const authHeaders = {};
      for (const handler of this.authHandlers.values()) {
        Object.assign(authHeaders, await handler.getAdditionalAuthHeader(request));
      }
      return authHeaders;
    }
  }
  getCookie(request, authInfo) {
    // TODO: This logic is only applicable for JWT auth type
    for (const handler of this.authHandlers.values()) {
      if (handler.requestIncludesAuthInfo(request)) {
        return handler.getCookie(request, authInfo);
      }
    }
    return {};
  }
  getKeepAliveExpiry(cookie, request) {
    var _cookie$authType2;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).getKeepAliveExpiry(cookie, request);
    } else {
      // default to TTL setting
      return Date.now() + this.config.session.ttl;
    }
  }
  async isValidCookie(cookie, request) {
    var _cookie$authType3;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie, request);
    } else {
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$authType4;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType4 = cookie.authType) === null || _cookie$authType4 === void 0 ? void 0 : _cookie$authType4.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie, request);
    } else {
      return {};
    }
  }
}
exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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