#ifndef LOGGING_H_
#define LOGGING_H_

#include <QMutexLocker>
#include <QMutex>
#include <QQueue>
#include <QPointer>
#include <QCoreApplication>

#include <cstdint>
#include <cstdlib>

#include "mythconfig.h"
#include "mythbaseexp.h"  //  MBASE_PUBLIC , etc.
#include "verbosedefs.h"
#include "mthread.h"
#include "referencecounter.h"
#include "compat.h"
#include "mythchrono.h"

#ifdef _MSC_VER
#  include <unistd.h>	// pid_t
#endif

#if CONFIG_SYSTEMD_JOURNAL
static constexpr int SYSTEMD_JOURNAL_FACILITY { -99 };
#endif

class QString;
class MSqlQuery;
class LoggingItem;

void loggingRegisterThread(const QString &name);
void loggingDeregisterThread(void);

class QWaitCondition;

enum LoggingType {
    kMessage       = 0x01,
    kRegistering   = 0x02,
    kDeregistering = 0x04,
    kFlush         = 0x08,
    kStandardIO    = 0x10,
    kInitializing  = 0x20,
};

class LoggerThread;

using tmType = struct tm;

/// \brief The logging items that are generated by LOG() and are sent to the
///        console
class LoggingItem: public QObject, public ReferenceCounter
{
    Q_OBJECT

    Q_PROPERTY(int pid READ pid WRITE setPid)
    Q_PROPERTY(qlonglong tid READ tid WRITE setTid)
    Q_PROPERTY(qulonglong threadId READ threadId WRITE setThreadId)
    Q_PROPERTY(int line READ line WRITE setLine)
    Q_PROPERTY(int type READ type WRITE setType)
    Q_PROPERTY(int level READ level WRITE setLevel)
    Q_PROPERTY(int facility READ facility WRITE setFacility)
    Q_PROPERTY(qlonglong epoch READ epochInt WRITE setEpochInt)
    Q_PROPERTY(QString file READ file WRITE setFile)
    Q_PROPERTY(QString function READ function WRITE setFunction)
    Q_PROPERTY(QString threadName READ threadName WRITE setThreadName)
    Q_PROPERTY(QString appName READ appName WRITE setAppName)
    Q_PROPERTY(QString table READ table WRITE setTable)
    Q_PROPERTY(QString logFile READ logFile WRITE setLogFile)
    Q_PROPERTY(QString message READ message WRITE setMessage)

    friend class LoggerThread;
    friend MBASE_PUBLIC void LogPrintLine(uint64_t mask, LogLevel_t level, const char *file, int line,
                             const char *function, QString message);

  public:
    QString getThreadName(void);
    int64_t getThreadTid(void);
    void setThreadTid(void);
    static LoggingItem *create(const char *_file, const char *_function, int _line, LogLevel_t _level,
                               LoggingType _type);
    static LoggingItem *create(QByteArray &buf);
    QByteArray toByteArray(void);
    QString getTimestamp(const char *format = "yyyy-MM-dd HH:mm:ss") const;
    QString getTimestampUs(const char *format = "yyyy-MM-dd HH:mm:ss") const;
    char getLevelChar(void);

    int                 pid() const         { return m_pid; };
    qlonglong           tid() const         { return m_tid; };
    qulonglong          threadId() const    { return m_threadId; };
    int                 line() const        { return m_line; };
    int                 type() const        { return (int)m_type; };
    int                 level() const       { return (int)m_level; };
    int                 facility() const    { return m_facility; };
    std::chrono::microseconds epoch() const { return m_epoch; };
    qlonglong epochInt() const              { return m_epoch.count(); };
    QString             file() const        { return m_file; };
    QString             function() const    { return m_function; };
    QString             threadName() const  { return m_threadName; };
    QString             appName() const     { return m_appName; };
    QString             table() const       { return m_table; };
    QString             logFile() const     { return m_logFile; };
    QString             message() const     { return m_message; };

    void setPid(const int val)              { m_pid = val; };
    void setTid(const qlonglong val)        { m_tid = val; };
    void setThreadId(const qulonglong val)  { m_threadId = val; };
    void setLine(const int val)             { m_line = val; };
    void setType(const int val)             { m_type = (LoggingType)val; };
    void setLevel(const int val)            { m_level = (LogLevel_t)val; };
    void setFacility(const int val)         { m_facility = val; };
    void setEpoch(std::chrono::microseconds val) { m_epoch = val; };
    void setEpochInt(const qlonglong val)   { setEpoch(std::chrono::microseconds(val)); };
    void setFile(const QString &val)        { m_file = val; };
    void setFunction(const QString &val)    { m_function = val; };
    void setThreadName(const QString &val)  { m_threadName = val; };
    void setAppName(const QString &val)     { m_appName = val; };
    void setTable(const QString &val)       { m_table = val; };
    void setLogFile(const QString &val)     { m_logFile = val; };
    void setMessage(const QString &val)     { m_message = val; };

  protected:
    int                 m_pid        {-1};
    qlonglong           m_tid        {-1};
    qulonglong          m_threadId   {UINT64_MAX};
    int                 m_line       {0};
    LoggingType         m_type       {kMessage};
    LogLevel_t          m_level      {LOG_INFO};
    int                 m_facility   {0};
    std::chrono::microseconds m_epoch {0us};
    QString             m_file       {};
    QString             m_function   {};
    QString             m_threadName {};
    QString             m_appName    {};
    QString             m_table      {};
    QString             m_logFile    {};
    QString             m_message    {};

  private:
    LoggingItem()
        : ReferenceCounter("LoggingItem", false) {};
    LoggingItem(const char *_file, const char *_function,
                int _line, LogLevel_t _level, LoggingType _type);
    Q_DISABLE_COPY(LoggingItem);
};

/// \brief The logging thread that consumes the logging queue and dispatches
///        each LoggingItem
class LoggerThread : public QObject, public MThread
{
    Q_OBJECT

    friend MBASE_PUBLIC void LogPrintLine(uint64_t mask, LogLevel_t level, const char *file, int line,
                             const char *function, QString message);
  public:
    LoggerThread(QString filename, bool progress, bool quiet, QString table,
                 int facility);
    ~LoggerThread() override;
    void run(void) override; // MThread
    void stop(void);
    bool flush(int timeoutMS = 200000);
    static void handleItem(LoggingItem *item);
    void fillItem(LoggingItem *item);
  private:
    Q_DISABLE_COPY(LoggerThread);
    QWaitCondition *m_waitNotEmpty {nullptr};
                                    ///< Condition variable for waiting
                                    ///  for the queue to not be empty
                                    ///  Protected by logQueueMutex
    QWaitCondition *m_waitEmpty    {nullptr};
                                    ///< Condition variable for waiting
                                    ///  for the queue to be empty
                                    ///  Protected by logQueueMutex
    bool    m_aborted {false};      ///< Flag to abort the thread.
                                    ///  Protected by logQueueMutex
    QString m_filename;    ///< Filename of debug logfile
    bool    m_progress;    ///< show only LOG_ERR and more important (console only)
    bool    m_quiet;       ///< silence the console (console only)
    QString m_appname {QCoreApplication::applicationName()};
                           ///< Cached application name
    QString m_tablename;   ///< Cached table name for db logging
    int     m_facility;    ///< Cached syslog facility (or -1 to disable)
    pid_t   m_pid;         ///< Cached pid value

  protected:
    bool logConsole(LoggingItem *item) const;
};

#endif

/*
 * vim:ts=4:sw=4:ai:et:si:sts=4
 */
