/***************************************************************
 * Name:      dxtimepickerctrl.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2020
 * License:   GPL3
 **************************************************************/

#include <wx/gbsizer.h>
#include "dxtimepickerctrl.h"
#include "dxsettings.h"
#include "dxdefs.h"
#include "dxutils.h"
#include "dxicons.h"

BEGIN_EVENT_TABLE(dxTimePickerCtrl, wxPanel)
    EVT_BUTTON(ID_TIME_SELECT, dxTimePickerCtrl::OnSelect)
END_EVENT_TABLE()

dxTimePickerCtrl::dxTimePickerCtrl(wxWindow *parent, const wxDateTime &time)
    : wxPanel(parent), m_time(time)
{
    m_mainSizer = new wxBoxSizer(wxHORIZONTAL);
    m_text = new wxStaticText(this, wxID_ANY, dxutils::formatTime(m_time, dxsettings.timeFormat(), true));
    m_mainSizer->Add(m_text, 0, wxALIGN_CENTER_VERTICAL);
    m_mainSizer->AddSpacer(5);
    m_select = new wxBitmapButton(this, ID_TIME_SELECT, ICO_TIME);
    m_select->SetBitmapDisabled(ICO_GTIME);
    m_select->SetToolTip(_("Select time"));
    m_mainSizer->Add(m_select, 0, wxEXPAND|wxALL);
    this->SetSizer(m_mainSizer);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

wxDateTime dxTimePickerCtrl::time() const
{
    return m_time;
}

void dxTimePickerCtrl::setTime(const wxDateTime &time)
{
    m_time = time;
    m_text->SetLabel(dxutils::formatTime(m_time, dxsettings.timeFormat(), true));
}

bool dxTimePickerCtrl::Enable(bool enable)
{
    m_text->Enable(enable);
    m_select->Enable(enable);
    return true;
}

void dxTimePickerCtrl::OnSelect(wxCommandEvent &/*event*/)
{
    TimeDialog dialog(this,m_time);
    if(dialog.ShowModal() == wxID_OK)
    {
        setTime(dialog.time());
        m_mainSizer->Layout();
        m_mainSizer->Fit(this);
        m_mainSizer->SetSizeHints(this);
    }
}

IMPLEMENT_DYNAMIC_CLASS(TimeDialog, wxDialog)

TimeDialog::TimeDialog(wxWindow *parent, const wxDateTime &time)
    : wxDialog(parent, wxID_ANY, _("Select time")), m_time(time)
{
    m_time.SetSecond(0);
    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);
    wxGridBagSizer *gridSizer = new wxGridBagSizer(5, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Hours:")), wxGBPosition(0,0), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_hours = new wxSpinCtrl(this, ID_TIME_HOURS, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS|wxALIGN_RIGHT|wxSP_WRAP);
    m_hours->SetRange(dxsettings.useAMPM()?1:0,dxsettings.useAMPM()?12:23);
    changeHours();
    gridSizer->Add(m_hours, wxGBPosition(1,0), wxDefaultSpan, wxEXPAND|wxALL, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Minutes:")), wxGBPosition(0,1), wxDefaultSpan, wxALIGN_CENTER_VERTICAL, 5);
    m_minutes = new wxSpinCtrl(this, ID_TIME_MINUTES, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS|wxALIGN_RIGHT|wxSP_WRAP);
    m_minutes->SetRange(0,59);
    m_minutes->SetValue(m_time.GetMinute());
    gridSizer->Add(m_minutes, wxGBPosition(1,1), wxDefaultSpan, wxEXPAND|wxALL, 5);
    if(dxsettings.useAMPM())
    {
        wxBoxSizer *boxSizer = new wxBoxSizer(wxHORIZONTAL);
        m_AMPM = new wxStaticText(this, wxID_ANY, m_time.GetHour()>11?_("PM"):_("AM"));
        boxSizer->Add(m_AMPM, 0, wxALIGN_CENTER_VERTICAL);
        m_AMPMSpin = new wxSpinButton(this, ID_TIME_AMPM);
        boxSizer->Add(m_AMPMSpin);
        gridSizer->Add(boxSizer, wxGBPosition(1,2), wxDefaultSpan, wxEXPAND|wxALL, 5);
    }
    mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);
    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);
    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);

    Connect(ID_TIME_HOURS, wxEVT_SPINCTRL, wxSpinEventHandler(TimeDialog::OnSpinCtrl), nullptr, this);
    Connect(ID_TIME_MINUTES, wxEVT_SPINCTRL, wxSpinEventHandler(TimeDialog::OnSpinCtrl), nullptr, this);
    Connect(ID_TIME_SECONDS, wxEVT_SPINCTRL, wxSpinEventHandler(TimeDialog::OnSpinCtrl), nullptr, this);
    Connect(ID_TIME_AMPM, wxEVT_SPIN_UP, wxSpinEventHandler(TimeDialog::OnSpinUp), nullptr, this);
    Connect(ID_TIME_AMPM, wxEVT_SPIN_DOWN, wxSpinEventHandler(TimeDialog::OnSpinDown), nullptr, this);
}

wxDateTime TimeDialog::time() const
{
    return m_time;
}

void TimeDialog::OnSpinCtrl(wxSpinEvent &event)
{
    if(event.GetId() == ID_TIME_HOURS)
    {
        if(dxsettings.useAMPM())
        {
            if(m_AMPM->GetLabelText() == _("AM"))
            {
                if(m_hours->GetValue() == 12) m_time.SetHour(0);
                else m_time.SetHour(m_hours->GetValue());
            }
            else
            {
                if(m_hours->GetValue() == 12) m_time.SetHour(12);
                else m_time.SetHour(m_hours->GetValue()+12);
            }
        }
        else
        {
            m_time.SetHour(m_hours->GetValue());
        }
    }
    if(event.GetId() == ID_TIME_MINUTES)
    {
        m_time.SetMinute(m_minutes->GetValue());
    }
}

void TimeDialog::OnSpinUp(wxSpinEvent &/*event*/)
{
    m_time.SetHour((m_time.GetHour() + 12) % 24);
    m_AMPM->SetLabel(m_time.GetHour()>11?_("PM"):_("AM"));
    changeHours();
}

void TimeDialog::OnSpinDown(wxSpinEvent &/*event*/)
{
    m_time.SetHour((m_time.GetHour() + 12) % 24);
    m_AMPM->SetLabel(m_time.GetHour()>11?_("PM"):_("AM"));
    changeHours();
}

void TimeDialog::changeHours()
{
    if(dxsettings.useAMPM())
    {
        if(m_time.GetHour()==0 || m_time.GetHour()==12)
        {
            m_hours->SetValue(12);
            return;
        }
        if(m_time.GetHour()<12) m_hours->SetValue(m_time.GetHour());
        else m_hours->SetValue(m_time.GetHour()-12);
    }
    else
    {
        m_hours->SetValue(m_time.GetHour());
    }
}
