/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN mul

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/mul.h>

static int
test_simde_mm512_mul_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT32_C(  1303646110), -INT32_C(  1991094019), -INT32_C(    60179606), -INT32_C(  1143961245),  INT32_C(    53151249),  INT32_C(  1139066569), -INT32_C(  1576434937), -INT32_C(  2053958169),
         INT32_C(   584206116),  INT32_C(    44770456),  INT32_C(  1123947743), -INT32_C(  1342320995), -INT32_C(  1632491307), -INT32_C(   169699602),  INT32_C(     9956121),  INT32_C(   612706816) },
      {  INT32_C(   172382066),  INT32_C(  1527574908), -INT32_C(  1549923834), -INT32_C(  1387095080), -INT32_C(  1303640636), -INT32_C(  1247269221),  INT32_C(   431308569), -INT32_C(   599966870),
         INT32_C(   249988242),  INT32_C(  2070606453), -INT32_C(   736163588),  INT32_C(  1736602019),  INT32_C(   303746678),  INT32_C(   365413116),  INT32_C(  1798208513),  INT32_C(  1246194615) },
      {  INT64_C(  224725209774663260),  INT64_C(   93273805660129404), -INT64_C(   69290128050554364), -INT64_C(  679929896799075153),
         INT64_C(  146044659904488072), -INT64_C(  827409403211381884), -INT64_C(  495863811365128146),  INT64_C(   17903181538658073) } },
    { {  INT32_C(  1700277743),  INT32_C(   467714591),  INT32_C(  1861287882), -INT32_C(   405441935),  INT32_C(  1023012672), -INT32_C(  1286487887), -INT32_C(   199327939), -INT32_C(   633444630),
        -INT32_C(  1287678061),  INT32_C(   617488217), -INT32_C(  1869431265),  INT32_C(  1886873392),  INT32_C(   145518935),  INT32_C(  1857813809),  INT32_C(  1734597244), -INT32_C(   750673600) },
      { -INT32_C(  1836679112), -INT32_C(  1095346785),  INT32_C(  1145980947),  INT32_C(   112510639),  INT32_C(  1745838391), -INT32_C(   606614946), -INT32_C(   465421660),  INT32_C(   347636699),
        -INT32_C(  1566163453), -INT32_C(  1486791533),  INT32_C(  1407954852), -INT32_C(  1403347083),  INT32_C(  1595238656), -INT32_C(   667227085),  INT32_C(    29129766),  INT32_C(    34960639) },
      { -INT64_C( 3122864615166604216),  INT64_C( 2133000449653984254),  INT64_C( 1786014797257090752),  INT64_C(   92771540253758740),
         INT64_C( 2016714318368104633), -INT64_C( 2632074820037247780),  INT64_C(  232137430291951360),  INT64_C(   50528411821964904) } },
    { {  INT32_C(  1185201075), -INT32_C(  1142094569),  INT32_C(   705681589),  INT32_C(  2027383160),  INT32_C(    98036946), -INT32_C(    19066408), -INT32_C(  1929405811), -INT32_C(  1047653106),
        -INT32_C(   402115632), -INT32_C(   308021960),  INT32_C(  1176023758), -INT32_C(   306254053), -INT32_C(  1275881765),  INT32_C(   900845735),  INT32_C(  2042736746), -INT32_C(  1774563131) },
      { -INT32_C(  1149353341),  INT32_C(   111747384), -INT32_C(   280182316), -INT32_C(  1948448080),  INT32_C(  1212076192),  INT32_C(   192802720),  INT32_C(  1703165599),  INT32_C(   301710990),
         INT32_C(   969767169),  INT32_C(  1849652890), -INT32_C(   413234377),  INT32_C(   913456021), -INT32_C(  1417760757),  INT32_C(  1052179359), -INT32_C(   928826823), -INT32_C(    86401287) },
      { -INT64_C( 1362214815308041575), -INT64_C(  197719501964580124),  INT64_C(  118828248182989632), -INT64_C( 3286097603805895789),
        -INT64_C(  389958538055285808), -INT64_C(  485973444974328766),  INT64_C( 1808895096988896105), -INT64_C( 1897348682012537958) } },
    { { -INT32_C(  1305237993),  INT32_C(  1394635292), -INT32_C(  1841660163), -INT32_C(   993481543), -INT32_C(    76528036),  INT32_C(  2067408449),  INT32_C(  1514397025), -INT32_C(  1823204228),
        -INT32_C(   549091389), -INT32_C(   164403463), -INT32_C(  1635226140),  INT32_C(  1986154778), -INT32_C(   646786409),  INT32_C(  1515498745),  INT32_C(    95721353),  INT32_C(  1989740723) },
      { -INT32_C(  2007572849),  INT32_C(   696158532),  INT32_C(   281478902), -INT32_C(   276354729), -INT32_C(   171378180),  INT32_C(   776936613),  INT32_C(  1714684851), -INT32_C(  1696740085),
        -INT32_C(   266194005), -INT32_C(  1306943300), -INT32_C(    20717402),  INT32_C(    99437065), -INT32_C(   386222781), -INT32_C(  2078914095),  INT32_C(  1474972236), -INT32_C(  1007565033) },
      {  INT64_C( 2620360356230052057), -INT64_C(  518388480538381026),  INT64_C(   13115235528654480),  INT64_C( 2596713637166968275),
         INT64_C(  146164835948922945),  INT64_C(   33877637303288280),  INT64_C(  249803645596983429),  INT64_C(  141186338067355308) } },
    { { -INT32_C(  1229777926),  INT32_C(  1516883123), -INT32_C(  1252512596), -INT32_C(  1178909322), -INT32_C(   878594566),  INT32_C(  1263515647),  INT32_C(   430127362), -INT32_C(    69430271),
         INT32_C(  1538428840),  INT32_C(   129309531), -INT32_C(  1111683769),  INT32_C(  1282832466),  INT32_C(   739710765), -INT32_C(   797415730), -INT32_C(  1578493024),  INT32_C(  1469892271) },
      { -INT32_C(  1313649066), -INT32_C(  1330026391), -INT32_C(   932350346),  INT32_C(   454419438), -INT32_C(   918016774),  INT32_C(   865714323), -INT32_C(  1965784101), -INT32_C(  1595772854),
         INT32_C(   676435391),  INT32_C(  1943603965),  INT32_C(  1849443968),  INT32_C(   613044522),  INT32_C(   284021373),  INT32_C(  1833142162),  INT32_C(  1425479434),  INT32_C(  1207228808) },
      {  INT64_C( 1615496623877317116),  INT64_C( 1167780552249958216),  INT64_C(  806564549133250084), -INT64_C(  845537529624671562),
         INT64_C( 1040647713911076440), -INT64_C( 2055996840900555392),  INT64_C(  210093667098180345), -INT64_C( 2250109342424468416) } },
    { {  INT32_C(  1819231854), -INT32_C(   773896112), -INT32_C(  1187046513), -INT32_C(   354563732),  INT32_C(   771410843),  INT32_C(  1553612370), -INT32_C(   575565227), -INT32_C(   635132565),
         INT32_C(  1011258603),  INT32_C(  1796023772), -INT32_C(  1390130111), -INT32_C(  1315503594),  INT32_C(   534745805),  INT32_C(   628849104),  INT32_C(  1996696587), -INT32_C(  1118754862) },
      { -INT32_C(  1728473157), -INT32_C(    50133316),  INT32_C(  1772824659), -INT32_C(   149274070), -INT32_C(  1541998124),  INT32_C(  2127204723), -INT32_C(  1862939202), -INT32_C(  1337112844),
        -INT32_C(  1706539043), -INT32_C(  1600697523), -INT32_C(  1660337549),  INT32_C(  1502880901), -INT32_C(  1862358499), -INT32_C(    99628996), -INT32_C(  1987443563),  INT32_C(   674879307) },
      { -INT64_C( 3144493425998343078), -INT64_C( 2104425329626364067), -INT64_C( 1189514072739258532),  INT64_C( 1072243024686328854),
        -INT64_C( 1725752288589136929),  INT64_C( 2308085221288837939), -INT64_C(  995888394746346695), -INT64_C( 3968321779097219481) } },
    { {  INT32_C(  1824686366),  INT32_C(  1074551501),  INT32_C(   568202908),  INT32_C(  1467707962),  INT32_C(  1508407581), -INT32_C(   699140287),  INT32_C(  1180687867), -INT32_C(   747660876),
        -INT32_C(   415289062),  INT32_C(   673729419), -INT32_C(  1689713055), -INT32_C(  1779186568),  INT32_C(  2129582909), -INT32_C(   850116142), -INT32_C(   753617890),  INT32_C(  1738965837) },
      {  INT32_C(  1078977972), -INT32_C(  1838647504), -INT32_C(   181554819), -INT32_C(  1282727818), -INT32_C(   852329989), -INT32_C(   644118853), -INT32_C(  2018726086), -INT32_C(   420523470),
         INT32_C(  1747336759),  INT32_C(   855281333), -INT32_C(  1238948032),  INT32_C(  1131000392), -INT32_C(   418276564),  INT32_C(  1556130850), -INT32_C(  1914409637), -INT32_C(   143404097) },
      {  INT64_C( 1968796394722729752), -INT64_C(  103159976117213652), -INT64_C( 1285661016921246609), -INT64_C( 2383485396536598562),
        -INT64_C(  725649843643230058),  INT64_C( 2093466664136957760), -INT64_C(  890754621929644676),  INT64_C( 1442733351231605930) } },
    { { -INT32_C(   966813167),  INT32_C(  1761106216), -INT32_C(   937549952), -INT32_C(    32732974),  INT32_C(  1172643107),  INT32_C(   614639049), -INT32_C(   760117742),  INT32_C(  1791566937),
        -INT32_C(   416274242),  INT32_C(    21964929),  INT32_C(   432696903),  INT32_C(   420992758), -INT32_C(  1134560013), -INT32_C(  1260387934), -INT32_C(   528051833),  INT32_C(  1951027125) },
      { -INT32_C(   128222601), -INT32_C(   369448286),  INT32_C(   235127832), -INT32_C(  1926751590), -INT32_C(  1186363625),  INT32_C(   258812296),  INT32_C(  1877996730), -INT32_C(  1142736573),
         INT32_C(  1437810355),  INT32_C(    20884969),  INT32_C(   185614705),  INT32_C(  1939355740),  INT32_C(  1194123711), -INT32_C(  1000957686), -INT32_C(   785103475),  INT32_C(   831264638) },
      {  INT64_C(  123967298953787367), -INT64_C(  220444087605464064), -INT64_C( 1391181127251782875), -INT64_C( 1427498633890983660),
        -INT64_C(  598523415667375910),  INT64_C(   80314908004758615), -INT64_C( 1354805013075768243),  INT64_C(  414575329068419675) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mul_epi32(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_mul_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 8259215308803572895), INT64_C( 5002417564910761422),
                            INT64_C( 4825945910792190995), INT64_C(-3854692997504557014),
                            INT64_C(-5029859126276555558), INT64_C(-6821356987634887986),
                            INT64_C( 4779218217009886481), INT64_C( 4803075487209946977)),
      UINT8_C(116),
      simde_mm512_set_epi32(INT32_C(-1946312318), INT32_C(  344802157), INT32_C( 1813783552), INT32_C(  136328242),
                            INT32_C( -821129857), INT32_C( -162465728), INT32_C(-2000203088), INT32_C( 1070574927),
                            INT32_C( 1169396458), INT32_C(-1183467128), INT32_C(-1936534523), INT32_C( 1077760263),
                            INT32_C( 1655422492), INT32_C(  623663138), INT32_C(   29163868), INT32_C(-2106787203)),
      simde_mm512_set_epi32(INT32_C( 1815416844), INT32_C( 1421058674), INT32_C( -653559880), INT32_C(-1986451398),
                            INT32_C(   72292265), INT32_C( -878074326), INT32_C(    9737947), INT32_C( 1206062441),
                            INT32_C(-1794166530), INT32_C(  629576453), INT32_C(-1534533514), INT32_C(  511428134),
                            INT32_C(  206078130), INT32_C(  438975617), INT32_C(  828494683), INT32_C( -995405524)),
      simde_mm512_set_epi64(INT64_C( 8259215308803572895), INT64_C( -270809426907782316),
                            INT64_C(  142656984611699328), INT64_C( 1291180209731016807),
                            INT64_C(-5029859126276555558), INT64_C(  551196920205439242),
                            INT64_C( 4779218217009886481), INT64_C( 4803075487209946977)) },
    { simde_mm512_set_epi64(INT64_C(-2096589957716905410), INT64_C( 5843054744461330548),
                            INT64_C( 6974024204978419548), INT64_C( 3764508487718737373),
                            INT64_C( 3631623951055726390), INT64_C( 3085354128371369606),
                            INT64_C(-8195484891056484583), INT64_C( -289672705472788318)),
      UINT8_C(192),
      simde_mm512_set_epi32(INT32_C(  750022106), INT32_C( -164409985), INT32_C( 1508401218), INT32_C(  326736787),
                            INT32_C( 1376534770), INT32_C(  433267140), INT32_C(-1811276142), INT32_C( -423374585),
                            INT32_C(  316128811), INT32_C( 1066215241), INT32_C( -742321807), INT32_C(  771493183),
                            INT32_C( 1158801588), INT32_C( 1819952183), INT32_C( 1670004358), INT32_C(-1282166628)),
      simde_mm512_set_epi32(INT32_C(-1440467958), INT32_C(  673155116), INT32_C( 1160373089), INT32_C(-1560331288),
                            INT32_C( 1954920850), INT32_C(  443628207), INT32_C(-1358199964), INT32_C( 1999231031),
                            INT32_C( -922196474), INT32_C(-1304354759), INT32_C( 1973769080), INT32_C(-1007908450),
                            INT32_C(-1145320026), INT32_C( 1345168267), INT32_C(-1647348591), INT32_C(-2009668702)),
      simde_mm512_set_epi64(INT64_C( -110673422524233260), INT64_C( -509817631696691656),
                            INT64_C( 6974024204978419548), INT64_C( 3764508487718737373),
                            INT64_C( 3631623951055726390), INT64_C( 3085354128371369606),
                            INT64_C(-8195484891056484583), INT64_C( -289672705472788318)) },
    { simde_mm512_set_epi64(INT64_C(-5821818953001636176), INT64_C( 1997894375206593641),
                            INT64_C(-1188496888106000468), INT64_C( 4574447963200493304),
                            INT64_C( 3346200385521264609), INT64_C(-5642979348732921527),
                            INT64_C( -695384029725146025), INT64_C(-7115212454065332556)),
      UINT8_C(247),
      simde_mm512_set_epi32(INT32_C(  133614784), INT32_C( 1104524722), INT32_C( -405494742), INT32_C(   33506731),
                            INT32_C( 1866794314), INT32_C(-1942831246), INT32_C(-2066004046), INT32_C(-1057119888),
                            INT32_C(-1508387315), INT32_C( 2140894614), INT32_C(-1227481958), INT32_C( -545548506),
                            INT32_C( 2041568161), INT32_C(-1734631316), INT32_C( -578665178), INT32_C(  976865378)),
      simde_mm512_set_epi32(INT32_C( 1938874012), INT32_C( -315470352), INT32_C(   25227789), INT32_C(  348972975),
                            INT32_C( -119098852), INT32_C( -117531009), INT32_C(-1256068989), INT32_C( -330670492),
                            INT32_C(-1342579595), INT32_C( 1663139463), INT32_C( 1519977261), INT32_C(-2010660089),
                            INT32_C(  811843811), INT32_C( 1950445467), INT32_C( -632407557), INT32_C(  132507618)),
      simde_mm512_set_epi64(INT64_C( -348444802842042144), INT64_C(   11692943599594725),
                            INT64_C(  228342916659107214), INT64_C(  349558353467944896),
                            INT64_C( 3346200385521264609), INT64_C( 1096912607627777034),
                            INT64_C(-3383303787208444572), INT64_C(  129442104345449604)) },
    { simde_mm512_set_epi64(INT64_C(-6399381415989804252), INT64_C(-4072646889620133673),
                            INT64_C(-3499367054553152785), INT64_C(-2596410489019354993),
                            INT64_C( 3709612225265967420), INT64_C( 1617021521015256349),
                            INT64_C( 8518903223542129770), INT64_C(-7495998104551122449)),
      UINT8_C(125),
      simde_mm512_set_epi32(INT32_C( 1369528234), INT32_C(-2013461915), INT32_C(  979595496), INT32_C(-1220154251),
                            INT32_C(  305231144), INT32_C(  243633364), INT32_C(  911946112), INT32_C(  158189864),
                            INT32_C(  126572094), INT32_C(   -5395242), INT32_C(-1561205257), INT32_C(  515958610),
                            INT32_C( 1545815628), INT32_C( 1042892620), INT32_C( -956025439), INT32_C( -181963588)),
      simde_mm512_set_epi32(INT32_C( 1859688708), INT32_C( 1308950804), INT32_C(-1932687023), INT32_C( 2111441590),
                            INT32_C( 1137586884), INT32_C( 1521953186), INT32_C( 1589240826), INT32_C( 1097366673),
                            INT32_C( -573799426), INT32_C( 1360758617), INT32_C(-1894824063), INT32_C( -305567235),
                            INT32_C(-1488593619), INT32_C( 1052029738), INT32_C(-1777311621), INT32_C( -525756513)),
      simde_mm512_set_epi64(INT64_C(-6399381415989804252), INT64_C(-2576284431776699090),
                            INT64_C(  370798574555697704), INT64_C(  173592284760002472),
                            INT64_C(   -7341622042300314), INT64_C( -157660045832143350),
                            INT64_C( 8518903223542129770), INT64_C(   95668541519848644)) },
    { simde_mm512_set_epi64(INT64_C( 6860124546956220466), INT64_C(-1265261131078623514),
                            INT64_C( 5737379338676836508), INT64_C(-3711065605003334500),
                            INT64_C(-8479853253989282483), INT64_C( 7964407686671565496),
                            INT64_C( 7785652122788440203), INT64_C(-3096894189429138445)),
      UINT8_C(214),
      simde_mm512_set_epi32(INT32_C( -814208176), INT32_C( 1449013393), INT32_C(  623550410), INT32_C( -805020885),
                            INT32_C(-1088320756), INT32_C( 2022589200), INT32_C(  839176386), INT32_C( 1343270967),
                            INT32_C(  111940457), INT32_C( 1537061703), INT32_C(-1460061235), INT32_C( 1515709350),
                            INT32_C( 1650058892), INT32_C(   69963651), INT32_C(  758490839), INT32_C(  180779892)),
      simde_mm512_set_epi32(INT32_C(-1893053059), INT32_C( -525508532), INT32_C( 1089028030), INT32_C(  641037603),
                            INT32_C(  776284580), INT32_C(  143220066), INT32_C(  609964739), INT32_C(  739061585),
                            INT32_C( 1296320934), INT32_C( 1641387359), INT32_C(  450216201), INT32_C( -102009462),
                            INT32_C(  184231048), INT32_C(   68801332), INT32_C( 1909515723), INT32_C(-1558553543)),
      simde_mm512_set_epi64(INT64_C( -761468901003769076), INT64_C( -516048658485338655),
                            INT64_C( 5737379338676836508), INT64_C(  992759969955502695),
                            INT64_C(-8479853253989282483), INT64_C( -154616695341869700),
                            INT64_C(    4813592380383132), INT64_C(-3096894189429138445)) },
    { simde_mm512_set_epi64(INT64_C(-2621488480535608616), INT64_C(-6848868720227948061),
                            INT64_C( 6279616399573024356), INT64_C(  745095038278958047),
                            INT64_C(-1323215695156753279), INT64_C( -383012613214998281),
                            INT64_C( 1460565887768366290), INT64_C(-5348367197220594908)),
      UINT8_C( 92),
      simde_mm512_set_epi32(INT32_C(-1537831012), INT32_C(-1136146129), INT32_C(  928255499), INT32_C( 1369020603),
                            INT32_C( 1021713905), INT32_C(-1374572733), INT32_C(  981266194), INT32_C( -209600569),
                            INT32_C( -856684622), INT32_C( 1444842251), INT32_C(-1223337348), INT32_C(-1314813402),
                            INT32_C(  630708065), INT32_C( 1782361994), INT32_C(  982404882), INT32_C(  968278192)),
      simde_mm512_set_epi32(INT32_C( -560531037), INT32_C( 2016874130), INT32_C( 1909033660), INT32_C( -288062633),
                            INT32_C( 1926487797), INT32_C(-1384808965), INT32_C(  650303852), INT32_C( 1591608188),
                            INT32_C( 1071082983), INT32_C( 1207794171), INT32_C(-2085192565), INT32_C(  656256578),
                            INT32_C(-1465520335), INT32_C( 2093271192), INT32_C(  315880197), INT32_C( 1596114493)),
      simde_mm512_set_epi64(INT64_C(-2621488480535608616), INT64_C( -394363679531427699),
                            INT64_C( 6279616399573024356), INT64_C( -333601981829858972),
                            INT64_C( 1745072048772318921), INT64_C( -862854943905058356),
                            INT64_C( 1460565887768366290), INT64_C(-5348367197220594908)) },
    { simde_mm512_set_epi64(INT64_C( 1319224608096301911), INT64_C(-6587132379427165760),
                            INT64_C(-1318415648940904266), INT64_C( 5083686936283500523),
                            INT64_C( 2916706726526170303), INT64_C( 1232072806289907439),
                            INT64_C(-4244069429267903156), INT64_C( 1868613955189624367)),
      UINT8_C( 45),
      simde_mm512_set_epi32(INT32_C( 1044553244), INT32_C(  448636134), INT32_C(  422274875), INT32_C(-1037497281),
                            INT32_C(  533714637), INT32_C(-1738371545), INT32_C(  -17938559), INT32_C(-1389744139),
                            INT32_C(  827695522), INT32_C(-1482919408), INT32_C( 1233158285), INT32_C(  343037625),
                            INT32_C(-1483824200), INT32_C( -901390751), INT32_C( -727066099), INT32_C( -648215186)),
      simde_mm512_set_epi32(INT32_C( 1981159106), INT32_C(  410835312), INT32_C( 2072880481), INT32_C(  105988514),
                            INT32_C(  751462668), INT32_C( 1834849576), INT32_C( -217803098), INT32_C(-1411746849),
                            INT32_C(-1237635210), INT32_C( -311304150), INT32_C( -986441771), INT32_C( 1680967167),
                            INT32_C(  746636010), INT32_C(-2078030023), INT32_C(  843084787), INT32_C(  759454903)),
      simde_mm512_set_epi64(INT64_C( 1319224608096301911), INT64_C(-6587132379427165760),
                            INT64_C(-3189650292273714920), INT64_C( 5083686936283500523),
                            INT64_C(  461638965825943200), INT64_C(  576634984670658375),
                            INT64_C(-4244069429267903156), INT64_C( -492290201206756958)) },
    { simde_mm512_set_epi64(INT64_C(-1017619325410469279), INT64_C( 7670597165848860921),
                            INT64_C(-5135734722746288063), INT64_C( 8555281953176040262),
                            INT64_C( 2622398452638226743), INT64_C( 2072647407054444460),
                            INT64_C( 5884644356355100584), INT64_C(-3677310731734481669)),
      UINT8_C(226),
      simde_mm512_set_epi32(INT32_C(  390006051), INT32_C(  789765807), INT32_C( -514015364), INT32_C( -970761836),
                            INT32_C( -378978470), INT32_C(  -73123202), INT32_C(-1325609418), INT32_C( 1232280698),
                            INT32_C( 1916265121), INT32_C( 1820507576), INT32_C( -792248141), INT32_C( -262685644),
                            INT32_C( 1624847858), INT32_C( -403255584), INT32_C( 1568995237), INT32_C( 1227106212)),
      simde_mm512_set_epi32(INT32_C( -769652371), INT32_C( -261880602), INT32_C(   85687930), INT32_C(  432371064),
                            INT32_C( 1626214727), INT32_C( 1845517289), INT32_C(-2002810442), INT32_C(-2069468881),
                            INT32_C(-1294326872), INT32_C(-1409401131), INT32_C(-1446683671), INT32_C( 2011451607),
                            INT32_C( 1570003547), INT32_C(-1564123603), INT32_C( -200447069), INT32_C( -676297563)),
      simde_mm512_set_epi64(INT64_C( -206824344976175814), INT64_C( -419729327921913504),
                            INT64_C( -134950133518039378), INT64_C( 8555281953176040262),
                            INT64_C( 2622398452638226743), INT64_C( 2072647407054444460),
                            INT64_C(  630741576975949152), INT64_C(-3677310731734481669)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_mul_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_mul_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
   {  UINT8_C(138),
      simde_mm512_set_epi32(INT32_C( 1098716707), INT32_C(-1080185167), INT32_C(  796032668), INT32_C( 1756455873),
                            INT32_C(-1031023150), INT32_C(  313996055), INT32_C(-1552434635), INT32_C(   82580470),
                            INT32_C( -868810524), INT32_C(-1501290792), INT32_C( -628539172), INT32_C(  286404385),
                            INT32_C(-2116183242), INT32_C(  925268541), INT32_C( 1423169798), INT32_C(  472979926)),
      simde_mm512_set_epi32(INT32_C(-1589762727), INT32_C( 1342398972), INT32_C( -162164967), INT32_C( 1184007139),
                            INT32_C( 1973410894), INT32_C(  837116435), INT32_C(-1912965227), INT32_C( -221173809),
                            INT32_C(-1524627531), INT32_C(  505638542), INT32_C( 1789154769), INT32_C( 1707140994),
                            INT32_C(  111719139), INT32_C( 1287257616), INT32_C(-1103747425), INT32_C( 1951299418)),
      simde_mm512_set_epi64(INT64_C(-1450039457750448324), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( -759110487184905264), INT64_C(                   0),
                            INT64_C( 1191058976247458256), INT64_C(                   0)) },
   {  UINT8_C(226),
      simde_mm512_set_epi32(INT32_C( 1851172912), INT32_C(  432012768), INT32_C(-1336678725), INT32_C(  141506650),
                            INT32_C(  576471669), INT32_C(-2021849973), INT32_C(  610549751), INT32_C(  470887358),
                            INT32_C( 1210740282), INT32_C( -720782218), INT32_C(  967227355), INT32_C(-1907082749),
                            INT32_C( -376079371), INT32_C(  615957162), INT32_C(  189423181), INT32_C(  750118943)),
      simde_mm512_set_epi32(INT32_C(-1194827437), INT32_C( 1644918495), INT32_C(-1387747393), INT32_C(-1434123267),
                            INT32_C( 1354817839), INT32_C( 1324343890), INT32_C( -595811004), INT32_C(-1790143018),
                            INT32_C( -914188665), INT32_C( -647124032), INT32_C(  792952903), INT32_C( 2106780254),
                            INT32_C(  -65103351), INT32_C( -572558150), INT32_C( -801231269), INT32_C( -650481868)),
      simde_mm512_set_epi64(INT64_C(  710625792159344160), INT64_C( -202937979200225550),
                            INT64_C(-2677624658239214970), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( -352671293153970300), INT64_C(                   0)) },
   {  UINT8_C(205),
      simde_mm512_set_epi32(INT32_C( 1389849027), INT32_C( -968733779), INT32_C( -903985535), INT32_C( -505052458),
                            INT32_C(  849997016), INT32_C( -665823569), INT32_C(  843681453), INT32_C( 1241052856),
                            INT32_C( 1218361488), INT32_C( -511802096), INT32_C( 2056154947), INT32_C( -475453332),
                            INT32_C( 1793883682), INT32_C( 1281268084), INT32_C(-1443305318), INT32_C(-2002775301)),
      simde_mm512_set_epi32(INT32_C( -480306273), INT32_C(-1942698584), INT32_C( 1404753532), INT32_C(  237623409),
                            INT32_C( 1343658265), INT32_C(-1947335016), INT32_C( 1618135889), INT32_C(  726476998),
                            INT32_C( 1324737144), INT32_C( 1048817456), INT32_C(  319312471), INT32_C( 1894816689),
                            INT32_C( 1848939745), INT32_C(-1295730322), INT32_C( 1089929027), INT32_C( -785534579)),
      simde_mm512_set_epi64(INT64_C( 1881957740736268936), INT64_C( -120012286793789322),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( -536786972302187776), INT64_C( -900896908314257748),
                            INT64_C(                   0), INT64_C( 1573249252902633279)) },
   {  UINT8_C(206),
      simde_mm512_set_epi32(INT32_C(  163723168), INT32_C(   94537413), INT32_C( 1298848275), INT32_C(  -99870655),
                            INT32_C( 1537532032), INT32_C(-1949556986), INT32_C( -894015664), INT32_C(-1324496729),
                            INT32_C(  850348293), INT32_C(  906352618), INT32_C(-1965873722), INT32_C(-2107953605),
                            INT32_C(  559881293), INT32_C(   -5815681), INT32_C(-1173896203), INT32_C( 1760080316)),
      simde_mm512_set_epi32(INT32_C( 1202706763), INT32_C(-1110213669), INT32_C( 1229627598), INT32_C( -147072860),
                            INT32_C( 1883759514), INT32_C(-1191387298), INT32_C( 1673499534), INT32_C(  640453183),
                            INT32_C(-1171836364), INT32_C( -982522972), INT32_C(-1480196612), INT32_C(-2077854762),
                            INT32_C( -872251595), INT32_C( 1154127488), INT32_C(  896971913), INT32_C(-1585180342)),
      simde_mm512_set_epi64(INT64_C( -104956728144498297), INT64_C(   14688262860923300),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( -890512267917340696), INT64_C( 4380021436224317010),
                            INT64_C(   -6712037303539328), INT64_C(                   0)) },
   {  UINT8_C(197),
      simde_mm512_set_epi32(INT32_C( -650041052), INT32_C( -647184441), INT32_C(-1880618021), INT32_C(-1812158288),
                            INT32_C(  264100196), INT32_C( -263091932), INT32_C(-1182134909), INT32_C(-1890025577),
                            INT32_C( 1421768266), INT32_C(  936126513), INT32_C( -213174057), INT32_C( -588951079),
                            INT32_C(  217188364), INT32_C( 1950574682), INT32_C( -530860484), INT32_C(  713083418)),
      simde_mm512_set_epi32(INT32_C(-1419088193), INT32_C(  155768054), INT32_C(  575537364), INT32_C(-1651547513),
                            INT32_C(-1310582959), INT32_C( 1366625247), INT32_C(  375333442), INT32_C(  516971366),
                            INT32_C( 1204467496), INT32_C(-1684524880), INT32_C( 2029390656), INT32_C( 1244178650),
                            INT32_C(-1765716319), INT32_C(  -62663523), INT32_C(  233795696), INT32_C(  614711137)),
      simde_mm512_set_epi64(INT64_C( -100810660953647814), INT64_C( 2992865513708737744),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( -732760358386263350),
                            INT64_C(                   0), INT64_C(  438340318654626266)) },
   {  UINT8_C(221),
      simde_mm512_set_epi32(INT32_C(-1439155961), INT32_C(  680420491), INT32_C( 1277792130), INT32_C(-1440597466),
                            INT32_C( -965757835), INT32_C( 2017786190), INT32_C(-1072056911), INT32_C( -897882665),
                            INT32_C(    1375688), INT32_C( 1420052414), INT32_C( -331914389), INT32_C(-1649119241),
                            INT32_C( -706010264), INT32_C(  713383150), INT32_C(  896627462), INT32_C( 1020243588)),
      simde_mm512_set_epi32(INT32_C( -359263092), INT32_C(-1196237833), INT32_C( 1322331949), INT32_C(   43567177),
                            INT32_C(  414081468), INT32_C( 1462500900), INT32_C( 1902422273), INT32_C( -902219192),
                            INT32_C(-1637598569), INT32_C( -626716515), INT32_C( 1485910176), INT32_C(-1246912099),
                            INT32_C(-1032397276), INT32_C(-1436895618), INT32_C(-1408100359), INT32_C( -678052935)),
      simde_mm512_set_epi64(INT64_C( -813944733682636003), INT64_C(  -62762764786973482),
                            INT64_C(                   0), INT64_C(  810086972527106680),
                            INT64_C( -889970300019417210), INT64_C( 2056306734296596859),
                            INT64_C(                   0), INT64_C( -691779159258330780)) },
   {  UINT8_C(176),
      simde_mm512_set_epi32(INT32_C(  806025559), INT32_C(  277323133), INT32_C(-2040845209), INT32_C( 1514048177),
                            INT32_C( 1299301232), INT32_C( 1804349866), INT32_C(  505045603), INT32_C(-1270991510),
                            INT32_C(-1365476185), INT32_C( -470279784), INT32_C( 1957249393), INT32_C(  966280187),
                            INT32_C(  550173580), INT32_C( 1419279519), INT32_C(  120074737), INT32_C( -623354205)),
      simde_mm512_set_epi32(INT32_C(  182628708), INT32_C( -711074484), INT32_C(  700640568), INT32_C( -182451726),
                            INT32_C( 1928956599), INT32_C( 1423054326), INT32_C( 1016030809), INT32_C(-1086945734),
                            INT32_C( 1392670038), INT32_C(    3796661), INT32_C( -232044152), INT32_C( 1236628648),
                            INT32_C(   19789106), INT32_C(-2055501126), INT32_C( 1118019036), INT32_C( -365745616)),
      simde_mm512_set_epi64(INT64_C( -197197403699238372), INT64_C(                   0),
                            INT64_C( 2567687882428820316), INT64_C( 1381498799744718340),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
   {  UINT8_C(158),
      simde_mm512_set_epi32(INT32_C( 1841667236), INT32_C( 1484771017), INT32_C( -884031658), INT32_C(-1839125718),
                            INT32_C(-1569039961), INT32_C( 1041098150), INT32_C( 1388925681), INT32_C(  863701002),
                            INT32_C(  128435058), INT32_C( -263295419), INT32_C(-1184146866), INT32_C( 1159115917),
                            INT32_C(  866281726), INT32_C(-1295662984), INT32_C( -351675537), INT32_C( -710944336)),
      simde_mm512_set_epi32(INT32_C(-1968953227), INT32_C(  227585281), INT32_C( -737334168), INT32_C( 1230090038),
                            INT32_C(-1805794302), INT32_C( 1379277168), INT32_C(  744356262), INT32_C(-1333512317),
                            INT32_C( -486348180), INT32_C( 1206532716), INT32_C( 1803086042), INT32_C(-1291499422),
                            INT32_C( 1358104641), INT32_C(-2056773451), INT32_C(-1326911147), INT32_C( 1579123656)),
      simde_mm512_set_epi64(INT64_C(  337912029124600777), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-1151755924372241634),
                            INT64_C( -317674536996428004), INT64_C(-1496997536836499974),
                            INT64_C( 2664885226934637784), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_mul_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_mul_epu32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
   {  UINT8_C(166),
      simde_x_mm512_set_epu32(UINT32_C(4120514587), UINT32_C(1586964835), UINT32_C(1689003642), UINT32_C(2702971618),
                              UINT32_C(2798377561), UINT32_C( 356472812), UINT32_C(2899999566), UINT32_C(3229978818),
                              UINT32_C(1853356574), UINT32_C(  66069374), UINT32_C(1309687627), UINT32_C(1439779852),
                              UINT32_C(  51916795), UINT32_C(2259276195), UINT32_C( 994758469), UINT32_C(3337558808)),
      simde_x_mm512_set_epu32(UINT32_C(2258908256), UINT32_C(3395137062), UINT32_C(4215249296), UINT32_C(2133148800),
                              UINT32_C(2933274444), UINT32_C(1851927716), UINT32_C(4190919751), UINT32_C(1746820983),
                              UINT32_C( 828520274), UINT32_C( 635229603), UINT32_C(1544367040), UINT32_C(1918376547),
                              UINT32_C( 892222123), UINT32_C(3243361756), UINT32_C( 803359976), UINT32_C( 163739728)),
      simde_x_mm512_set_epu64(UINT64_C( 5387963127399214770), UINT64_C(                   0),
                              UINT64_C(  660161880543257392), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C( 2762039900919931044),
                              UINT64_C( 7327650007104198420), UINT64_C(                   0)) },
   {  UINT8_C(219),
      simde_x_mm512_set_epu32(UINT32_C(1410010955), UINT32_C( 503921354), UINT32_C(1418189156), UINT32_C( 444221777),
                              UINT32_C( 148285537), UINT32_C(2288722231), UINT32_C( 344338098), UINT32_C(3454728003),
                              UINT32_C(1478480780), UINT32_C(1743148264), UINT32_C(3822764711), UINT32_C(1636469832),
                              UINT32_C(3280064546), UINT32_C( 933016241), UINT32_C(1726799481), UINT32_C(3824577952)),
      simde_x_mm512_set_epu32(UINT32_C(1037963842), UINT32_C(2512709916), UINT32_C( 937723538), UINT32_C(2708935661),
                              UINT32_C(3914488889), UINT32_C(2550814880), UINT32_C(2900652427), UINT32_C(2176349091),
                              UINT32_C(3028439158), UINT32_C(2773320535), UINT32_C( 444719300), UINT32_C(3233757255),
                              UINT32_C(1004532908), UINT32_C(  43390785), UINT32_C( 738451500), UINT32_C(3536836475)),
      simde_x_mm512_set_epu64(UINT64_C( 1266208183079946264), UINT64_C( 1203368213108089597),
                              UINT64_C(                   0), UINT64_C( 7518694148981295273),
                              UINT64_C( 4834308876100801240), UINT64_C(                   0),
                              UINT64_C(   40484307114739185), UINT64_C(13526906802114399200)) },
   {  UINT8_C(194),
      simde_x_mm512_set_epu32(UINT32_C(1176316177), UINT32_C(1751826934), UINT32_C(3378345958), UINT32_C( 543404964),
                              UINT32_C(2579785136), UINT32_C(2416322328), UINT32_C(  75139728), UINT32_C(2416880998),
                              UINT32_C(4234686409), UINT32_C(2660004756), UINT32_C(2106185379), UINT32_C( 797059438),
                              UINT32_C(2372191392), UINT32_C( 269555244), UINT32_C(1767269404), UINT32_C(1625455101)),
      simde_x_mm512_set_epu32(UINT32_C(1851434759), UINT32_C(3604871411), UINT32_C(1327258228), UINT32_C(1673018657),
                              UINT32_C( 305970996), UINT32_C(2819644549), UINT32_C(3721065063), UINT32_C(1020891301),
                              UINT32_C(4202682046), UINT32_C(4200645833), UINT32_C( 117038385), UINT32_C(3020070100),
                              UINT32_C(1353160823), UINT32_C( 440057333), UINT32_C(3056423613), UINT32_C(1896622192)),
      simde_x_mm512_set_epu64(UINT64_C( 6315110831396383874), UINT64_C(  909126643078413348),
                              UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C(  118619761770804252), UINT64_C(                   0)) },
   {  UINT8_C( 67),
      simde_x_mm512_set_epu32(UINT32_C(2693954212), UINT32_C( 277998850), UINT32_C(4169077113), UINT32_C(3680111513),
                              UINT32_C(4156583103), UINT32_C(4105987148), UINT32_C(  11818996), UINT32_C( 514873926),
                              UINT32_C(1191268288), UINT32_C(3638344486), UINT32_C(2361786195), UINT32_C( 500533201),
                              UINT32_C(3058957194), UINT32_C( 190737734), UINT32_C(3837187385), UINT32_C(4003123598)),
      simde_x_mm512_set_epu32(UINT32_C(4122956852), UINT32_C(3007076678), UINT32_C(1011742851), UINT32_C( 831857768),
                              UINT32_C(2217989187), UINT32_C(3681606305), UINT32_C(3147415754), UINT32_C( 236426985),
                              UINT32_C( 216160186), UINT32_C(3279967715), UINT32_C(3874145825), UINT32_C(2203854710),
                              UINT32_C(3676418261), UINT32_C( 401038296), UINT32_C(3825112812), UINT32_C(2929607534)),
      simde_x_mm512_set_epu64(UINT64_C(                   0), UINT64_C( 3061329349195282984),
                              UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C(   76493135826261264), UINT64_C(11727581052233987332)) },
   {  UINT8_C(169),
      simde_x_mm512_set_epu32(UINT32_C( 428651380), UINT32_C(1050238262), UINT32_C(4167120113), UINT32_C( 669121916),
                              UINT32_C(1457539263), UINT32_C(3520615042), UINT32_C(1174118849), UINT32_C(1102257957),
                              UINT32_C(1414101989), UINT32_C(3097425534), UINT32_C(1024087984), UINT32_C(1792583521),
                              UINT32_C(3354797839), UINT32_C( 580554502), UINT32_C(1472515666), UINT32_C(3870057603)),
      simde_x_mm512_set_epu32(UINT32_C(2083050486), UINT32_C(1796942025), UINT32_C( 755961532), UINT32_C(2214717680),
                              UINT32_C(3706324798), UINT32_C(1039769945), UINT32_C(3555811997), UINT32_C( 761202637),
                              UINT32_C(1863011574), UINT32_C(1454498620), UINT32_C( 958628441), UINT32_C(2380256526),
                              UINT32_C(4218133731), UINT32_C(3449338768), UINT32_C(3115502206), UINT32_C(2279816507)),
      simde_x_mm512_set_epu64(UINT64_C( 1887217269250760550), UINT64_C(                   0),
                              UINT64_C( 3660629708586512690), UINT64_C(                   0),
                              UINT64_C( 4505201164755763080), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C( 8823021206360252721)) },
   {  UINT8_C(203),
      simde_x_mm512_set_epu32(UINT32_C( 968785729), UINT32_C(3446816529), UINT32_C(1989948608), UINT32_C(3935090572),
                              UINT32_C(2260595137), UINT32_C(3809743538), UINT32_C(1768049062), UINT32_C(1253090843),
                              UINT32_C(4000901225), UINT32_C(2487234584), UINT32_C( 840765913), UINT32_C(1202598978),
                              UINT32_C(2886819484), UINT32_C(2063363126), UINT32_C(2370412425), UINT32_C(1978444200)),
      simde_x_mm512_set_epu32(UINT32_C(1758358159), UINT32_C( 259726788), UINT32_C(1062244813), UINT32_C(1397736159),
                              UINT32_C(1484315275), UINT32_C(2101001099), UINT32_C(2659688367), UINT32_C(1816554597),
                              UINT32_C(1863116741), UINT32_C(3211066307), UINT32_C( 496281550), UINT32_C(3010953410),
                              UINT32_C(1914417911), UINT32_C(1058492483), UINT32_C(1785378717), UINT32_C(1261746977)),
      simde_x_mm512_set_epu64(UINT64_C(  895230585902478852), UINT64_C( 5500218381424392948),
                              UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C( 7986675170287561288), UINT64_C(                   0),
                              UINT64_C( 2184054358570381858), UINT64_C( 2496295988513183400)) },
   {  UINT8_C( 89),
      simde_x_mm512_set_epu32(UINT32_C( 244202415), UINT32_C(1696418382), UINT32_C(4253734840), UINT32_C(1521382913),
                              UINT32_C(2523120367), UINT32_C( 719365215), UINT32_C( 746887847), UINT32_C( 329869757),
                              UINT32_C(2935442647), UINT32_C(3965449572), UINT32_C(   2046702), UINT32_C(3055578856),
                              UINT32_C(2614828885), UINT32_C(2261447742), UINT32_C( 379053160), UINT32_C(1474182998)),
      simde_x_mm512_set_epu32(UINT32_C( 180314942), UINT32_C(3784268734), UINT32_C(2189933725), UINT32_C(1759707651),
                              UINT32_C(4017470040), UINT32_C( 528482752), UINT32_C(2637497058), UINT32_C(3574995683),
                              UINT32_C(2110412704), UINT32_C( 661885013), UINT32_C(3935066909), UINT32_C( 163101530),
                              UINT32_C(3963037657), UINT32_C( 399559486), UINT32_C( 875430591), UINT32_C(1854318955)),
      simde_x_mm512_set_epu64(UINT64_C(                   0), UINT64_C( 2677189152106767363),
                              UINT64_C(                   0), UINT64_C( 1179282957227259031),
                              UINT64_C( 2624671641514064436), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C( 2733605476330127090)) },
   {  UINT8_C( 27),
      simde_x_mm512_set_epu32(UINT32_C(3300721541), UINT32_C(3440866090), UINT32_C(3838602911), UINT32_C(1016597887),
                              UINT32_C( 287068752), UINT32_C(1521867279), UINT32_C(2420112012), UINT32_C(2417142414),
                              UINT32_C( 344709524), UINT32_C(1803316517), UINT32_C( 467213234), UINT32_C( 327864893),
                              UINT32_C(2661940215), UINT32_C(4228328219), UINT32_C(1396080639), UINT32_C(4001917131)),
      simde_x_mm512_set_epu32(UINT32_C(4155157678), UINT32_C(1774567103), UINT32_C(1949309963), UINT32_C( 729844445),
                              UINT32_C(2587732272), UINT32_C( 138621029), UINT32_C(2703994882), UINT32_C(1904478113),
                              UINT32_C(2402800240), UINT32_C( 959065024), UINT32_C(2558227042), UINT32_C(3067418732),
                              UINT32_C(1398342314), UINT32_C(3263383247), UINT32_C(3963437622), UINT32_C(1585677583)),
      simde_x_mm512_set_epu64(UINT64_C(                   0), UINT64_C(                   0),
                              UINT64_C(                   0), UINT64_C( 4603394823466984782),
                              UINT64_C( 1729497798656201408), UINT64_C(                   0),
                              UINT64_C(13798655472701947093), UINT64_C( 6345750283650374373)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_mul_epu32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mul_epu32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu32(UINT32_C( 768255153), UINT32_C(3116504916), UINT32_C(2849349603), UINT32_C(3380602699),
                              UINT32_C(3667150171), UINT32_C(2606748140), UINT32_C( 256440763), UINT32_C(4236376754),
                              UINT32_C( 137611130), UINT32_C(3608004165), UINT32_C(  23379469), UINT32_C( 634104346),
                              UINT32_C( 752813209), UINT32_C( 304980816), UINT32_C( 873141861), UINT32_C(  42552677)),
      simde_x_mm512_set_epu32(UINT32_C(3537218346), UINT32_C( 554508152), UINT32_C(2540529856), UINT32_C(1492162297),
                              UINT32_C(4027600118), UINT32_C( 897760677), UINT32_C( 832131936), UINT32_C(1226979254),
                              UINT32_C(2823494192), UINT32_C(1370605687), UINT32_C(3043623483), UINT32_C( 310819447),
                              UINT32_C(3262156609), UINT32_C(4133822383), UINT32_C( 153673388), UINT32_C( 507486240)),
      simde_x_mm512_set_epu64(UINT64_C( 1728127381670075232), UINT64_C( 5044407888584239603),
                              UINT64_C( 2340235974934890780), UINT64_C( 5197946389285861516),
                              UINT64_C( 4945151027268686355), UINT64_C(  197091962164016662),
                              UINT64_C( 1260736523566404528), UINT64_C(   21594898052664480)) },
    { simde_x_mm512_set_epu32(UINT32_C(3038228522), UINT32_C(2177263565), UINT32_C(1580156717), UINT32_C(3293644153),
                              UINT32_C(3877520946), UINT32_C(1914222601), UINT32_C(2300352870), UINT32_C(3239916612),
                              UINT32_C(2045429998), UINT32_C(2834457902), UINT32_C(2502406118), UINT32_C(3874567768),
                              UINT32_C(1624909929), UINT32_C( 552025498), UINT32_C(3781080866), UINT32_C( 445279347)),
      simde_x_mm512_set_epu32(UINT32_C(3686614578), UINT32_C(2012284249), UINT32_C(4065027833), UINT32_C(2391570441),
                              UINT32_C(2365238876), UINT32_C(1345195249), UINT32_C(3876854758), UINT32_C(2935715346),
                              UINT32_C( 298066676), UINT32_C( 616522972), UINT32_C(2787002250), UINT32_C(2890879290),
                              UINT32_C(2445030057), UINT32_C(1418356119), UINT32_C( 876914337), UINT32_C(2884175418)),
      simde_x_mm512_set_epu64(UINT64_C( 4381273177771087685), UINT64_C( 7876981999487281473),
                              UINT64_C( 2575003148393622649), UINT64_C( 9511472917608727752),
                              UINT64_C( 1747508409749924744), UINT64_C(11200907718212724720),
                              UINT64_C(  782968742932322262), UINT64_C( 1284263746760492046)) },
    { simde_x_mm512_set_epu32(UINT32_C(1327609198), UINT32_C(1911763444), UINT32_C(3596479631), UINT32_C(3933061513),
                              UINT32_C(3463498323), UINT32_C( 687111330), UINT32_C( 635542403), UINT32_C(1153148129),
                              UINT32_C(3815988413), UINT32_C(2677217701), UINT32_C( 271158343), UINT32_C(1601359912),
                              UINT32_C(4276983578), UINT32_C(2555948345), UINT32_C(1846034446), UINT32_C(1127199678)),
      simde_x_mm512_set_epu32(UINT32_C( 657487984), UINT32_C(2570668084), UINT32_C( 753507331), UINT32_C(1705841966),
                              UINT32_C(3937986766), UINT32_C(3019564702), UINT32_C(2409588030), UINT32_C( 467930148),
                              UINT32_C(4115653696), UINT32_C(3587871831), UINT32_C(1753201197), UINT32_C( 778129766),
                              UINT32_C(1742436576), UINT32_C(2505253535), UINT32_C(2950628411), UINT32_C(2064165361)),
      simde_x_mm512_set_epu64(UINT64_C( 4914509269648721296), UINT64_C( 6709181383734854558),
                              UINT64_C( 2074777118412273660), UINT64_C(  539592774668893092),
                              UINT64_C( 9605513974872480531), UINT64_C( 1246065813606340592),
                              UINT64_C( 6403298626588649575), UINT64_C( 2326726530257953758)) },
    { simde_x_mm512_set_epu32(UINT32_C(2919345837), UINT32_C(4236345846), UINT32_C(3784567990), UINT32_C(4027374119),
                              UINT32_C(3288320277), UINT32_C( 390224653), UINT32_C(3910835486), UINT32_C( 865013699),
                              UINT32_C(2748043226), UINT32_C( 449666617), UINT32_C( 232557914), UINT32_C( 838031623),
                              UINT32_C(3874792609), UINT32_C( 112848728), UINT32_C( 173110782), UINT32_C(1917463852)),
      simde_x_mm512_set_epu32(UINT32_C(1885981625), UINT32_C( 167105269), UINT32_C(1642818873), UINT32_C( 795737603),
                              UINT32_C( 583404702), UINT32_C(1590901338), UINT32_C(2535047020), UINT32_C(1665933152),
                              UINT32_C( 271178891), UINT32_C( 492170774), UINT32_C(4067923128), UINT32_C(2457242362),
                              UINT32_C(4206213856), UINT32_C(1232486026), UINT32_C(2718472187), UINT32_C(4168006676)),
      simde_x_mm512_set_epu64(UINT64_C(  707915712172862574), UINT64_C( 3204733027837296757),
                              UINT64_C(  620808922578285714), UINT64_C( 1441054998098249248),
                              UINT64_C(  221312766930851558), UINT64_C( 2059246804731213526),
                              UINT64_C(  139084480311874928), UINT64_C( 7992002136124675952)) },
    { simde_x_mm512_set_epu32(UINT32_C(  27104904), UINT32_C(4150065749), UINT32_C(3990632930), UINT32_C( 634032004),
                              UINT32_C(2048919564), UINT32_C(1865014244), UINT32_C( 549754386), UINT32_C(2522098959),
                              UINT32_C(2696620961), UINT32_C( 891563523), UINT32_C(2188909902), UINT32_C(2179241133),
                              UINT32_C(1743310130), UINT32_C( 377093787), UINT32_C(2755680804), UINT32_C(3712100521)),
      simde_x_mm512_set_epu32(UINT32_C( 672807047), UINT32_C(2773804867), UINT32_C(4088841569), UINT32_C( 619049193),
                              UINT32_C( 593052350), UINT32_C( 730103388), UINT32_C(1414198306), UINT32_C(4002179273),
                              UINT32_C(1269785901), UINT32_C( 747703241), UINT32_C( 347333415), UINT32_C( 968811996),
                              UINT32_C(4236163540), UINT32_C(2123412997), UINT32_C(  33837735), UINT32_C(1851248845)),
      simde_x_mm512_set_epu64(UINT64_C(11511472572946200383), UINT64_C(  392497000412372772),
                              UINT64_C( 1361653218212658672), UINT64_C(10093892178164676807),
                              UINT64_C(  666624935704478043), UINT64_C( 2111274951827031468),
                              UINT64_C(  800725848403749639), UINT64_C( 6872021802025148245)) },
    { simde_x_mm512_set_epu32(UINT32_C(3744995587), UINT32_C(2704878999), UINT32_C(2216207729), UINT32_C(3174220609),
                              UINT32_C(2276590134), UINT32_C( 284809778), UINT32_C(2003404586), UINT32_C(1707085270),
                              UINT32_C(2713648433), UINT32_C(2786430472), UINT32_C( 397019195), UINT32_C( 630796576),
                              UINT32_C(1959866953), UINT32_C( 629006272), UINT32_C(2429347726), UINT32_C(3247824799)),
      simde_x_mm512_set_epu32(UINT32_C(3020299794), UINT32_C(2488516068), UINT32_C(3326847413), UINT32_C(1426347053),
                              UINT32_C(3015511399), UINT32_C( 258677619), UINT32_C(3923020384), UINT32_C( 835454201),
                              UINT32_C(3228303109), UINT32_C( 994730831), UINT32_C(2437482082), UINT32_C(1004732602),
                              UINT32_C(3078918689), UINT32_C(1633253517), UINT32_C(1920589043), UINT32_C(3888518352)),
      simde_x_mm512_set_epu64(UINT64_C( 6731134851007255932), UINT64_C( 4527540211219015277),
                              UINT64_C(   73673915240958582), UINT64_C( 1426191560286719270),
                              UINT64_C( 2771748298936282232), UINT64_C(  633781885137170752),
                              UINT64_C( 1027326705959058624), UINT64_C(12629226334992211248)) },
    { simde_x_mm512_set_epu32(UINT32_C( 237961802), UINT32_C(1124052031), UINT32_C(3408632402), UINT32_C(1936321731),
                              UINT32_C(3188356992), UINT32_C( 413227284), UINT32_C(1767960975), UINT32_C(2214647351),
                              UINT32_C(4011124733), UINT32_C(3189426671), UINT32_C(3040561164), UINT32_C(3376223700),
                              UINT32_C(2268266209), UINT32_C( 155837480), UINT32_C(1377610501), UINT32_C(1504228568)),
      simde_x_mm512_set_epu32(UINT32_C(1573768507), UINT32_C( 476780671), UINT32_C(2153500842), UINT32_C(1201914669),
                              UINT32_C(1130822801), UINT32_C(3370243267), UINT32_C(1286308912), UINT32_C(2062398363),
                              UINT32_C(1095401713), UINT32_C(4089334856), UINT32_C(2597794703), UINT32_C(2139321595),
                              UINT32_C(2505322640), UINT32_C(2764790171), UINT32_C(3415336749), UINT32_C(  59419438)),
      simde_x_mm512_set_epu64(UINT64_C(  535926281579092801), UINT64_C( 2327293492392372039),
                              UINT64_C( 1392676471641696828), UINT64_C( 4567485071324686413),
                              UINT64_C(13042633656376344376), UINT64_C( 7222828270960801500),
                              UINT64_C(  430857932977409080), UINT64_C(   89380416134104784)) },
    { simde_x_mm512_set_epu32(UINT32_C( 493235400), UINT32_C( 189383962), UINT32_C(2622533649), UINT32_C( 943550019),
                              UINT32_C( 227224723), UINT32_C(1724057992), UINT32_C(4133039778), UINT32_C(3416450213),
                              UINT32_C(1064097074), UINT32_C(1615527431), UINT32_C( 106890087), UINT32_C(3131878508),
                              UINT32_C(4228916541), UINT32_C(2298347901), UINT32_C(2681451816), UINT32_C( 956711717)),
      simde_x_mm512_set_epu32(UINT32_C(1959431707), UINT32_C(3425635109), UINT32_C(3493232750), UINT32_C(2950665544),
                              UINT32_C(1223627161), UINT32_C(3625235337), UINT32_C( 456501342), UINT32_C(2245318318),
                              UINT32_C(3915087897), UINT32_C(4086538960), UINT32_C(2510683850), UINT32_C(3689243003),
                              UINT32_C(2734380582), UINT32_C(3715382302), UINT32_C(3353219492), UINT32_C(1828163673)),
      simde_x_mm512_set_epu64(UINT64_C(  648760349308721858), UINT64_C( 2784100530103845336),
                              UINT64_C( 6250115955635663304), UINT64_C( 7671018245783901734),
                              UINT64_C( 6601915787730211760), UINT64_C(11554260871885079524),
                              UINT64_C( 8539241115214248102), UINT64_C( 1749025606552856541)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mul_epu32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_mul_epu32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu64(UINT64_C(11617731129322750966), UINT64_C( 2428231924375211538),
                              UINT64_C(14175135673172244792), UINT64_C( 5480651963328574733),
                              UINT64_C(12032129819668007160), UINT64_C( 4424822542185790875),
                              UINT64_C(14867552050688999361), UINT64_C(18178319731812771647)),
      UINT8_C(193),
      simde_x_mm512_set_epu32(UINT32_C(3263550415), UINT32_C(3722438839), UINT32_C(2250736680), UINT32_C(1691034658),
                              UINT32_C(2554266733), UINT32_C(3217520562), UINT32_C(1461265118), UINT32_C( 788514619),
                              UINT32_C(2189225773), UINT32_C(1631863219), UINT32_C(1750226365), UINT32_C( 568487836),
                              UINT32_C(2400146531), UINT32_C(3640095823), UINT32_C( 206125598), UINT32_C( 929630688)),
      simde_x_mm512_set_epu32(UINT32_C(2599352978), UINT32_C(2499798093), UINT32_C(1296552276), UINT32_C( 187401912),
                              UINT32_C(2689124664), UINT32_C(3631388300), UINT32_C( 697343737), UINT32_C(2062287739),
                              UINT32_C(2446945975), UINT32_C( 568195668), UINT32_C( 631862630), UINT32_C( 893101483),
                              UINT32_C( 859175541), UINT32_C( 104178488), UINT32_C(4045255037), UINT32_C(4203900757)),
      simde_x_mm512_set_epu64(UINT64_C( 9305345511041334027), UINT64_C(  316903128167466096),
                              UINT64_C(14175135673172244792), UINT64_C( 5480651963328574733),
                              UINT64_C(12032129819668007160), UINT64_C( 4424822542185790875),
                              UINT64_C(14867552050688999361), UINT64_C( 3908075153013630816)) },
    { simde_x_mm512_set_epu64(UINT64_C(13472523368263323530), UINT64_C( 9759174632444686247),
                              UINT64_C(13193200280295594145), UINT64_C(   61830035715779390),
                              UINT64_C( 3749566472430999385), UINT64_C( 4387274564618060685),
                              UINT64_C( 6292382073951294857), UINT64_C( 2998314933539498774)),
      UINT8_C( 51),
      simde_x_mm512_set_epu32(UINT32_C(2328158325), UINT32_C(4044751844), UINT32_C(3983880162), UINT32_C(1865776280),
                              UINT32_C( 663966634), UINT32_C(3835216878), UINT32_C(1416309334), UINT32_C(3569688362),
                              UINT32_C( 671765842), UINT32_C(4201434406), UINT32_C(2145277385), UINT32_C( 232005336),
                              UINT32_C(3364267605), UINT32_C(1204199577), UINT32_C(1905702483), UINT32_C(1405245895)),
      simde_x_mm512_set_epu32(UINT32_C(1978530737), UINT32_C(3532128238), UINT32_C(2586558058), UINT32_C(2555375701),
                              UINT32_C(1328514887), UINT32_C(2070014178), UINT32_C(2738790052), UINT32_C(1824660691),
                              UINT32_C(3424488035), UINT32_C(3798301173), UINT32_C( 335648721), UINT32_C( 829536855),
                              UINT32_C( 101359129), UINT32_C( 500535839), UINT32_C(3768468917), UINT32_C(3000753624)),
      simde_x_mm512_set_epu64(UINT64_C(13472523368263323530), UINT64_C( 9759174632444686247),
                              UINT64_C( 7938953313164896284), UINT64_C( 6513470033261578142),
                              UINT64_C( 3749566472430999385), UINT64_C( 4387274564618060685),
                              UINT64_C(  602745045597140103), UINT64_C( 4216796712032373480)) },
    { simde_x_mm512_set_epu64(UINT64_C( 9490244949648135949), UINT64_C( 3952247228721925392),
                              UINT64_C( 4800241040971682796), UINT64_C( 9619996883527725324),
                              UINT64_C(15935750477416943804), UINT64_C(  545362928884482916),
                              UINT64_C(13559318363578452842), UINT64_C( 7722701545450284407)),
      UINT8_C(134),
      simde_x_mm512_set_epu32(UINT32_C( 500898194), UINT32_C(4078085990), UINT32_C(2494049110), UINT32_C(1592224201),
                              UINT32_C( 111635698), UINT32_C(    186713), UINT32_C(1765622469), UINT32_C(4017148467),
                              UINT32_C(2543052619), UINT32_C(1161807732), UINT32_C(1925351794), UINT32_C(2298119068),
                              UINT32_C( 457010151), UINT32_C(2589010019), UINT32_C( 502276479), UINT32_C(1967748710)),
      simde_x_mm512_set_epu32(UINT32_C(1919012105), UINT32_C(2908857333), UINT32_C(1122604656), UINT32_C(3433647442),
                              UINT32_C(2386428500), UINT32_C( 463161035), UINT32_C( 504317420), UINT32_C(3353921428),
                              UINT32_C(1582348389), UINT32_C(1180932658), UINT32_C(1476554796), UINT32_C(2783736621),
                              UINT32_C( 330646602), UINT32_C(1769150036), UINT32_C(2020624655), UINT32_C(3683994282)),
      simde_x_mm512_set_epu64(UINT64_C(11862570336616064670), UINT64_C( 3952247228721925392),
                              UINT64_C( 4800241040971682796), UINT64_C( 9619996883527725324),
                              UINT64_C(15935750477416943804), UINT64_C( 6397358209009989228),
                              UINT64_C( 4580347168318210684), UINT64_C( 7722701545450284407)) },
    { simde_x_mm512_set_epu64(UINT64_C( 8956593975554634232), UINT64_C( 9593792923362730078),
                              UINT64_C(  751700862087837721), UINT64_C( 7205298436209283097),
                              UINT64_C( 7151721520472513082), UINT64_C( 8910303953543094872),
                              UINT64_C(  533657364826431938), UINT64_C(13265804505255182490)),
      UINT8_C(251),
      simde_x_mm512_set_epu32(UINT32_C(1468179080), UINT32_C(3172744829), UINT32_C(1457928522), UINT32_C(1192418034),
                              UINT32_C( 105193191), UINT32_C( 430546192), UINT32_C(1509518002), UINT32_C( 354607881),
                              UINT32_C(3139371107), UINT32_C(2393204313), UINT32_C(1496510794), UINT32_C(3916080313),
                              UINT32_C(3933358732), UINT32_C(2965437178), UINT32_C(2440098689), UINT32_C( 675981365)),
      simde_x_mm512_set_epu32(UINT32_C(3471092536), UINT32_C(4213288110), UINT32_C(3288478343), UINT32_C( 269318758),
                              UINT32_C(2757016548), UINT32_C( 404238758), UINT32_C(3038240298), UINT32_C(3153052129),
                              UINT32_C(1906833283), UINT32_C(1593207408), UINT32_C(  59630942), UINT32_C(3403525194),
                              UINT32_C(2850644791), UINT32_C(1343686045), UINT32_C(1484433553), UINT32_C(2500647723)),
      simde_x_mm512_set_epu64(UINT64_C(13367688064089683190), UINT64_C(  321140543933681772),
                              UINT64_C(  174043457915709536), UINT64_C( 1118097134147228649),
                              UINT64_C( 3812870840329150704), UINT64_C( 8910303953543094872),
                              UINT64_C( 3984616553402781010), UINT64_C( 1690391261177681895)) },
    { simde_x_mm512_set_epu64(UINT64_C( 9350173910558210368), UINT64_C( 8451791018593404629),
                              UINT64_C( 5111327021160397113), UINT64_C( 8067526547900849939),
                              UINT64_C( 4397190784689926414), UINT64_C(15730677711069966608),
                              UINT64_C(15155555711952095903), UINT64_C( 4601095961680188139)),
      UINT8_C( 77),
      simde_x_mm512_set_epu32(UINT32_C( 289816884), UINT32_C(3139170300), UINT32_C(2611086568), UINT32_C( 981312265),
                              UINT32_C(2310975133), UINT32_C(1423285786), UINT32_C(4145860146), UINT32_C( 849682935),
                              UINT32_C(3098522529), UINT32_C(1778767618), UINT32_C(2469498326), UINT32_C(3407697658),
                              UINT32_C(4257688348), UINT32_C(1327333484), UINT32_C(4206795397), UINT32_C(1498113253)),
      simde_x_mm512_set_epu32(UINT32_C(3119696014), UINT32_C( 934565143), UINT32_C( 659452226), UINT32_C(3987623713),
                              UINT32_C(3941044651), UINT32_C(3075534691), UINT32_C( 348385654), UINT32_C(3299605274),
                              UINT32_C( 734145932), UINT32_C(2544261168), UINT32_C(1332327027), UINT32_C(3348556299),
                              UINT32_C(3524888946), UINT32_C(2026143937), UINT32_C(3684996090), UINT32_C(3613250397)),
      simde_x_mm512_set_epu64(UINT64_C( 9350173910558210368), UINT64_C( 3913104057771739945),
                              UINT64_C( 5111327021160397113), UINT64_C( 8067526547900849939),
                              UINT64_C( 4525649377373257824), UINT64_C(11410867457783447742),
                              UINT64_C(15155555711952095903), UINT64_C( 5413058306153211441)) },
    { simde_x_mm512_set_epu64(UINT64_C(16773967285187515106), UINT64_C( 3477992427783883408),
                              UINT64_C( 1002604261497217766), UINT64_C( 1352136840172993944),
                              UINT64_C(10899831745595212891), UINT64_C( 2236619794744991665),
                              UINT64_C( 4130838651210953091), UINT64_C(12223797258734177268)),
      UINT8_C(188),
      simde_x_mm512_set_epu32(UINT32_C(3198361131), UINT32_C(1105007823), UINT32_C(3912476736), UINT32_C(2446995251),
                              UINT32_C(2582844574), UINT32_C(2764726563), UINT32_C(1724064135), UINT32_C( 994964469),
                              UINT32_C( 257756540), UINT32_C( 980676724), UINT32_C(2274290616), UINT32_C(4142129112),
                              UINT32_C(1407509141), UINT32_C(1593753754), UINT32_C(3346961920), UINT32_C(3859603033)),
      simde_x_mm512_set_epu32(UINT32_C(1883521655), UINT32_C( 997816392), UINT32_C( 186891147), UINT32_C(2813182686),
                              UINT32_C(3818488413), UINT32_C(1262292349), UINT32_C(2615667202), UINT32_C(3433123518),
                              UINT32_C(2317895719), UINT32_C(1795398992), UINT32_C(1050555729), UINT32_C(1774700454),
                              UINT32_C( 350094657), UINT32_C(2222937199), UINT32_C( 539823167), UINT32_C( 663093254)),
      simde_x_mm512_set_epu64(UINT64_C( 1102594919077634616), UINT64_C( 3477992427783883408),
                              UINT64_C( 3489893187551966487), UINT64_C( 3415835918098281942),
                              UINT64_C( 1760706001747462208), UINT64_C( 7351038415593016848),
                              UINT64_C( 4130838651210953091), UINT64_C(12223797258734177268)) },
    { simde_x_mm512_set_epu64(UINT64_C( 9161315007163903385), UINT64_C(  901926328951971839),
                              UINT64_C(11374042021460658344), UINT64_C(14235844241233139061),
                              UINT64_C(16689996302050367513), UINT64_C(17811135944692719319),
                              UINT64_C( 7952138000462838282), UINT64_C(15106420877923679668)),
      UINT8_C(197),
      simde_x_mm512_set_epu32(UINT32_C(1215894565), UINT32_C( 325247992), UINT32_C(3808486726), UINT32_C(3829410744),
                              UINT32_C(1276796092), UINT32_C( 483034698), UINT32_C(3265794508), UINT32_C( 145210622),
                              UINT32_C(4212031611), UINT32_C(3325547336), UINT32_C(1445017193), UINT32_C(2689093900),
                              UINT32_C(4273435877), UINT32_C( 524026689), UINT32_C(3618756570), UINT32_C(3961201514)),
      simde_x_mm512_set_epu32(UINT32_C(3460615822), UINT32_C(2842020471), UINT32_C(1351189519), UINT32_C(2329879373),
                              UINT32_C(3974357402), UINT32_C(2816300347), UINT32_C( 773721318), UINT32_C(3997442937),
                              UINT32_C(2436503902), UINT32_C(3242344117), UINT32_C(1149812233), UINT32_C( 907108945),
                              UINT32_C(1385675283), UINT32_C(3399903430), UINT32_C(2550192792), UINT32_C(3214774192)),
      simde_x_mm512_set_epu64(UINT64_C(  924361451415644232), UINT64_C( 8922065103190183512),
                              UINT64_C(11374042021460658344), UINT64_C(14235844241233139061),
                              UINT64_C(16689996302050367513), UINT64_C( 2439301130634935500),
                              UINT64_C( 7952138000462838282), UINT64_C(12734368396518526688)) },
    { simde_x_mm512_set_epu64(UINT64_C(10381435592908454864), UINT64_C( 7683972863259161915),
                              UINT64_C(  312335983814548083), UINT64_C( 3934167861393427795),
                              UINT64_C(15803008790257017530), UINT64_C(12384685209313245301),
                              UINT64_C(17881738201070197485), UINT64_C(14224003016858721277)),
      UINT8_C( 76),
      simde_x_mm512_set_epu32(UINT32_C(3028673683), UINT32_C(2581675996), UINT32_C(3969199228), UINT32_C(1709618805),
                              UINT32_C(3286547215), UINT32_C(2496179327), UINT32_C(2647114121), UINT32_C(2818621113),
                              UINT32_C( 879830851), UINT32_C(3024057012), UINT32_C( 247658746), UINT32_C(1778653183),
                              UINT32_C( 608002580), UINT32_C(2912110970), UINT32_C(2119947745), UINT32_C( 102275654)),
      simde_x_mm512_set_epu32(UINT32_C(3762799031), UINT32_C(1035026982), UINT32_C( 282468805), UINT32_C( 635023104),
                              UINT32_C(1863059331), UINT32_C(4265385561), UINT32_C( 804673998), UINT32_C(2920963576),
                              UINT32_C(1218801842), UINT32_C(1010987093), UINT32_C(3172703974), UINT32_C(1792395250),
                              UINT32_C(3430253324), UINT32_C( 714780216), UINT32_C(4029344470), UINT32_C(2590869425)),
      simde_x_mm512_set_epu64(UINT64_C(10381435592908454864), UINT64_C( 1085647440207870720),
                              UINT64_C(  312335983814548083), UINT64_C( 3934167861393427795),
                              UINT64_C( 3057282607628146116), UINT64_C( 3188049516606580750),
                              UINT64_C(17881738201070197485), UINT64_C(14224003016858721277)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_mul_epu32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mul_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -775.40), SIMDE_FLOAT32_C(  -210.92), SIMDE_FLOAT32_C(   987.42), SIMDE_FLOAT32_C(   542.45),
                         SIMDE_FLOAT32_C(  -745.60), SIMDE_FLOAT32_C(   -50.38), SIMDE_FLOAT32_C(   163.82), SIMDE_FLOAT32_C(  -164.62),
                         SIMDE_FLOAT32_C(  -736.65), SIMDE_FLOAT32_C(  -764.30), SIMDE_FLOAT32_C(   675.25), SIMDE_FLOAT32_C(  -182.15),
                         SIMDE_FLOAT32_C(  -748.44), SIMDE_FLOAT32_C(    82.10), SIMDE_FLOAT32_C(   684.52), SIMDE_FLOAT32_C(  -343.09)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   263.91), SIMDE_FLOAT32_C(  -350.89), SIMDE_FLOAT32_C(  -318.01), SIMDE_FLOAT32_C(  -980.00),
                         SIMDE_FLOAT32_C(   872.18), SIMDE_FLOAT32_C(    80.96), SIMDE_FLOAT32_C(   145.89), SIMDE_FLOAT32_C(   832.89),
                         SIMDE_FLOAT32_C(  -267.96), SIMDE_FLOAT32_C(  -536.57), SIMDE_FLOAT32_C(  -934.00), SIMDE_FLOAT32_C(   653.62),
                         SIMDE_FLOAT32_C(   984.11), SIMDE_FLOAT32_C(   140.30), SIMDE_FLOAT32_C(  -580.05), SIMDE_FLOAT32_C(  -915.75)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-204635.83), SIMDE_FLOAT32_C( 74009.72), SIMDE_FLOAT32_C(-314009.44), SIMDE_FLOAT32_C(-531601.00),
                         SIMDE_FLOAT32_C(-650297.38), SIMDE_FLOAT32_C( -4078.76), SIMDE_FLOAT32_C( 23899.70), SIMDE_FLOAT32_C(-137110.34),
                         SIMDE_FLOAT32_C(197392.73), SIMDE_FLOAT32_C(410100.44), SIMDE_FLOAT32_C(-630683.50), SIMDE_FLOAT32_C(-119056.88),
                         SIMDE_FLOAT32_C(-736547.25), SIMDE_FLOAT32_C( 11518.63), SIMDE_FLOAT32_C(-397055.84), SIMDE_FLOAT32_C(314184.66)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -910.74), SIMDE_FLOAT32_C(  -302.10), SIMDE_FLOAT32_C(   937.08), SIMDE_FLOAT32_C(   618.13),
                         SIMDE_FLOAT32_C(    85.12), SIMDE_FLOAT32_C(     3.50), SIMDE_FLOAT32_C(  -122.84), SIMDE_FLOAT32_C(   290.22),
                         SIMDE_FLOAT32_C(   606.76), SIMDE_FLOAT32_C(  -664.92), SIMDE_FLOAT32_C(   454.81), SIMDE_FLOAT32_C(   299.40),
                         SIMDE_FLOAT32_C(  -524.63), SIMDE_FLOAT32_C(    40.68), SIMDE_FLOAT32_C(   218.77), SIMDE_FLOAT32_C(    35.82)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   392.21), SIMDE_FLOAT32_C(   139.00), SIMDE_FLOAT32_C(  -878.97), SIMDE_FLOAT32_C(   778.57),
                         SIMDE_FLOAT32_C(  -810.83), SIMDE_FLOAT32_C(   413.49), SIMDE_FLOAT32_C(   505.44), SIMDE_FLOAT32_C(   291.58),
                         SIMDE_FLOAT32_C(  -757.25), SIMDE_FLOAT32_C(   594.07), SIMDE_FLOAT32_C(   304.96), SIMDE_FLOAT32_C(  -155.47),
                         SIMDE_FLOAT32_C(   635.03), SIMDE_FLOAT32_C(   654.85), SIMDE_FLOAT32_C(   777.61), SIMDE_FLOAT32_C(  -598.19)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-357201.31), SIMDE_FLOAT32_C(-41991.90), SIMDE_FLOAT32_C(-823665.19), SIMDE_FLOAT32_C(481257.47),
                         SIMDE_FLOAT32_C(-69017.85), SIMDE_FLOAT32_C(  1447.21), SIMDE_FLOAT32_C(-62088.25), SIMDE_FLOAT32_C( 84622.34),
                         SIMDE_FLOAT32_C(-459469.03), SIMDE_FLOAT32_C(-395009.03), SIMDE_FLOAT32_C(138698.86), SIMDE_FLOAT32_C(-46547.72),
                         SIMDE_FLOAT32_C(-333155.81), SIMDE_FLOAT32_C( 26639.30), SIMDE_FLOAT32_C(170117.73), SIMDE_FLOAT32_C(-21427.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   202.90), SIMDE_FLOAT32_C(  -396.66), SIMDE_FLOAT32_C(  -364.01), SIMDE_FLOAT32_C(    56.81),
                         SIMDE_FLOAT32_C(  -881.59), SIMDE_FLOAT32_C(   212.81), SIMDE_FLOAT32_C(  -968.64), SIMDE_FLOAT32_C(  -657.19),
                         SIMDE_FLOAT32_C(   232.02), SIMDE_FLOAT32_C(   984.70), SIMDE_FLOAT32_C(  -800.83), SIMDE_FLOAT32_C(  -826.63),
                         SIMDE_FLOAT32_C(   822.26), SIMDE_FLOAT32_C(  -892.21), SIMDE_FLOAT32_C(  -651.70), SIMDE_FLOAT32_C(  -380.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -221.35), SIMDE_FLOAT32_C(  -305.38), SIMDE_FLOAT32_C(   546.45), SIMDE_FLOAT32_C(  -697.03),
                         SIMDE_FLOAT32_C(    93.97), SIMDE_FLOAT32_C(   975.92), SIMDE_FLOAT32_C(   876.47), SIMDE_FLOAT32_C(   762.37),
                         SIMDE_FLOAT32_C(   880.83), SIMDE_FLOAT32_C(  -763.06), SIMDE_FLOAT32_C(  -540.57), SIMDE_FLOAT32_C(  -512.55),
                         SIMDE_FLOAT32_C(   -32.98), SIMDE_FLOAT32_C(   700.87), SIMDE_FLOAT32_C(  -425.19), SIMDE_FLOAT32_C(  -849.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-44911.91), SIMDE_FLOAT32_C(121132.03), SIMDE_FLOAT32_C(-198913.28), SIMDE_FLOAT32_C(-39598.28),
                         SIMDE_FLOAT32_C(-82843.02), SIMDE_FLOAT32_C(207685.53), SIMDE_FLOAT32_C(-848983.88), SIMDE_FLOAT32_C(-501021.94),
                         SIMDE_FLOAT32_C(204370.19), SIMDE_FLOAT32_C(-751385.19), SIMDE_FLOAT32_C(432904.69), SIMDE_FLOAT32_C(423689.19),
                         SIMDE_FLOAT32_C(-27118.13), SIMDE_FLOAT32_C(-625323.25), SIMDE_FLOAT32_C(277096.34), SIMDE_FLOAT32_C(323227.12)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -25.40), SIMDE_FLOAT32_C(  -267.80), SIMDE_FLOAT32_C(   353.79), SIMDE_FLOAT32_C(   -35.72),
                         SIMDE_FLOAT32_C(   125.21), SIMDE_FLOAT32_C(   137.22), SIMDE_FLOAT32_C(   310.88), SIMDE_FLOAT32_C(  -724.55),
                         SIMDE_FLOAT32_C(  -538.86), SIMDE_FLOAT32_C(    39.65), SIMDE_FLOAT32_C(  -229.28), SIMDE_FLOAT32_C(  -842.78),
                         SIMDE_FLOAT32_C(   -14.75), SIMDE_FLOAT32_C(  -859.98), SIMDE_FLOAT32_C(   215.44), SIMDE_FLOAT32_C(   762.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -551.49), SIMDE_FLOAT32_C(   -42.33), SIMDE_FLOAT32_C(  -926.18), SIMDE_FLOAT32_C(    36.96),
                         SIMDE_FLOAT32_C(   954.39), SIMDE_FLOAT32_C(   874.71), SIMDE_FLOAT32_C(  -375.00), SIMDE_FLOAT32_C(   949.07),
                         SIMDE_FLOAT32_C(   -16.18), SIMDE_FLOAT32_C(  -931.82), SIMDE_FLOAT32_C(  -687.15), SIMDE_FLOAT32_C(  -416.23),
                         SIMDE_FLOAT32_C(  -313.36), SIMDE_FLOAT32_C(   905.90), SIMDE_FLOAT32_C(     1.93), SIMDE_FLOAT32_C(  -464.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( 14007.85), SIMDE_FLOAT32_C( 11335.97), SIMDE_FLOAT32_C(-327673.22), SIMDE_FLOAT32_C( -1320.21),
                         SIMDE_FLOAT32_C(119499.17), SIMDE_FLOAT32_C(120027.71), SIMDE_FLOAT32_C(-116580.00), SIMDE_FLOAT32_C(-687648.69),
                         SIMDE_FLOAT32_C(  8718.75), SIMDE_FLOAT32_C(-36946.66), SIMDE_FLOAT32_C(157549.75), SIMDE_FLOAT32_C(350790.34),
                         SIMDE_FLOAT32_C(  4622.06), SIMDE_FLOAT32_C(-779055.88), SIMDE_FLOAT32_C(   415.80), SIMDE_FLOAT32_C(-354700.72)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -861.86), SIMDE_FLOAT32_C(   437.62), SIMDE_FLOAT32_C(  -236.27), SIMDE_FLOAT32_C(   439.08),
                         SIMDE_FLOAT32_C(   476.60), SIMDE_FLOAT32_C(  -725.80), SIMDE_FLOAT32_C(   626.57), SIMDE_FLOAT32_C(  -848.67),
                         SIMDE_FLOAT32_C(  -961.54), SIMDE_FLOAT32_C(  -999.94), SIMDE_FLOAT32_C(   788.38), SIMDE_FLOAT32_C(  -928.14),
                         SIMDE_FLOAT32_C(   779.51), SIMDE_FLOAT32_C(   846.68), SIMDE_FLOAT32_C(  -858.45), SIMDE_FLOAT32_C(   292.21)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -182.57), SIMDE_FLOAT32_C(  -580.78), SIMDE_FLOAT32_C(   704.32), SIMDE_FLOAT32_C(  -124.94),
                         SIMDE_FLOAT32_C(  -725.23), SIMDE_FLOAT32_C(  -635.58), SIMDE_FLOAT32_C(   196.48), SIMDE_FLOAT32_C(  -485.66),
                         SIMDE_FLOAT32_C(  -906.19), SIMDE_FLOAT32_C(   -43.74), SIMDE_FLOAT32_C(   899.84), SIMDE_FLOAT32_C(  -720.16),
                         SIMDE_FLOAT32_C(   576.76), SIMDE_FLOAT32_C(   994.06), SIMDE_FLOAT32_C(  -108.56), SIMDE_FLOAT32_C(   212.62)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(157349.78), SIMDE_FLOAT32_C(-254160.95), SIMDE_FLOAT32_C(-166409.69), SIMDE_FLOAT32_C(-54858.66),
                         SIMDE_FLOAT32_C(-345644.62), SIMDE_FLOAT32_C(461303.97), SIMDE_FLOAT32_C(123108.47), SIMDE_FLOAT32_C(412165.06),
                         SIMDE_FLOAT32_C(871337.94), SIMDE_FLOAT32_C( 43737.38), SIMDE_FLOAT32_C(709415.88), SIMDE_FLOAT32_C(668409.31),
                         SIMDE_FLOAT32_C(449590.19), SIMDE_FLOAT32_C(841650.69), SIMDE_FLOAT32_C( 93193.33), SIMDE_FLOAT32_C( 62129.69)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   593.71), SIMDE_FLOAT32_C(  -601.77), SIMDE_FLOAT32_C(  -110.10), SIMDE_FLOAT32_C(   145.40),
                         SIMDE_FLOAT32_C(   740.85), SIMDE_FLOAT32_C(   970.19), SIMDE_FLOAT32_C(  -854.26), SIMDE_FLOAT32_C(  -208.21),
                         SIMDE_FLOAT32_C(   769.57), SIMDE_FLOAT32_C(  -297.46), SIMDE_FLOAT32_C(  -845.75), SIMDE_FLOAT32_C(  -517.72),
                         SIMDE_FLOAT32_C(  -240.19), SIMDE_FLOAT32_C(  -763.89), SIMDE_FLOAT32_C(  -197.03), SIMDE_FLOAT32_C(   -33.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -533.43), SIMDE_FLOAT32_C(   541.52), SIMDE_FLOAT32_C(  -293.53), SIMDE_FLOAT32_C(   215.59),
                         SIMDE_FLOAT32_C(  -844.97), SIMDE_FLOAT32_C(  -755.33), SIMDE_FLOAT32_C(   468.59), SIMDE_FLOAT32_C(  -772.73),
                         SIMDE_FLOAT32_C(   919.17), SIMDE_FLOAT32_C(   772.56), SIMDE_FLOAT32_C(  -506.06), SIMDE_FLOAT32_C(   848.47),
                         SIMDE_FLOAT32_C(   289.91), SIMDE_FLOAT32_C(    20.43), SIMDE_FLOAT32_C(   -64.43), SIMDE_FLOAT32_C(  -706.80)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-316702.72), SIMDE_FLOAT32_C(-325870.50), SIMDE_FLOAT32_C( 32317.65), SIMDE_FLOAT32_C( 31346.78),
                         SIMDE_FLOAT32_C(-625996.00), SIMDE_FLOAT32_C(-732813.62), SIMDE_FLOAT32_C(-400297.69), SIMDE_FLOAT32_C(160890.11),
                         SIMDE_FLOAT32_C(707365.62), SIMDE_FLOAT32_C(-229805.69), SIMDE_FLOAT32_C(428000.25), SIMDE_FLOAT32_C(-439269.84),
                         SIMDE_FLOAT32_C(-69633.48), SIMDE_FLOAT32_C(-15606.27), SIMDE_FLOAT32_C( 12694.64), SIMDE_FLOAT32_C( 23571.78)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -180.32), SIMDE_FLOAT32_C(  -914.18), SIMDE_FLOAT32_C(  -674.78), SIMDE_FLOAT32_C(   230.92),
                         SIMDE_FLOAT32_C(   619.73), SIMDE_FLOAT32_C(  -630.60), SIMDE_FLOAT32_C(  -418.47), SIMDE_FLOAT32_C(  -865.96),
                         SIMDE_FLOAT32_C(  -670.71), SIMDE_FLOAT32_C(    17.47), SIMDE_FLOAT32_C(    61.90), SIMDE_FLOAT32_C(   647.63),
                         SIMDE_FLOAT32_C(  -455.42), SIMDE_FLOAT32_C(  -850.08), SIMDE_FLOAT32_C(   132.45), SIMDE_FLOAT32_C(  -354.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -729.37), SIMDE_FLOAT32_C(  -945.53), SIMDE_FLOAT32_C(   198.36), SIMDE_FLOAT32_C(   643.43),
                         SIMDE_FLOAT32_C(  -994.87), SIMDE_FLOAT32_C(  -154.38), SIMDE_FLOAT32_C(  -536.68), SIMDE_FLOAT32_C(  -548.49),
                         SIMDE_FLOAT32_C(  -292.06), SIMDE_FLOAT32_C(  -771.61), SIMDE_FLOAT32_C(  -487.89), SIMDE_FLOAT32_C(  -482.82),
                         SIMDE_FLOAT32_C(   131.08), SIMDE_FLOAT32_C(   366.17), SIMDE_FLOAT32_C(   127.55), SIMDE_FLOAT32_C(  -936.85)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(131520.00), SIMDE_FLOAT32_C(864384.62), SIMDE_FLOAT32_C(-133849.36), SIMDE_FLOAT32_C(148580.86),
                         SIMDE_FLOAT32_C(-616550.75), SIMDE_FLOAT32_C( 97352.02), SIMDE_FLOAT32_C(224584.48), SIMDE_FLOAT32_C(474970.41),
                         SIMDE_FLOAT32_C(195887.56), SIMDE_FLOAT32_C(-13480.03), SIMDE_FLOAT32_C(-30200.39), SIMDE_FLOAT32_C(-312688.72),
                         SIMDE_FLOAT32_C(-59696.46), SIMDE_FLOAT32_C(-311273.81), SIMDE_FLOAT32_C( 16894.00), SIMDE_FLOAT32_C(332385.00)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   405.10), SIMDE_FLOAT32_C(   457.94), SIMDE_FLOAT32_C(   120.94), SIMDE_FLOAT32_C(   944.02),
                         SIMDE_FLOAT32_C(  -205.34), SIMDE_FLOAT32_C(   155.90), SIMDE_FLOAT32_C(  -913.86), SIMDE_FLOAT32_C(   170.83),
                         SIMDE_FLOAT32_C(  -194.64), SIMDE_FLOAT32_C(   505.24), SIMDE_FLOAT32_C(   874.71), SIMDE_FLOAT32_C(  -847.65),
                         SIMDE_FLOAT32_C(   -72.00), SIMDE_FLOAT32_C(   772.81), SIMDE_FLOAT32_C(  -151.00), SIMDE_FLOAT32_C(  -489.53)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   923.98), SIMDE_FLOAT32_C(   597.38), SIMDE_FLOAT32_C(  -237.17), SIMDE_FLOAT32_C(  -159.78),
                         SIMDE_FLOAT32_C(   -45.00), SIMDE_FLOAT32_C(  -167.53), SIMDE_FLOAT32_C(   681.28), SIMDE_FLOAT32_C(  -654.80),
                         SIMDE_FLOAT32_C(   504.91), SIMDE_FLOAT32_C(  -353.27), SIMDE_FLOAT32_C(  -789.06), SIMDE_FLOAT32_C(  -566.71),
                         SIMDE_FLOAT32_C(  -516.77), SIMDE_FLOAT32_C(   957.42), SIMDE_FLOAT32_C(  -465.35), SIMDE_FLOAT32_C(   491.11)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(374304.28), SIMDE_FLOAT32_C(273564.19), SIMDE_FLOAT32_C(-28683.34), SIMDE_FLOAT32_C(-150835.52),
                         SIMDE_FLOAT32_C(  9240.30), SIMDE_FLOAT32_C(-26117.93), SIMDE_FLOAT32_C(-622594.56), SIMDE_FLOAT32_C(-111859.48),
                         SIMDE_FLOAT32_C(-98275.68), SIMDE_FLOAT32_C(-178486.12), SIMDE_FLOAT32_C(-690198.69), SIMDE_FLOAT32_C(480371.75),
                         SIMDE_FLOAT32_C( 37207.44), SIMDE_FLOAT32_C(739903.75), SIMDE_FLOAT32_C( 70267.85), SIMDE_FLOAT32_C(-240413.08)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mul_ps(test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_mul_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
       { simde_mm512_set_ps(SIMDE_FLOAT32_C(   229.27), SIMDE_FLOAT32_C(  -114.91), SIMDE_FLOAT32_C(   520.43), SIMDE_FLOAT32_C(  -755.19),
                         SIMDE_FLOAT32_C(   -68.64), SIMDE_FLOAT32_C(   632.30), SIMDE_FLOAT32_C(    98.14), SIMDE_FLOAT32_C(   455.87),
                         SIMDE_FLOAT32_C(  -873.22), SIMDE_FLOAT32_C(  -223.86), SIMDE_FLOAT32_C(   181.32), SIMDE_FLOAT32_C(   364.92),
                         SIMDE_FLOAT32_C(   946.51), SIMDE_FLOAT32_C(    22.05), SIMDE_FLOAT32_C(   444.47), SIMDE_FLOAT32_C(  -746.17)),
      UINT16_C( 6152),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   702.34), SIMDE_FLOAT32_C(   368.95), SIMDE_FLOAT32_C(   161.91), SIMDE_FLOAT32_C(   459.04),
                         SIMDE_FLOAT32_C(  -828.47), SIMDE_FLOAT32_C(   943.39), SIMDE_FLOAT32_C(   202.26), SIMDE_FLOAT32_C(   112.87),
                         SIMDE_FLOAT32_C(   382.91), SIMDE_FLOAT32_C(   124.14), SIMDE_FLOAT32_C(   954.24), SIMDE_FLOAT32_C(  -214.34),
                         SIMDE_FLOAT32_C(  -998.93), SIMDE_FLOAT32_C(  -255.92), SIMDE_FLOAT32_C(    57.01), SIMDE_FLOAT32_C(  -391.73)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -206.12), SIMDE_FLOAT32_C(  -322.43), SIMDE_FLOAT32_C(  -456.42), SIMDE_FLOAT32_C(   258.98),
                         SIMDE_FLOAT32_C(   291.55), SIMDE_FLOAT32_C(  -459.80), SIMDE_FLOAT32_C(   286.61), SIMDE_FLOAT32_C(    15.13),
                         SIMDE_FLOAT32_C(  -772.68), SIMDE_FLOAT32_C(  -503.52), SIMDE_FLOAT32_C(  -599.88), SIMDE_FLOAT32_C(   107.93),
                         SIMDE_FLOAT32_C(    -3.35), SIMDE_FLOAT32_C(  -993.69), SIMDE_FLOAT32_C(  -325.33), SIMDE_FLOAT32_C(   755.40)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   229.27), SIMDE_FLOAT32_C(  -114.91), SIMDE_FLOAT32_C(   520.43), SIMDE_FLOAT32_C(118882.19),
                         SIMDE_FLOAT32_C(-241540.41), SIMDE_FLOAT32_C(   632.30), SIMDE_FLOAT32_C(    98.14), SIMDE_FLOAT32_C(   455.87),
                         SIMDE_FLOAT32_C(  -873.22), SIMDE_FLOAT32_C(  -223.86), SIMDE_FLOAT32_C(   181.32), SIMDE_FLOAT32_C(   364.92),
                         SIMDE_FLOAT32_C(  3346.42), SIMDE_FLOAT32_C(    22.05), SIMDE_FLOAT32_C(   444.47), SIMDE_FLOAT32_C(  -746.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   974.52), SIMDE_FLOAT32_C(    42.80), SIMDE_FLOAT32_C(  -282.69), SIMDE_FLOAT32_C(  -590.27),
                         SIMDE_FLOAT32_C(   398.09), SIMDE_FLOAT32_C(   498.04), SIMDE_FLOAT32_C(   449.97), SIMDE_FLOAT32_C(  -357.92),
                         SIMDE_FLOAT32_C(  -441.74), SIMDE_FLOAT32_C(  -180.77), SIMDE_FLOAT32_C(  -289.47), SIMDE_FLOAT32_C(  -620.49),
                         SIMDE_FLOAT32_C(   763.75), SIMDE_FLOAT32_C(  -763.91), SIMDE_FLOAT32_C(  -576.44), SIMDE_FLOAT32_C(   698.61)),
      UINT16_C(15973),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -512.47), SIMDE_FLOAT32_C(  -526.12), SIMDE_FLOAT32_C(   488.92), SIMDE_FLOAT32_C(   -99.48),
                         SIMDE_FLOAT32_C(   579.58), SIMDE_FLOAT32_C(   601.36), SIMDE_FLOAT32_C(   900.16), SIMDE_FLOAT32_C(   871.84),
                         SIMDE_FLOAT32_C(   797.21), SIMDE_FLOAT32_C(   523.84), SIMDE_FLOAT32_C(  -923.94), SIMDE_FLOAT32_C(   -14.85),
                         SIMDE_FLOAT32_C(  -320.00), SIMDE_FLOAT32_C(  -463.51), SIMDE_FLOAT32_C(  -980.83), SIMDE_FLOAT32_C(  -194.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   558.10), SIMDE_FLOAT32_C(  -796.63), SIMDE_FLOAT32_C(  -676.04), SIMDE_FLOAT32_C(   908.46),
                         SIMDE_FLOAT32_C(  -870.82), SIMDE_FLOAT32_C(   691.21), SIMDE_FLOAT32_C(  -550.67), SIMDE_FLOAT32_C(   268.52),
                         SIMDE_FLOAT32_C(   837.19), SIMDE_FLOAT32_C(  -677.60), SIMDE_FLOAT32_C(  -171.06), SIMDE_FLOAT32_C(   -56.18),
                         SIMDE_FLOAT32_C(   490.37), SIMDE_FLOAT32_C(   -61.61), SIMDE_FLOAT32_C(  -109.46), SIMDE_FLOAT32_C(  -710.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   974.52), SIMDE_FLOAT32_C(    42.80), SIMDE_FLOAT32_C(-330529.47), SIMDE_FLOAT32_C(-90373.61),
                         SIMDE_FLOAT32_C(-504709.88), SIMDE_FLOAT32_C(415666.06), SIMDE_FLOAT32_C(-495691.06), SIMDE_FLOAT32_C(  -357.92),
                         SIMDE_FLOAT32_C(  -441.74), SIMDE_FLOAT32_C(-354954.00), SIMDE_FLOAT32_C(158049.17), SIMDE_FLOAT32_C(  -620.49),
                         SIMDE_FLOAT32_C(   763.75), SIMDE_FLOAT32_C( 28556.85), SIMDE_FLOAT32_C(  -576.44), SIMDE_FLOAT32_C(138212.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -360.39), SIMDE_FLOAT32_C(   643.16), SIMDE_FLOAT32_C(  -441.22), SIMDE_FLOAT32_C(   409.21),
                         SIMDE_FLOAT32_C(   666.49), SIMDE_FLOAT32_C(   893.19), SIMDE_FLOAT32_C(  -859.97), SIMDE_FLOAT32_C(  -253.09),
                         SIMDE_FLOAT32_C(  -516.49), SIMDE_FLOAT32_C(  -209.00), SIMDE_FLOAT32_C(  -119.77), SIMDE_FLOAT32_C(    -6.76),
                         SIMDE_FLOAT32_C(   978.44), SIMDE_FLOAT32_C(   847.98), SIMDE_FLOAT32_C(   812.41), SIMDE_FLOAT32_C(  -887.11)),
      UINT16_C(51212),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   923.25), SIMDE_FLOAT32_C(   251.46), SIMDE_FLOAT32_C(   -49.04), SIMDE_FLOAT32_C(  -876.73),
                         SIMDE_FLOAT32_C(  -239.21), SIMDE_FLOAT32_C(   952.17), SIMDE_FLOAT32_C(  -247.44), SIMDE_FLOAT32_C(  -278.60),
                         SIMDE_FLOAT32_C(   877.29), SIMDE_FLOAT32_C(  -266.07), SIMDE_FLOAT32_C(  -839.50), SIMDE_FLOAT32_C(  -281.99),
                         SIMDE_FLOAT32_C(  -652.15), SIMDE_FLOAT32_C(  -877.11), SIMDE_FLOAT32_C(   527.90), SIMDE_FLOAT32_C(  -842.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -453.51), SIMDE_FLOAT32_C(   863.74), SIMDE_FLOAT32_C(   571.67), SIMDE_FLOAT32_C(   552.19),
                         SIMDE_FLOAT32_C(  -903.81), SIMDE_FLOAT32_C(  -274.48), SIMDE_FLOAT32_C(   891.44), SIMDE_FLOAT32_C(    28.40),
                         SIMDE_FLOAT32_C(   753.09), SIMDE_FLOAT32_C(   415.38), SIMDE_FLOAT32_C(  -974.66), SIMDE_FLOAT32_C(  -864.92),
                         SIMDE_FLOAT32_C(  -696.24), SIMDE_FLOAT32_C(  -279.21), SIMDE_FLOAT32_C(  -548.00), SIMDE_FLOAT32_C(     3.33)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-418703.12), SIMDE_FLOAT32_C(217196.06), SIMDE_FLOAT32_C(  -441.22), SIMDE_FLOAT32_C(   409.21),
                         SIMDE_FLOAT32_C(216200.39), SIMDE_FLOAT32_C(   893.19), SIMDE_FLOAT32_C(  -859.97), SIMDE_FLOAT32_C(  -253.09),
                         SIMDE_FLOAT32_C(  -516.49), SIMDE_FLOAT32_C(  -209.00), SIMDE_FLOAT32_C(  -119.77), SIMDE_FLOAT32_C(    -6.76),
                         SIMDE_FLOAT32_C(454052.94), SIMDE_FLOAT32_C(244897.88), SIMDE_FLOAT32_C(   812.41), SIMDE_FLOAT32_C(  -887.11)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   325.22), SIMDE_FLOAT32_C(   570.14), SIMDE_FLOAT32_C(   680.58), SIMDE_FLOAT32_C(  -511.81),
                         SIMDE_FLOAT32_C(  -117.17), SIMDE_FLOAT32_C(  -613.97), SIMDE_FLOAT32_C(   182.50), SIMDE_FLOAT32_C(   480.12),
                         SIMDE_FLOAT32_C(  -750.83), SIMDE_FLOAT32_C(   220.35), SIMDE_FLOAT32_C(   724.25), SIMDE_FLOAT32_C(   984.66),
                         SIMDE_FLOAT32_C(   871.75), SIMDE_FLOAT32_C(  -772.37), SIMDE_FLOAT32_C(   130.52), SIMDE_FLOAT32_C(   736.76)),
      UINT16_C(42108),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   960.66), SIMDE_FLOAT32_C(  -509.07), SIMDE_FLOAT32_C(   477.59), SIMDE_FLOAT32_C(  -132.21),
                         SIMDE_FLOAT32_C(   254.98), SIMDE_FLOAT32_C(   600.06), SIMDE_FLOAT32_C(    43.49), SIMDE_FLOAT32_C(   466.19),
                         SIMDE_FLOAT32_C(    22.31), SIMDE_FLOAT32_C(  -551.17), SIMDE_FLOAT32_C(  -167.87), SIMDE_FLOAT32_C(   278.33),
                         SIMDE_FLOAT32_C(  -232.38), SIMDE_FLOAT32_C(   650.45), SIMDE_FLOAT32_C(  -297.78), SIMDE_FLOAT32_C(  -280.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   562.20), SIMDE_FLOAT32_C(  -287.04), SIMDE_FLOAT32_C(   876.78), SIMDE_FLOAT32_C(   669.33),
                         SIMDE_FLOAT32_C(   940.57), SIMDE_FLOAT32_C(  -280.66), SIMDE_FLOAT32_C(    24.42), SIMDE_FLOAT32_C(  -147.12),
                         SIMDE_FLOAT32_C(   -57.84), SIMDE_FLOAT32_C(   841.25), SIMDE_FLOAT32_C(  -446.10), SIMDE_FLOAT32_C(  -973.24),
                         SIMDE_FLOAT32_C(   869.66), SIMDE_FLOAT32_C(   982.80), SIMDE_FLOAT32_C(  -763.04), SIMDE_FLOAT32_C(  -245.47)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(540083.06), SIMDE_FLOAT32_C(   570.14), SIMDE_FLOAT32_C(418741.38), SIMDE_FLOAT32_C(  -511.81),
                         SIMDE_FLOAT32_C(  -117.17), SIMDE_FLOAT32_C(-168412.84), SIMDE_FLOAT32_C(   182.50), SIMDE_FLOAT32_C(   480.12),
                         SIMDE_FLOAT32_C(  -750.83), SIMDE_FLOAT32_C(-463671.75), SIMDE_FLOAT32_C( 74886.80), SIMDE_FLOAT32_C(-270881.88),
                         SIMDE_FLOAT32_C(-202091.59), SIMDE_FLOAT32_C(639262.25), SIMDE_FLOAT32_C(   130.52), SIMDE_FLOAT32_C(   736.76)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -770.00), SIMDE_FLOAT32_C(   327.27), SIMDE_FLOAT32_C(  -642.48), SIMDE_FLOAT32_C(   864.18),
                         SIMDE_FLOAT32_C(  -775.21), SIMDE_FLOAT32_C(   -92.35), SIMDE_FLOAT32_C(  -864.16), SIMDE_FLOAT32_C(    80.78),
                         SIMDE_FLOAT32_C(  -974.40), SIMDE_FLOAT32_C(  -299.06), SIMDE_FLOAT32_C(  -754.35), SIMDE_FLOAT32_C(  -147.65),
                         SIMDE_FLOAT32_C(  -797.65), SIMDE_FLOAT32_C(   829.71), SIMDE_FLOAT32_C(   269.35), SIMDE_FLOAT32_C(   372.83)),
      UINT16_C(61342),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -220.25), SIMDE_FLOAT32_C(   802.77), SIMDE_FLOAT32_C(  -755.69), SIMDE_FLOAT32_C(   -58.33),
                         SIMDE_FLOAT32_C(   587.03), SIMDE_FLOAT32_C(   375.88), SIMDE_FLOAT32_C(   775.50), SIMDE_FLOAT32_C(  -179.11),
                         SIMDE_FLOAT32_C(   184.41), SIMDE_FLOAT32_C(  -603.91), SIMDE_FLOAT32_C(  -170.90), SIMDE_FLOAT32_C(  -781.45),
                         SIMDE_FLOAT32_C(  -860.97), SIMDE_FLOAT32_C(  -616.84), SIMDE_FLOAT32_C(   704.72), SIMDE_FLOAT32_C(  -251.07)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -62.78), SIMDE_FLOAT32_C(  -149.86), SIMDE_FLOAT32_C(   102.32), SIMDE_FLOAT32_C(  -271.05),
                         SIMDE_FLOAT32_C(  -465.52), SIMDE_FLOAT32_C(   979.82), SIMDE_FLOAT32_C(   499.92), SIMDE_FLOAT32_C(    32.84),
                         SIMDE_FLOAT32_C(   792.53), SIMDE_FLOAT32_C(   466.38), SIMDE_FLOAT32_C(  -301.08), SIMDE_FLOAT32_C(  -381.33),
                         SIMDE_FLOAT32_C(  -752.23), SIMDE_FLOAT32_C(    18.86), SIMDE_FLOAT32_C(  -462.80), SIMDE_FLOAT32_C(  -168.70)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( 13827.29), SIMDE_FLOAT32_C(-120303.12), SIMDE_FLOAT32_C(-77322.20), SIMDE_FLOAT32_C(   864.18),
                         SIMDE_FLOAT32_C(-273274.22), SIMDE_FLOAT32_C(368294.75), SIMDE_FLOAT32_C(387687.97), SIMDE_FLOAT32_C( -5881.97),
                         SIMDE_FLOAT32_C(146150.47), SIMDE_FLOAT32_C(  -299.06), SIMDE_FLOAT32_C(  -754.35), SIMDE_FLOAT32_C(297990.31),
                         SIMDE_FLOAT32_C(647647.44), SIMDE_FLOAT32_C(-11633.60), SIMDE_FLOAT32_C(-326144.41), SIMDE_FLOAT32_C(   372.83)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -62.71), SIMDE_FLOAT32_C(   768.74), SIMDE_FLOAT32_C(   172.84), SIMDE_FLOAT32_C(  -481.12),
                         SIMDE_FLOAT32_C(   290.59), SIMDE_FLOAT32_C(  -598.30), SIMDE_FLOAT32_C(  -212.50), SIMDE_FLOAT32_C(   657.51),
                         SIMDE_FLOAT32_C(  -400.85), SIMDE_FLOAT32_C(   353.00), SIMDE_FLOAT32_C(  -898.98), SIMDE_FLOAT32_C(  -461.75),
                         SIMDE_FLOAT32_C(  -690.46), SIMDE_FLOAT32_C(  -171.93), SIMDE_FLOAT32_C(   135.84), SIMDE_FLOAT32_C(  -604.52)),
      UINT16_C(61129),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -267.02), SIMDE_FLOAT32_C(   -31.24), SIMDE_FLOAT32_C(  -859.84), SIMDE_FLOAT32_C(  -104.89),
                         SIMDE_FLOAT32_C(   -39.96), SIMDE_FLOAT32_C(   179.68), SIMDE_FLOAT32_C(   -71.01), SIMDE_FLOAT32_C(   557.26),
                         SIMDE_FLOAT32_C(   127.39), SIMDE_FLOAT32_C(   271.58), SIMDE_FLOAT32_C(  -162.76), SIMDE_FLOAT32_C(   248.01),
                         SIMDE_FLOAT32_C(   856.68), SIMDE_FLOAT32_C(   762.32), SIMDE_FLOAT32_C(   432.07), SIMDE_FLOAT32_C(   743.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -846.50), SIMDE_FLOAT32_C(   988.92), SIMDE_FLOAT32_C(   696.05), SIMDE_FLOAT32_C(   647.58),
                         SIMDE_FLOAT32_C(   780.95), SIMDE_FLOAT32_C(    46.01), SIMDE_FLOAT32_C(   -77.15), SIMDE_FLOAT32_C(  -747.70),
                         SIMDE_FLOAT32_C(  -416.84), SIMDE_FLOAT32_C(   679.81), SIMDE_FLOAT32_C(  -124.78), SIMDE_FLOAT32_C(  -976.50),
                         SIMDE_FLOAT32_C(  -745.93), SIMDE_FLOAT32_C(   116.64), SIMDE_FLOAT32_C(  -479.84), SIMDE_FLOAT32_C(   919.24)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(226032.42), SIMDE_FLOAT32_C(-30893.86), SIMDE_FLOAT32_C(-598491.62), SIMDE_FLOAT32_C(  -481.12),
                         SIMDE_FLOAT32_C(-31206.76), SIMDE_FLOAT32_C(  8267.08), SIMDE_FLOAT32_C(  5478.42), SIMDE_FLOAT32_C(   657.51),
                         SIMDE_FLOAT32_C(-53101.25), SIMDE_FLOAT32_C(184622.80), SIMDE_FLOAT32_C(  -898.98), SIMDE_FLOAT32_C(  -461.75),
                         SIMDE_FLOAT32_C(-639023.31), SIMDE_FLOAT32_C(  -171.93), SIMDE_FLOAT32_C(   135.84), SIMDE_FLOAT32_C(683050.44)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   722.50), SIMDE_FLOAT32_C(     1.66), SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(  -417.42),
                         SIMDE_FLOAT32_C(   237.94), SIMDE_FLOAT32_C(   860.12), SIMDE_FLOAT32_C(  -692.46), SIMDE_FLOAT32_C(     6.00),
                         SIMDE_FLOAT32_C(  -979.01), SIMDE_FLOAT32_C(    20.38), SIMDE_FLOAT32_C(    85.42), SIMDE_FLOAT32_C(  -156.50),
                         SIMDE_FLOAT32_C(    23.29), SIMDE_FLOAT32_C(  -569.89), SIMDE_FLOAT32_C(    24.40), SIMDE_FLOAT32_C(   257.32)),
      UINT16_C(53230),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -804.04), SIMDE_FLOAT32_C(  -689.70), SIMDE_FLOAT32_C(   435.74), SIMDE_FLOAT32_C(    49.95),
                         SIMDE_FLOAT32_C(   554.19), SIMDE_FLOAT32_C(   936.14), SIMDE_FLOAT32_C(   554.63), SIMDE_FLOAT32_C(  -242.02),
                         SIMDE_FLOAT32_C(  -909.08), SIMDE_FLOAT32_C(  -184.42), SIMDE_FLOAT32_C(  -668.15), SIMDE_FLOAT32_C(   202.23),
                         SIMDE_FLOAT32_C(   620.00), SIMDE_FLOAT32_C(   -11.65), SIMDE_FLOAT32_C(  -295.73), SIMDE_FLOAT32_C(  -637.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -700.26), SIMDE_FLOAT32_C(   -48.09), SIMDE_FLOAT32_C(  -661.22), SIMDE_FLOAT32_C(  -536.85),
                         SIMDE_FLOAT32_C(  -172.68), SIMDE_FLOAT32_C(   263.32), SIMDE_FLOAT32_C(  -189.80), SIMDE_FLOAT32_C(  -595.66),
                         SIMDE_FLOAT32_C(   244.26), SIMDE_FLOAT32_C(  -637.08), SIMDE_FLOAT32_C(  -871.35), SIMDE_FLOAT32_C(  -417.36),
                         SIMDE_FLOAT32_C(  -313.14), SIMDE_FLOAT32_C(  -902.95), SIMDE_FLOAT32_C(  -801.13), SIMDE_FLOAT32_C(  -357.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(563037.06), SIMDE_FLOAT32_C( 33167.67), SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(  -417.42),
                         SIMDE_FLOAT32_C(-95697.52), SIMDE_FLOAT32_C(246504.39), SIMDE_FLOAT32_C(-105268.77), SIMDE_FLOAT32_C(144161.62),
                         SIMDE_FLOAT32_C(-222051.88), SIMDE_FLOAT32_C(117490.30), SIMDE_FLOAT32_C(582192.50), SIMDE_FLOAT32_C(  -156.50),
                         SIMDE_FLOAT32_C(-194146.81), SIMDE_FLOAT32_C( 10519.37), SIMDE_FLOAT32_C(236918.19), SIMDE_FLOAT32_C(   257.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   113.45), SIMDE_FLOAT32_C(   -47.38), SIMDE_FLOAT32_C(   434.74), SIMDE_FLOAT32_C(   133.40),
                         SIMDE_FLOAT32_C(    37.09), SIMDE_FLOAT32_C(  -195.77), SIMDE_FLOAT32_C(  -878.67), SIMDE_FLOAT32_C(   758.19),
                         SIMDE_FLOAT32_C(   -87.72), SIMDE_FLOAT32_C(  -903.51), SIMDE_FLOAT32_C(  -821.22), SIMDE_FLOAT32_C(  -102.72),
                         SIMDE_FLOAT32_C(   329.70), SIMDE_FLOAT32_C(   752.97), SIMDE_FLOAT32_C(  -341.79), SIMDE_FLOAT32_C(  -130.85)),
      UINT16_C(62361),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -994.03), SIMDE_FLOAT32_C(  -716.03), SIMDE_FLOAT32_C(  -435.51), SIMDE_FLOAT32_C(  -960.04),
                         SIMDE_FLOAT32_C(    32.26), SIMDE_FLOAT32_C(  -474.76), SIMDE_FLOAT32_C(  -182.77), SIMDE_FLOAT32_C(  -229.72),
                         SIMDE_FLOAT32_C(  -949.63), SIMDE_FLOAT32_C(  -938.60), SIMDE_FLOAT32_C(  -855.41), SIMDE_FLOAT32_C(  -231.99),
                         SIMDE_FLOAT32_C(   115.21), SIMDE_FLOAT32_C(   716.21), SIMDE_FLOAT32_C(  -407.80), SIMDE_FLOAT32_C(   373.68)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   186.12), SIMDE_FLOAT32_C(  -224.01), SIMDE_FLOAT32_C(   199.06), SIMDE_FLOAT32_C(  -162.12),
                         SIMDE_FLOAT32_C(  -212.56), SIMDE_FLOAT32_C(  -205.93), SIMDE_FLOAT32_C(  -577.22), SIMDE_FLOAT32_C(  -567.47),
                         SIMDE_FLOAT32_C(  -916.44), SIMDE_FLOAT32_C(   780.43), SIMDE_FLOAT32_C(  -604.79), SIMDE_FLOAT32_C(   540.03),
                         SIMDE_FLOAT32_C(  -974.56), SIMDE_FLOAT32_C(  -517.05), SIMDE_FLOAT32_C(  -241.22), SIMDE_FLOAT32_C(   102.85)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-185008.86), SIMDE_FLOAT32_C(160397.89), SIMDE_FLOAT32_C(-86692.62), SIMDE_FLOAT32_C(155641.67),
                         SIMDE_FLOAT32_C(    37.09), SIMDE_FLOAT32_C(  -195.77), SIMDE_FLOAT32_C(105498.50), SIMDE_FLOAT32_C(130359.20),
                         SIMDE_FLOAT32_C(870278.94), SIMDE_FLOAT32_C(  -903.51), SIMDE_FLOAT32_C(  -821.22), SIMDE_FLOAT32_C(-125281.57),
                         SIMDE_FLOAT32_C(-112279.05), SIMDE_FLOAT32_C(   752.97), SIMDE_FLOAT32_C(  -341.79), SIMDE_FLOAT32_C( 38432.99)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_mul_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_mul_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(47289),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -658.59), SIMDE_FLOAT32_C(  -110.05), SIMDE_FLOAT32_C(  -529.45), SIMDE_FLOAT32_C(    46.72),
                         SIMDE_FLOAT32_C(   -62.14), SIMDE_FLOAT32_C(   483.09), SIMDE_FLOAT32_C(   301.22), SIMDE_FLOAT32_C(  -113.80),
                         SIMDE_FLOAT32_C(  -597.24), SIMDE_FLOAT32_C(    55.35), SIMDE_FLOAT32_C(   938.56), SIMDE_FLOAT32_C(   -50.24),
                         SIMDE_FLOAT32_C(    49.65), SIMDE_FLOAT32_C(  -991.96), SIMDE_FLOAT32_C(   606.92), SIMDE_FLOAT32_C(   149.59)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   375.27), SIMDE_FLOAT32_C(  -498.85), SIMDE_FLOAT32_C(  -342.99), SIMDE_FLOAT32_C(   861.28),
                         SIMDE_FLOAT32_C(   526.60), SIMDE_FLOAT32_C(  -759.33), SIMDE_FLOAT32_C(   328.64), SIMDE_FLOAT32_C(   698.74),
                         SIMDE_FLOAT32_C(   615.23), SIMDE_FLOAT32_C(   873.23), SIMDE_FLOAT32_C(   127.27), SIMDE_FLOAT32_C(   719.43),
                         SIMDE_FLOAT32_C(  -625.99), SIMDE_FLOAT32_C(  -942.07), SIMDE_FLOAT32_C(   458.53), SIMDE_FLOAT32_C(   322.40)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-247149.08), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(181596.05), SIMDE_FLOAT32_C( 40239.00),
                         SIMDE_FLOAT32_C(-32722.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(-367439.94), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(119450.53), SIMDE_FLOAT32_C(-36144.16),
                         SIMDE_FLOAT32_C(-31080.40), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( 48227.81)) },
    { UINT16_C(37892),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -265.18), SIMDE_FLOAT32_C(   394.19), SIMDE_FLOAT32_C(   565.39), SIMDE_FLOAT32_C(  -167.00),
                         SIMDE_FLOAT32_C(   350.77), SIMDE_FLOAT32_C(   863.35), SIMDE_FLOAT32_C(  -537.11), SIMDE_FLOAT32_C(  -601.68),
                         SIMDE_FLOAT32_C(  -980.35), SIMDE_FLOAT32_C(  -851.86), SIMDE_FLOAT32_C(  -959.52), SIMDE_FLOAT32_C(  -856.72),
                         SIMDE_FLOAT32_C(   393.09), SIMDE_FLOAT32_C(  -263.92), SIMDE_FLOAT32_C(   261.53), SIMDE_FLOAT32_C(   409.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -261.28), SIMDE_FLOAT32_C(  -762.17), SIMDE_FLOAT32_C(   850.55), SIMDE_FLOAT32_C(  -684.91),
                         SIMDE_FLOAT32_C(    69.61), SIMDE_FLOAT32_C(   771.73), SIMDE_FLOAT32_C(  -506.14), SIMDE_FLOAT32_C(  -578.92),
                         SIMDE_FLOAT32_C(   322.24), SIMDE_FLOAT32_C(   192.10), SIMDE_FLOAT32_C(  -768.24), SIMDE_FLOAT32_C(  -528.40),
                         SIMDE_FLOAT32_C(  -871.80), SIMDE_FLOAT32_C(   -55.77), SIMDE_FLOAT32_C(   401.18), SIMDE_FLOAT32_C(  -914.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( 69286.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(114379.97),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(666273.06), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( 14718.82), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(21270),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   124.49), SIMDE_FLOAT32_C(   701.35), SIMDE_FLOAT32_C(   498.79), SIMDE_FLOAT32_C(   832.83),
                         SIMDE_FLOAT32_C(  -974.32), SIMDE_FLOAT32_C(  -582.20), SIMDE_FLOAT32_C(  -288.73), SIMDE_FLOAT32_C(   146.91),
                         SIMDE_FLOAT32_C(   866.64), SIMDE_FLOAT32_C(   902.02), SIMDE_FLOAT32_C(   -35.40), SIMDE_FLOAT32_C(  -390.90),
                         SIMDE_FLOAT32_C(  -670.61), SIMDE_FLOAT32_C(  -294.26), SIMDE_FLOAT32_C(   904.08), SIMDE_FLOAT32_C(  -920.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   282.01), SIMDE_FLOAT32_C(   688.91), SIMDE_FLOAT32_C(   333.36), SIMDE_FLOAT32_C(   260.07),
                         SIMDE_FLOAT32_C(   576.07), SIMDE_FLOAT32_C(   133.85), SIMDE_FLOAT32_C(   534.76), SIMDE_FLOAT32_C(  -643.54),
                         SIMDE_FLOAT32_C(  -999.40), SIMDE_FLOAT32_C(   257.62), SIMDE_FLOAT32_C(   420.35), SIMDE_FLOAT32_C(  -394.28),
                         SIMDE_FLOAT32_C(   211.89), SIMDE_FLOAT32_C(   496.82), SIMDE_FLOAT32_C(  -993.25), SIMDE_FLOAT32_C(  -590.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(483167.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(216594.11),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-154401.27), SIMDE_FLOAT32_C(-94542.46),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(154124.05),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-146194.27), SIMDE_FLOAT32_C(-897977.50), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(31632),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    50.34), SIMDE_FLOAT32_C(   -97.56), SIMDE_FLOAT32_C(   263.08), SIMDE_FLOAT32_C(  -308.40),
                         SIMDE_FLOAT32_C(   354.47), SIMDE_FLOAT32_C(   -70.93), SIMDE_FLOAT32_C(   486.01), SIMDE_FLOAT32_C(  -938.29),
                         SIMDE_FLOAT32_C(   -47.71), SIMDE_FLOAT32_C(  -345.27), SIMDE_FLOAT32_C(    12.62), SIMDE_FLOAT32_C(   733.96),
                         SIMDE_FLOAT32_C(   753.32), SIMDE_FLOAT32_C(  -397.23), SIMDE_FLOAT32_C(   708.66), SIMDE_FLOAT32_C(   404.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   875.93), SIMDE_FLOAT32_C(  -911.86), SIMDE_FLOAT32_C(   784.71), SIMDE_FLOAT32_C(    64.13),
                         SIMDE_FLOAT32_C(  -731.87), SIMDE_FLOAT32_C(  -647.62), SIMDE_FLOAT32_C(   107.77), SIMDE_FLOAT32_C(   557.50),
                         SIMDE_FLOAT32_C(  -491.55), SIMDE_FLOAT32_C(   414.15), SIMDE_FLOAT32_C(  -504.43), SIMDE_FLOAT32_C(   -27.13),
                         SIMDE_FLOAT32_C(  -947.21), SIMDE_FLOAT32_C(  -164.39), SIMDE_FLOAT32_C(   287.82), SIMDE_FLOAT32_C(   414.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( 88961.05), SIMDE_FLOAT32_C(206441.50), SIMDE_FLOAT32_C(-19777.69),
                         SIMDE_FLOAT32_C(-259425.95), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( 52377.30), SIMDE_FLOAT32_C(-523096.66),
                         SIMDE_FLOAT32_C( 23451.85), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-19912.33),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(47299),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -544.18), SIMDE_FLOAT32_C(  -903.28), SIMDE_FLOAT32_C(   559.95), SIMDE_FLOAT32_C(  -483.39),
                         SIMDE_FLOAT32_C(  -994.67), SIMDE_FLOAT32_C(  -750.48), SIMDE_FLOAT32_C(   312.50), SIMDE_FLOAT32_C(   110.85),
                         SIMDE_FLOAT32_C(  -430.65), SIMDE_FLOAT32_C(    39.80), SIMDE_FLOAT32_C(   -26.24), SIMDE_FLOAT32_C(   378.89),
                         SIMDE_FLOAT32_C(  -139.95), SIMDE_FLOAT32_C(  -775.11), SIMDE_FLOAT32_C(  -758.69), SIMDE_FLOAT32_C(   318.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   548.46), SIMDE_FLOAT32_C(   682.86), SIMDE_FLOAT32_C(  -635.50), SIMDE_FLOAT32_C(   737.24),
                         SIMDE_FLOAT32_C(   707.78), SIMDE_FLOAT32_C(  -907.72), SIMDE_FLOAT32_C(  -791.08), SIMDE_FLOAT32_C(   176.45),
                         SIMDE_FLOAT32_C(    64.55), SIMDE_FLOAT32_C(    55.56), SIMDE_FLOAT32_C(  -108.86), SIMDE_FLOAT32_C(   505.77),
                         SIMDE_FLOAT32_C(   224.25), SIMDE_FLOAT32_C(   639.22), SIMDE_FLOAT32_C(   369.92), SIMDE_FLOAT32_C(  -708.31)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-298460.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-355848.22), SIMDE_FLOAT32_C(-356374.44),
                         SIMDE_FLOAT32_C(-704007.56), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(-27798.46), SIMDE_FLOAT32_C(  2211.29), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-280654.62), SIMDE_FLOAT32_C(-225603.83)) },
    { UINT16_C(40773),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -86.71), SIMDE_FLOAT32_C(  -432.63), SIMDE_FLOAT32_C(  -433.37), SIMDE_FLOAT32_C(  -539.66),
                         SIMDE_FLOAT32_C(   559.54), SIMDE_FLOAT32_C(  -287.88), SIMDE_FLOAT32_C(  -991.42), SIMDE_FLOAT32_C(  -690.07),
                         SIMDE_FLOAT32_C(   345.70), SIMDE_FLOAT32_C(   616.00), SIMDE_FLOAT32_C(   341.79), SIMDE_FLOAT32_C(  -307.10),
                         SIMDE_FLOAT32_C(   709.24), SIMDE_FLOAT32_C(  -920.15), SIMDE_FLOAT32_C(   404.20), SIMDE_FLOAT32_C(    52.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   815.79), SIMDE_FLOAT32_C(  -788.62), SIMDE_FLOAT32_C(  -370.04), SIMDE_FLOAT32_C(  -939.88),
                         SIMDE_FLOAT32_C(  -591.16), SIMDE_FLOAT32_C(  -819.13), SIMDE_FLOAT32_C(   932.20), SIMDE_FLOAT32_C(  -243.11),
                         SIMDE_FLOAT32_C(   -87.62), SIMDE_FLOAT32_C(   725.93), SIMDE_FLOAT32_C(  -374.67), SIMDE_FLOAT32_C(   301.09),
                         SIMDE_FLOAT32_C(  -174.47), SIMDE_FLOAT32_C(  -898.14), SIMDE_FLOAT32_C(  -924.02), SIMDE_FLOAT32_C(  -333.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(-70737.15), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(507215.62),
                         SIMDE_FLOAT32_C(-330777.62), SIMDE_FLOAT32_C(235811.16), SIMDE_FLOAT32_C(-924201.75), SIMDE_FLOAT32_C(167762.92),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(447172.88), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(826423.56), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-17520.49)) },
    { UINT16_C(61172),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -690.89), SIMDE_FLOAT32_C(  -270.55), SIMDE_FLOAT32_C(   476.48), SIMDE_FLOAT32_C(   219.51),
                         SIMDE_FLOAT32_C(   642.18), SIMDE_FLOAT32_C(  -588.29), SIMDE_FLOAT32_C(  -762.74), SIMDE_FLOAT32_C(   -33.80),
                         SIMDE_FLOAT32_C(  -786.44), SIMDE_FLOAT32_C(  -855.21), SIMDE_FLOAT32_C(   145.12), SIMDE_FLOAT32_C(    50.96),
                         SIMDE_FLOAT32_C(   710.85), SIMDE_FLOAT32_C(   234.05), SIMDE_FLOAT32_C(   345.96), SIMDE_FLOAT32_C(   118.24)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -143.72), SIMDE_FLOAT32_C(  -461.06), SIMDE_FLOAT32_C(   866.17), SIMDE_FLOAT32_C(  -706.49),
                         SIMDE_FLOAT32_C(   594.76), SIMDE_FLOAT32_C(   424.92), SIMDE_FLOAT32_C(   166.20), SIMDE_FLOAT32_C(   776.85),
                         SIMDE_FLOAT32_C(  -191.32), SIMDE_FLOAT32_C(  -329.15), SIMDE_FLOAT32_C(  -651.62), SIMDE_FLOAT32_C(   -22.33),
                         SIMDE_FLOAT32_C(  -429.53), SIMDE_FLOAT32_C(   758.36), SIMDE_FLOAT32_C(   926.10), SIMDE_FLOAT32_C(    17.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C( 99294.71), SIMDE_FLOAT32_C(124739.77), SIMDE_FLOAT32_C(412712.69), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(381942.97), SIMDE_FLOAT32_C(-249976.19), SIMDE_FLOAT32_C(-126767.38), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(150461.70), SIMDE_FLOAT32_C(281492.38), SIMDE_FLOAT32_C(-94563.09), SIMDE_FLOAT32_C( -1137.94),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(177494.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(31704),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   761.96), SIMDE_FLOAT32_C(   858.35), SIMDE_FLOAT32_C(   360.42), SIMDE_FLOAT32_C(   321.87),
                         SIMDE_FLOAT32_C(  -444.03), SIMDE_FLOAT32_C(  -177.67), SIMDE_FLOAT32_C(  -802.25), SIMDE_FLOAT32_C(   408.95),
                         SIMDE_FLOAT32_C(  -337.63), SIMDE_FLOAT32_C(   948.17), SIMDE_FLOAT32_C(   248.80), SIMDE_FLOAT32_C(   170.02),
                         SIMDE_FLOAT32_C(   939.41), SIMDE_FLOAT32_C(  -580.14), SIMDE_FLOAT32_C(   237.93), SIMDE_FLOAT32_C(  -698.11)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   786.48), SIMDE_FLOAT32_C(  -475.23), SIMDE_FLOAT32_C(   258.84), SIMDE_FLOAT32_C(   824.39),
                         SIMDE_FLOAT32_C(  -282.56), SIMDE_FLOAT32_C(   765.76), SIMDE_FLOAT32_C(  -709.23), SIMDE_FLOAT32_C(   701.93),
                         SIMDE_FLOAT32_C(  -367.75), SIMDE_FLOAT32_C(   404.32), SIMDE_FLOAT32_C(  -447.00), SIMDE_FLOAT32_C(   864.94),
                         SIMDE_FLOAT32_C(   954.31), SIMDE_FLOAT32_C(   410.35), SIMDE_FLOAT32_C(  -565.19), SIMDE_FLOAT32_C(  -545.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(-407913.66), SIMDE_FLOAT32_C( 93291.12), SIMDE_FLOAT32_C(265346.41),
                         SIMDE_FLOAT32_C(125465.12), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(568979.75), SIMDE_FLOAT32_C(287054.28),
                         SIMDE_FLOAT32_C(124163.44), SIMDE_FLOAT32_C(383364.09), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(147057.11),
                         SIMDE_FLOAT32_C(896488.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_mul_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mul_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -736.65), SIMDE_FLOAT64_C( -764.30),
                         SIMDE_FLOAT64_C(  675.25), SIMDE_FLOAT64_C( -182.15),
                         SIMDE_FLOAT64_C( -748.44), SIMDE_FLOAT64_C(   82.10),
                         SIMDE_FLOAT64_C(  684.52), SIMDE_FLOAT64_C( -343.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -267.96), SIMDE_FLOAT64_C( -536.57),
                         SIMDE_FLOAT64_C( -934.00), SIMDE_FLOAT64_C(  653.62),
                         SIMDE_FLOAT64_C(  984.11), SIMDE_FLOAT64_C(  140.30),
                         SIMDE_FLOAT64_C( -580.05), SIMDE_FLOAT64_C( -915.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(197392.73), SIMDE_FLOAT64_C(410100.45),
                         SIMDE_FLOAT64_C(-630683.50), SIMDE_FLOAT64_C(-119056.88),
                         SIMDE_FLOAT64_C(-736547.29), SIMDE_FLOAT64_C(11518.63),
                         SIMDE_FLOAT64_C(-397055.83), SIMDE_FLOAT64_C(314184.67)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -775.40), SIMDE_FLOAT64_C( -210.92),
                         SIMDE_FLOAT64_C(  987.42), SIMDE_FLOAT64_C(  542.45),
                         SIMDE_FLOAT64_C( -745.60), SIMDE_FLOAT64_C(  -50.38),
                         SIMDE_FLOAT64_C(  163.82), SIMDE_FLOAT64_C( -164.62)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  263.91), SIMDE_FLOAT64_C( -350.89),
                         SIMDE_FLOAT64_C( -318.01), SIMDE_FLOAT64_C( -980.00),
                         SIMDE_FLOAT64_C(  872.18), SIMDE_FLOAT64_C(   80.96),
                         SIMDE_FLOAT64_C(  145.89), SIMDE_FLOAT64_C(  832.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-204635.81), SIMDE_FLOAT64_C(74009.72),
                         SIMDE_FLOAT64_C(-314009.43), SIMDE_FLOAT64_C(-531601.00),
                         SIMDE_FLOAT64_C(-650297.41), SIMDE_FLOAT64_C(-4078.76),
                         SIMDE_FLOAT64_C(23899.70), SIMDE_FLOAT64_C(-137110.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  606.76), SIMDE_FLOAT64_C( -664.92),
                         SIMDE_FLOAT64_C(  454.81), SIMDE_FLOAT64_C(  299.40),
                         SIMDE_FLOAT64_C( -524.63), SIMDE_FLOAT64_C(   40.68),
                         SIMDE_FLOAT64_C(  218.77), SIMDE_FLOAT64_C(   35.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -757.25), SIMDE_FLOAT64_C(  594.07),
                         SIMDE_FLOAT64_C(  304.96), SIMDE_FLOAT64_C( -155.47),
                         SIMDE_FLOAT64_C(  635.03), SIMDE_FLOAT64_C(  654.85),
                         SIMDE_FLOAT64_C(  777.61), SIMDE_FLOAT64_C( -598.19)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-459469.01), SIMDE_FLOAT64_C(-395009.02),
                         SIMDE_FLOAT64_C(138698.86), SIMDE_FLOAT64_C(-46547.72),
                         SIMDE_FLOAT64_C(-333155.79), SIMDE_FLOAT64_C(26639.30),
                         SIMDE_FLOAT64_C(170117.74), SIMDE_FLOAT64_C(-21427.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -910.74), SIMDE_FLOAT64_C( -302.10),
                         SIMDE_FLOAT64_C(  937.08), SIMDE_FLOAT64_C(  618.13),
                         SIMDE_FLOAT64_C(   85.12), SIMDE_FLOAT64_C(    3.50),
                         SIMDE_FLOAT64_C( -122.84), SIMDE_FLOAT64_C(  290.22)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  392.21), SIMDE_FLOAT64_C(  139.00),
                         SIMDE_FLOAT64_C( -878.97), SIMDE_FLOAT64_C(  778.57),
                         SIMDE_FLOAT64_C( -810.83), SIMDE_FLOAT64_C(  413.49),
                         SIMDE_FLOAT64_C(  505.44), SIMDE_FLOAT64_C(  291.58)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-357201.34), SIMDE_FLOAT64_C(-41991.90),
                         SIMDE_FLOAT64_C(-823665.21), SIMDE_FLOAT64_C(481257.47),
                         SIMDE_FLOAT64_C(-69017.85), SIMDE_FLOAT64_C( 1447.22),
                         SIMDE_FLOAT64_C(-62088.25), SIMDE_FLOAT64_C(84622.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  232.02), SIMDE_FLOAT64_C(  984.70),
                         SIMDE_FLOAT64_C( -800.83), SIMDE_FLOAT64_C( -826.63),
                         SIMDE_FLOAT64_C(  822.26), SIMDE_FLOAT64_C( -892.21),
                         SIMDE_FLOAT64_C( -651.70), SIMDE_FLOAT64_C( -380.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  880.83), SIMDE_FLOAT64_C( -763.06),
                         SIMDE_FLOAT64_C( -540.57), SIMDE_FLOAT64_C( -512.55),
                         SIMDE_FLOAT64_C(  -32.98), SIMDE_FLOAT64_C(  700.87),
                         SIMDE_FLOAT64_C( -425.19), SIMDE_FLOAT64_C( -849.48)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(204370.18), SIMDE_FLOAT64_C(-751385.18),
                         SIMDE_FLOAT64_C(432904.67), SIMDE_FLOAT64_C(423689.21),
                         SIMDE_FLOAT64_C(-27118.13), SIMDE_FLOAT64_C(-625323.22),
                         SIMDE_FLOAT64_C(277096.32), SIMDE_FLOAT64_C(323227.14)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  202.90), SIMDE_FLOAT64_C( -396.66),
                         SIMDE_FLOAT64_C( -364.01), SIMDE_FLOAT64_C(   56.81),
                         SIMDE_FLOAT64_C( -881.59), SIMDE_FLOAT64_C(  212.81),
                         SIMDE_FLOAT64_C( -968.64), SIMDE_FLOAT64_C( -657.19)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.35), SIMDE_FLOAT64_C( -305.38),
                         SIMDE_FLOAT64_C(  546.45), SIMDE_FLOAT64_C( -697.03),
                         SIMDE_FLOAT64_C(   93.97), SIMDE_FLOAT64_C(  975.92),
                         SIMDE_FLOAT64_C(  876.47), SIMDE_FLOAT64_C(  762.37)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-44911.92), SIMDE_FLOAT64_C(121132.03),
                         SIMDE_FLOAT64_C(-198913.26), SIMDE_FLOAT64_C(-39598.27),
                         SIMDE_FLOAT64_C(-82843.01), SIMDE_FLOAT64_C(207685.54),
                         SIMDE_FLOAT64_C(-848983.90), SIMDE_FLOAT64_C(-501021.94)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -538.86), SIMDE_FLOAT64_C(   39.65),
                         SIMDE_FLOAT64_C( -229.28), SIMDE_FLOAT64_C( -842.78),
                         SIMDE_FLOAT64_C(  -14.75), SIMDE_FLOAT64_C( -859.98),
                         SIMDE_FLOAT64_C(  215.44), SIMDE_FLOAT64_C(  762.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -16.18), SIMDE_FLOAT64_C( -931.82),
                         SIMDE_FLOAT64_C( -687.15), SIMDE_FLOAT64_C( -416.23),
                         SIMDE_FLOAT64_C( -313.36), SIMDE_FLOAT64_C(  905.90),
                         SIMDE_FLOAT64_C(    1.93), SIMDE_FLOAT64_C( -464.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 8718.75), SIMDE_FLOAT64_C(-36946.66),
                         SIMDE_FLOAT64_C(157549.75), SIMDE_FLOAT64_C(350790.32),
                         SIMDE_FLOAT64_C( 4622.06), SIMDE_FLOAT64_C(-779055.88),
                         SIMDE_FLOAT64_C(  415.80), SIMDE_FLOAT64_C(-354700.69)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  -25.40), SIMDE_FLOAT64_C( -267.80),
                         SIMDE_FLOAT64_C(  353.79), SIMDE_FLOAT64_C(  -35.72),
                         SIMDE_FLOAT64_C(  125.21), SIMDE_FLOAT64_C(  137.22),
                         SIMDE_FLOAT64_C(  310.88), SIMDE_FLOAT64_C( -724.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -551.49), SIMDE_FLOAT64_C(  -42.33),
                         SIMDE_FLOAT64_C( -926.18), SIMDE_FLOAT64_C(   36.96),
                         SIMDE_FLOAT64_C(  954.39), SIMDE_FLOAT64_C(  874.71),
                         SIMDE_FLOAT64_C( -375.00), SIMDE_FLOAT64_C(  949.07)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(14007.85), SIMDE_FLOAT64_C(11335.97),
                         SIMDE_FLOAT64_C(-327673.22), SIMDE_FLOAT64_C(-1320.21),
                         SIMDE_FLOAT64_C(119499.17), SIMDE_FLOAT64_C(120027.71),
                         SIMDE_FLOAT64_C(-116580.00), SIMDE_FLOAT64_C(-687648.67)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mul_pd(test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_mul_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -821.30), SIMDE_FLOAT64_C( -768.64),
                         SIMDE_FLOAT64_C(  -18.18), SIMDE_FLOAT64_C( -679.16),
                         SIMDE_FLOAT64_C( -992.98), SIMDE_FLOAT64_C( -764.30),
                         SIMDE_FLOAT64_C(  419.74), SIMDE_FLOAT64_C(  970.61)),
      UINT8_C( 76),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -167.78), SIMDE_FLOAT64_C( -432.98),
                         SIMDE_FLOAT64_C( -407.63), SIMDE_FLOAT64_C(  -78.73),
                         SIMDE_FLOAT64_C( -377.24), SIMDE_FLOAT64_C( -338.63),
                         SIMDE_FLOAT64_C( -681.32), SIMDE_FLOAT64_C( -483.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -931.82), SIMDE_FLOAT64_C( -180.10),
                         SIMDE_FLOAT64_C( -213.80), SIMDE_FLOAT64_C( -618.07),
                         SIMDE_FLOAT64_C(  922.09), SIMDE_FLOAT64_C( -681.84),
                         SIMDE_FLOAT64_C( -317.54), SIMDE_FLOAT64_C(  448.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -821.30), SIMDE_FLOAT64_C(77979.70),
                         SIMDE_FLOAT64_C(  -18.18), SIMDE_FLOAT64_C( -679.16),
                         SIMDE_FLOAT64_C(-347849.23), SIMDE_FLOAT64_C(230891.48),
                         SIMDE_FLOAT64_C(  419.74), SIMDE_FLOAT64_C(  970.61)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -706.27), SIMDE_FLOAT64_C( -736.90),
                         SIMDE_FLOAT64_C(  388.85), SIMDE_FLOAT64_C( -452.26),
                         SIMDE_FLOAT64_C( -983.38), SIMDE_FLOAT64_C( -800.62),
                         SIMDE_FLOAT64_C(  310.59), SIMDE_FLOAT64_C(  810.60)),
      UINT8_C( 87),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -613.25), SIMDE_FLOAT64_C(  846.16),
                         SIMDE_FLOAT64_C(  824.90), SIMDE_FLOAT64_C( -554.53),
                         SIMDE_FLOAT64_C( -163.66), SIMDE_FLOAT64_C(  923.31),
                         SIMDE_FLOAT64_C( -996.35), SIMDE_FLOAT64_C( -303.21)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -11.46), SIMDE_FLOAT64_C( -598.19),
                         SIMDE_FLOAT64_C(  495.52), SIMDE_FLOAT64_C(  117.93),
                         SIMDE_FLOAT64_C(  291.55), SIMDE_FLOAT64_C(  189.90),
                         SIMDE_FLOAT64_C( -859.41), SIMDE_FLOAT64_C(    9.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -706.27), SIMDE_FLOAT64_C(-506164.45),
                         SIMDE_FLOAT64_C(  388.85), SIMDE_FLOAT64_C(-65395.72),
                         SIMDE_FLOAT64_C( -983.38), SIMDE_FLOAT64_C(175336.57),
                         SIMDE_FLOAT64_C(856273.15), SIMDE_FLOAT64_C(-2959.33)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -563.02), SIMDE_FLOAT64_C( -883.18),
                         SIMDE_FLOAT64_C(  852.82), SIMDE_FLOAT64_C( -331.20),
                         SIMDE_FLOAT64_C( -286.53), SIMDE_FLOAT64_C( -422.71),
                         SIMDE_FLOAT64_C( -717.56), SIMDE_FLOAT64_C( -209.20)),
      UINT8_C( 30),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -241.93), SIMDE_FLOAT64_C( -343.53),
                         SIMDE_FLOAT64_C(  736.91), SIMDE_FLOAT64_C( -835.83),
                         SIMDE_FLOAT64_C( -444.99), SIMDE_FLOAT64_C(  943.16),
                         SIMDE_FLOAT64_C(   17.49), SIMDE_FLOAT64_C(  -26.72)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -614.80), SIMDE_FLOAT64_C( -251.11),
                         SIMDE_FLOAT64_C(  421.22), SIMDE_FLOAT64_C( -961.92),
                         SIMDE_FLOAT64_C(  971.24), SIMDE_FLOAT64_C( -348.19),
                         SIMDE_FLOAT64_C( -171.56), SIMDE_FLOAT64_C( -420.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -563.02), SIMDE_FLOAT64_C( -883.18),
                         SIMDE_FLOAT64_C(  852.82), SIMDE_FLOAT64_C(804001.59),
                         SIMDE_FLOAT64_C(-432192.09), SIMDE_FLOAT64_C(-328398.88),
                         SIMDE_FLOAT64_C(-3000.58), SIMDE_FLOAT64_C( -209.20)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  442.66), SIMDE_FLOAT64_C(  -69.80),
                         SIMDE_FLOAT64_C(  151.84), SIMDE_FLOAT64_C(  278.19),
                         SIMDE_FLOAT64_C( -105.37), SIMDE_FLOAT64_C( -898.05),
                         SIMDE_FLOAT64_C(  104.61), SIMDE_FLOAT64_C(  131.40)),
      UINT8_C( 92),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -598.49), SIMDE_FLOAT64_C(  226.31),
                         SIMDE_FLOAT64_C(   -6.29), SIMDE_FLOAT64_C(  443.90),
                         SIMDE_FLOAT64_C( -544.30), SIMDE_FLOAT64_C( -925.04),
                         SIMDE_FLOAT64_C(  484.35), SIMDE_FLOAT64_C( -740.68)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   72.46), SIMDE_FLOAT64_C(  -87.05),
                         SIMDE_FLOAT64_C( -714.68), SIMDE_FLOAT64_C(  393.49),
                         SIMDE_FLOAT64_C(  651.31), SIMDE_FLOAT64_C(  480.47),
                         SIMDE_FLOAT64_C(  373.84), SIMDE_FLOAT64_C(  843.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  442.66), SIMDE_FLOAT64_C(-19700.29),
                         SIMDE_FLOAT64_C(  151.84), SIMDE_FLOAT64_C(174670.21),
                         SIMDE_FLOAT64_C(-354508.03), SIMDE_FLOAT64_C(-444453.97),
                         SIMDE_FLOAT64_C(  104.61), SIMDE_FLOAT64_C(  131.40)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  988.68), SIMDE_FLOAT64_C(  353.35),
                         SIMDE_FLOAT64_C( -309.14), SIMDE_FLOAT64_C( -266.17),
                         SIMDE_FLOAT64_C(  819.45), SIMDE_FLOAT64_C(  592.47),
                         SIMDE_FLOAT64_C(  382.11), SIMDE_FLOAT64_C(  516.02)),
      UINT8_C( 87),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   51.49), SIMDE_FLOAT64_C( -696.81),
                         SIMDE_FLOAT64_C(  178.38), SIMDE_FLOAT64_C(  907.89),
                         SIMDE_FLOAT64_C(  646.15), SIMDE_FLOAT64_C(  281.27),
                         SIMDE_FLOAT64_C(  226.71), SIMDE_FLOAT64_C( -906.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -823.13), SIMDE_FLOAT64_C( -506.07),
                         SIMDE_FLOAT64_C( -848.31), SIMDE_FLOAT64_C( -467.13),
                         SIMDE_FLOAT64_C(  559.51), SIMDE_FLOAT64_C( -498.81),
                         SIMDE_FLOAT64_C(  598.24), SIMDE_FLOAT64_C(  523.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  988.68), SIMDE_FLOAT64_C(352634.64),
                         SIMDE_FLOAT64_C( -309.14), SIMDE_FLOAT64_C(-424102.66),
                         SIMDE_FLOAT64_C(  819.45), SIMDE_FLOAT64_C(-140300.29),
                         SIMDE_FLOAT64_C(135626.99), SIMDE_FLOAT64_C(-474963.09)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -811.79), SIMDE_FLOAT64_C(  522.04),
                         SIMDE_FLOAT64_C(  594.85), SIMDE_FLOAT64_C(    0.75),
                         SIMDE_FLOAT64_C( -855.43), SIMDE_FLOAT64_C(  660.82),
                         SIMDE_FLOAT64_C( -308.44), SIMDE_FLOAT64_C(  882.56)),
      UINT8_C( 62),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -252.73), SIMDE_FLOAT64_C( -915.63),
                         SIMDE_FLOAT64_C( -935.95), SIMDE_FLOAT64_C( -722.20),
                         SIMDE_FLOAT64_C( -497.29), SIMDE_FLOAT64_C( -166.63),
                         SIMDE_FLOAT64_C(  516.64), SIMDE_FLOAT64_C( -317.86)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -925.15), SIMDE_FLOAT64_C(  466.19),
                         SIMDE_FLOAT64_C(  263.72), SIMDE_FLOAT64_C(  424.85),
                         SIMDE_FLOAT64_C(  205.96), SIMDE_FLOAT64_C(  401.84),
                         SIMDE_FLOAT64_C(  361.23), SIMDE_FLOAT64_C(  807.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -811.79), SIMDE_FLOAT64_C(  522.04),
                         SIMDE_FLOAT64_C(-246828.73), SIMDE_FLOAT64_C(-306826.67),
                         SIMDE_FLOAT64_C(-102421.85), SIMDE_FLOAT64_C(-66958.60),
                         SIMDE_FLOAT64_C(186625.87), SIMDE_FLOAT64_C(  882.56)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  915.95), SIMDE_FLOAT64_C( -721.40),
                         SIMDE_FLOAT64_C( -153.01), SIMDE_FLOAT64_C(  377.63),
                         SIMDE_FLOAT64_C(  983.59), SIMDE_FLOAT64_C( -647.06),
                         SIMDE_FLOAT64_C(  224.30), SIMDE_FLOAT64_C(  -39.06)),
      UINT8_C( 70),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  724.37), SIMDE_FLOAT64_C( -108.80),
                         SIMDE_FLOAT64_C( -716.02), SIMDE_FLOAT64_C( -552.47),
                         SIMDE_FLOAT64_C(  411.46), SIMDE_FLOAT64_C( -439.29),
                         SIMDE_FLOAT64_C(  397.99), SIMDE_FLOAT64_C(  -31.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -763.99), SIMDE_FLOAT64_C(  279.59),
                         SIMDE_FLOAT64_C(  318.18), SIMDE_FLOAT64_C(   57.40),
                         SIMDE_FLOAT64_C(   13.78), SIMDE_FLOAT64_C( -535.45),
                         SIMDE_FLOAT64_C(   52.16), SIMDE_FLOAT64_C( -903.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  915.95), SIMDE_FLOAT64_C(-30419.39),
                         SIMDE_FLOAT64_C( -153.01), SIMDE_FLOAT64_C(  377.63),
                         SIMDE_FLOAT64_C(  983.59), SIMDE_FLOAT64_C(235217.83),
                         SIMDE_FLOAT64_C(20759.16), SIMDE_FLOAT64_C(  -39.06)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -219.27), SIMDE_FLOAT64_C(  499.44),
                         SIMDE_FLOAT64_C( -493.92), SIMDE_FLOAT64_C(  481.91),
                         SIMDE_FLOAT64_C(  270.70), SIMDE_FLOAT64_C(  857.18),
                         SIMDE_FLOAT64_C( -745.19), SIMDE_FLOAT64_C( -960.45)),
      UINT8_C(113),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -135.86), SIMDE_FLOAT64_C( -159.92),
                         SIMDE_FLOAT64_C(  756.29), SIMDE_FLOAT64_C( -526.68),
                         SIMDE_FLOAT64_C(    5.30), SIMDE_FLOAT64_C(  278.11),
                         SIMDE_FLOAT64_C(  884.85), SIMDE_FLOAT64_C(  638.85)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  726.26), SIMDE_FLOAT64_C(  110.13),
                         SIMDE_FLOAT64_C( -961.42), SIMDE_FLOAT64_C(   96.39),
                         SIMDE_FLOAT64_C(  930.93), SIMDE_FLOAT64_C( -241.35),
                         SIMDE_FLOAT64_C( -108.47), SIMDE_FLOAT64_C(  -69.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -219.27), SIMDE_FLOAT64_C(-17611.99),
                         SIMDE_FLOAT64_C(-727112.33), SIMDE_FLOAT64_C(-50766.69),
                         SIMDE_FLOAT64_C(  270.70), SIMDE_FLOAT64_C(  857.18),
                         SIMDE_FLOAT64_C( -745.19), SIMDE_FLOAT64_C(-44598.12)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_mul_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_mul_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
     { UINT8_C(  4),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  232.34), SIMDE_FLOAT64_C(  716.29),
                         SIMDE_FLOAT64_C(  520.56), SIMDE_FLOAT64_C( -458.82),
                         SIMDE_FLOAT64_C(  550.79), SIMDE_FLOAT64_C(  687.92),
                         SIMDE_FLOAT64_C( -593.10), SIMDE_FLOAT64_C( -620.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -156.55), SIMDE_FLOAT64_C( -441.38),
                         SIMDE_FLOAT64_C(  554.99), SIMDE_FLOAT64_C(  294.84),
                         SIMDE_FLOAT64_C( -270.30), SIMDE_FLOAT64_C( -228.66),
                         SIMDE_FLOAT64_C(  910.49), SIMDE_FLOAT64_C( -483.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-157299.79),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(165),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -526.05), SIMDE_FLOAT64_C(  453.25),
                         SIMDE_FLOAT64_C(  821.16), SIMDE_FLOAT64_C( -906.31),
                         SIMDE_FLOAT64_C( -873.91), SIMDE_FLOAT64_C( -472.79),
                         SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C( -955.90)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  305.84), SIMDE_FLOAT64_C( -400.58),
                         SIMDE_FLOAT64_C( -475.09), SIMDE_FLOAT64_C( -582.28),
                         SIMDE_FLOAT64_C( -849.06), SIMDE_FLOAT64_C( -392.73),
                         SIMDE_FLOAT64_C( -370.73), SIMDE_FLOAT64_C( -928.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-160887.13), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-390124.90), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(185678.82),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(887973.75)) },
    { UINT8_C(175),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  871.20), SIMDE_FLOAT64_C( -761.38),
                         SIMDE_FLOAT64_C( -106.42), SIMDE_FLOAT64_C( -228.29),
                         SIMDE_FLOAT64_C( -864.78), SIMDE_FLOAT64_C( -773.10),
                         SIMDE_FLOAT64_C(  984.91), SIMDE_FLOAT64_C( -982.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -488.22), SIMDE_FLOAT64_C(  361.25),
                         SIMDE_FLOAT64_C( -346.47), SIMDE_FLOAT64_C(  411.25),
                         SIMDE_FLOAT64_C(  117.68), SIMDE_FLOAT64_C(  448.38),
                         SIMDE_FLOAT64_C( -319.67), SIMDE_FLOAT64_C(  -97.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-425337.26), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(36871.34), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-101767.31), SIMDE_FLOAT64_C(-346642.58),
                         SIMDE_FLOAT64_C(-314846.18), SIMDE_FLOAT64_C(96244.77)) },
    { UINT8_C(195),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -43.54), SIMDE_FLOAT64_C(  387.84),
                         SIMDE_FLOAT64_C( -190.98), SIMDE_FLOAT64_C(  468.25),
                         SIMDE_FLOAT64_C( -832.05), SIMDE_FLOAT64_C( -600.33),
                         SIMDE_FLOAT64_C( -246.00), SIMDE_FLOAT64_C(  160.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -279.42), SIMDE_FLOAT64_C(  980.35),
                         SIMDE_FLOAT64_C(  897.98), SIMDE_FLOAT64_C( -354.38),
                         SIMDE_FLOAT64_C(  689.03), SIMDE_FLOAT64_C(  555.84),
                         SIMDE_FLOAT64_C(  823.79), SIMDE_FLOAT64_C( -979.93)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(12165.95), SIMDE_FLOAT64_C(380218.94),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-202652.34), SIMDE_FLOAT64_C(-157180.77)) },
    { UINT8_C(236),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  821.55), SIMDE_FLOAT64_C(  740.38),
                         SIMDE_FLOAT64_C( -934.60), SIMDE_FLOAT64_C(  694.91),
                         SIMDE_FLOAT64_C(  432.52), SIMDE_FLOAT64_C(  380.89),
                         SIMDE_FLOAT64_C(  -22.14), SIMDE_FLOAT64_C(  683.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  735.17), SIMDE_FLOAT64_C(  313.88),
                         SIMDE_FLOAT64_C( -529.80), SIMDE_FLOAT64_C( -869.79),
                         SIMDE_FLOAT64_C(  294.43), SIMDE_FLOAT64_C(  958.02),
                         SIMDE_FLOAT64_C(  383.81), SIMDE_FLOAT64_C(  520.19)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(603978.91), SIMDE_FLOAT64_C(232390.47),
                         SIMDE_FLOAT64_C(495151.08), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(127346.86), SIMDE_FLOAT64_C(364900.24),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(144),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -704.15), SIMDE_FLOAT64_C(  418.80),
                         SIMDE_FLOAT64_C( -562.82), SIMDE_FLOAT64_C(  910.01),
                         SIMDE_FLOAT64_C(  513.17), SIMDE_FLOAT64_C(  314.44),
                         SIMDE_FLOAT64_C(  866.48), SIMDE_FLOAT64_C(  466.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  663.36), SIMDE_FLOAT64_C(  883.11),
                         SIMDE_FLOAT64_C(  475.36), SIMDE_FLOAT64_C(  451.49),
                         SIMDE_FLOAT64_C(  246.05), SIMDE_FLOAT64_C( -122.55),
                         SIMDE_FLOAT64_C(  401.83), SIMDE_FLOAT64_C(  557.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-467104.94), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(410860.41),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(181),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -758.51), SIMDE_FLOAT64_C( -164.55),
                         SIMDE_FLOAT64_C(  334.89), SIMDE_FLOAT64_C( -549.60),
                         SIMDE_FLOAT64_C(  344.01), SIMDE_FLOAT64_C( -985.45),
                         SIMDE_FLOAT64_C( -235.88), SIMDE_FLOAT64_C(  450.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -408.01), SIMDE_FLOAT64_C(  723.62),
                         SIMDE_FLOAT64_C( -159.29), SIMDE_FLOAT64_C(  720.82),
                         SIMDE_FLOAT64_C( -893.97), SIMDE_FLOAT64_C(  826.45),
                         SIMDE_FLOAT64_C(   -3.06), SIMDE_FLOAT64_C(  902.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(309479.67), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-53344.63), SIMDE_FLOAT64_C(-396162.67),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-814425.15),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(406617.08)) },
    { UINT8_C(211),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  311.61), SIMDE_FLOAT64_C( -930.67),
                         SIMDE_FLOAT64_C(  465.44), SIMDE_FLOAT64_C( -366.35),
                         SIMDE_FLOAT64_C(  205.36), SIMDE_FLOAT64_C(  276.19),
                         SIMDE_FLOAT64_C(  975.10), SIMDE_FLOAT64_C( -338.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  817.02), SIMDE_FLOAT64_C( -834.50),
                         SIMDE_FLOAT64_C( -648.42), SIMDE_FLOAT64_C(  761.90),
                         SIMDE_FLOAT64_C(   24.27), SIMDE_FLOAT64_C(  838.31),
                         SIMDE_FLOAT64_C( -854.11), SIMDE_FLOAT64_C(  403.52)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(254591.60), SIMDE_FLOAT64_C(776644.11),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-279122.07),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(-832842.66), SIMDE_FLOAT64_C(-136575.38)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_mul_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mul_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_mul_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_mul_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mul_epu32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_mul_epu32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_mul_epu32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mul_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_mul_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_mul_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mul_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_mul_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_mul_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
