#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::SelectFileModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/select_file/select_file_module.rb'
# include ::Gtk::SelectFileModule
# =========================================================================== #
module Gtk

module SelectFileModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Select a file'

  # ========================================================================= #
  # === FONT_SIZE
  # ========================================================================= #
  FONT_SIZE = 18

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = "Liberation serif #{FONT_SIZE}"

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = '80% or minimum 700px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT =  900

  # ========================================================================= #
  # === initialize
  #
  # You have to pass the parent-widget to the instance of SelectFile,
  # as the first argument.
  # ========================================================================= #
  def initialize(
      parent_widget = nil,
      run_already   = true,
      &block
    )
    super(:vertical)
    set_parent_widget(parent_widget)
    reset
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      #
      # The Hash may look like this:
      #
      #   ::Gtk::SelectFile.new(this_widget) {{
      #     filter_for_these_file_types: :mp3,
      #     current_folder:              return_path_to_the_local_songs_directory_or_return_pwd,
      #     additional_directories:      return_pwd
      #   }}
      #
      # ===================================================================== #
      if yielded.is_a? Hash
        @hash = yielded
        # =================================================================== #
        # === :filter_for_these_file_types
        #
        # This entry point can be used to filter for specific file
        # types, such as .mp3 files.
        # =================================================================== #
        if @hash.has_key? :filter_for_these_file_types
          filter_for_these_file_types = @hash[:filter_for_these_file_types]
          @file_filter = ::Gtk::FileFilter.new
          @file_filter.name = "Filter for #{filter_for_these_file_types} files"
          if filter_for_these_file_types.is_a? Symbol
            filter_for_these_file_types = "*.#{filter_for_these_file_types.to_s}"
          else
            # =============================================================== #
            # This clause will ensure that this begins with a '*.'.
            # =============================================================== #
            if filter_for_these_file_types.is_a? String
              unless filter_for_these_file_types.start_with? '*.'
                if filter_for_these_file_types.frozen?
                  filter_for_these_file_types = filter_for_these_file_types.dup
                end
                unless filter_for_these_file_types.start_with? '.'
                  filter_for_these_file_types.prepend('.')
                end
                filter_for_these_file_types.prepend('*')
              end
            end
          end
          @file_filter.add_pattern(filter_for_these_file_types)
        end
      end
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @hash
    # ======================================================================= #
    @hash = {}
    set_use_this_font(USE_THIS_FONT)
    # ======================================================================= #
    # === @file_filter
    # ======================================================================= #
    @file_filter = nil
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === set_parent_widget
  # ========================================================================= #
  def set_parent_widget(i)
    @parent_widget = i
  end

  # ========================================================================= #
  # === selected_file?
  # ========================================================================= #
  def selected_file?
    @entry_show_selected_file.text.to_s
  end; alias file? selected_file? # === file?

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @entry_show_selected_file = gtk_entry
    # ======================================================================= #
    # We will make use of a button to trigger the open-file functionality.
    # ======================================================================= #
    @button_select_file = gtk_button('_Select File')
    @button_select_file.bblack2
    @button_select_file.on_clicked {
      do_click_the_button_select_file
    }
    @button_select_file.fancy_tooltip = 'Click this button in '\
      'order to <b>select a local file</b>.'
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    minimal(
      @button_select_file, 1
    )
    minimal( # Add the entry, but make it smaller.
      @entry_show_selected_file, 1
    )
  end

  # ========================================================================= #
  # === do_click_the_button_select_file
  # ========================================================================= #
  def do_click_the_button_select_file
    # ======================================================================= #
    # Delegate the parameters towards Gtk.select_file() next:
    # ======================================================================= #
    result = ::Gtk.select_file(@parent_widget, @file_filter) { @hash }
    @entry_show_selected_file.set_text(result)
  end; alias do_pick_file do_click_the_button_select_file # === do_pick_file

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    12
  end

  # ========================================================================= #
  # === favicon?
  # ========================================================================= #
  def favicon?
    :audio_favicon
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::SelectFileModule[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

  # ========================================================================= #
  # === Gtk::SelectFileModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    r = ::Gtk.run
    _ = ::Gtk::SelectFile.new(r)
    r << _
    r.top_left_then_run
  end

end; end