#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::PsauxViewerModule
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/psaux_viewer/psaux_viewer_module.rb'
# include ::Gtk::PsauxViewerModule
# =========================================================================== #
module Gtk

module PsauxViewerModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === REFRESH_EVERY_N_SECONDS
  # ========================================================================= #
  REFRESH_EVERY_N_SECONDS = 60

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE  =
    "Process Viewer - updating every "\
    "#{REFRESH_EVERY_N_SECONDS} seconds"

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH  = 1680

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 920

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title  = TITLE
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    set_font_size(18)
    use_gtk_paradise_css_file
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === modify_label
  # ========================================================================= #
  def modify_label(name, colour)
    ::Gtk::Label.new(name).set_markup( %Q[#{name} ], true).to_s
  end

  # ========================================================================= #
  # === ps_aux_result?
  # ========================================================================= #
  def ps_aux_result?
    `ps aux`
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    @liststore      = Gtk::ListStore.new(*([String] * 12))
    renderer        = gtk_cell_renderer_text
    treeview        = gtk_tree_view(@liststore)
    treeview.set_size_request(width?, height?)
    scrolled_window = gtk_scrolled_window(treeview) { :always }
    treeview.selection.multiple_selection # Allow multiple selects.
    # ======================================================================= #
    #       .insert_column(position, title, cell, attributes)
    # The -1 means that we will append at the end.
    # ======================================================================= #
    treeview.quick_append('User',       renderer,  text:  0)
    treeview.quick_append('PID',        renderer,  text:  1)
    treeview.quick_append('CPU%',       renderer,  text:  2)
    treeview.quick_append('Memory%',    renderer,  text:  3)
    treeview.quick_append('VSZ',        renderer,  text:  4)
    treeview.quick_append('RSS',        renderer,  text:  5)
    treeview.quick_append('TTY',        renderer,  text:  6)
    treeview.quick_append('Stat',       renderer,  text:  7)
    treeview.quick_append('Start',      renderer,  text:  8)
    treeview.quick_append('Time',       renderer,  text:  9)
    treeview.quick_append('Command',    renderer,  text: 10)
    treeview.make_resizable
    columns = treeview.columns?
    columns[1].sort_via(1)
    columns[2].sort_via(2)
    # ======================================================================= #
    # Create a separate thread that loops timely to make the
    # process update transparent.
    # ======================================================================= #
    Thread.new {
      loop {
        update_the_liststore
        # Refresh process list every 10 seconds.
        sleep REFRESH_EVERY_N_SECONDS
      }
    }
    # ======================================================================= #
    # Add the scrolled window to the main window next.
    # ======================================================================= #
    maximal(scrolled_window)
    create_the_update_button
    minimal(@button_update_the_content)
  end

  # ========================================================================= #
  # === create_the_update_button
  #
  # This is the button that can be used by the user to update the
  # process-listing.
  # ========================================================================= #
  def create_the_update_button
    @button_update_the_content = gtk_button('_Update')
    @button_update_the_content.bblack1
    @button_update_the_content.make_bold
    @button_update_the_content.hint = 'Click this button to <b>update</b> '\
                                      'the liststore shown above.'
    @button_update_the_content.set_background_colour :mintcream
    @button_update_the_content.on_clicked {
      update_the_liststore
    }
  end

  # ========================================================================= #
  # === update_the_liststore                                     (update tag)
  # ========================================================================= #
  def update_the_liststore
    @liststore.clear # Clear the liststore first.
    # ======================================================================= #
    # Get all available processes next.
    # ======================================================================= #
    processes = ps_aux_result?.split("\n")[1..-1].sort.map {|entry|
      entry.split(/\s+/, 11)
    }
    # ======================================================================= #
    # Go through each process creating a row for it. We sort by the
    # process id.
    # ======================================================================= #  
    processes.sort_by {|line| line[1].to_i }.each { |process|
      iter = @liststore.append
      # ===================================================================== #
      # Go through each process detail creating a column for it.
      # ===================================================================== #
      process.each_with_index { |ps, index| iter[index] = ps }
    }
  end

  # ========================================================================= #
  # === Gtk::PsauxViewerModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::PsauxViewer.new
    r = ::Gtk.run
    r << _
    r.automatic
    r.top_left_then_run
  end

end; end