#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::GoogleSearcherModule
#
# This module can be used to open a tab in the browser, "simulating"
# googling.
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/google_searcher/google_searcher_module.rb'
# include ::Gtk::GoogleSearcherModule
# =========================================================================== #
module Gtk

module GoogleSearcherModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  begin
    require 'open'
  rescue LoadError; end

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'GoogleSearcher'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1280

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 100

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:vertical)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @width
    # ======================================================================= #
    set_width(WIDTH)
    # ======================================================================= #
    # === @height
    # ======================================================================= #
    set_height(HEIGHT)
    # ======================================================================= #
    # === @title
    # ======================================================================= #
    @title = 'Google Searcher'
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
    set_font(:hack_20)
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_and_populate_gtk_tables
  # ========================================================================= #
  def create_and_populate_gtk_tables
    @main_table = gtk_table(1, 2, true)
    @main_table.set_column_spacings(1)
    @main_table.set_row_spacings(1)
    @main_table.attach_defaults(@button_google_for_this,0,1, 0, 1)
    @main_table.attach_defaults(@entry_google_for_this,1,2, 0, 1)
  end

  # ========================================================================= #
  # === create_gtk_buttons
  # ========================================================================= #
  def create_gtk_buttons
    @button_google_for_this = gtk_button('_GoogleSearch')
    @button_google_for_this.on_clicked { google_for_this }
    @button_google_for_this.modify_background(:active,   :coral)
    @button_google_for_this.modify_background(:normal,   :cornsilk)
    @button_google_for_this.modify_background(:prelight, :aliceblue)
    @button_google_for_this.set_name('button1')
    @button_google_for_this.set_size_request(200, 50)
    @button_google_for_this.fancy_tooltips = 'Click on this button '\
      'to <b>perform a google-search</b>.'
  end

  # ========================================================================= #
  # === google_for_this
  #
  # https://www.google.com/search?hl=de&q=&btnG=Google-Suche&meta=test
  # ========================================================================= #
  def google_for_this
    first  = 'https://www.google.com/search?hl=de&q='
    string = @entry_google_for_this.text.to_s.dup
    string.tr!(' ', '+') if string.include? ' '
    last = '&btnG=Google-Suche&lr='
    final_string = first+string+last
    open_in_browser(final_string)
  end

  # ========================================================================= #
  # === create_gtk_entries                                        (entry tag)
  #
  # This will create the main Gtk entry.
  # ========================================================================= #
  def create_gtk_entries
    # ======================================================================= #
    # === @entry_google_for_this
    # ======================================================================= #
    @entry_google_for_this = gtk_entry
    @entry_google_for_this.do_focus
    @entry_google_for_this.bblack1
    @entry_google_for_this.css_class('pad4px')
    @entry_google_for_this.set_max_length(80)
    @entry_google_for_this.set_text('')
    @entry_google_for_this.on_key_press_event { |w, event|
      case Gdk::Keyval.to_name(event.keyval)
      when 'Return','KP_Enter'
        google_for_this
      end
    }
    @entry_google_for_this.set_name('BG_papayawhip')
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_gtk_entries
    create_gtk_buttons
    create_and_populate_gtk_tables
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    Thread.abort_on_exception = true
    if use_gtk2?
      pack_start(@main_table, false, false)
    else
      pack_start(@main_table, expand: false, fill: false)
    end
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::GoogleSearcher.width?
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === Gtk::GoogleSearcher.height?
  # ========================================================================= #
  def self.height?
    HEIGHT
  end

  # ========================================================================= #
  # === Gtk::GoogleSearcherModule.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::GoogleSearcher.new(i)
    r = ::Gtk.run
    r << _
    r.set_size_request(
      _.width?,
      _.height?
    )
    r.top_left_then_run
  end

end; end