#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ErrorWidgetModule
#
# This widget shall capture "errors", or notifications, for the end user.
#
# The main idea behind its creation was to avoid the gtk-status-bar,
# since the API keeps on confusing me. So instead I created my own
# customized widget that keeps track of notifications.
#
# To append "errors" to it simply make use of the << method.
#
# Usage example:
#
#   Gtk::ErrorWidget.new(ARGV)
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/error_widget/error_widget_module.rb'
# include ::Gtk::ErrorWidgetModule
# =========================================================================== #
module Gtk

module ErrorWidgetModule # === Gtk::ErrorWidgetModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Error Widget'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 250

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 80

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true
    )
    super(:horizontal)
    reset
    set_commandline_arguments(i)
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height(TITLE, WIDTH, HEIGHT)
    # ======================================================================= #
    # === @make_bold
    # ======================================================================= #
    @make_bold = false
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @entry_for_notifications = create_entry
  end

  # ========================================================================= #
  # === <<
  # ========================================================================= #
  def <<(i)
    @entry_for_notifications.set_text(i.to_s)
    colourize(:steelblue)
  end; alias capture_this_error << # === capture_this_error

  # ========================================================================= #
  # === colourize
  #
  # The first argument to this method, as a String, should denote which
  # colour we will be using.
  # ========================================================================= #
  def colourize(
      use_this_colour = :slateblue
    )
    @entry_for_notifications.modify_foreground(use_this_colour, :normal)
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    maximal(
      @entry_for_notifications, 0
    )
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::ErrorWidget[]
  # ========================================================================= #
  def self.[](i = ARGV)
    new
  end

  # ========================================================================= #
  # === Gtk::ErrorWidgetModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    error_widget = Gtk::ErrorWidget.new(ARGV)
    r = ::Gtk.run
    r.set_size_request(error_widget.width?, error_widget.height?)
    r.signal_connect(:delete_event) { |_widget| Gtk.main_quit }
    r.enable_simple_exit
    error_widget << 'HI THERE - THIS IS A TEST ERROR'
    r << error_widget
    r.top_left_then_run
  end

  # ========================================================================= #
  # === Gtk::ErrorWidgetModule.widget
  # ========================================================================= #
  def self.widget(&block)
    ::Gtk::ErrorWidget.new(&block)
  end

end; end