#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::EmojiWidgetModule
#
# This small widget shall show the available (registered) emojis, from
# the gtk-paradise project.
#
# Usage example:
#
#   Gtk::EmojiWidget.new
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/emoji_widget/emoji_widget_module.rb'
# include Gtk::EmojiWidgetModule
# =========================================================================== #
module Gtk

module EmojiWidgetModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Emoji Widget'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1400

  # ========================================================================= #
  # === WIDTH_OF_THE_SMALL_EMOJI_BOX
  #
  # This constant specifies how wide the small emoji box should be - this
  # is the box that contains the Gtk::Frame, as well as the emoji itself,
  # as a Gtk::Label.
  # ========================================================================= #
  WIDTH_OF_THE_SMALL_EMOJI_BOX  = 220

  # ========================================================================= #
  # === HEIGHT_OF_SMALL_EMOJI_BOX
  # ========================================================================= #
  HEIGHT_OF_SMALL_EMOJI_BOX = 105

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 50

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super()
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, infer_the_namespace]
    title_width_height_font(TITLE, WIDTH, HEIGHT, :hack_22)
    use_gtk_paradise_project_css_file
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_entry
  end

  # ========================================================================= #
  # === return_sorted_emojis
  # ========================================================================= #
  def return_sorted_emojis
    ::Gtk::Emojis.constants.sort_by {|entry|
      entry = entry.to_s
      # ===================================================================== #
      # Next, we need to query whether it has a number or not. If so then 
      # we modify that number temporarily, so that "_1" becomes "_0001".
      # ===================================================================== #
      if entry =~ /\d+$/
        number_part = entry.scan(/\d+$/).flatten.first
        modified_number = number_part.rjust(4, '0')
        if entry.include? '_'
          splitted = entry.split('_')
          entry = splitted[0 .. -2].join('_')+'_'+modified_number
        end
      end
      entry
    }
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    outer_vbox = gtk_vbox
    outer_vbox.minimal(
      left_aligned_label('This widget shows the <b>registered emojis</b>, '\
      'as part of the <b>gtk_paradise gem</b>.'), 20
    )
    outer_vbox.maximal(hspacer)
    outer_vbox.minimal(@entry_on_top)
    outer_vbox.maximal(hspacer)
    sorted_emoji_constants = return_sorted_emojis
    # ======================================================================= #
    # We group the emojis into "rows" of 5.
    # ======================================================================= #
    group_of_five = sorted_emoji_constants.each_slice(5)
    group_of_five.each {|these_constants|
      new_hbox = gtk_hbox
      these_constants.each {|inner_this_constant|
        frame = gtk_frame
        frame.set_label(inner_this_constant.to_s)
        frame.css_class('bblack1')
        emoji_representation = ::Gtk::Emojis.const_get(inner_this_constant)
        label = gtk_label(emoji_representation)
        label.make_selectable # Make it "copy/pastable".
        frame.add(label)
        frame.set_border_width(2)
        frame.set_size_request(
          WIDTH_OF_THE_SMALL_EMOJI_BOX,
          HEIGHT_OF_SMALL_EMOJI_BOX
        )
        eventbox = gtk_eventbox(frame)
        eventbox.on_clicked {
          # ================================================================= #
          # Output that emoji on the commandline next.
          # ================================================================= #
          e emoji_representation.to_s
          @entry_on_top.set_text(emoji_representation.to_s)
        }
        new_hbox.minimal(eventbox, 1)
      }
      outer_vbox.add(new_hbox)
    }
    add_with_viewport(outer_vbox)
    show_all
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    # ======================================================================= #
    # Determine the size:
    # ======================================================================= #
    set_size_request(width?, height?)
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === create_entry
  # ========================================================================= #
  def create_entry
    @entry_on_top = create_entry
    @entry_on_top.align_to_the_center
    @entry_on_top.hint = 'This entry will show the emoji '\
      'that is selected (clicked upon) from the entry below.'
  end

  # ========================================================================= #
  # === Gtk::EmojiWidget[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

  # ========================================================================= #
  # === Gtk::EmojiWidget.run
  # ======================================================================== #
  def self.run
    require 'gtk_paradise/run.rb'
    emoji_widget = Gtk::EmojiWidget.new
    r = ::Gtk.run
    r << emoji_widget
    r.automatic
    r.set_background :white
    r.enable_quick_exit_then_top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::EmojiWidget.run
end