#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::BrowserModule
#
# This small widget allows the user to quickly open a remote URL via
# the browser, from within the ruby-gtk widget sets.
#
# Usage example:
#
#   Gtk::Browser.new(ARGV)
#
# =========================================================================== #
# require 'gtk_paradise/widgets/shared_code/browser/browser_module.rb'
# include ::Gtk::BrowserModule
# =========================================================================== #
module Gtk

module BrowserModule

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === USE_THIS_TEXT_FOR_THE_URL_OPEN_BUTTON
  # ========================================================================= #
  USE_THIS_TEXT_FOR_THE_URL_OPEN_BUTTON = '_URL-Open ⧉  ➡️ '

  # ========================================================================= #
  # === ARRAY_DEFAULT_URLS
  #
  # Here you can add new URLs. These will appear in a drop-down combo-box.
  # ========================================================================= #
  ARRAY_DEFAULT_URLS = YAML.load_file(
    "#{::Gtk.project_yaml_directory?}browser_URLs.yml"
  )

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1600

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 120

  # ========================================================================= #
  # === WIDTH_OF_THE_ENTRY_OPEN_THIS_URL
  # ========================================================================= #
  WIDTH_OF_THE_ENTRY_OPEN_THIS_URL  = 400

  # ========================================================================= #
  # === HEIGHT_OF_THE_ENTRY_OPEN_THIS_URL
  # ========================================================================= #
  HEIGHT_OF_THE_ENTRY_OPEN_THIS_URL =  55

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Open URL Widget'

  # ========================================================================= #
  # === FONT_TO_USE
  # ========================================================================= #
  FONT_TO_USE = :dejavu_condensed_20

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = nil,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    title_width_height_font(TITLE, WIDTH, HEIGHT, FONT_TO_USE)
    # ======================================================================= #
    # === @use_this_browser
    #
    # Denote which browser is in use here.
    # ======================================================================= #
    @use_this_browser = return_the_main_browser_in_use
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === return_the_main_browser_in_use
  #
  # This method will return the main browser that this widget shall use.
  # ========================================================================= #
  def return_the_main_browser_in_use(
      _ = 'thorium'
    )
    if ENV.has_key? 'USE_THIS_BROWSER'
      _ = ENV['USE_THIS_BROWSER'].to_s.dup
    end
    return _
  end

  # ========================================================================= #
  # === which_browser?
  # ========================================================================= #
  def which_browser?
    @use_this_browser
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    2
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  begin
    require 'open'
  rescue LoadError; end
  # ========================================================================= #
  # === open_this_url
  # ========================================================================= #
  def open_this_url(
      _ = @entry_open_this_url.text
    )
    unless _.empty?
      e
      e "#{::Colours.rev}Opening url #{sfancy(_)} #{rev}now:"
      e
      Open.in_browser(_) {{ use_this_browser: @use_this_browser }} # Delegate towards class Open.in_browser here.
    end
  end; alias do_open_the_remote_URL open_this_url # === do_open_the_remote_URL

  # ========================================================================= #
  # === create_gtk_buttons
  # ========================================================================= #
  def create_gtk_buttons
    @button_open_this_url = bold_button(USE_THIS_TEXT_FOR_THE_URL_OPEN_BUTTON)
    @button_open_this_url.modify_background(:active,   :coral)
    @button_open_this_url.modify_background(:normal,   :cornsilk)
    @button_open_this_url.modify_background(:prelight, :aliceblue)
    @button_open_this_url.bblack2
    @button_open_this_url.fancy_tooltips = '<b>Clicking</b> on this button will '\
      'open the URL (designated via the entry on the right side), via '\
      'the browser (which is '+which_browser?+').'
    @button_open_this_url.can_default = true
    @button_open_this_url.on_clicked { open_this_url }
  end

  # ========================================================================= #
  # === create_gtk_combobox
  #
  # Lets only add some of them we like.
  # ========================================================================= #
  def create_gtk_combobox
    # @combo_box = gtk_combo_box_entry(ARRAY_DEFAULT_URLS)
    @combo_box = gtk_combo_box_entry
    @combo_box.bblack2
    ARRAY_DEFAULT_URLS.each { |val| @combo_box.append_text(val) }
    @combo_box.active = 0
    @combo_box.signal_connect(:event) { |widget, event|
      # assume ruby-gtk3
      use_this_as_event_for_the_button_release_event = :button_release
      case event.event_type
      when use_this_as_event_for_the_button_release_event
        Thread.new { @combo_box.popup }
      end
    }
    @combo_box.on_changed {
      @entry_open_this_url.set_text(@combo_box.active_text)
    }
  end

  # ========================================================================= #
  # === Browser.width?
  #
  # The preferred (hardcoded) width. We assume oldschool desktop-computers
  # here.
  # ========================================================================= #
  def self.width?
    WIDTH
  end

  # ========================================================================= #
  # === create_the_main_hbox
  # ========================================================================= #
  def create_the_main_hbox
    @main_hbox = gtk_hbox    
    @main_hbox.minimal(@button_open_this_url, 3)
    @main_hbox.maximal(@entry_open_this_url,  3)
    @main_hbox.minimal(@combo_box,            3)
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    Thread.abort_on_exception = true
    add(@main_hbox)
    show_all
  end

  # ========================================================================= #
  # === create_gtk_entries
  #
  # Here you can add text thingies, aka gtk-entries.
  # ========================================================================= #
  def create_gtk_entries
    @entry_open_this_url = gtk_entry('')
    @entry_open_this_url.set_max_length(100)
    @entry_open_this_url.set_size_request(
      WIDTH_OF_THE_ENTRY_OPEN_THIS_URL,
      HEIGHT_OF_THE_ENTRY_OPEN_THIS_URL
    )
    @entry_open_this_url.do_focus
    @entry_open_this_url.align_to_the_center
    @entry_open_this_url.set_activates_default(true)
    @entry_open_this_url.select_all_on_click_event
    @entry_open_this_url.css_class('BG_light_yellow')
    @entry_open_this_url.bblack2
    @entry_open_this_url.on_key_press_event { |widget, event|
      if is_it_an_enter_key?(event)
        do_open_the_remote_URL
      end
    }
  end

  # ========================================================================= #
  # === assign_this_text
  # ========================================================================= #
  def assign_this_text(i)
    @entry_open_this_url.set_text(i.to_s)
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_gtk_buttons
    create_gtk_entries
    create_gtk_combobox
    create_the_main_hbox
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === Gtk::Browser[]
  # ========================================================================= #
  def self.[](i = '')
    new(i)
  end

  # ========================================================================= #
  # === Gtk::BrowserModule.run
  # ========================================================================= #
  def self.run
    require 'gtk_paradise/run'
    _ = ::Gtk::Browser.new
    r = ::Gtk.run
    r << _
    r.infer_the_title_from_the_child_widget
    r.set_size_request(_.width?, _.height?)
    r.top_left_then_run
  end

end; end