#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::SystemInformation
#
# This widget has been inspired by:
#
#   https://www.zdnet.com/a/img/resize/760995a08dae70693d20461340930ade9fd75516/2019/08/19/34818cd2-0da5-42e4-9a16-b03f38f0d4a5/webmin.png?width=1200&fit=bounds&auto=webp
#
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/gtk3_standalone/gtk3_standalone.rb'
# Gtk::SystemInformation.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class SystemInformation < ::Gtk::Box # === Gtk::SystemInformation 

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'System Information'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 1200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '15% or minimum 500px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_18

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file # or use use_project_css_file 
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    8
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    4
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
  end

  # ========================================================================= #
  # === add_time_on_system
  # ========================================================================= #
  def add_time_on_system
    minimal(
      left_aligned_selectable_text(
        'Time on system: '\
        '<b>'+dd_mm_yy+', '+
        return_current_time_in_hh_mm_ss+'</b>'
      ),
      2
    )
  end

  # ========================================================================= #
  # === add_operating_system
  # ========================================================================= #
  def add_operating_system
    require 'roebe/toplevel_methods/platform.rb'
    _ = Roebe.platform.to_s
    minimal(
      left_aligned_selectable_text('Operating system: <b>'+_+'</b>')
    )
  end

  # ========================================================================= #
  # === add_clock
  # ========================================================================= #
  def add_clock
    require 'gtk_paradise/widgets/gtk3/mini_clock/mini_clock.rb'
    clock = ::Gtk.mini_clock
    clock.align_to_right
    minimal(
      clock, 2
    )
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    add_clock
    add_operating_system
    add_time_on_system
    add_processor_information
    add_binary_type_information
    add_n_CPUs
  end

  # ========================================================================= #
  # === add_n_CPUs
  # ========================================================================= #
  def add_n_CPUs
    _ = `lscpu`.strip.split("\n").select {|line| line.start_with? 'CPU(s)' }
    if _ and _.is_a?(Array)
      _ = _.first
    end
    if _.include? ':'
      _ = _.split(':').last.strip
    end
    minimal(
      left_aligned_selectable_text("n CPUs: <b>#{_}</b>")
    )
  end

  # ========================================================================= #
  # === add_binary_type_information
  # ========================================================================= #
  def add_binary_type_information
    _ = `uname -m`.strip
    case _
    when /x86_64/
      _ = '64'
    end
    minimal(
      left_aligned_selectable_text("Binary type: <b>#{_} bit</b>")
    )
  end

  # ========================================================================= #
  # === add_processor_information
  # ========================================================================= #
  def add_processor_information
    _ = `lscpu`.split("\n").select {|entry| entry.include? 'Model name: ' }
    _ = _.first if _.is_a? Array
    _ = _.split(':').last.strip
    minimal(
      left_aligned_selectable_text("Processor information: <b>#{_}</b>")
    )
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::SystemInformation.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::SystemInformation.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.set_border_width(25)
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::SystemInformation.run
end