#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::NumberedLines
#
# To use a scrolled window as well, do:
#
#   Gtk::NumberedLines.new { :do_use_a_scrolled_window }
#
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/numbered_lines/numbered_lines.rb'
# Gtk::NumberedLines.new {{ upper_range: 100 }}
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class NumberedLines < ::Gtk::Box

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Numbered Lines'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 20

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 50

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :hack_20

  # ========================================================================= #
  # === ARROW_TO_USE
  # ========================================================================= #
  ARROW_TO_USE = '➡️'

  # ========================================================================= #
  # === POPULATE_N_ENTRIES_BY_DEFAULT
  # ========================================================================= #
  POPULATE_N_ENTRIES_BY_DEFAULT = 15

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV,
      run_already = true,
      &block
    )
    super(:vertical)
    reset
    set_commandline_arguments(i)
    # ======================================================================= #
    # === Handle blocks given next
    # ======================================================================= #
    if block_given?
      yielded = yield
      flattened = [yielded].flatten
      flattened.each {|this_entry|
        case this_entry
        # =================================================================== #
        # === :do_use_a_scrolled_window
        # =================================================================== #
        when :do_use_a_scrolled_window
          do_use_a_scrolled_window
        # =================================================================== #
        # === :do_not_use_arrow_icons
        # =================================================================== #
        when :do_not_use_arrow_icons
          do_not_use_arrow_icons
        else
          if this_entry.is_a? Hash
            # =============================================================== #
            # === :specify_n_lines
            # =============================================================== #
            if this_entry.has_key? :specify_n_lines
              set_n_lines(yielded.delete(:specify_n_lines))
            # =============================================================== #
            # === :upper_range
            # =============================================================== #
            elsif this_entry.has_key? :upper_range
              set_n_lines(yielded.delete(:upper_range))
            # =============================================================== #
            # === :n_lines
            # =============================================================== #
            elsif yielded.has_key? :n_lines
              set_n_lines(yielded.delete(:n_lines))
            end
          end
        end
      }
    end
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    # ======================================================================= #
    # === @up_to_n_times
    #
    # How many lines to display. This also specifies the maximum number
    # of lines the widget can hold. You can modify this from within
    # initialize() via :upper_range.
    # ======================================================================= #
    @up_to_n_times = POPULATE_N_ENTRIES_BY_DEFAULT
    # ======================================================================= #
    # === @use_a_scrolled_window
    #
    # This can be used to create a scrolled-window as-is.
    # ======================================================================= #
    @use_a_scrolled_window = false
    # ======================================================================= #
    # === @array_event_boxes
    # ======================================================================= #
    @array_event_boxes = []
    # ======================================================================= #
    # === @use_arrow_icons
    # ======================================================================= #
    @use_arrow_icons = true
    use_gtk_paradise_project_css_file
    infer_the_size_automatically
  end

  # ========================================================================= #
  # === use_arrow_icons?
  # ========================================================================= #
  def use_arrow_icons?
    @use_arrow_icons
  end

  # ========================================================================= #
  # === do_not_use_arrow_icons
  # ========================================================================= #
  def do_not_use_arrow_icons
    @use_arrow_icons = false
  end; alias no_arrows do_not_use_arrow_icons # === no_arrows

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
    create_the_main_vbox
  end

  # ========================================================================= #
  # === create_the_main_vbox
  # ========================================================================= #
  def create_the_main_vbox
    # ======================================================================= #
    # === @main_vbox
    # ======================================================================= #
    @main_vbox = create_vbox
    @main_vbox.css_class('BG_lightgrey')
  end

  # ========================================================================= #
  # === scrolled_window?
  # ========================================================================= #
  def scrolled_window?
    @scrolled_window
  end; alias mainview scrolled_window? # === mainview

  # ========================================================================= #
  # === do_use_a_scrolled_window
  # ========================================================================= #
  def do_use_a_scrolled_window
    @use_a_scrolled_window = true
  end; alias use_a_scrolling_widget do_use_a_scrolled_window # === use_a_scrolling_widget

  # ========================================================================= #
  # === hide_the_last_n_lines
  #
  # This variant does not seem to work.
  # ========================================================================= #
  def hide_the_last_n_lines(
      n_lines_to_remove = 10
    )
    inverted_array = @array_event_boxes.reverse
    n_lines_to_remove.times {|index|
      inverted_array[index].set_visible(false)
    }
  end

  # ========================================================================= #
  # === remove_the_last_n_lines
  #
  # This method can be specifically used to remove the last n-lines of
  # the widget.
  # ========================================================================= #
  def remove_the_last_n_lines(
      n_lines_to_remove = 10
    )
    if n_lines_to_remove < 0
      n_lines_to_remove = 0
    end
    inverted_array = @array_event_boxes.reverse
    n_lines_to_remove.times {|index|
      @main_vbox.remove(
        inverted_array[index]
      ) unless inverted_array[index].nil?
    }
  end; alias remove_last remove_the_last_n_lines # === remove_last

  # ========================================================================= #
  # === cut_down_to
  #
  # This method can be used to "cut-down" to a specific number.
  # ========================================================================= #
  def cut_down_to(i = 10)
    remove_n_entries = @array_event_boxes.size - i
    remove_the_last_n_lines(remove_n_entries)
    set_use_n_lines(i) # Set the new threshold here.
    recalculate
  end; alias set_use_n_entries cut_down_to # === set_use_n_entries

  # ========================================================================= #
  # === n_entries?
  # ========================================================================= #
  def n_entries?
    @up_to_n_times
  end; alias n_lines? n_entries? # === n_lines?

  # ========================================================================= #
  # === set_use_n_lines
  # ========================================================================= #
  def set_use_n_lines(n_lines)
    @up_to_n_times = n_lines.to_i
  end; alias n_lines=      set_use_n_lines # === n_lines=
       alias set_n_entries set_use_n_lines # === set_n_entries
       alias set_maximum   set_use_n_lines # === set_maximum
       alias set_n_lines   set_use_n_lines # === set_n_lines

  # ========================================================================= #
  # === repopulate_towards
  #
  # This method will simply combine two other methods.
  # ========================================================================= #
  def repopulate_towards(i = up_to_n_times?)
    set_maximum(i)
    repopulate
  end

  # ========================================================================= #
  # === up_to_n_times?
  # ========================================================================= #
  def up_to_n_times?
    @up_to_n_times
  end

  # ========================================================================= #
  # === repopulate
  #
  # This method will "repopulate" accordingly to how many lines are
  # assigned.
  # ========================================================================= #
  def repopulate(
      i = @array_event_boxes.size
    )
    remove_the_last_n_lines(i) # First, remove all of them.
    @up_to_n_times.times {|index| index += 1
      # ===================================================================== #
      # Use a gtk-label that is then put inside of an event-box.
      # ===================================================================== #
      label = left_aligned_label(
        index.to_s.ljust(2,' ')
      )
      event_box = create_eventbox(label)
      @array_event_boxes << event_box
      event_box.signal_connect(:event) {|widget, event|
        # =================================================================== #
        # Handle mouse-button-press events next
        # =================================================================== #
        if event.is_a?(Gdk::EventButton) and
          (event.event_type.name == 'GDK_BUTTON_PRESS') and
          use_arrow_icons?
          old_text = label.text.ljust(2,' ')
          if old_text.include? ARROW_TO_USE
            label.set_text("#{old_text.delete(ARROW_TO_USE)}")
          else
            label.set_text("#{old_text}#{ARROW_TO_USE}")
          end
          case event.button
          when 1 # left-mouse-button
            #label.set_colour(:black)
          when 2 # middle-mouse-button
          when 3 # right-mouse-button
            # Right mouse button clicked changes it to red colour.
            if label.text?.include? ARROW_TO_USE
              label.text = label.text?.sub(/#{ARROW_TO_USE}/,
                '<span foreground="tomato">'+ARROW_TO_USE+'</span>'
              )
            end
          end
          label.do_markify
        end
      }
      @main_vbox.minimal(event_box)
    }
    @main_vbox.show_all
  end; alias recalculate repopulate # === recalculate

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    repopulate
    decide_whether_to_use_a_scrolled_window_or_not
  end

  # ========================================================================= #
  # === decide_whether_to_use_a_scrolled_window_or_not
  # ========================================================================= #
  def decide_whether_to_use_a_scrolled_window_or_not
    # ======================================================================= #
    # Check whether we will use a scrolled window or not next:
    # ======================================================================= #
    if @use_a_scrolled_window
      # ===================================================================== #
      # The scrolled-window will contain the main vbox:
      # ===================================================================== #
      @scrolled_window = scrolled_window(@main_vbox) { :automatic }
      @scrolled_window.set_size_request(width?, height?)
      minimal(@scrolled_window, 0)
    else
      minimal(@main_vbox, 0)
    end
  end

  # ========================================================================= #
  # === remove_the_main_vbox
  # ========================================================================= #
  def remove_the_main_vbox
    remove(@main_vbox)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    super()
  end

  # ========================================================================= #
  # === Gtk::NumberedLines.run
  # ========================================================================= #
  def self.run(
      i = ARGV,
      &block
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::NumberedLines.new(i)
    r = ::Gtk.run
    if block_given?
      yielded = yield
      [yielded].flatten.each {|this_entry|
        case this_entry
        # ======================================================================= #
        # === :do_use_a_scrolled_window
        # ======================================================================= #
        when :do_use_a_scrolled_window
          # =================================================================== #
          # Let's add it to a scrolled window, for simpler navigation.
          # =================================================================== #
          scrolled_window = ::Gtk::ScrolledWindow.new
          scrolled_window.add(_)
          r << scrolled_window
        else
          _.send(:do_not_use_arrow_icons)
        end
      }
    else
      r << _
    end
    r.automatic_size_and_automatic_title
    # _.hide_the_last_n_lines(10)
    # _.remove_the_last_n_lines(10)
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  alias e puts
  Thread.abort_on_exception
  numbered_lines = Gtk::NumberedLines.run {
    [
      :do_not_use_arrow_icons,
      :do_use_a_scrolled_window
    ]
  }
  # ========================================================================= #
  # numbered_lines.do_not_use_arrow_icons
  # The next part is not invoked, due to the above code
  # blocking it, but we show it nonetheless to show how
  # this widget can be modified.
  # ========================================================================= #
  e 'Sleeping for 2 seconds next ... '
  Thread.new {
    sleep 2
    e 'Done! Next hiding the last 20 entries.'
    numbered_lines.hide_the_last_n_lines(20)
    e 'Done doing so!'
    numbered_lines.set_maximum(10)
    numbered_lines.repopulate
  }
end