#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::FilesInDirectory
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk3/files_in_directory/files_in_directory.rb'
# Gtk::FilesInDirectory.run
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class FilesInDirectory < ::Gtk::Box # === Gtk::FilesInDirectory 

  include ::Gtk::BaseModule

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Files in a directory'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '15% or minimum 150px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '15%'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_14

  # ========================================================================= #
  # === DIRECTORY_TO_MY_LOCAL_SONGS
  # ========================================================================= #
  DIRECTORY_TO_MY_LOCAL_SONGS = '/home/x/songs/'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      commandline_arguments = ARGV,
      run_already           = true
    )
    super(:vertical)
    reset
    set_commandline_arguments(
      commandline_arguments
    )
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    handle_CSS_rules
    infer_the_size_automatically
    # ======================================================================= #
    # === @array_buttons
    # ======================================================================= #
    @array_buttons = []
    # ======================================================================= #
    # === @current_filename
    # ======================================================================= #
    @current_filename = ''.dup
    # ======================================================================= #
    # === @parent_widget
    # ======================================================================= #
    @parent_widget = nil
    # ======================================================================= #
    # === @on_click_use_green_border
    # ======================================================================= #
    @on_click_use_green_border = false
  end

  # ========================================================================= #
  # === parent_widget?
  # ========================================================================= #
  def parent_widget?
    @parent_widget
  end

  # ========================================================================= #
  # === set_parent_widget
  # ========================================================================= #
  def set_parent_widget(i)
    @parent_widget = i
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    0
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    0
  end

  # ========================================================================= #
  # === create_skeleton                                          (create tag)
  # ========================================================================= #
  def create_skeleton
  end
 
  # ========================================================================= #
  # === set_current_filename
  # ========================================================================= #
  def set_current_filename(i)
    @current_filename = i.to_s
  end

  # ========================================================================= #
  # === enforce_short_basename
  # ========================================================================= #
  def enforce_short_basename
    @array_buttons.each {|this_button|
      old_text = this_button.text?
      this_button.set_text(File.basename(old_text))
    }
  end; alias basename enforce_short_basename # === basename

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === current_filename?
  # ========================================================================= #
  def current_filename?
    @current_filename
  end; alias filename? current_filename? # === filename?

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    most_outer_vbox = padded_vbox
    these_files_are_known(:obtain_local_songs)
    outer_box = gtk_vbox
    @array_buttons.each {|this_button|
      this_button.set_size_request(540, 25)
      hbox = gtk_hbox
      hbox.minimal(this_button, 1)
      outer_box.minimal(hbox, 1)
    }
    most_outer_vbox.minimal(outer_box, 1)
    scrolled_window = gtk_scrolled_window(most_outer_vbox)
    scrolled_window.width_height(140, 260)
    minimal(scrolled_window, 1)
  end

  # ========================================================================= #
  # === handle_CSS_rules                                            (CSS tag)
  #
  # Add some CSS rules for this widget.
  # ========================================================================= #
  def handle_CSS_rules
    use_gtk_paradise_project_css_file
    add_and_apply_these_custom_CSS_rules '

.action_button_containing_the_filename_of_the_song {
  transition: 2000ms linear;
  background-color: white;
}

.action_button_containing_the_filename_of_the_song:hover {
  transition: 2000ms linear;
  background-color: lightblue;
}

.border_green_1px {
  border: 1px solid green;
}

'
  end

  # ========================================================================= #
  # === on_click_use_green_border
  # ========================================================================= #
  def on_click_use_green_border
    @on_click_use_green_border = true
  end

  # ========================================================================= #
  # === these_files_are_known
  # ========================================================================= #
  def these_files_are_known(i)
    case i
    # ======================================================================= #
    # === :obtain_local_songs
    # ======================================================================= #
    when :obtain_local_songs
      i = Dir["#{DIRECTORY_TO_MY_LOCAL_SONGS}*"]
      begin
        require 'multimedia_paradise/toplevel_methods/is_audio_file.rb'
        i.select! {|entry|
          ::MultimediaParadise.is_an_audio_file?(entry)
        }
      rescue LoadError; end
    end
    i.select! {|entry| File.file? entry }
    i.size.times {
      button = gtk_button
      button.clear_background
      button.bblack1
      button.pad1px
      button.css_class('action_button_containing_the_filename_of_the_song')
      button.width_height(25, 5)
      button.on_clicked {
        run_this_code_on_clicked_action(button)
      }
      @array_buttons << button 
    }
    # ======================================================================= #
    # Sort alphabetically, then iterate over our collection:
    # ======================================================================= #
    i.sort.each_with_index {|entry, index|
      _ = @array_buttons[index]
      if index < @array_buttons.size
        @array_buttons[index].set_text(entry.to_s)
      end
    }
    enforce_short_basename
  end

  # ========================================================================= #
  # === run_this_code_on_clicked_action
  #
  # This is the code that will be run when the user clicks on the button.
  # ========================================================================= #
  def run_this_code_on_clicked_action(
      button                 = nil,
      directory_to_the_songs = DIRECTORY_TO_MY_LOCAL_SONGS
    )
    if button
      _ = button.text?.to_s # Obtain the text on that button here.
      unless File.exist? _
        _ = directory_to_the_songs+_ if File.exist?(directory_to_the_songs+_)
      end
      set_current_filename(_)
      # ===================================================================== #
      # Allow feedback onto the parent-widget via set_main_entry().
      # ===================================================================== #
      if @parent_widget and
         @parent_widget.respond_to?(:set_main_entry)
        @parent_widget.set_main_entry(_)
      end
      if @on_click_use_green_border
        @array_buttons.each {|this_button|
          this_button.remove_css_class('bblack1')
          this_button.remove_css_class('border_green_1px')
          this_button.add_css_class('bblack1')
        }
        button.remove_css_class('bblack1')
        button.set_css_class('border_green_1px')
      end
    end
  end

  # ========================================================================= #
  # === Gtk::FilesInDirectory.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    require 'gtk_paradise/run'
    _ = ::Gtk::FilesInDirectory.new(i)
    r = ::Gtk.run
    r << _
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::FilesInDirectory.run
end