#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::ColouredButton
#
# This is a subclass of Gtk::Button.
#
# Specific usage example in ruby code:
#
#   Gtk::ColouredButton.new('_NIHSearch_darkslategray', true)
#
# =========================================================================== #
# require 'gtk_paradise/gui/gtk3/coloured_button/coloured_button.rb'
# Gtk::ColouredButton.new('Test_cyan')
# =========================================================================== #
require 'gtk_paradise/require_gtk3'

module Gtk

class ColouredButton < ::Gtk::BaseModuleButton # === Gtk::ColouredButton 

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Coloured button'

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = '25% or minimum 200px'

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = '15% or minimum 100px'

  # ========================================================================= #
  # === USE_THIS_FONT
  # ========================================================================= #
  USE_THIS_FONT = :dejavu_condensed_20

  attr_accessor :colour_active
  attr_accessor :colour_normal
  attr_accessor :colour_prelight
  attr_accessor :can_be_clicked # true or false. NO other value allowed!

  # ========================================================================= #
  # === @default_colour
  # ========================================================================= #
  @default_colour = 'slateblue'

  # ========================================================================= #
  # === Gtk::ColouredButton.default_colour?
  # ========================================================================= #
  def self.default_colour?
    @default_colour
  end

  # ========================================================================= #
  # === initialize
  #
  # This method will initialize a new instance of Gtk::ColouredButton.
  #
  # Here we 'cheat' a little. Because arg is special, the last part must
  # be a String such as:
  #
  #   '_coral'
  #
  # It will THEN be used as constant name for the colour, in this case
  # the coral colour.
  #
  # The second argument to this method determines whether we have a quit
  # button or whether we do not. By default we will not use a quit
  # button there. (A quit button is one that, upon a click-event, will
  # exit the application.)
  #
  # Usage Examples:
  #
  #   @button_quit = ColouredButton.new('_NIHSearch_darkslategray', true)
  #   @button_quit = ColouredButton.new('_NIHSearch_darkorchid',    true)
  #
  # ========================================================================= #
  def initialize(
      arg                       = "_NIHSearch_#{ColouredButton.default_colour?}", 
      is_a_quit_button          = false,
      colourize                 = true,
      optional_array_with_icons = nil
    )
    super()
    reset
    case is_a_quit_button
    when :this_is_a_quit_button,
         :quit
      is_a_quit_button = true
    end
    @flag_ignored_first_char = false # default
    if arg.start_with? '_'
      arg = arg[1, arg.size]
      @flag_ignored_first_char = true
    end
    if arg.empty?
      remaining_arg = ''
      colour_to_use = 'slateblue' # default
    else
      remaining_arg = arg.split('_').first
      colour_to_use = arg.split('_').last
    end
    remaining_arg = "_#{remaining_arg}" if @flag_ignored_first_char
    if ::Gtk.use_gtk2?
      super(remaining_arg)
    else
      super(label: remaining_arg)
    end
    @text = remaining_arg
    # Currently not so much in use hmmmmm
    colour_to_use = 'slateblue' if colour_to_use.nil?
    # Dont colourize if 3. argument is false 
    colourize_this_button(colour_to_use) if colourize
    if is_a_quit_button # If we have a quit-button:
      on_clicked { ::Gtk.main_quit }
    else 
      # Not sure if we will need this else clause.
    end
     set_new_icons(optional_array_with_icons) if optional_array_with_icons
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    reset_the_internal_variables
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    # ======================================================================= #
    # === Set the title, width, height and the font in use.
    # ======================================================================= #
    title_width_height_font(TITLE, WIDTH, HEIGHT, USE_THIS_FONT)
    use_gtk_paradise_project_css_file 
    infer_the_size_automatically
    @can_be_clicked = true # on default, every button can be clicked
  end

  # ========================================================================= #
  # === padding?
  # ========================================================================= #
  def padding?
    4
  end

  # ========================================================================= #
  # === border_size?
  # ========================================================================= #
  def border_size?
    2
  end

  # ========================================================================= #
  # === create_skeleton                            (create tag, skeleton tag)
  # ========================================================================= #
  def create_skeleton
  end

  # ========================================================================= #
  # === connect_skeleton                                        (connect tag)
  # ========================================================================= #
  def connect_skeleton
    abort_on_exception
    minimal(left_aligned_label('Hello world! This is a simple test.'))
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton
  end

  # ========================================================================= #
  # === set_new_icon
  #
  # Set one or more images for our button. Input can be an Array or a
  # String.
  # ========================================================================= #
  def set_new_icon(i)
    if i.is_a? String
      image = gtk_image(location)
      image.set_padding(4,0) # left-right axis, top-down axis
      set_image(image)
    elsif i.is_a? Array
      i.each {|location|
        set_new_icon(location)
      }
    end
  end; alias set_new_icons set_new_icon # === set_new_icons
  
  # ========================================================================= #
  # === return_random_colour
  # ========================================================================= #
  def return_random_colour
    return ::Colours.random_html_colour if Object.const_defined? :Colours
    return nil # else return nil.
  end

  # ========================================================================= #
  # === colourize_this_button
  #
  # This method is called whenever you want to colourize this widget.
  #
  # Need to default onto slateblue.
  # ========================================================================= #
  def colourize_this_button(
      colour_to_use = 'slateblue'
    )
    colour_to_use = 'slateblue' if colour_to_use.nil?
    colour_to_use = return_random_colour if colour_to_use == 'random'
    original_colour_to_use = colour_to_use.dup 
    # ===================================================================== #
    # Now modify button colours. This needs to be handled differently
    # depending on the gtk-version at hand.
    # ===================================================================== #
    begin
      colour1 = Gdk::Color.parse(colour_to_use)
    rescue ArgumentError # rescue
      colour1 = Gdk::Color.parse('slateblue')
    end
    if ::Gtk.use_gtk3? and colour1.is_a?(Gdk::Color)
      colour1 = ::Gtk.convert_gdk_colour_to_gdk_rgba_colour(colour1)
    end
    @colour_active = colour1
    modify_background(:active, colour1)
    # ===================================================================== #
    # Next determine colour2
    # ===================================================================== #
    begin
      colour2 = Gdk::Color.parse(colour_to_use)
    rescue ArgumentError # rescue
      colour2 = Gdk::Color.parse('slateblue')
    end
    if ::Gtk.use_gtk3? and colour2.is_a?(Gdk::Color)
      colour2 = ::Gtk.convert_gdk_colour_to_gdk_rgba_colour(colour2)
    end
    @colour_normal = colour2
    modify_background(:normal, colour2)
    begin
      colour_to_use = Gdk::Color.parse(colour_to_use)
      if ::Gtk.use_gtk3? and colour_to_use.is_a?(Gdk::Color)
        colour_to_use = ::Gtk.convert_gdk_colour_to_gdk_rgba_colour(colour_to_use)
      end
      modify_background(:prelight, colour_to_use)
    rescue ArgumentError => error
      p error
      Gdk::Color.parse('slateblue')
    end
    # ===================================================================== #
    # This will be for the slightly lighter colour:
    # ===================================================================== #
    constant = original_colour_to_use.capitalize.to_sym
    # ===================================================================== #
    # If the user has HtmlColours enabled.
    # ===================================================================== #
    if Object.const_defined?(:Colours) and
     ::Colours.html_colours.include?(constant.to_s.downcase.to_sym)
       html_colour_to_rgb_value
      _ = Gdk::Color.new(*::Colours.html_colour_to_rgb[constant.to_s.downcase])
      # Somehow this here does not work - it makes the background too dark.
      # modify_bg(Gtk::StateType::PRELIGHT, _)
    else # use a default
      modify_background(:prelight, :dodgerblue)
      @colour_prelight = 'dodgerblue'
    end
  end

  # ========================================================================= #
  # === toggle_focus
  # ========================================================================= #
  def toggle_focus
    self.can_focus = self.can_focus? ? false : true
  end

  # ========================================================================= #
  # === text?
  # ========================================================================= #
  def text? # the @text var just stores label.
    @text
  end; alias text text? # === text

  # ========================================================================= #
  # === Gtk::ColouredButton.run
  # ========================================================================= #
  def self.run(
      i = ARGV
    )
    begin
      require 'colours'
    rescue LoadError; end
    require 'gtk_paradise/run'
    coloured_button = Gtk::ColouredButton.new(
      "_Test_#{::Colours.random_html_colour}"
    )
    coloured_button.toggle_focus
    r = ::Gtk.run
    r << coloured_button
    r.automatic_size_then_automatic_title
    r.top_left_then_run
  end

end; end

if __FILE__ == $PROGRAM_NAME
  Gtk::ColouredButton.run
end # gtkcolouredbutton