#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::YoutubeDl
#
# This class allows us to download remote videos from e. g. youtube,
# via youtube-dl. The main use case is to then embed this small
# widget into a larger widget.
#
# Usage example:
#
#   Gtk::YoutubeDl.new
#
# =========================================================================== #
# require 'gtk_paradise/widgets/gtk2/youtube_dl.rb'
# Gtk::YoutubeDl.new
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class YoutubeDl < ::Gtk::Frame # === Gtk::YoutubeDl

  require 'fileutils'

  include ::Colours::E

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = 24

  # ========================================================================= #
  # === MAIN_FONT_SIZE
  # ========================================================================= #
  MAIN_FONT_SIZE = 24

  # ========================================================================= #
  # === WIDTH
  # ========================================================================= #
  WIDTH = 1220

  # ========================================================================= #
  # === HEIGHT
  # ========================================================================= #
  HEIGHT = 280

  # ========================================================================= #
  # === TITLE
  # ========================================================================= #
  TITLE = 'Youtube-dl wrapper'

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = ARGV, # <- This is ignored for the time being.
      run_already = true
    )
    super(TITLE)
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
    # ======================================================================= #
    # === @cd_into_this_directory
    #
    # Specify into which directory we will cd into.
    # ======================================================================= #
    @cd_into_this_directory = '/Depot/j/'
    @title  = TITLE
    set_width(WIDTH)
    set_height(HEIGHT)
    infer_the_size_automatically
    increase_font_size
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    @statusbar = gtk_statusbar { :resize_grip }
    @statusbar_context_id = @statusbar.get_context_id('youtube-dl')
    make_bold # Make the main GTK frame bold.
    vbox = gtk_vbox
    @entry_for_the_remote_url = gtk_entry

    hbox = gtk_hbox {{ spacing: 5 }}
    youtubedl_version = `youtube-dl --version`.strip
    hbox.minimal(
      gtk_label(
        'The <b>youtube-dl version</b> in use is: '+
        youtubedl_version
      )
    )
    vbox.minimal(hbox)
    vbox.minimal(spacer)

    vbox.minimal(
      gtk_label('Input the remote URL into the following input-widget:').make_bold, 0
    )
    vbox.minimal(@entry_for_the_remote_url, 0)
    @button_do_download = gtk_coloured_button(
      '_Download the above ˆˆˆ remote URL via youtube-dl_lightgreen'
    )
    @button_do_download.tooltip = ' Pressing this button will '\
      'download the above ˆˆˆ URL. '
    @button_do_download.on_clicked {
      do_download(@entry_for_the_remote_url.text)
    }
    vbox.minimal(@button_do_download, 0)
    @entry_for_the_command_that_will_be_run = gtk_entry { :read_only }
    vbox.minimal(@entry_for_the_command_that_will_be_run, 0)

    # ======================================================================= #
    # Next, the small widget notifying the user into which local
    # file that video was stored:
    # ======================================================================= #
    @entry_merged_into_this_file = gtk_entry { :read_only }
    saved_into_which_file_mini_widget = gtk_hbox
    saved_into_which_file_mini_widget.minimal(
      gtk_bold_text('Saved into this file:'),
       0
    )
    saved_into_which_file_mini_widget.maximal(
      @entry_merged_into_this_file,
      0
    )
    vbox.minimal(saved_into_which_file_mini_widget, 0)

    download_into_which_dir_mini_widget = gtk_hbox
    download_into_which_dir_mini_widget.minimal(
      gtk_bold_text('Download into this directory:'),
      0
    )
    download_into_which_dir_mini_widget.maximal(
      @entry_download_into_this_directory = gtk_entry,
      0
    )
    # Set an initial value:
    @entry_download_into_this_directory.set_text(@cd_into_this_directory)
    vbox.minimal(download_into_which_dir_mini_widget, 0)
    vbox.minimal(@statusbar, 0)
    add(vbox)
  end

  # ========================================================================= #
  # === do_download                                            (download tag)
  #
  # This is the action that will download the remote video.
  # ========================================================================= #
  def do_download(
      i = @entry_for_the_remote_url.text
    )
    cd(
      @entry_download_into_this_directory.text
    )
    if i.empty?
      add_to_statusbar('Please provide a valid remote URL.')
    else
      e ::Colours.steelblue(i)
      cmd = "youtube-dl #{i}"
      @entry_for_the_command_that_will_be_run.set_text(cmd)
      e ::Colours.tomato(cmd)
      Thread.new {
        result = `#{cmd}`
        if result.include? '[ffmpeg]'
          scanned = result.scan(
            /\[ffmpeg\] Merging formats into (.+)\n/
          ).flatten.first.to_s.delete('"')
          if File.exist? scanned
            new_value = scanned.tr(' ','_').delete('&()').squeeze('_')
            FileUtils.mv(scanned, new_value)
            scanned = new_value
          end
          e 'Merging into the file '+
            ::Colours.steelblue(File.absolute_path(scanned))
          @entry_merged_into_this_file.set_text(scanned)
        end
      }.join
    end
  end

  # ========================================================================= #
  # === add_to_statusbar
  # ========================================================================= #
  def add_to_statusbar(i = '')
    @statusbar.add_this_message(@statusbar_context_id, i)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton
  end

  # ========================================================================= #
  # === Gtk::YoutubeDl[]
  # ========================================================================= #
  def self.[](i = '')
    new
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  r = Gtk.run
  youtubedl = Gtk::YoutubeDl.new(ARGV)
  r << youtubedl
  r.width = youtubedl.width?
  r.height = youtubedl.height?
  r.top_left_then_run
end # gtkyoutubedl