#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::StatusIconExample
#
# This class shows how to use a Gtk::StatusIcon image.
#
# A StatusIcon is essentially an icon in tray; additionally it allows
# for simple interactions with the mouse.
#
# Documentations on StatusIcon class is provided at:
# 
#   http://ruby-gnome2.sourceforge.jp/hiki.cgi?cmd=view&p=Gtk%3A%3AStatusIcon
#
# Usage example:
#
#   Gtk::StatusIconExample.new(ARGV)
#
# =========================================================================== #
# require 'gtk_paradise/classes/status_icon_example.rb'
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class StatusIconExample < Gtk::VBox # === Gtk::StatusIconExample

  include Colours

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  alias ee print

  # ========================================================================= #
  # === MAIN_FONT_SIZE
  # ========================================================================= #
  MAIN_FONT_SIZE = 32

  # ========================================================================= #
  # === PREFERRED_WIDTH
  # ========================================================================= #
  PREFERRED_WIDTH  = 550

  # ========================================================================= #
  # === PREFERRED_HEIGHT
  # ========================================================================= #
  PREFERRED_HEIGHT = 550

  # ========================================================================= #
  # === USE_THIS_IMAGE
  # ========================================================================= #
  USE_THIS_IMAGE = 
    "#{::Gtk.directory_to_the_images?}misc/weather.png"

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      i           = nil,
      run_already = true
    )
    super()
    register_sigint
    reset
    run if run_already
  end

  # ========================================================================= #
  # === reset                                                     (reset tag)
  # ========================================================================= #
  def reset
    infer_the_namespace
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, namespace?]
    increase_font_size
    set_size_request(
      PREFERRED_WIDTH,
      PREFERRED_HEIGHT,
    )
    # ======================================================================= #
    # === StatusIcon
    #
    # Displayed Icon - from class StatusIcon.
    # ======================================================================= #
    @use_this_image = USE_THIS_IMAGE
    # ======================================================================= #
    # === @status_icon
    # ======================================================================= #
    @status_icon = gtk_status_icon(@use_this_image)
    @status_icon.tooltip = ' Status Icon - we can even blink! '
    # ======================================================================= #
    # Handle left click on icon. If clicked the icon will
    # begin to blink.
    # ======================================================================= #
    @status_icon.signal_connect(:activate) {|icon| 
      e 'We start to blink!'
      icon.blinking = !icon.blinking? # ← Toggle it.
    }
    # ======================================================================= #
    # === information
    #
    # Pop up a menu on right click.
    # ======================================================================= #
    # Build the menu:
    information = Gtk::ImageMenuItem.new(Gtk::Stock::INFO) # info button.
    information.on_activate {
      e
      cliner
      e "Embedded: #{@status_icon.embedded?}"
      e "Visible:  #{@status_icon.visible?}"
      e "Blinking: #{@status_icon.blinking?}"
      cliner
      e
    }
    quit = Gtk::ImageMenuItem.new(Gtk::Stock::QUIT) # quit button.
    quit.signal_connect(:activate) { Gtk.main_quit }
    menu = gtk_menu # Here comes the Gtk Menu.
    menu.append(information)
    menu.append(Gtk::SeparatorMenuItem.new)
    menu.append(quit)
    menu.show_all
    # ======================================================================= #
    # Show menu on right mouse-button click.
    # ======================================================================= #
    @status_icon.signal_connect(:popup_menu) {|tray, button, time|
      menu.popup(nil, nil, button, time)
    }
    # ======================================================================= #
    # Handle scroll events here. We need to press a modifier key like
    # Control. Needs a recent gtk version (>=2.16 )
    # ======================================================================= #
    @status_icon.signal_connect(:scroll_event){|icon, event|
	    modifier = event.state#A GdkModifierType indicating the state of modifier keys and mouse buttons
      # Handle only control and shift key
	    ctrl_shift = (Gdk::Window::CONTROL_MASK|Gdk::Window::SHIFT_MASK)
	    mod = modifier & ctrl_shift
	    case mod
	    when 0
		    ee '(None)'
	    when Gdk::Window::CONTROL_MASK
		    ee 'Control+'
	    when Gdk::Window::SHIFT_MASK # This is 
		    ee 'Shift+'
	    when (Gdk::Window::CONTROL_MASK|Gdk::Window::SHIFT_MASK)
		    ee 'Control+Shift+'
	    end
      # Check for direction
	    case event.direction
	    when Gdk::EventScroll::UP
		    ee 'up'+N
	    when Gdk::EventScroll::DOWN
		    ee 'down'+N
	    end
    }
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    vbox = gtk_vbox
    @label = gtk_bold_label(
     "Hello world!\n\nNotice that a small icon may \nappear "\
     "as status, on the \nbottom area of the desktop."
    )
    vbox.maximal(@label, 1)
    minimal(vbox, 2)
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton
  end

  # ========================================================================= #
  # === Gtk::StatusIconExample[]
  # ========================================================================= #
  def self.[](i = '')
    new
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'
  r = Gtk.run(width: 200, height: 200)
  r << Gtk::StatusIconExample.new(ARGV)
  r.enable_quick_exit
  r.top_left_then_run
end # status_icon_example