#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::MenuDemo 
#
# The primary purpose of this class is to showcase how a menu is used
# in ruby-gtk.
# =========================================================================== #
require 'gtk_paradise/require_gtk2'

module Gtk

class MenuDemo < Gtk::VBox # === Gtk::MenuDemo

  require 'gtk_paradise/requires/require_the_base_module.rb'
  include ::Gtk::BaseModule

  # ========================================================================= #
  # === NAMESPACE
  # ========================================================================= #
  NAMESPACE = inspect

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super()
    self.border_width = 10
    reset
    increase_font_size
    run
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @configuration
    # ======================================================================= #
    @configuration = [true, __dir__, NAMESPACE]
  end

  # ========================================================================= #
  # === create_menu
  # ========================================================================= #
  def create_menu(depth)
    return nil if depth < 1
    menu = gtk_menu
    group = nil
    0.upto(4) {|i|
      text = sprintf('Item %2d - %d', depth, i + 1)
      # ===================================================================== #
      # Group the menu into radio menu item entries:
      # ===================================================================== #
      menuitem = gtk_radio_menu_item(group, text)
      group = menuitem.group
      menu.append(menuitem) # Append the radio-menu-item here.
      if depth > 1
        menuitem.set_submenu(
          create_menu(depth - 1)
        )
      end
    }
    menu
  end

  # ========================================================================= #
  # === title?
  # ========================================================================= #
  def title?
    'MenuDemo'
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    box1 = gtk_vbox
    # ======================================================================= #
    # Next comes the main menu bar:
    # ======================================================================= #
    menu_bar = gtk_menu_bar
    box1.pack_start(menu_bar, false, true, 0)
    menu = create_menu(2)
    menuitem = gtk_menu_item("First\nline")
    menuitem.set_submenu(menu)
    menu_bar.append(menuitem)
    menuitem = gtk_menu_item('Second')
    menuitem.set_submenu(create_menu(3))
    menu_bar.append(menuitem)
    menuitem = gtk_menu_item('Third')
    menuitem.set_submenu(create_menu(4))
    menu_bar.append(menuitem)

    optionmenu = gtk_option_menu
    optionmenu.set_menu(create_menu(1))
    optionmenu.set_history(4)

    separator2 = gtk_hseparator
    box1.maximal(separator2, 0)
    box1.maximal(optionmenu, 0)
    separator = gtk_hseparator
    box1.minimal(separator, 0)
    pack_maximal(box1, 0)
  end

  # ========================================================================= #
  # === width?
  # ========================================================================= #
  def width?
    800
  end

end; end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb' 
  x = Gtk::MenuDemo.new
  r = Gtk.run(
    x.width?,560,
    :tabble,
    x.title?,0 
  )
  r << x
  r.modify_background :normal, :beige
  r.enable_quick_exit
  r.top_left_then_run
end # $RUBY_GTK/menu_demo.rb