#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/toplevel_methods/keys.rb'
# include ::Gtk::Keys
# =========================================================================== #
module Gtk

# =========================================================================== #
# === Gtk::Keys
#
# Contains key specific stuff for easier reuse.
# =========================================================================== #
module Keys # === Gtk::Keys

  # ========================================================================= #
  # === @arrow_keys
  # ========================================================================= #
  @arrow_keys = []

  # ========================================================================= #
  # === Gtk::Keys.arrow_keys?
  # ========================================================================= #
  def self.arrow_keys?
    @arrow_keys
  end

  # ========================================================================= #
  # === Gtk::Keys.populate_arrow_keys
  #
  # This method will populate the @arrow_keys Array.
  # ========================================================================= #
  def self.populate_arrow_keys
    # ========================================================================= #
    # === ARROW_KEYS
    #
    # Do note that the arrow-keys have changed in ruby-gtk3, from
    # e. g. 'Gdk::Keyval::GDK_KEY_Left' to 'Gdk::Keyval::KEY_Left'.
    #
    # So the new variants in ruby-gtk3 are:
    #
    #   Gdk::Keyval::KEY_Left
    #   Gdk::Keyval::KEY_Right
    #   Gdk::Keyval::KEY_Up
    #   Gdk::Keyval::KEY_Down
    #
    # The old variants in ruby-gtk2 were:
    #
    #   Gdk::Keyval::GDK_KEY_Left
    #   Gdk::Keyval::GDK_KEY_Right
    #   Gdk::Keyval::GDK_KEY_Up
    #   Gdk::Keyval::GDK_KEY_Down
    #
    # ========================================================================= #
    if Object.const_defined?(:Gtk) and ::Gtk::Version::MAJOR.to_i < 3
      @arrow_keys = [
        Gdk::Keyval::GDK_KEY_Left,
        Gdk::Keyval::GDK_KEY_Right,
        Gdk::Keyval::GDK_KEY_Up,
        Gdk::Keyval::GDK_KEY_Down
      ]
    else # else assume gtk3 here, for now.
      @arrow_keys = [
        Gdk::Keyval::KEY_Left,
        Gdk::Keyval::KEY_Right,
        Gdk::Keyval::KEY_Up,
        Gdk::Keyval::KEY_Down
      ]
    end
  end
end

# =========================================================================== #
# === Gtk.get_shortcut_key
#
# This just wraps around Gdk::Keyval.from_name
# =========================================================================== #
def self.get_shortcut_key(of_this_key)
  case of_this_key
  when '←'
    return 65361 # Left-arrow.
  when '→'
    return 65363 # Right-arrow.
  end
  Gdk::Keyval.from_name(of_this_key.to_s)
end

end

if __FILE__ == $PROGRAM_NAME
  require 'gtk3'
  Gtk::Keys.populate_arrow_keys
  include ::Gtk::Keys
  pp Gtk::Keys.arrow_keys?
end