#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/threads/threads.rb'
# =========================================================================== #

# =========================================================================== #
# Call Gtk.init_thread_protect method first when you start things up, then,
# whenever you need to do GUI update stuff, just wrap it in a
# Gtk.thread_protect {} block. Voila! No more crashes.
#
#   Gtk.init_thread_protect
#   Gtk.thread_protect {}
#
# =========================================================================== #
module Gtk

  require 'thread'

  # ========================================================================= #
  # === Gtk::PENDING_CALLS_MUTEX
  # ========================================================================= #
  PENDING_CALLS_MUTEX = Mutex.new

  # ========================================================================= #
  # === Gtk::PENDING_CALLS
  # ========================================================================= #
  PENDING_CALLS = []

  # ========================================================================= #
  # === Gtk.init_thread_protect
  # ========================================================================= #
  def self.init_thread_protect(
      timeout_to_use = 100
    )
    Gtk.timeout_add(timeout_to_use) {
      PENDING_CALLS_MUTEX.synchronize {
        PENDING_CALLS.each { |closure| closure.call }
        PENDING_CALLS.clear
      }
      true
    }
  end

  # ========================================================================= #
  # === Gtk.thread_protect
  # ========================================================================= #
  def self.thread_protect(
      &block
    )
    if Thread.current == Thread.main
      block.call
    else
      PENDING_CALLS_MUTEX.synchronize {
        PENDING_CALLS << block
      }
    end
  end

  # ========================================================================= #
  # === Gtk.thread_flush
  # ========================================================================= #
  def self.thread_flush
    if PENDING_CALLS_MUTEX.try_lock
      PENDING_CALLS.each { |closure| closure.call }
      PENDING_CALLS.clear
      PENDING_CALLS_MUTEX.unlock
    end
  end

end