#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::TextBufferExample
# =========================================================================== #
# $GTK_PARADISE/examples/gtk3/019_text_buffer_example.rb
# =========================================================================== #
require 'gtk3'

module Gtk

class TextBufferExample < ::Gtk::Box

  alias e puts

  # ========================================================================= #
  # === ARRAY_USE_THESE_COLOURS
  # ========================================================================= #
  ARRAY_USE_THESE_COLOURS = %w( red gold green steelblue white )

  # ========================================================================= #
  # === CSS_STRING
  # ========================================================================= #
  CSS_STRING = <<-EOF

/* The following variants do not all seem to work that well ... */
textview {
  border: none;
  background-color: white;
  padding: 8px;
  border:  8px solid darkblue;
  border-width: 12px 12px 12px 12px;
  border-style: solid;
  border-color: #000;
  border-radius: 3px;
}

#scrolledwindow1 {
  border: 3px solid darkblue;
}

/* And if you want to style the text as well */
textview text {
  color: darkgreen;
}

.larger_font {
  font-size: 30px;
}

EOF

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize(
      run_already = true
    )
    super(:horizontal)
    reset
    add_css_rules
    run if run_already
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
  end

  # ========================================================================= #
  # === create_textview
  # ========================================================================= #
  def create_textview
    @grid.attach(@scrolled_window, 0, 1, 3, 1)

    @text_view = ::Gtk::TextView.new

    @text_buffer = @text_view.buffer

    @text_view.pixels_above_lines = 2
    @text_view.pixels_below_lines = 2
    @text_view.pixels_inside_wrap = 2

    start_iter = @text_buffer.start_iter # Obtain the start iter.

    @text_buffer.insert(start_iter, 'This is some text ') # Insert some regular text.
    # ======================================================================= #
    # Next, insert some markup text - bold text.
    # ======================================================================= #
    @text_buffer.insert_markup(
      @text_buffer.end_iter,
      "<b>and some bold text</b>",
      -1
    )
    @text_buffer.insert_markup(
      @text_buffer.end_iter,
      "... <i>and italic, too!</i> \\o/\n\n",
      -1
    )

    ARRAY_USE_THESE_COLOURS.each {|entry|
      @text_buffer.insert_markup(
        @text_buffer.end_iter,
        "<span color=\"#{entry}\">This is a test message.</span>\n",
        -1
      )
    }
    iter = return_current_iter
    @text_buffer.insert(iter, "This is more text that "\
      "is just added (appended, actually).\n")
    iter = return_current_iter
    darkblue = @text_buffer.create_tag(nil, { foreground: 'darkblue' })
    @text_buffer.insert(
      iter, 'A hyperlink: <a href="blog.fefe.de">here</a>', tags: [darkblue]
    )

    @scrolled_window.add(@text_view)
  end

  # ========================================================================= #
  # === return_current_iter
  # ========================================================================= #
  def return_current_iter
    @text_buffer.get_iter_at(offset: -1)
  end

  # ========================================================================= #
  # === create_grid
  # ========================================================================= #
  def create_grid
    @grid = ::Gtk::Grid.new
  end

  # ========================================================================= #
  # === connect_skeleton
  # ========================================================================= #
  def connect_skeleton
    add(@grid)
    consider_allowing_the_text_buffer_to_respond_to_certain_events
  end

  # ========================================================================= #
  # === create_scrolled_window
  # ========================================================================= #
  def create_scrolled_window
    @scrolled_window = ::Gtk::ScrolledWindow.new
    @scrolled_window.set_hexpand(true)
    @scrolled_window.set_vexpand(true)
    @scrolled_window.set_name('scrolledwindow1')
    @scrolled_window.shadow_type = :in
    @scrolled_window.style_context.add_class('larger_font')
  end

  # ========================================================================= #
  # === add_css_rules
  # ========================================================================= #
  def add_css_rules
    provider = ::Gtk::CssProvider.new
    provider.load(data: CSS_STRING)
    ::Gtk::StyleContext.add_provider_for_screen(
      Gdk::Screen.default,
      provider,
      Gtk::StyleProvider::PRIORITY_APPLICATION
    )
  end

  # ========================================================================= #
  # === consider_allowing_the_text_buffer_to_respond_to_certain_events (event tag)
  # ========================================================================= #
  def consider_allowing_the_text_buffer_to_respond_to_certain_events
    @text_buffer.signal_connect(:delete_range) {|widget, event|
      e 'Something was deleted.'
    }
    @text_buffer.signal_connect(:changed) {|widget, event|
      e 'The cursor position is at: '+
         widget.cursor_position.to_s
    }
    @text_buffer.signal_connect('notify::cursor-position') {|widget, event|
      e 'The cursor position was changed. It is now at: '+
         widget.cursor_position.to_s
    }
    # @text_buffer.signal_connect(:insert_text) {|widget, iter| # widget is text-buffer, logically.
    @text_view.signal_connect(:key_press_event) {|widget, event|
      e 'The key that was added was: '+
         Gdk::Keyval.to_name(event.keyval).to_s
      false
    }
  end

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
    create_grid
    create_scrolled_window
    create_textview
  end

  # ========================================================================= #
  # === run
  # ========================================================================= #
  def run
    create_skeleton
    connect_skeleton
  end

end; end

if __FILE__ == $PROGRAM_NAME
  window = Gtk::Window.new
  text_buffer_example = Gtk::TextBufferExample.new
  window.add(text_buffer_example)
  window.signal_connect(:delete_event) { Gtk.main_quit }
  window.border_width = 6
  window.show_all
  window.set_default_size(1400, 600)
  window.set_size_request(800, 550)
  window.move(0, 0)
  Gtk.main
end