#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# Use this file to do pixbuf manipulations. We can place a rectangular
# object on an image.
#
# The file used to be named 'pixbuf_manipulations.rb'.
# =========================================================================== #
# Upstream documentation is available here:
#
#   https://ruby-gnome2.osdn.jp/hiki.cgi?Gdk%3A%3APixbuf
#
# =========================================================================== #
require 'gdk_pixbuf2'

alias e puts

# =========================================================================== #
# === DEFAULT_FILE
# =========================================================================== #
DEFAULT_FILE = '/home/x/data/images/NJOY/VidaGuerra.jpg'
Rect = Struct.new('Rect', :x, :y, :w, :h)
path = ARGV.first || DEFAULT_FILE
pixbuf = GdkPixbuf::Pixbuf.new(file: path)

# =========================================================================== #
# Sample crop area - We use Rect.new, which we defined above.
# =========================================================================== #
crop = Rect.new(
  pixbuf.width / 4,
  pixbuf.height / 4,
  pixbuf.width / 2,
  pixbuf.height / 2
)
# =========================================================================== #
# Make pixbuf red (for the border part)
# =========================================================================== #
red_pixbuf = pixbuf.dup
red_pixbuf.fill!(0xff0000ff) # Fill it with our colour here.
# Make greyed out/pixelated pixbuf for background
grey_pixbuf = pixbuf.saturate_and_pixelate(0.2, true)
# =========================================================================== #
# Box for border
# =========================================================================== #
box = Rect.new(crop.x - 2, crop.y - 2, crop.w + 4, crop.h + 4)
# Ensure that border doesn't overflow image
box.x = [box.x,0].max
box.y = [box.y,0].max
box.w = [box.w, red_pixbuf.width  - box.x].min
box.h = [box.h, red_pixbuf.height - box.y].min
# Use red pixbuf as border background
red_pixbuf.copy_area(box.x, box.y, box.w, box.h, grey_pixbuf, box.x, box.y)
# Copy pristine image into the crop area
pixbuf.copy_area(crop.x, crop.y, crop.w, crop.h, grey_pixbuf, crop.x, crop.y)
pixbuf = grey_pixbuf
store_where = ('/'+path+'.out'+File.extname(DEFAULT_FILE)).squeeze '/'
e "Stored here: #{store_where}"
# =========================================================================== #
# See:
#
#   https://developer.gnome.org/gdk-pixbuf/stable/gdk-pixbuf-File-saving.html#gdk-pixbuf-save
#
# gdk_pixbuf_save (pixbuf, handle, "jpeg", &error, "quality", "100", NULL);
#
# And in ruby it is:
#
#   save(filename, type, options = nil)
#
# =========================================================================== #
pixbuf.save(store_where, 'jpeg', quality: 80)
# rb $RUBY_GTK/pixbuf_manipulations.rb