#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Drawing::Turtle
# =========================================================================== #
# require 'gtk_paradise/drawing/turtle.rb'
# =========================================================================== #
module Gtk

module Drawing

require 'gtk_paradise/drawing/tool.rb'

# =========================================================================== #
# The turtle can move to the left and to the right.
# =========================================================================== #
class Turtle < Tool

  attr_accessor :pen_po
  attr_accessor :dir

  # ========================================================================= #
  # === initialize
  # ========================================================================= #
  def initialize
    super()
    reset
    @pen_po = Vector[0, 0]
    @dir    = Vector[1, 0]
    @width, @height = @pen.get_window_size
  end

  # ========================================================================= #
  # === reset
  # ========================================================================= #
  def reset
    # ======================================================================= #
    # === @pen
    #
    # The pen is used to "draw" on the area.
    # ======================================================================= #
    @pen = Tool.new
    # ======================================================================= #
    # === @color_to_use
    # ======================================================================= #
    @color_to_use = [65535, 65535, 65535]
  end

  # ========================================================================= #
  # === left
  # ========================================================================= #
  def left(deg)
    # ======================================================================= #
    # The Greek letter θ (theta) is used in math as a variable to represent
    # a measured angle.
    # ======================================================================= #
    θ = Math::PI * deg / 180
    @dir = Matrix[
      [
        Math.cos(θ),
        -Math.sin(θ)
      ],
      [
        Math.sin(θ),
        Math.cos(θ)
      ]
    ] * @dir
  end

  # ========================================================================= #
  # === right
  # ========================================================================= #
  def right(deg)
    left(-deg)
  end

  # ========================================================================= #
  # === forward
  #
  # Let the turtle move forward.
  # ========================================================================= #
  def forward(
      length,
      shall_we_draw = true
    )
    next_po = @pen_po + @dir * length
    if shall_we_draw
      @pen.color(*@color_to_use)
      @pen.line(
        @width / 2 + next_po[0], @height / 2 - next_po[1],
        @width / 2 + @pen_po[0], @height / 2 - @pen_po[1]
      )
    end
    @pen_po = next_po
  end

  # ========================================================================= #
  # === back
  # ========================================================================= #
  def back(length)
    forward(-length, false)
  end

  # ========================================================================= #
  # === color
  #
  # The method accepts three arguments: r, g and b.
  # ========================================================================= #
  def color(r, g, b)
    @color_to_use = [r, g, b]
    # Delegate the colour onto the pen-tool.
    @pen.color(*@color_to_use)
  end

  # ========================================================================= #
  # === circle
  # ========================================================================= #
  def circle(
      radius, fill = false
    )
    @pen.color(*@color_to_use)
    @pen.circle(
      fill,
      @width / 2 + @pen_po[0],
      @height / 2 - @pen_po[1],
      radius
    )
  end

  # ========================================================================= #
  # === move
  #
  # Position the turtle via Vector.
  # ========================================================================= #    
  def move(x, y)
     @pen_po = Vector[x, y]
  end

end; end; end