#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Toolbar
#
# Gtk::Toolbar is a bar that holds buttons and other widgets, usually 
# placed on the very top of a widget.
#
# Note that Gtk::Toolbar has changed in gtk3. For example, append_space()
# is no longer available.
# =========================================================================== #
# require 'gtk_paradise/core_classes/toolbar.rb'
# =========================================================================== #
module Gtk

class Toolbar # === Gtk::Toolbar

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === Gtk::Toolbar<<
  # ========================================================================= #
  # alias << append # Add an alias between << and append.

  # ========================================================================= #
  # === add_space
  # ========================================================================= #
  def add_space
    if respond_to?(:append_space) # and ::Gtk.use_gtk2?
      append_space
    end
  end

  # ========================================================================= #
  # === append_with_space
  # ========================================================================= #
  def append_with_space(
      text,
      optional_tooltip = nil,
      style            = nil,
      &block
    )
    # The following two methods no longer exist in gtk3.
    if ::Gtk.use_gtk2?
      append(
        text,
        optional_tooltip,
        style,
        &block
      )
      append_space
    end
  end; alias << append_with_space # === <<

  # ========================================================================= #
  # === add_open_file_entry
  # ========================================================================= #
  def add_open_file_entry(parent_widget)
    append_with_space(
      ::Gtk::Stock::OPEN,
      'Open File'
    ) {
        dialog = ::Gtk::FileChooserDialog.new(
                   'Open File',
                   parent_widget,
                   ::Gtk::FileChooser::ACTION_OPEN,
                   nil,
                   [::Gtk::Stock::CANCEL, ::Gtk::Dialog::RESPONSE_CANCEL],
                   [::Gtk::Stock::OPEN,   ::Gtk::Dialog::RESPONSE_ACCEPT]
                 )
        parent_widget.filename = dialog.filename
        case dialog.run
        when Gtk::Dialog::RESPONSE_ACCEPT
          # This is entered when the user clicked "ok".
          # We will be silent here, but perhaps this may change
          # in the future.
        end
        dialog.destroy
      }
  end

end

# =========================================================================== #
# === Gtk.toolbar
# =========================================================================== #
def self.toolbar
  ::Gtk::Toolbar.new
end

end