#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::TextView
# =========================================================================== #
# require 'gtk_paradise/core_classes/text_view.rb'
# Gtk.textview
# =========================================================================== #
module Gtk

class TextView # === Gtk::TextView

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === do_focus
  # ========================================================================= #
  def do_focus
    set_focus(true)
  end; alias focus do_focus # === focus

  # ========================================================================= #
  # === set_left_right_margin
  # ========================================================================= #
  def set_left_right_margin(i)
    set_left_margin(i)
    set_right_margin(i)
  end

  # ========================================================================= #
  # === buffer?
  # ========================================================================= #
  alias buffer? buffer

  # ========================================================================= #
  # === on_changed
  # ========================================================================= #
  # def on_changed(&block)
  #   signal_connect(:key_press_event, &block)
  # end

  # ========================================================================= #
  # === clear
  # ========================================================================= #
  def clear
    buffer.set_text('')
  end; alias clear_the_text clear # === clear_the_text

  # ========================================================================= #
  # === do_wrap_properly
  # ========================================================================= #
  def do_wrap_properly
    set_wrap_mode(::Gtk::WrapMode::CHAR)
  end

  # ========================================================================= #
  # === wrap_mode_word
  # ========================================================================= #
  def wrap_mode_word
    set_wrap_mode(::Gtk::WrapMode::WORD)
  end

  # ========================================================================= #
  # === text?
  #
  # This variant makes sure that a String will be returned.
  # ========================================================================= #
  def text?
    buffer.text.to_s
  end; alias getText text? # === getText

  # ========================================================================= #
  # === do_markify
  # ========================================================================= #
  def do_markify
    buffer.do_markify
  end

  # ========================================================================= #
  # === word_wrap
  # ========================================================================= #
  def word_wrap(
      i = :word
    )
    # if ::Gtk.use_gtk2?
    #   i = ::Gtk::TextTag::WRAP_WORD
    set_wrap_mode(i)
  end

  # ========================================================================= #
  # === set_text
  #
  # This will set the text-content in a TextView widget.
  # ========================================================================= #
  def set_text(i)
    buffer.set_text(i) # Delegate towards the TextBuffer here.
  end

  # ========================================================================= #
  # === read_only
  # ========================================================================= #
  def read_only
    set_property('editable', false)
  end; alias it_can_not_be_edited read_only # === it_can_not_be_edited

  # ========================================================================= #
  # === append_this_text
  # ========================================================================= #
  def append_this_text(i)
    old_text = buffer.text
    buffer.set_text("#{old_text}#{i}")
  end; alias add_this_text append_this_text # === add_this_text

  # ========================================================================= #
  # === select_everything
  #
  # This method can be used like so, for a Gtk::TextView instance:
  #
  #   @text_view.select_everything(event)
  #
  # ========================================================================= #
  def select_everything(event)
    signal_emit(:select_all, event)
  end

  # ========================================================================= #
  # === visible_cursor
  #
  # This method can be used to make the cursor visible.
  # ========================================================================= #
  def visible_cursor
    self.cursor_visible = true 
  end; alias the_cursor_is_visible visible_cursor # === the_cursor_is_visible

  # ========================================================================= #
  # === the_cursor_is_not_visible
  # ========================================================================= #
  def the_cursor_is_not_visible
    self.cursor_visible = false
  end

end

# =========================================================================== #
# === Gtk.create_view
#
# This creates a TextView object. You must input a buffer for this
# to work. Hence this is different to Gtk.textview which will
# create its own buffer.
# =========================================================================== #
def self.create_view(buffer)
  _ = ::Gtk::TextView.new(buffer)
  _.set_editable(true)
  _.cursor_visible = true
  return _
end

# =========================================================================== #
# === Gtk.textview
#
# This will instantiate a new Gtk::TextView widget.
#
# It will automatically create a new empty buffer too. Since as of April
# 2018, you can also pass in your own text-buffer, as fourth argument
# to this method.
# =========================================================================== #
def self.textview(
    use_this_text_buffer = ::Gtk::TextBuffer.new,
    left_margin          = 2,
    right_margin         = 2,
    is_editable          = true
  )
  case left_margin
  when :default
    left_margin = 2
  end
  case right_margin
  # ========================================================================= #
  # === :default
  # ========================================================================= #
  when :default
    right_margin = 2
  end
  case is_editable
  # ========================================================================= #
  # === :default
  # ========================================================================= #
  when :default
    is_editable = true
  end
  if use_this_text_buffer.nil?
    use_this_text_buffer = ::Gtk::TextBuffer.new # In this case create a new text buffer.
  end
  # ========================================================================= #
  # Convert Strings into TextBuffer instances next.
  # ========================================================================= #
  if use_this_text_buffer.is_a? String
    _ = ::Gtk::TextBuffer.new
    _.set_text(use_this_text_buffer)
    use_this_text_buffer = _
  elsif use_this_text_buffer.is_a? Symbol
    case use_this_text_buffer
    # ======================================================================= #
    # === :default_buffer
    #
    # This variant has to use an empty buffer, so it will not call
    # .set_Text(), unlike the code block above.
    # ======================================================================= #
    when :default_buffer,
         :default
      _ = ::Gtk::TextBuffer.new
      use_this_text_buffer = _
    end
  end
  text_buffer = use_this_text_buffer
  text_view   = ::Gtk::TextView.new(text_buffer)
  text_view.set_editable(true) if is_editable
  text_view.left_margin    = left_margin.to_i
  text_view.right_margin   = right_margin.to_i
  text_view.cursor_visible = true
  # ========================================================================= #
  # === Handle blocks next
  # ========================================================================= #
  if block_given?
    yielded = yield
    if yielded.is_a? Hash
      # ===================================================================== #
      # === :indent
      # ===================================================================== #
      if yielded.has_key? :indent
        text_view.indent = yielded.delete(:indent)
      end
    end
  end
  return text_view # Finally return it here.
end; self.instance_eval { alias textarea               textview } # === Gtk.textarea
     self.instance_eval { alias create_textview        textview } # === Gtk.create_textview
     self.instance_eval { alias return_textarea        textview } # === Gtk.return_textarea
     self.instance_eval { alias create_textarea        textview } # === Gtk.create_textarea
     self.instance_eval { alias text_view              textview } # === Gtk.text_view
     self.instance_eval { alias text_view_widget       textview } # === Gtk.text_view_widget
     self.instance_eval { alias return_textview_widget textview } # === Gtk.return_textview_widget

# =========================================================================== #
# === Gtk.display_this_file
#
# This method will quickly make use of TextBuffer and TextView to
# show the content of a particular local file. In other words,
# this is a bit like File.read() in pure ruby, except that we
# use it in regards to a gtk-widget, so that the user can look
# at this file.
#
# Usage example:
#
#   Gtk.display_this_file('/Depot/j/TODO.md')
#
# =========================================================================== #
def self.display_this_file(this_file)
  if this_file.is_a? Array
    this_file = this_file.join(' ').strip
  end
  if File.exist? this_file
    use_this_text_buffer = ::Gtk::TextBuffer.new
    dataset = File.read(this_file)
    use_this_text_buffer.set_text(
      dataset
    )
    # ========================================================================= #
    # Next we have to return this widget.
    # ========================================================================= #
    Gtk.textview(
      use_this_text_buffer,
      :default,
      :default,
      :default
    )
  else
    puts "No file exists at #{this_file}."
  end
end; self.instance_eval { alias show_this_file display_this_file } # === Gtk.show_this_file

# =========================================================================== #
# === Gtk.display_this_file_and_then_run_it
# =========================================================================== #
def self.display_this_file_and_then_run_it(i)
  require 'gtk_paradise/run'
  r = Gtk.run
  _ = Gtk.display_this_file(i)
  frame = ::Gtk::Frame.new(" #{i} ")
  frame.add(_)
  frame.set_border_width(2)
  r << frame
  r.easy_exit
  r.top_left_then_run
end; self.instance_eval { alias display_this_file_and_then_run display_this_file_and_then_run_it } # === Gtk.display_this_file_and_then_run
     self.instance_eval { alias show_this_file_and_then_run    display_this_file_and_then_run_it } # === Gtk.show_this_file_and_then_run

end