#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# require 'gtk_paradise/core_classes/notebook.rb'
# Gtk.add_notebook
# .focus_on_the_first_tab
# =========================================================================== #
# API documentation:
#
#   https://lazka.github.io/pgi-docs/Gtk-3.0/classes/Notebook.html
#
# =========================================================================== #
module Gtk

class Notebook # === Gtk::Notebook

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  alias get_current_page current_page
  alias current_page?    current_page

  # ========================================================================= #
  # === the_first_tab_is_active
  # ========================================================================= #
  def the_first_tab_is_active
    set_current_page(0)
  end; alias first_is_active                  the_first_tab_is_active # === first_is_active
       alias first_tab_is_active              the_first_tab_is_active # === first_tab_is_active
       alias the_first_notebook_tab_is_active the_first_tab_is_active # === the_first_notebook_tab_is_active

  # ========================================================================= #
  # === do_show_the_tabs
  # ========================================================================= #
  def do_show_the_tabs
    set_show_tabs(true)
  end

  # ========================================================================= #
  # === focus_on_the_first_tab
  # ========================================================================= #
  def focus_on_the_first_tab
    set_page(0)
  end

  # ========================================================================= #
  # === add_tab
  #
  # Note that append_page() can not handle Strings, which is why we have to
  # convert this into a Gtk::Label widget.
  #
  # The first argument to the method .add_tab() must be the widget that has
  # to be added; the second argument to this method is the title for
  # that widget.
  #
  # Usage examples:
  #
  #   .add_tab(gtk_button('Test'), 'A test button')
  #   notebook.add_flexible_tab(widget: gtk_button('OK 1'), title: '#1')
  #
  # ========================================================================= #
  def add_tab(
      widget,
      title = ''
    )
    # ======================================================================= #
    # === widget
    #
    # Handle the situation when the widget, aka the first argument,
    # is a Hash.
    # ======================================================================= #
    if widget.is_a? Hash
      # ===================================================================== #
      # === :title
      # ===================================================================== #
      if widget.has_key? :title
        title = widget[:title]
      end
      # ===================================================================== #
      # === :widget
      # ===================================================================== #
      if widget.has_key? :widget
        widget = widget[:widget]
      end
    end
    if title.is_a? Hash
      # ===================================================================== #
      # === :title
      # ===================================================================== #
      if title.has_key? :title
        title = title[:title]
      end
    end
    if title.is_a? String
      title = ::Gtk::Label.new(title)
    end
    append_page(widget, title)
    set_tab_reorderable(widget, true)
  end; alias append_this_widget add_tab # === append_this_widget
       alias add_flexible_tab   add_tab # === add_flexible_tab
       alias tab1=              add_tab # === tab1=
       alias tab2=              add_tab # === tab2=
       alias tab3=              add_tab # === tab3=
       alias tab4=              add_tab # === tab4=
       alias tab5=              add_tab # === tab5=
       alias tab6=              add_tab # === tab6=
       alias tab7=              add_tab # === tab7=
       alias tab8=              add_tab # === tab8=
       alias tab9=              add_tab # === tab9=

  # ========================================================================= #
  # === remove_the_last_tab
  #
  # This method must be guaranteed to remove the very last tab of a notebook.
  # ========================================================================= #
  def remove_the_last_tab
    _ = n_pages? - 1
    last_tab = get_nth_page(_)
    remove_page(_)
    return last_tab # Return it as well here.
  end

  # ========================================================================= #
  # === append_page_with_index
  #
  # This is currently not in use. The idea was that we can pass an index,
  # which can then be used to remove the tab at hand via a number.
  # ========================================================================= #
  def append_page_with_index(
      widget1,
      widget2,
      index = 0
    )
    widget2.set_index(index)
    append_page(widget1, widget2)
  end

  # ========================================================================= #
  # === all_pages?
  #
  # This method will return all pages. In gtk4 this is no longer needed,
  # so the method serves primarily a use case on gtk3.
  #
  # Actually this returns an incorrect result ...
  # ========================================================================= #
  def all_pages?
    array = []
    (0..100).each {|position|
      result = get_nth_page(position)
      if result
        array << result
      elsif result.nil?
        return array # Early return in this case.
      end
    }
    return array
  end

  # ========================================================================= #
  # === can_be_rearranged
  # ========================================================================= #
  def can_be_rearranged(i = children)
    i.each {|this_tab|
      set_tab_reorderable(this_tab, true)
    }
  end; alias allow_rearrangement_of_the_tabs can_be_rearranged # === allow_rearrangement_of_the_tabs

  # ========================================================================= #
  # === the_tabs_appear_on_top
  #
  # This method will make the tabs appear on top.
  # ========================================================================= #
  def the_tabs_appear_on_top(
      tab_position = :top
    )
    set_tab_pos(tab_position)
  end; alias position_on_top the_tabs_appear_on_top # === position_on_top
       alias on_top          the_tabs_appear_on_top # === on_top

  # ========================================================================= #
  # === first_tab?
  #
  # This method will return the first tab.
  # ========================================================================= #
  def first_tab?
    get_nth_page(0)
  end

  # ========================================================================= #
  # === nth_page
  # ========================================================================= #
  def nth_page(i = 0)
    get_nth_page(i)
  end

  # ========================================================================= #
  # === entry
  #
  # This is like add_tab() but in reverse order.
  # ========================================================================= #
  def entry(
      text        = '',
      this_widget = ::Gtk::Box.new # The argument is just a dummy.
    )
    add_tab(this_widget, text)
  end

  # ========================================================================= #
  # === focus_entry
  #
  # This method allows you to focus on the notebook tab.
  #
  # Some aliases exist to this method; note that the alias name called
  # `focus_on_tab_number` will not necessarily be guaranteed to start
  # counting at 0; we will simply use whatever the method set_page()
  # uses there.
  # ========================================================================= #
  def focus_entry(i = 1)
    set_page(i.to_i)
  end; alias focus_on_this_tab   focus_entry # === focus_on_this_tab
       alias focus_on_tab_number focus_entry # === focus_on_tab_number
       alias focus_on_tab        focus_entry # === focus_on_tab
       alias focus               focus_entry # === focus
       alias activate_this_tab   focus_entry # === activate_this_tab

  # ========================================================================= #
  # === add_page
  #
  # This is similar to add_tab() but the arguments are reversed.
  # ========================================================================= #
  def add_page(a, b)
    add_tab(b, a)
  end

  # ========================================================================= #
  # === position_the_tabs_right
  # ========================================================================= #
  def position_the_tabs_right(
      tab_position = :right
    )
    set_tab_pos(tab_position)
  end; alias position_on_the_right_hand_side position_the_tabs_right # === position_on_the_right_hand_side
       alias position_to_the_right           position_the_tabs_right # === position_to_the_right

  # ========================================================================= #
  # === position_the_tabs_left
  # ========================================================================= #
  def position_the_tabs_left(
      tab_position = :left
    )
    set_tab_pos(tab_position)
  end; alias position_on_the_left_hand_side position_the_tabs_left # === position_on_the_left_hand_side
       alias position_to_the_left           position_the_tabs_left # === position_to_the_left

  # ========================================================================= #
  # === n_pages?
  # ========================================================================= #
  def n_pages?
    n_pages
  end

  # ========================================================================= #
  # === do_show_the_border
  #
  # This method enables "show the border" on a notebook.
  # ========================================================================= #
  def do_show_the_border
    set_show_border(true)
  end; alias do_show_border do_show_the_border # === do_show_border

  # ========================================================================= #
  # === is_scrollable
  # ========================================================================= #
  def is_scrollable
    set_scrollable(true)
  end; alias allow_scrolling is_scrollable # === allow_scrolling

  # ========================================================================= #
  # === enable_drag_and_drop_support
  #
  # This method will require an argument - the widget.
  # ========================================================================= #
  def enable_drag_and_drop_support(
      for_this_widget = self
    )
    set_tab_reorderable(for_this_widget, true)
  end

  # ========================================================================= #
  # === useful_defaults
  # ========================================================================= #
  def useful_defaults
    do_show_the_border
    do_show_the_tabs
    # is_scrollable
    # focus_on_the_first_tab
    # enable_drag_and_drop_support # ← This can not be done as no tab exists yet.
  end

  # ========================================================================= #
  # === remove_tab
  # ========================================================================= #
  alias remove_tab remove_page

end

# =========================================================================== #
# === Gtk.add_reorderable_notebook
# =========================================================================== #
def self.add_reorderable_notebook(*widgets, &block)
  ::Gtk.add_notebook(widgets) { :reorderable }
end

# =========================================================================== #
# === Gtk.add_notebook
#
# Add a Gtk::Notebook via this method.
#
# The first input argument to this method should contain the sub-widgets
# that are part of this Notebook.
#
# Usage example:
#
#   Gtk.add_notebook(label1, label2, label3) { :reorderable }
#
# =========================================================================== #
def self.add_notebook(*widgets, &block) # ← These are the subwidgets.
  enable_drag_and_drop_support_for_this_widget = false
  if block_given?
    yielded = yield
    case yielded
    # ======================================================================= #
    # === :enable_drag_and_support
    # ======================================================================= #
    when :enable_drag_and_support,
         :reorderable
      enable_drag_and_drop_support_for_this_widget = true
    end
  end
  notebook = ::Gtk::Notebook.new
  notebook.set_scrollable(true)
  notebook.set_show_tabs(true)
  widgets.flatten.each {|widget|
    name = widget.class.to_s
    if name.include? '::'
      name = name.split('::').last
    end
    notebook.append_page(widget, ::Gtk::Label.new(name))
    if enable_drag_and_drop_support_for_this_widget
      notebook.enable_drag_and_drop_support(widget)
    end
  }
  return notebook
end; self.instance_eval { alias create_notebook add_notebook } # === Gtk.create_notebook

# =========================================================================== #
# === Gtk.notebook
# =========================================================================== #
def self.notebook
  ::Gtk::Notebook.new
end

end