#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::Grid
# =========================================================================== #
# require 'gtk_paradise/core_classes/grid.rb'
# =========================================================================== #
module Gtk

class Grid # === Gtk::Grid

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === two_width
  # ========================================================================= #
  def two_width(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 2,
      height        = 1,
      &block
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    # === :default
    when :default
      top_position = top_position?
    end
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    width.times { Gtk::Grid.increment_left_position }
    return gtk_widget_that_is_to_be_placed
  end

  # ========================================================================= #
  # === @full_width
  # ========================================================================= #
  @full_width = 2

  # ========================================================================= #
  # === Gtk::Grid.full_width?
  # ========================================================================= #
  def self.full_width?
    @full_width
  end

  # ========================================================================= #
  # === Gtk::Grid.full_width=
  # ========================================================================= #
  def self.full_width=(i = 2)
    @full_width = i
  end

  # ========================================================================= #
  # === full_row
  #
  # This method attempts to simulate a full line in a grid, but for now
  # this is just hardcoded to a width of 2. At a later time this may
  # become more sophisticated than that hardcoded value.
  # ========================================================================= #
  def full_row(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = full_width?,
      height        = 1,
      &block
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      top_position = top_position?
    end
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :new_row
      # ===================================================================== #
      when :new_row
        new_row
      end
    end
    return new_row
  end; alias full_line full_row # === full_line

  # ========================================================================= #
  # === reset_the_counter
  #
  # Delegate towards Gtk::Grid.reset() here.
  # ========================================================================= #
  def reset_the_counter
    Gtk::Grid.reset
  end; alias reset_all_counters reset_the_counter # === reset_all_counters
       alias reset              reset_the_counter # === reset
       alias clear              reset_the_counter # === clear

  # ========================================================================= #
  # === full_width?
  # ========================================================================= #
  def full_width?
    Gtk::Grid.full_width?
  end

  # ========================================================================= #
  # === append_this_widget
  #
  # For now this is simply an alias to left(). However had, the meaning
  # is different - here we simply want to call an "append" functionality
  # without having to worry about left, middle or right.
  # ========================================================================= #
  def append_this_widget(i)
    left(i)
  end; alias append_this append_this_widget # === append_this

  # ========================================================================= #
  # === cellspacing=
  #
  # Since as of October 2023 this uses a default value (currently 8), to
  # provide some default spacing - see the last aliased method name. I
  # found it easier to memorize it like that.
  # ========================================================================= #
  def cellspacing=(
      i = 8
    )
    set_column_spacing(i)
    set_row_spacing(i)
  end; alias spacing=        cellspacing= # === spacing=
       alias default_spacing cellspacing= # === default_spacing
       alias uniform_spacing cellspacing= # === uniform_spacing

  # ========================================================================= #
  # === text
  #
  # This is, kind of, an alias towards .left(), but assuming a Gtk::Label
  # as its content.
  # ========================================================================= #
  def text(i)
    if i.is_a? String
      i = ::Gtk::Label.new(i)
    end
    left(i)
  end

  # ========================================================================= #
  # === add_then_new_row
  # ========================================================================= #
  def add_then_new_row(i)
    full_row(i)
    new_row
  end

  # ========================================================================= #
  # === six_width
  # ========================================================================= #
  def six_width(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 6,
      height        = 1,
      &block
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      top_position = top_position?
    end
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    width.times { Gtk::Grid.increment_left_position }
    return gtk_widget_that_is_to_be_placed
  end

  # ========================================================================= #
  # === five_width
  # ========================================================================= #
  def five_width(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 5,
      height        = 1,
      &block
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    when :default
      top_position = top_position?
    end
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    width.times { Gtk::Grid.increment_left_position }
    return gtk_widget_that_is_to_be_placed
  end

  # ========================================================================= #
  # === Gtk::Grid.reset_the_grid_left_position
  # ========================================================================= #
  def self.reset_the_grid_left_position
    @hash_grid_positions[:left_position] = 0
  end

  # ========================================================================= #
  # === Gtk::Grid.reset                                           (reset tag)
  #
  # This method can be used to reset the counters defined above again.
  # ========================================================================= #
  def self.reset
    reset_the_grid_left_position
    reset_the_grid_top_position
  end

  # ========================================================================= #
  # === reset_left_position
  # ========================================================================= #
  def reset_left_position
    ::Gtk::Grid.reset_the_grid_left_position
  end

  # ========================================================================= #
  # === set_left                                                   (left tag)
  #
  # This method will position the gtk-widget on the left side of the
  # gtk-grid layout, by default.
  #
  # The first argument to this method must be the gtk-widget that is to
  # be placed in the grid-cell. After it was placed the left-position
  # counter will be incremented.
  # ========================================================================= #
  def set_left(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 1,
      height        = 1
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      top_position  = top_position?
    end
    # ======================================================================= #
    # .attach(child_widget, left_pos, top_pos, width, height)
    # ======================================================================= #
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    Gtk::Grid.increment_left_position
  end; alias left set_left # === left

  # ========================================================================= #
  # === set_middle                                               (middle tag)
  # ========================================================================= #
  def set_middle(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 1,
      height        = 1
    )
    case top_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      top_position = top_position?
    end
    # ======================================================================= #
    # .attach(child_widget, left_pos, top_pos, width, height)
    # ======================================================================= #
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    # ======================================================================= #
    # We also have to increment the left position of the grid afterwards.
    # Otherwise we would get incorrect left-positions.
    # ======================================================================= #
    left_position += 1
    ::Gtk::Grid.set_grid_left_position(left_position)
  end; alias middle set_middle # === middle

  # ========================================================================= #
  # === line_spacing=
  # ========================================================================= #
  def line_spacing=(i = 5)
    set_column_spacing(i)
  end

  # ========================================================================= #
  # === spacing1=
  #
  # This will pad from left-to-right aka horizontal, so it is applicable
  # when you put widgets next to one another.
  # ========================================================================= #
  def spacing1=(i)
    set_column_spacing(i)
  end; alias columnspacing= spacing1= # === columnspacing=

  # ========================================================================= #
  # === spacing2=
  # ========================================================================= #
  def spacing2=(i)
    set_row_spacing(i)
  end; alias rowspacing= spacing2= # === rowspacing=

  # ========================================================================= #
  # === new_row
  #
  # This will add a new row to the grid.
  # ========================================================================= #
  def new_row
    ::Gtk::Grid.increment_the_grid_top_position
    ::Gtk::Grid.reset_the_grid_left_position # Must reset the left-counter as well.
  end; alias add_new_row new_row # === add_new_row
       alias new_line    new_row # === new_line
       alias newrow      new_row # === newrow

  # ========================================================================= #
  # === left_right
  #
  # This method combines left() and right() into one aggregate method.
  # ========================================================================= #
  def left_right(
      argument_to_the_left_side,
      argument_to_the_right_side
    )
    left(argument_to_the_left_side)
    right(argument_to_the_right_side)
  end

  # ========================================================================= #
  # === @hash_grid_positions
  #
  # This hash can be used to specify positions in the grid 2D layout.
  # Elements can be placed accordingly - and automatically - based
  # on the values stored in this hash.
  # ========================================================================= #
  @hash_grid_positions = {
    left_position: 0, # Initial value must be 0.
    top_position:  0  # Initial value must be 0.
  }

  # ========================================================================= #
  # === Gtk::Grid.reset_the_grid_top_position
  # ========================================================================= #
  def self.reset_the_grid_top_position
    @hash_grid_positions[:top_position] = 0
  end

  # ========================================================================= #
  # === Gtk::Grid.increment_left_position
  #
  # This method will increment the gtk-grid left position.
  # ========================================================================= #
  def self.increment_left_position(by = 1)
    @hash_grid_positions[:left_position] += by
  end; self.instance_eval { alias increment_the_left_position      increment_left_position } # === Gtk::Grid.increment_the_left_position
       self.instance_eval { alias increment_the_grid_left_position increment_left_position } # === Gtk::Grid.increment_the_grid_left_position

  # ========================================================================= #
  # === Gtk::Grid.increment_top_position
  #
  # It is better to call .new_row() rather than this method.
  # ========================================================================= #
  def self.increment_top_position(by = 1)
    @hash_grid_positions[:top_position] += by
  end; self.instance_eval { alias increment_the_top_position      increment_top_position } # === Gtk::Grid.increment_the_top_position
       self.instance_eval { alias increment_the_grid_top_position increment_top_position } # === Gtk::Grid.increment_the_grid_top_position

  # ========================================================================= #
  # === Gtk::Grid.left_position?
  #
  # Toplevel-query method to find out what the left_position's current
  # value is.
  # ========================================================================= #
  def self.left_position?
    @hash_grid_positions[:left_position]
  end; self.instance_eval { alias grid_left_position? left_position? } # === Gtk::Grid.grid_left_position?

  # ========================================================================= #
  # === Gtk::Grid.set_left_position
  # ========================================================================= #
  def self.set_left_position(by = 1)
    @hash_grid_positions[:left_position] = by
  end; self.instance_eval { alias set_grid_left_position set_left_position } # === Gtk::Grid.set_grid_left_position

  # ========================================================================= #
  # === no_row_spacing
  # ========================================================================= #
  def no_row_spacing
    set_row_spacing(0)
  end

  # ========================================================================= #
  # === right_double_width
  # ========================================================================= #
  def right_double_width(i)
    set_right(i, :default, :default, 2, 1) 
  end

  # ========================================================================= #
  # === no_column_spacing
  # ========================================================================= #
  def no_column_spacing
    set_column_spacing(0)
  end

  # ========================================================================= #
  # === debug                                                     (debug tag)
  #
  # A small debug-related method.
  # ========================================================================= #
  def debug
    "gtk-grid left position is: #{left_position?}\n"\
    "gtk-grid top  position is: #{top_position?}\n"
  end; alias counter? debug # === counter?

  # ========================================================================= #
  # === left_position?
  #
  # Simply delegate towards Gtk::Grid.left_position? via this method.
  # ========================================================================= #
  def left_position?
    Gtk::Grid.left_position?
  end; alias gtk_left_position?  left_position? # === gtk_left_position?
       alias grid_left_position? left_position? # === grid_left_position?
       alias left_counter?       left_position? # === left_counter?

  # ========================================================================= #
  # === Gtk::Grid.top_position?
  # ========================================================================= #
  def self.top_position?
    @hash_grid_positions[:top_position]
  end; self.instance_eval { alias grid_top_position? top_position? } # === Gtk::Grid.grid_top_position?

  # ========================================================================= #
  # === top_position?
  #
  # Simply delegate towards Gtk::Grid.top_position?.
  # ========================================================================= #
  def top_position?
    ::Gtk::Grid.grid_top_position?
  end; alias grid_top_position? top_position? # === grid_top_position?

  # ========================================================================= #
  # === set_right                                                 (right tag)
  #
  # This will position on the right side, by default.
  #
  # Note that this method will also increment the top position. If you don't
  # want that, use .middle() instead - see this method being defined in
  # this file here. The way to increment properly is by simply calling the
  # method .new_row().
  # ========================================================================= #
  def set_right(
      gtk_widget_that_is_to_be_placed,
      left_position = left_position?,
      top_position  = top_position?,
      width         = 1,
      height        = 1
    )
    case left_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      left_position = left_position?
    end
    case top_position
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default
      top_position  = top_position?
    end
    # ======================================================================= #
    # .attach(child_widget, left_pos, top_pos, width, height)
    # ======================================================================= #
    attach(
      gtk_widget_that_is_to_be_placed,
      left_position,
      top_position,
      width,
      height
    )
    new_row
  end; alias right set_right # === right

end

# =========================================================================== #
# === Gtk.grid_counter?
# =========================================================================== #
def self.grid_counter?
  ::Gtk::Grid.top_position?
end

# =========================================================================== #
# === Gtk.grid
#
# Toplevel method to create a new grid-widget.
# =========================================================================== #
def self.grid(i = nil)
  ::Gtk.reset_the_grid_counter
  grid = ::Gtk::Grid.new
  if i.is_a?(String) and i.include?('x')
    splitted = i.split('x')
    ::Gtk::Grid.full_width = splitted.first.to_i
  end
  return grid
end

end