#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::CheckButton
# =========================================================================== #
# require 'gtk_paradise/core_classes/check_button.rb'
# =========================================================================== #
module Gtk

class CheckButton # === Gtk::CheckButton

  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'

  # ========================================================================= #
  # === checked?
  # ========================================================================= #
  alias checked?      active? # === checked?

  # ========================================================================= #
  # === is_it_active?
  # ========================================================================= #
  alias is_it_active? active? # === is_it_active?

  # ========================================================================= #
  # === is_active?
  # ========================================================================= #
  alias is_active?    active? # === is_active?

  # ========================================================================= #
  # === switch_state
  #
  # This method can be used to simply switch the state.
  # ========================================================================= #
  def switch_state
    if is_it_active?
      set_active(false)
    else
      set_active(true)
    end
  end

  # ========================================================================= #
  # === on_toggled
  #
  # This is the method that will be called when the user clicks on its
  # active area.
  # ========================================================================= #
  def on_toggled(&block)
    signal_connect(:toggled, &block)
  end; alias toggle     on_toggled # === toggle
       alias on_changed on_toggled # === on_changed

  # ========================================================================= #
  # === is_active
  # ========================================================================= #
  def is_active
    set_active(true)
  end; alias mark_as_checked is_active # === mark_as_checked
       alias is_now_active   is_active # === is_now_active
       alias is_checked      is_active # === is_checked
       alias is_checked?     is_active # === is_checked?

  # ========================================================================= #
  # === mark_as_unchecked
  # ========================================================================= #
  def mark_as_unchecked
    set_active(false)
  end; alias is_inactive     mark_as_unchecked # === is_inactive
       alias is_now_inactive mark_as_unchecked # === is_now_inactive

end

# =========================================================================== #
# === Gtk.check_button
#
# This method is an easier toplevel way to create a new Gtk::CheckButton
# instance.
#
# See here for official documentation of Gtk::CheckButton:
#
#   https://ruby-gnome2.osdn.jp/hiki.cgi?Gtk%3A%3ACheckButton
#
# Usage example:
#
#   check_button = Gtk.check_button('Cat or Dog')
#
# =========================================================================== #
def self.check_button(
    i             = '',
    use_underline = :do_not_underline,
    &block
  )
  is_checked = false
  case use_underline
  # ========================================================================= #
  # === :do_underline
  # ========================================================================= #
  when :do_underline
    use_underline = true
  # ========================================================================= #
  # === :do_not_underline
  # ========================================================================= #
  when :do_not_underline
    use_underline = false
  end
  case i
  # ========================================================================= #
  # === :is_checked
  # ========================================================================= #
  when :is_checked
    i = ''
    is_checked = true
  end
  # ========================================================================= #
  # === Handle gtk2-versions first
  #
  # Disabled as of October 2023.
  # ========================================================================= #
  # if Gtk.are_we_using_gtk2?
  #   _ = ::Gtk::CheckButton.new(i, use_underline)
  # else # else assume ruby-gtk3.
    _ = ::Gtk::CheckButton.new(i)
    _.use_underline = true if use_underline # Handle it explicitly in this case.
  # end
  # ========================================================================= #
  # === Handle block arguments given to this method
  # ========================================================================= #
  if block_given?
    yielded = yield
    case yielded
    # ======================================================================= #
    # === :is_checked
    # ======================================================================= #
    when :is_checked
      is_checked = true
    end
  end
  _.is_checked if is_checked
  return _
end; self.instance_eval { alias create_checkbox  check_button } # === Gtk.create_checkbox
     self.instance_eval { alias create_check_box check_button } # === Gtk.create_check_box

end

if __FILE__ == $PROGRAM_NAME
  require 'gtk_paradise/run.rb'  
  r = Gtk.run
  r << Gtk::CheckButton.new
  r.top_left_run
end