#!/usr/bin/ruby -w
# Encoding: UTF-8
# frozen_string_literal: true
# =========================================================================== #
# === Gtk::BaseModule
#
# This convenient module can be included into other gtk-related classes.
#
# Once this module is included, you can use its simpler accessor-methods,
# such as gtk_button or create_button() to create a Gtk::Button.new
# instance. Or, even simpler, just use button() or button without the
# () parens.
#
# These methods typically will have the prefix "gtk_", which is quite
# similar to the C-methods gtk is originally written in. Most of these
# classes will work without the "gtk_" prefix, but sometimes this may
# lead to conflicts, so as a general rule, it may be better to include
# the "gtk_" prefix if in doubt. But you can also try to see whether
# a "pure" variant without "gtk_" prefix may work for your use case -
# ideally we can have a pure ruby-gtk DSL one day that can be used
# for other toolkits as well. But for now, it is recommended to be
# more conservative and retain the "gtk_" prefix, unless you know
# exactly what you are doing.
#
# The .rb file here will load all the other .rb files residing under
# gtk_paradise/base_module/*. It is normally required indirectly,
# via the line shown below this comment.
#
# This file also contains "html-tags" that may be used - and useful - for
# ruby-gtk applications.
#
# Some of these html-tags are tested via ruby-code. For example, the
# headers (h1, h2, h3, h4, h5 and h6) are tested in the file called
# testing_the_html_pseudo_tags.rb, in the test/gtk3/ subdirectory.
# =========================================================================== #
# require 'gtk_paradise/requires/require_the_base_module.rb'
# include ::Gtk::BaseModule
# =========================================================================== #
module Gtk

module BaseModule # === Gtk::BaseModule

  alias ee print

  require 'gtk_paradise/project/project.rb'
  require 'gtk_paradise/constants/constants.rb'
  require 'gtk_paradise/base_module/experimental.rb'
  require 'gtk_paradise/requires/require_the_base_module_widgets.rb'

  # ========================================================================= #
  # Pull in Gtk::InternalHashModule next:
  # ========================================================================= #
  require 'gtk_paradise/base_module/internal_hash_module/internal_hash_module.rb'
  include ::Gtk::InternalHashModule

  require 'gtk_paradise/fonts/fonts.rb' # This will also pull in 'gtk_paradise/constants/fonts.rb'

  # ========================================================================= #
  # Next, add toplevel_methods/ files and other toplevel files.
  # ========================================================================= #
  require 'gtk_paradise/toplevel_methods/css.rb'
  require 'gtk_paradise/toplevel_methods/determine_which_gtk_version_is_in_use.rb'
  require 'gtk_paradise/toplevel_methods/toplevel_methods.rb'
  require 'gtk_paradise/toplevel_methods/keys.rb'

  require 'gtk_paradise/core_classes/widget.rb' # Pull this one in first.
  # ========================================================================= #
  # Pull in the all Gtk core classes modifications next.
  # ========================================================================= #
  require 'gtk_paradise/core_classes/box.rb'
  require 'gtk_paradise/core_classes/button.rb'
  require 'gtk_paradise/core_classes/button_box.rb'
  require 'gtk_paradise/core_classes/calendar.rb'
  require 'gtk_paradise/core_classes/combo_box_text.rb'
  require 'gtk_paradise/core_classes/entry.rb'
  require 'gtk_paradise/core_classes/entry_completion.rb'
  require 'gtk_paradise/core_classes/event_box.rb'
  require 'gtk_paradise/core_classes/event_button.rb'
  require 'gtk_paradise/core_classes/event_key.rb'
  require 'gtk_paradise/core_classes/grid.rb'
  require 'gtk_paradise/core_classes/hbox.rb'
  require 'gtk_paradise/core_classes/header_bar.rb'
  require 'gtk_paradise/core_classes/image.rb'
  require 'gtk_paradise/core_classes/list_store.rb'
  require 'gtk_paradise/core_classes/menu.rb'
  require 'gtk_paradise/core_classes/menu_bar.rb'
  require 'gtk_paradise/core_classes/paned.rb'
  require 'gtk_paradise/core_classes/popover.rb'
  require 'gtk_paradise/core_classes/scale_button.rb'
  require 'gtk_paradise/core_classes/spinner.rb'
  require 'gtk_paradise/core_classes/status_icon.rb'
  require 'gtk_paradise/core_classes/text_buffer.rb'
  require 'gtk_paradise/core_classes/text_view.rb'
  require 'gtk_paradise/core_classes/toggle_button.rb'
  require 'gtk_paradise/core_classes/tree_selection.rb'
  require 'gtk_paradise/core_classes/tree_view.rb'
  require 'gtk_paradise/core_classes/tree_view_column.rb'
  require 'gtk_paradise/core_classes/window.rb'

  # ========================================================================= #
  # Add the very important emoji support next :)
  # ========================================================================= #
  require 'gtk_paradise/emojis/emojis.rb'

  # ========================================================================= #
  # === IMG_DIR
  #
  # This constant refers to the gtk-paradise's projects main image directory.
  # ========================================================================= #
  IMG_DIR = "#{::Gtk.project_base_directory?}images/"

  # ========================================================================= #
  # === @default_gtk_adjustment
  #
  # This is the default gtk-adjustment to use, if a gtk-adjustment is
  # to be used. In other words, the default value for a Gtk::Adjustment.
  # ========================================================================= #
  @default_gtk_adjustment = 
    ::Gtk::Adjustment.new(
      0.0,
      0.0,
      105.0,
      0.2,
      1.0,
      1.0
    )

  # ========================================================================= #
  # === Gtk::BaseModule::HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS
  #
  # This hash will set the corresponding font-size, as a percentage value.
  # The default is 100%, so all headers must be larger than the 100%
  # minimum.
  #
  # The following link has some percentage values:
  #
  #   https://www.metahead.com/converting-font-sizes-from-pixel-point-em-percent/
  #
  # So 12pt is 100%, 18pt is 150%, 36pt is 300%.
  #
  # And on this page:
  #
  #   https://zuga.net/articles/html-heading-elements/
  #
  # The listing for the headers defined in that Hash is:
  #
  #   32px 24px 18.72px 16px 13.28px 10.72px
  #
  # ========================================================================= #
  HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS = {
    h1: '200%', # 32    px
    h2: '180%', # 24    px
    h3: '150%', # 18.72 px
    h4: '130%', # 16    px
    h5: '125%', # 13.28 px
    h6: '115%'  # 10.72 px
  }

  # ========================================================================= #
  # Try to pull in the colours-gem next:
  # ========================================================================= #
  begin
    require 'colours/html_colours'
  rescue LoadError
  end

  # ========================================================================= #
  # === append_project_css_file
  #
  # This will find the file called project.css and then append it
  # onto the main CSS in use.
  #
  # Since as of August 2022 this method will properly handle Arrays
  # given as input.
  # ========================================================================= #
  def append_project_css_file(
      project_internal_css_files = 
        [
          ::Gtk.ruby_sitedir?+
          ::Gtk.infer_the_name_of_the_project_based_on_this_directory(configuration?[1])+
          '/css/project.css'
        ]
    )
    if project_internal_css_files.is_a? Array
      project_internal_css_files.each {|this_file|
        append_project_css_file(this_file)
      }
    else
      if File.exist? project_internal_css_files
        dataset = File.read(project_internal_css_files)
        append_these_new_CSS_rules(dataset)
      else
        e "No file exists at #{project_internal_css_files}."
      end
    end
  end; alias use_project_css_file    append_project_css_file # === use_project_css_file
       alias append_project_CSS_file append_project_css_file # === append_project_CSS_file

  # ========================================================================= #
  # === append_onto_the_CSS_rules                                (append tag)
  #
  # Use this method when you wish to append more CSS rules.
  # ========================================================================= #
  def append_onto_the_CSS_rules(i)
    if debug?
      e 'Adding the following CSS rule:'
      e '-' * 80
      e i
      e '-' * 80
    end
    if i and i.is_a?(String) and File.exist?(i)
      i = File.read(i)
    end
    ::Gtk.append_onto_the_CSS_rules(i)
  end; alias append_css                           append_onto_the_CSS_rules # === append_css
       alias add_to_the_css_string                append_onto_the_CSS_rules # === add_to_the_css_string
       alias append_these_CSS_rules               append_onto_the_CSS_rules # === append_these_CSS_rules
       alias append_these_custom_CSS_rules        append_onto_the_CSS_rules # === append_these_custom_CSS_rules
       alias add_and_apply_these_custom_CSS_rules append_onto_the_CSS_rules # === add_and_apply_these_custom_CSS_rules
       alias add_this_css                         append_onto_the_CSS_rules # === add_this_css
       alias add_css                              append_onto_the_CSS_rules # === add_this_css
       alias more_css                             append_onto_the_CSS_rules # === more_css
       alias more_CSS                             append_onto_the_CSS_rules # === more_CSS
       alias more_CSS_from                        append_onto_the_CSS_rules # === more_CSS_from
       alias append_these_new_CSS_rules           append_onto_the_CSS_rules # === append_these_new_CSS_rules
       alias add_custom_CSS_rules                 append_onto_the_CSS_rules # === add_custom_CSS_rules
       alias add_these_custom_CSS_rules           append_onto_the_CSS_rules # === add_these_custom_CSS_rules
       alias adhoc_css                            append_onto_the_CSS_rules # === adhoc_css
       alias in_file_css                          append_onto_the_CSS_rules # === in_file_css
       alias more_CSS_then_apply_it               append_onto_the_CSS_rules # === more_CSS_then_apply_it

  # ========================================================================= #
  # === hcentered_entry
  # ========================================================================= #
  def hcentered_entry(
      optional_text = nil
    )
    optional_text = optional_text.to_s
    new_entry = entry
    unless optional_text.empty?
      new_entry.set_text(optional_text)
    end
    new_entry.align_to_center
    return new_entry
  end

  # ========================================================================= #
  # === try_to_require_the_open_gem
  # ========================================================================= #
  def try_to_require_the_open_gem
    begin
      require 'open'
    rescue LoadError; end
  end

  # ========================================================================= #
  # === e                                                             (e tag)
  # ========================================================================= #
  def e(i = '')
    puts i
  end

  # ========================================================================= #
  # === gtk_centered_spin_button
  # ========================================================================= #
  def gtk_centered_spin_button(
      start_position =   0,
      end_position   = 100,
      increment_size =   1,
      &block
    )
    _ = gtk_spin_button(
      start_position,
      end_position,
      increment_size,
      &block
    )
    _.center # Center the spin-button here.
    return _
  end; alias centered_spin_button gtk_centered_spin_button # === centered_spin_button

  # ========================================================================= #
  # === hbox_text
  #
  # This method combines a hbox, and then text inside that hbox. It was
  # added in September 2022.
  # ========================================================================= #
  def hbox_text(
      i = ''
    )
    if use_gtk3?
      return hbox.add(text(i))
    elsif use_gtk4?
      return hbox.append(text(i))
    end
  end

  # ========================================================================= #
  # === default_application
  #
  # This method will only work for ruby-gtk4.
  # ========================================================================= #
  def default_application
    ::Gtk.default_application {{
      widget_to_use: self
    }}
  end

  # ========================================================================= #
  # === populate_the_internal_hash_with_the_default_values
  #
  # In order for this method to work, @internal_hash must exist.
  # ========================================================================= #
  def populate_the_internal_hash_with_the_default_values
    # ======================================================================= #
    # === :title
    #
    # This variable will keep track of the title of the widget at hand.
    # ======================================================================= #
    @internal_hash[:title]  = nil
    # ======================================================================= #
    # === :namespace
    # ======================================================================= #
    @internal_hash[:namespace]  = nil
    # ======================================================================= #
    # === :buttons
    # ======================================================================= #
    @internal_hash[:buttons] = []
    # ======================================================================= #
    # === :check_buttons
    # ======================================================================= #
    @internal_hash[:check_buttons] = []
    # ======================================================================= #
    # === :toggle_buttons
    # ======================================================================= #
    @internal_hash[:toggle_buttons] = []
    # ======================================================================= #
    # === :entries
    # ======================================================================= #
    @internal_hash[:entries] = []
    # ======================================================================= #
    # === :label
    # ======================================================================= #
    @internal_hash[:label] = []
    # ======================================================================= #
    # === :debug
    # ======================================================================= #
    @internal_hash[:debug] = false
    # ======================================================================= #
    # === :width
    # ======================================================================= #
    @internal_hash[:width]  = 0
    # ======================================================================= #
    # === :height
    # ======================================================================= #
    @internal_hash[:height] = 0
  end

  # ========================================================================= #
  # === reset_the_internal_variables
  # ========================================================================= #
  def reset_the_internal_variables(
      i = nil
    )
    # ======================================================================= #
    # === @configuration
    #
    # This variable keeps track of any "meta-data" associated with any
    # class that includes this module. In the past this variable was
    # called @config_file, but @configuration as name is, in my opinion,
    # more appropriate.
    #
    # By default @configuration will be set to nil, so it is up to
    # the responsibility of the subclass-implementation to initialize
    # this variable properly, via their own reset() method usually.
    #
    # A typical value for @configuration may look like this:
    #
    #   @configuration = [true, __dir__, NAMESPACE]
    #
    # - The first entry determines whether the class will make use of
    #   a config file.
    #
    # - The second entry gives us information about the base directory
    #   of the application at hand.
    #
    # - The last entry will yield the Namespace, such as Foo::Bar.
    #
    # ======================================================================= #
    unless i == :do_not_reset_the_configuration_variable
      @configuration = nil
    end
    # ======================================================================= #
    # Reset the toplevel-defined CSS rules first, so that we can have a
    # clean state again:
    # ======================================================================= #
    ::Gtk.reset_css_rules
    # ======================================================================= #
    # === @internal_hash
    #
    # This is the Hash that can be used to keep track of internal-states,
    # such as all buttons and so forth..
    # ======================================================================= #
    reset_the_internal_hash
    populate_the_internal_hash_with_the_default_values
    # ======================================================================= #
    # The next method-call may have to be re-evaluated in the future
    # (Feb 2021).
    # ======================================================================= #
    # check_for_a_manifest_yaml_file
  end

  # ========================================================================= #
  # === main_file?
  #
  # This is a wrapper over the toplevel-method that normally keeps track
  # of the "active" file.
  # ========================================================================= #
  def main_file?(
      i = ::Gtk.main_file?
    )
    i
  end; alias assigned_main_file? main_file? # === assigned_main_file?
       alias filename?           main_file? # === filename?

  # ========================================================================= #
  # === gtk_file_chooser_button
  #
  # The first argument to this method is the default text that is to
  # appear on the file-chooser button.
  #
  # Currently (as of November 2020) you have to handle the click-event
  # in your application.
  #
  # You can use the following line of code for this:
  #
  #   file_chooser = gtk_file_chooser_button
  #   file_chooser.on_click { create_a_new_file_chooser_dialog(this_widget, use_this_text) }
  #
  # ========================================================================= #
  def gtk_file_chooser_button(
      use_this_text = 'Open File',
      options_hash = {
        use_tooltip_hint: true
      },
      &block
    )
    # ======================================================================= #
    # === Handle blocks first
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      if yielded.is_a? Hash
        options_hash.merge!(yielded)
      end
    end
    if options_hash.is_a?(Hash) and
       options_hash.has_key?(:is_an_accelerator) and
      (options_hash[:is_an_accelerator] == true)
      use_this_text = use_this_text.dup
      use_this_text.prepend('_')
    end
    _ = create_button(use_this_text)
    # ======================================================================= #
    # === Handle the Hash, the assumed second argument to this method
    # ======================================================================= #
    if options_hash.is_a? Hash
      # ===================================================================== #
      # === :use_tooltip_hint
      #
      # This is the default handler for tooltips.
      # ===================================================================== #
      if options_hash[:use_tooltip_hint] and (options_hash[:use_tooltip_hint] == true)
        _.hint = 'Click on this button to open a local file.'
      end
      # ===================================================================== #
      # === :connect_to_this_widget
      #
      # This will tie the open-file functionality to a particular widget.
      # ===================================================================== #
      if options_hash.has_key? :connect_to_this_widget
        connect_to_this_widget = options_hash[:connect_to_this_widget]
        _.on_clicked {
          use_this_filename = new_open_file_dialog {{
            extra_directory: '/Depot/ISOs/'
          }}
          if use_this_filename and File.exist? use_this_filename
            connect_to_this_widget.set_text(use_this_filename.to_s)
          end
        }
      end
    end
    return _ # And finally return it here.
  end; alias gtk_file_open_button       gtk_file_chooser_button # === gtk_file_open_button
       alias return_file_chooser_button gtk_file_chooser_button # === return_file_chooser_button
       alias default_choose_filer       gtk_file_chooser_button # === default_choose_filer
       alias file_chooser_button        gtk_file_chooser_button # === file_chooser_button
       alias create_file_chooser_button gtk_file_chooser_button # === create_file_chooser_button
       alias file_open_button           gtk_file_chooser_button # === file_open_button

  # ========================================================================= #
  # === parse_this_file_into_a_table
  #
  # Note that Gtk::Table has been deprecated. Use GtkGrid instead. It
  # provides the same capabilities as GtkTable for arranging widgets
  # in a rectangular grid, but does support height-for-width geometry
  # management.
  #
  # Usage example:
  #
  #   parse_into_table '/Depot/j/animals.yml'
  #
  # ========================================================================= #
  def parse_this_file_into_a_table(this_file)
    if this_file and File.exist?(this_file)
      table = create_grid
      table.uniform_spacing(8)
      table.pad8px
      if this_file.end_with?('.yml')
        dataset = YAML.load_file(this_file)
      else
        dataset = File.readlines(this_file)
      end
      if dataset.is_a? Hash
        dataset = dataset.to_a.flatten
      end
      dataset.flatten! if dataset.respond_to?(:flatten!)
      dataset.each_with_index {|entry, index|
        entry = entry.to_s
        case index % 2
        when 0
          table.left(text(entry))
        when 1
          table.right(text(entry))
        end
      }
      return table
    end
  end; alias parse_into_table parse_this_file_into_a_table # === parse_into_table

  # ========================================================================= #
  # === return_default_grid
  # ========================================================================= #
  def return_default_grid
    return gtk_grid { :default }
  end; alias default_grid        return_default_grid # === default_grid
       alias default_gtk_grid    return_default_grid # === default_gtk_grid
       alias gtk_default_grid    return_default_grid # === gtk_default_grid
       alias create_default_grid return_default_grid # === create_default_grid

  # ========================================================================= #
  # === modify_label
  # ========================================================================= #
  def modify_label(
      name,
      colour  = 'darkblue',
      justify = nil
    )
    ::Gtk.modify_label(name, colour, justify)
  end

  # ========================================================================= #
  # === return_scale_button
  # ========================================================================= #
  def return_scale_button(
      i = :default_icon
    )
    ::Gtk.return_scale_button(i)
  end; alias scale_button     return_scale_button # === scale_button
       alias gtk_scale_button return_scale_button # === gtk_scale_button

  # ========================================================================= #
  # === gtk_bold_text_left_aligned
  # ========================================================================= #
  def gtk_bold_text_left_aligned(
      use_this_text = '',
      use_mnemonics = false
    )
    _ = create_bold_label(use_this_text, use_mnemonics)
    _.left_align
    return _
  end; alias gtk_bold_label_left_aligned gtk_bold_text_left_aligned # === gtk_bold_label_left_aligned
       alias bold_label_left_aligned     gtk_bold_text_left_aligned # === bold_label_left_aligned
       alias bold_text_left_aligned      gtk_bold_text_left_aligned # === bold_text_left_aligned
       alias left_aligned_bold_text      gtk_bold_text_left_aligned # === left_aligned_bold_text

  # ========================================================================= #
  # === left_aligned_selectable_text
  # ========================================================================= #
  def left_aligned_selectable_text(i = nil)
    _ = left_aligned_text(i.to_s)
    _.make_selectable
    return _
  end

  # ========================================================================= #
  # === gtk_list_store_table3
  #
  # Note that this adds elements in an uniform manner, e. g. Strings
  # only or Integers only and so forth. For now this is hardcoded
  # towards Strings.
  #
  # This method expects (and assumes) that you pass an Array.
  #
  # The first three elements then constitute the "heading" section;
  # the other parts that follow are the various entries in the
  # columns.
  # ========================================================================= #
  def gtk_list_store_table3(*i)
    i = i.flatten.compact
    list_store = ::Gtk::ListStore.new(
      *[String] * 3
    )
    tree_view = gtk_tree_view(list_store)
    renderer = ::Gtk::CellRendererText.new
    renderer.editable = true
    tree_view << [i.shift, renderer, text: 0]
    tree_view << [i.shift, renderer, text: 1]
    tree_view << [i.shift, renderer, text: 2]
    i.each_cons(3) {|entry|
      iter = list_store.append
      iter.set_value(0, entry[0])
      iter.set_value(1, entry[1]) if entry[1]
      iter.set_value(2, entry[2]) if entry[2]
    }
    # ======================================================================= #
    # Next, we need to style this tree view a little bit.
    # ======================================================================= #
    tree_view.hadjustment = ::Gtk::BaseModule.default_gtk_adjustment?
    tree_view.clickable_headers
    tree_view.enable_drag_and_drop # Enable drag and drop support of the treeview.
    # ======================================================================= #
    # tree_view.signal_connect(:button_press_event) { |widget, event|
    #   if event.event_type == Gdk::Event::BUTTON2_PRESS and event.button == 1 # event.event_type.name
    # tree_view.signal_connect(:key_press_event) { |widget, event|
    #  case Gdk::Keyval.to_name(event.keyval)
    #  when 'Return','KP_Enter' # enter drücken
    # tree_view.signal_connect(:button_press_event) { |widget, event|
    # ======================================================================= #
    #  if event.event_type == Gdk::Event::BUTTON_PRESS # normaler mouse button
    return tree_view
  end

  # ========================================================================= #
  # === gtk_list_store_table4
  # ========================================================================= #
  def gtk_list_store_table4(*i)
    i = i.flatten.compact
    list_store = ::Gtk::ListStore.new(
      *[String] * 4
    )
    tree_view = gtk_tree_view(list_store)
    renderer = ::Gtk::CellRendererText.new
    renderer.editable = true
    tree_view << [i.shift, renderer, text: 0]
    tree_view << [i.shift, renderer, text: 1]
    tree_view << [i.shift, renderer, text: 2]
    tree_view << [i.shift, renderer, text: 3]
    i.each_cons(4) {|entry|
      iter = list_store.append
      iter.set_value(0, entry[0])
      iter.set_value(1, entry[1]) if entry[1]
      iter.set_value(2, entry[2]) if entry[2]
      iter.set_value(3, entry[3]) if entry[3]
    }
    # ======================================================================= #
    # Next, we need to style this tree view a little bit.
    # ======================================================================= #
    tree_view.hadjustment = ::Gtk::BaseModule.default_gtk_adjustment?
    tree_view.clickable_headers
    tree_view.enable_drag_and_drop
    return tree_view
  end

  # ========================================================================= #
  # === table3
  #
  # Will return a table with a 3 columns layout.
  # ========================================================================= #
  def table3(*all_args)
    table = ::Gtk::Table.new(all_args.size / 3, 2, true)
    counter = 0
    slow_counter = 0
    for element in all_args
      # print (counter % 2),' ',(counter % 2)+1,' ',slow_counter,' ',slow_counter+1,"\n"
      table.attach_defaults( 
        element,
        (counter % 3), (counter % 3)+1, 
        slow_counter,   slow_counter+1
      )
      counter += 1
      slow_counter += 1 if  counter % 3 == 0
    end
    return table
  end

  # ========================================================================= #
  # === gtk_text_left_aligned
  #
  # This variant is actually just gtk_text() where we align the text onto
  # the left-hand side. So, left-centric text.
  # ========================================================================= #
  def gtk_text_left_aligned(i = nil)
    return ::Gtk.left_aligned_text(i)
  end; alias gtk_left_aligned_label gtk_text_left_aligned # === gtk_left_aligned_label
       alias left_aligned_text      gtk_text_left_aligned # === left_aligned_text
       alias left_aligned_label     gtk_text_left_aligned # === left_aligned_label
       alias gtk_left_aligned_text  gtk_text_left_aligned # === gtk_left_aligned_text

  # ========================================================================= #
  # === event_middle_mouse_button_click?
  #
  # Query whether the middle mouse button was clicked via this method.
  #
  # Note that a gdk-event has to be passed into this method.
  # ========================================================================= #
  def event_middle_mouse_button_click?(
      event
    )
    event.is_a?(Gdk::EventButton) and
   (event.event_type.name == 'GDK_BUTTON_PRESS') and
   (event.button == 2)
  end; alias middle_mouse_button_clicked?        event_middle_mouse_button_click? # === middle_mouse_button_clicked?
       alias event_middle_mouse_button?          event_middle_mouse_button_click? # === event_middle_mouse_button?
       alias on_middle_mouse_button_click        event_middle_mouse_button_click? # === on_middle_mouse_button_click
       alias is_middle_mouse_button_click_event? event_middle_mouse_button_click? # === is_middle_mouse_button_click_event?

  # ========================================================================= #
  # === uses_gtk2?
  # ========================================================================= #
  def uses_gtk2?
    we_use_this_gtk_version? == :gtk2
  end; alias use_gtk2? uses_gtk2? # === use_gtk2?
       alias gtk2?     uses_gtk2? # === gtk2?

  # ========================================================================= #
  # === uses_gtk3?
  # ========================================================================= #
  def uses_gtk3?
    we_use_this_gtk_version? == :gtk3
  end; alias use_gtk3? uses_gtk3? # === use_gtk3?
       alias gtk3?     uses_gtk3? # === gtk3?

  # ========================================================================= #
  # === uses_gtk4?
  # ========================================================================= #
  def uses_gtk4?
    ::Gtk.do_we_use_gtk4?
  end; alias use_gtk4? uses_gtk4? # === use_gtk4?

  # ========================================================================= #
  # === event_left_mouse_button_click?
  #
  # The ruby-gtk2 name may be:
  #
  #   event.event_type == Gdk::Event::BUTTON_PRESS
  #
  # But I think checking via .name explicitly is better.
  # ========================================================================= #
  def event_left_mouse_button_click?(
      event
    )
    event.is_a?(Gdk::EventButton) and
   (event.event_type.name == 'GDK_BUTTON_PRESS') and
   (event.button == 1)
  end; alias left_mouse_button_clicked? event_left_mouse_button_click? # === left_mouse_button_clicked?
       alias event_left_mouse_button?   event_left_mouse_button_click? # === event_left_mouse_button?
       alias on_left_mouse_button_click event_left_mouse_button_click? # === on_left_mouse_button_click
       alias is_left_mouse_button?      event_left_mouse_button_click? # === is_left_mouse_button?
       alias left_mouse_click?          event_left_mouse_button_click? # === left_mouse_click?

  # ========================================================================= #
  # === return_the_html_colours
  #
  # This method will return all html-colours. In order for this to work,
  # the colours gem has to be installed.
  # ========================================================================= #
  def return_the_html_colours
    ::Colours.html_colours
  end

  # ========================================================================= #
  # === current_event?
  # ========================================================================= #
  def current_event?
    return ::Gtk.current_event
  end

  # ========================================================================= #
  # === drag_start
  #
  # The source-widget must connect to the 'drag-data-get' signal.
  #
  # This probably no longer works in ruby-gtk4.
  # ========================================================================= #
  def drag_start
    drag_source_set(Gdk::ModifierType::BUTTON1_MASK |
                    Gdk::ModifierType::BUTTON2_MASK,
                    [['test', Gtk::TargetFlags::SAME_APP, 12345]],
                    Gdk::DragAction::COPY |
                    Gdk::DragAction::MOVE)
  end

  # ========================================================================= #
  # === gtk_image
  #
  # This method allows the user to make use of some Symbols.
  # ========================================================================= #
  def gtk_image(
      this_image         = '',
      optional_icon_size = nil
    )
    if this_image == :random
      # ===================================================================== #
      # Handle this variant quickly here.
      # ===================================================================== #
      this_image = FILE_TABBLE
    end
    if this_image.is_a? Symbol
      # ===================================================================== #
      # This may be a stock-item, or something like :weather_icon. Or it
      # may be an Emoji. Since as of December 2020, we will preferentially
      # use an emoji first.
      # ===================================================================== #
      if ::Gtk::Emojis.is_this_an_emoji_symbol?(this_image, :be_quiet)
        this_image = ::Gtk::Emojis.map_emoji(this_image, :be_quiet)
      else
        this_image = return_corresponding_gtk_stock_item(this_image)
      end
    end
    # ======================================================================= #
    # Next, we will intercept GdkPixbuf::Pixbuf and GdkPixbuf::PixbufAnimation
    # ======================================================================= #
    if this_image.is_a?(GdkPixbuf::Pixbuf) or
       this_image.is_a?(GdkPixbuf::PixbufAnimation)
      image = ::Gtk::Image.new
      # if use_gtk2?
      #   image.set(this_image, optional_icon_size)
      if use_gtk4?
        # =================================================================== #
        # ruby-gtk4 seems to ignore the icon-size setting
        # =================================================================== #
        image.set_from_pixbuf(this_image)
      else # else assume gtk4 here
        image.set_pixbuf(this_image)
      end
      return image
    end
    # ======================================================================= #
    # Next assume a remote image. This will be downloaded into /tmp/.
    # ======================================================================= #
    if this_image.respond_to?(:start_with?) and this_image.start_with?('http')
      if File.exist?('/tmp/'+File.basename(this_image))
        this_image = '/tmp/'+File.basename(this_image)
      else
        Dir.chdir('/tmp/')
        esystem 'wget '+this_image
        this_image = File.absolute_path(File.basename(this_image))
      end
    end
    # ======================================================================= #
    # The next if-clause is entered when an argument was supplied to this
    # method.
    # ======================================================================= #
    if this_image and this_image.respond_to?(:empty?) and !this_image.empty?
      # ===================================================================== #
      # The next assumes that gtk-related symbols look like this:
      #
      #   :"gtk-open"
      #
      # Hopefully this assumption will work; if not we may have to re-evaluate
      # it at a later time (written in November 2020).
      # ===================================================================== #
      if this_image.is_a?(Symbol) and this_image.start_with?('gtk-')
        image = ::Gtk::Image.new
        # =================================================================== #
        # In that case we will modify optional_icon_size if it is nil.
        # =================================================================== #
        if optional_icon_size.nil?
          optional_icon_size = ::Gtk::IconSize::SMALL_TOOLBAR # BUTTON # MENU
        end
      end
      if optional_icon_size
        image.set(this_image, optional_icon_size)
      else
        # =================================================================== #
        # === Handle gtk2 first
        # =================================================================== #
        if use_gtk2?
          image = ::Gtk::Image.new(this_image)
        else # else it is for ruby-gtk3.
          if this_image.is_a? String
            # =============================================================== #
            # We must rescue a few errors next, because handling images is
            # quite error-prone from my experience.
            # =============================================================== #
            begin
              this_image = GdkPixbuf::Pixbuf.new(file: this_image)
            rescue GLib::FileError => error
              pp error if is_on_roebe?
              if this_image.is_a? String
                this_image = { file: this_image }
              end
              return ::Gtk::Image.new(this_image)
            rescue Exception => error
              pp error if is_on_roebe?
              pp error.class if is_on_roebe?
              if this_image.is_a? String
                this_image = { file: this_image }
              end
              return ::Gtk::Image.new(this_image)
            end
          end
          image = ::Gtk::Image.new(pixbuf: this_image)
        end
      end
    else
      image = ::Gtk::Image.new
    end
    return image
  end; alias image        gtk_image # === image
       alias create_image gtk_image # === create_image

  # ========================================================================= #
  # === drag_stop
  #
  # The destination-widget must connect to the 'drag-data-received' signal. 
  # ========================================================================= #
  def drag_stop
    drag_dest_set(Gtk::DestDefaults::MOTION |
                  Gtk::DestDefaults::HIGHLIGHT,
                  [['test', :same_app, 12345]],
                  Gdk::DragAction::COPY |
                  Gdk::DragAction::MOVE)
  end

  # ========================================================================= #
  # === Gtk::BaseModule.default_gtk_adjustment?
  # ========================================================================= #
  def self.default_gtk_adjustment?
    @default_gtk_adjustment
  end

  require 'gtk_paradise/core_classes/file_chooser_dialog.rb'
  # ========================================================================= #
  # === gtk_file_chooser_dialog
  #
  # This method is simply a wrapper over Gtk::FileChooserDialog.
  # ========================================================================= #
  def gtk_file_chooser_dialog(
      label_to_use = '',
      which_widget_to_use,
      which_action_type,
      whatever_here,
      array_cancel,
      array_accept
    )
    ::Gtk::FileChooserDialog.new(
      label_to_use,
      which_widget_to_use,
      which_action_type,
      whatever_here,
      array_cancel,
      array_accept
    )
  end

  # ========================================================================= #
  # === button_themed_icon
  # ========================================================================= #
  def button_themed_icon(
      i = 'edit-paste-symbolic.symbolic'
    )
    button = create_button
    use_this_icon = gtk_themed_icon(i)
    button.set_image(use_this_icon)
    return button
  end

  # ========================================================================= #
  # === gtk_button_with_this_image
  # ========================================================================= #
  def gtk_button_with_this_image(image)
    _ = create_button
    _.add(image)
    return _
  end; alias button_with_image      gtk_button_with_this_image # === button_with_image
       alias button_with_this_image gtk_button_with_this_image # === button_with_this_image

  require 'gtk_paradise/core_classes/search_bar.rb'
  # ========================================================================= #
  # === gtk_search_bar
  #
  # The first argument, which is optional, should be of class
  # Gtk::SearchEntry if provided.
  #
  # Upstream API documentation can be seen here:
  #
  #   https://developer.gnome.org/gtk3/stable/GtkSearchBar.html
  #
  # ========================================================================= #
  def gtk_search_bar(
      optional_widget = nil
    )
    search_bar = ::Gtk::SearchBar.new
    if optional_widget
      search_bar.connect_entry(optional_widget)
      search_bar.add(optional_widget)
    end
    return search_bar
  end

  # ========================================================================= #
  # === gtk_toggle_button
  # ========================================================================= #
  def gtk_toggle_button(
      optional_content = ''
    )
    if optional_content and optional_content.is_a?(String) and use_gtk3?
      optional_content = { label: optional_content }
    end
    if use_gtk3?    # for ruby-gtk3
      toggle_button = ::Gtk::ToggleButton.new(optional_content)
    elsif use_gtk4? # for ruby-gtk4
      toggle_button = ::Gtk::ToggleButton.new(label: optional_content)
    end
    @internal_hash[:toggle_buttons] << toggle_button if @internal_hash
    return toggle_button
  end; alias toggle_button gtk_toggle_button # === toggle_button

  require 'gtk_paradise/core_classes/combo_box_entry.rb'
  # ========================================================================= #
  # === gtk_combo_box_entry
  #
  # Note that ComboBoxEntry no longer exists in GTK3, hence the check inside
  # of this method.
  # ========================================================================= #
  def gtk_combo_box_entry(
      array_or_nil = nil,
      i            = :strings_only
    )
    # if use_gtk2?
    #   _ = ::Gtk::ComboBoxEntry.new
    # else assume ruby-gtk3
    _ = ::Gtk::ComboBoxText.new
    if array_or_nil
      array_or_nil.each {|entry|
        if _.respond_to? :append_text
          _.append_text(entry)
        else
          _.append(entry)
        end
      }
    end
    return _
  end; alias gtk_dropdown_box       gtk_combo_box_entry # === gtk_dropdown_box
       alias combo_box_entry        gtk_combo_box_entry # === combo_box_entry
       alias create_combo_box_entry gtk_combo_box_entry # === create_combo_box_entry

  # ========================================================================= #
  # === remove_html
  # ========================================================================= #
  def remove_html(i)
    return i.gsub(/\<.+?\>/,'')
  end

  # ========================================================================= #
  # === gtk_password
  #
  # This variant can be used like a "password-protected input device".
  # ========================================================================= #
  def gtk_password(
      use_this_as_invisibility_char = 42
    )
    password = create_entry
    password.set_visibility(false)
    password.invisible_char = use_this_as_invisibility_char # This is '*' in unicode.
    return password 
  end

  # ========================================================================= #
  # === gtk_spinner
  # ========================================================================= #
  def gtk_spinner
    return ::Gtk::Spinner.new
  end; alias default_spinner gtk_spinner # === default_spinner
       alias new_spinner     gtk_spinner # === new_spinner
       alias create_spinner  gtk_spinner # === create_spinner

  # ========================================================================= #
  # === do_popup
  #
  # This variant expects a Hash as input.
  #
  # Typical usage example would go like this:
  #
  #   do_popup(
  #     message:          _+' is now in use.',
  #     over_this_widget: @entry_multimedia_player_in_use
  #   )
  #
  # ========================================================================= #
  def do_popup(
      i = {}
    )
    title_to_use   = ''
    this_widget    = nil
    padding_to_use = 15
    if i.is_a? Hash
      # ===================================================================== #
      # === :message
      # ===================================================================== #
      if i.has_key? :message
        title_to_use = i[:message]
      end
      # ===================================================================== #
      # === :over_this_widget
      # ===================================================================== #
      if i.has_key? :over_this_widget
        this_widget = i[:over_this_widget]
      end
    end
    return_popover(
      title_to_use,
      this_widget,
      padding_to_use
    ).popup
  end

  # ========================================================================= #
  # === gtk_table3
  #
  # This method will create a table with 3 rows per line, based on
  # ListStore, TreeView and TreeViewColumn. Otherwise it is identical
  # to gtk_table2, so please look at the documentation there.
  # ========================================================================= #
  def gtk_table3(*i)
    i.flatten! # Must keep our Array flat.
    list_store = ::Gtk::ListStore.new(String, String, String)
    # ======================================================================= #
    # Designate the cell-renderers in use here.
    # ======================================================================= #
    cell_renderer_text1 = ::Gtk::CellRendererText.new
    cell_renderer_text2 = ::Gtk::CellRendererText.new
    cell_renderer_text3 = ::Gtk::CellRendererText.new
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      if yielded.is_a?(Hash)
        # =================================================================== #
        # === :list_store
        # =================================================================== #
        if yielded.has_key?(:list_store)
          list_store = yielded[:list_store]
        end
        # =================================================================== #
        # === :cell_renderer_text1
        # =================================================================== #
        if yielded.has_key? :cell_renderer_text1
          cell_renderer_text1.send(yielded[:cell_renderer_text1])
        end
        # =================================================================== #
        # === :cell_renderer_text2
        # =================================================================== #
        if yielded.has_key? :cell_renderer_text2
          cell_renderer_text2.send(yielded[:cell_renderer_text2])
        end
      end
    end
    tree_view  = ::Gtk::TreeView.new(list_store)
    # ======================================================================= #
    # Get the first two members and make them the column-headers.
    # ======================================================================= #
    column1 = ::Gtk::TreeViewColumn.new(i.shift.to_s, cell_renderer_text1, { text: 0 })
    column2 = ::Gtk::TreeViewColumn.new(i.shift.to_s, cell_renderer_text2, { text: 1 })
    column3 = ::Gtk::TreeViewColumn.new(i.shift.to_s, cell_renderer_text3, { text: 2 })
    # ======================================================================= #
    # Add the two columns to the tree-view next.
    # ======================================================================= #
    tree_view.append_column(column1)
    tree_view.append_column(column2)
    tree_view.append_column(column3)
    # ======================================================================= #
    # Next we will feed the members of that Array in pairs-of-2 into
    # the TreeView widget, which explains why we use .each_slice().
    # ======================================================================= #
    i.each_slice(3) { |first, second, third|
      iter = list_store.append
      iter.set_value(0, first)
      iter.set_value(1, second)
      iter.set_value(2, third)
    } unless i.empty?
    # Assume ruby-GTK3 past this point here:
    tree_view.enable_model_drag_source(:button1_mask,
      [ ['GTK_TREE_MODEL_ROW', 0, 0] ], 
      Gdk::DragAction::COPY|Gdk::DragAction::MOVE
    )
    tree_view.enable_model_drag_dest(
      [ ['GTK_TREE_MODEL_ROW', 0, 0] ],
      Gdk::DragAction::COPY|Gdk::DragAction::MOVE
    )
    # ======================================================================= #
    # Two results will be returned next, via an Array.
    # ======================================================================= #
    return [tree_view, list_store] # ← Last but not least, return the tree-view here, together with the list store.
  end; alias tree_view_table3                      gtk_table3 # === table3
       alias tree_widget_with_two_columns_per_row3 gtk_table3 # === tree_widget_with_two_columns_per_row3

  # ========================================================================= #
  # === new_open_file_dialog
  #
  # This variant is only guaranteed to work for ruby-gtk3, not ruby-gtk2.
  # ========================================================================= #
  def new_open_file_dialog(
      use_this_title                            = 'Choose a file',
      use_this_name_for_the_file_chooser_dialog = 'custom_dialog',
      use_this_as_the_current_folder            = return_pwd,
      &block
    )
    # response_accept = ::Gtk::ResponseType::ACCEPT
    yielded = nil
    # ======================================================================= #
    # === Handle blocks given to this method next
    # ======================================================================= #
    if block_given?
      yielded = yield
    end
    # ======================================================================= #
    # Create a new instance of the file-chooser-dialog:
    # ======================================================================= #
    file_chooser_dialog = ::Gtk::FileChooserDialog.new(
      title:   use_this_title,
      action:  :open,
      buttons: [[::Gtk::Stock::OPEN,   :accept],
                [::Gtk::Stock::CANCEL, :cancel]]
    )
    # ======================================================================= #
    # === Handle Hashes next
    # ======================================================================= #
    if yielded and yielded.is_a?(Hash)
      # ===================================================================== #
      # === :name_for_the_file_chooser_dialog
      # ===================================================================== #
      if yielded.has_key? :name_for_the_file_chooser_dialog
        use_this_name_for_the_file_chooser_dialog = yielded.delete(:name_for_the_file_chooser_dialog)
      end
      # ===================================================================== #
      # === :use_this_as_the_current_folder
      # ===================================================================== #
      if yielded.has_key? :use_this_as_the_current_folder
        use_this_as_the_current_folder = yielded.delete(:use_this_as_the_current_folder)
      end
      # =================================================================== #
      # === additional_folders
      #
      # This can be used like so:
      #
      #   additional_folders: '/Depot/j/'
      #
      # =================================================================== #
      if yielded.has_key? :additional_folders
        _ = yielded[:additional_folders]
        if _.is_a? Array # Handle arrays here.
          _.each {|this_entry|
            file_chooser_dialog.add_this_directory(this_entry)
          }
        else
          file_chooser_dialog.add_this_directory(_)
        end
      # =================================================================== #
      # === extra_directory
      # =================================================================== #
      elsif yielded.has_key? :extra_directory
        _ = yielded[:extra_directory]
        if _.is_a? Array # Handle arrays here.
          _.each {|this_entry|
            file_chooser_dialog.add_this_directory(this_entry)
          }
        else
          file_chooser_dialog.add_this_directory(_)
        end
      # =================================================================== #
      # === additional_directories
      #
      # This is an alias to the above.
      # =================================================================== #
      elsif yielded.has_key? :additional_directories
        _ = yielded[:additional_directories]
        if _.is_a? Array
          _.flatten.sort_by {|entry| File.basename(entry) }.each {|this_entry|
            file_chooser_dialog.add_this_directory(this_entry)
          }
        else
          file_chooser_dialog.add_this_directory(_)
        end
      end
      # ===================================================================== #
      # === :filter_for
      # ===================================================================== #
      if yielded.has_key? :filter_for
        array = [yielded.delete(:filter_for)].flatten.compact # Always work on an Array.
        file_filter = ::Gtk::FileFilter.new
        array.each {|use_this_pattern|
          use_this_pattern = use_this_pattern.dup
          use_this_pattern.prepend('*') unless use_this_pattern.start_with?('*')
          file_filter.add_pattern(use_this_pattern)
        }
        file_chooser_dialog.add_filter(file_filter)
      end
    end
    file_chooser_dialog.set_name(use_this_name_for_the_file_chooser_dialog)
    file_chooser_dialog.current_folder = use_this_as_the_current_folder
    case file_chooser_dialog.run
    when ::Gtk::ResponseType::ACCEPT,
        ::Gtk::ResponseType::OK
      filename = File.absolute_path(file_chooser_dialog.filename.to_s)
      ::Gtk.set_main_file(filename) if File.exist?(filename)
    end
    file_chooser_dialog.destroy
    return filename
  end

  # ========================================================================= #
  # === image_starred
  #
  # This is a "star" image essentially.
  # ========================================================================= #
  def image_starred
    return_this_image_based_on_the_default_icon_theme(:starred)
  end

  # ========================================================================= #
  # === always_scrolled_window
  # ========================================================================= #
  def always_scrolled_window(i)
    return gtk_scrolled_window(i) { :always }
  end

  # ========================================================================= #
  # === initialize_gtk3
  #
  # Simply require gtk3 here.
  # ========================================================================= #
  def initialize_gtk3
    ::Gtk.initialize_gtk3
  end

  # ========================================================================= #
  # === white_background
  # ========================================================================= #
  def white_background
    modify_background(:normal, :white)
  end

  # ========================================================================= #
  # === gtk_hbutton_box
  #
  # This variant is specifically for ruby-gtk2.
  # ========================================================================= #
  def gtk_hbutton_box(*optional_arguments)
    box = ::Gtk::HButtonBox.new
    if optional_arguments and !optional_arguments.empty?
      optional_arguments.each {|entry| box.pack_start(entry, true, true) }
    end
    return box
  end

  # ========================================================================= #
  # === is_on_roebe?
  # ========================================================================= #
  def is_on_roebe?
    ::Gtk.is_on_roebe?
  end

  # ========================================================================= #
  # === gtk_grid
  #
  # Usage examples:
  #
  #   g = Gtk::Grid.new
  #   g.attach(Gtk::Button.new(label: 'Button'), 0, 0, 1, 1)
  #   g.attach(Gtk::CheckButton.new(''), 1, 0, 1, 1)
  #
  # Or:
  #
  #   grid = gtk_grid { :default }
  #   grid.left(text('This is left.'))
  #   grid.middle(text('This is middle.'))
  #   grid.right(text('This is right.'))
  #
  # ========================================================================= #
  def gtk_grid(
      optional_arguments = '',
      &block
    )
    _ = ::Gtk::Grid.new
    if optional_arguments and optional_arguments.include?('x')
      # ===================================================================== #
      # For instance:
      #
      #   grid1 = create_grid('1x2')
      #
      # This is currently ignored, though, as grids in gtk behave slightly
      # differently.
      # ===================================================================== #
    end
    # ======================================================================= #
    # === Handle blocks given to this method next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :default
      #
      # This entry point specifies a few defaults that I consider somewhat
      # useful.
      # ===================================================================== #
      when :default
        enable_default_grid_settings_for(_)
      end
    end
    return _
  end; alias grid        gtk_grid # === grid
       alias create_grid gtk_grid # === create_grid

  # ========================================================================= #
  # === enable_default_grid_settings_for
  # ========================================================================= #
  def enable_default_grid_settings_for(i)
    i.set_column_spacing(2)
    i.set_row_spacing(2)
  end

  # ========================================================================= #
  # === max_width?
  #
  # Alternatively we could use ::Gtk.max_width?. 
  # ========================================================================= #
  def max_width?
    ::Gtk.max_width?
  end; alias original_width? max_width? # === original_width?

  # ========================================================================= #
  # === max_resolution?
  #
  # This method will return an Array such as [1920, 1080] - maximum width
  # first, then maximum height.
  # ========================================================================= #
  def max_resolution?
    ::Gtk.resolution?
  end

  # ========================================================================= #
  # === max_height?
  # ========================================================================= #
  def max_height?
    return Gtk.resolution?.last # .last is height.
  end

  # ========================================================================= #
  # === set_width
  # ========================================================================= #
  def set_width(i = 0)
    i = ::Gtk.calculate_the_true_value_represented_via_a_percentage_value(i)
    @internal_hash[:width] = i.to_i if i.respond_to?(:to_i)
  end

  require 'gtk_paradise/core_classes/spin_button.rb'
  # ========================================================================= #
  # === gtk_spin_button
  #
  # Note that the values that are given to Gtk::SpinButton specify
  # min, max and range. The default that is used in the method
  # must make "sense", meaning it has to refer to "the most commonly
  # assumed use case" for a spin-button. I somewhat arbitrarily
  # decided that this means to start at 0, count up to 100, in
  # steps of 1. Obviously this is arbitrary, but other values are
  # just about as arbitrary as well - so I thought counting up to
  # 100 would make sense here.
  #
  # Usage examples:
  #
  #   gtk_spin_button(0, 300, 1)
  #   spinner_button = gtk_spinner_button {{ start_point: 1 }}
  #
  # ========================================================================= #
  def gtk_spin_button(
      start_position = 0,
      end_position   = 100,
      increment_size = 1,
      &block
    )
    return ::Gtk.spin_button(start_position, end_position, increment_size)
  end; alias spin_button             gtk_spin_button # === spin_button
       alias spinbutton              gtk_spin_button # === spinbutton
       alias gtk_spinner_button      gtk_spin_button # === gtk_spinner_button
       alias gtk_spinbutton          gtk_spin_button # === gtk_spinbutton
       alias create_spinbutton       gtk_spin_button # === create_spinbutton
       alias gtk_up_and_down_counter gtk_spin_button # === gtk_up_and_down_counter
       alias up_and_down_counter     gtk_spin_button # === up_and_down_counter

  require 'gtk_paradise/core_classes/expander.rb'
  # ========================================================================= #
  # === gtk_expander
  #
  # The first argument is the text on the label that is to be used.
  # ========================================================================= #
  def gtk_expander(
      text_on_the_label,
      use_markup = false,
      &block
    )
    expander = ::Gtk::Expander.new(text_on_the_label)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === :widget
      # ===================================================================== #
      if yielded.has_key? :widget
        expander.add(yielded.delete(:widget))
      end
    end
    return expander
  end

  # ========================================================================= #
  # === default_gtk_settings?
  #
  # This probably no longer works on ruby-gtk4.
  # ========================================================================= #
  def default_gtk_settings?
    ::Gtk::Settings.default
  end; alias gtk_settings? default_gtk_settings? # === gtk_settings?

  # ========================================================================= #
  # === html_table2
  # ========================================================================= #
  def html_table2(*i)
    _ = gtk_list_store_table2(*i)
    maximal(_, 0)
  end

  # ========================================================================= #
  # === gtk_hpaned
  #
  # Simply pass the widgets into this method - ideally only use two
  # widgets though.
  #
  # An usage example for this method follows:
  #
  #   hpaned = gtk_hpaned(
  #     widget1,
  #     widget2
  #   )
  #
  # Or:
  #
  #   hpaned = gtk_hpaned(
  #     widget1,
  #     widget2
  #   ) {{ border_width: 5 }}
  #
  # ========================================================================= #
  def gtk_hpaned(
      *optional_arguments
    )
    optional_arguments.flatten!
    hpaned = ::Gtk::Paned.new(:horizontal)
    if optional_arguments and !optional_arguments.empty?
      optional_arguments.each {|widget|
        hpaned.add(widget)
      }
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :border_width
        # =================================================================== #
        if yielded.has_key? :border_width
          set_border_width(
            yielded.delete(:border_width)
          )
        end
      end
    end
    return hpaned
  end; alias gtk_draggable_vpane gtk_hpaned # === gtk_draggable_vpane
       alias gtk_horizontal_pane gtk_hpaned # === gtk_horizontal_pane

  require 'gtk_paradise/core_classes/scrolled_window.rb'
  # ========================================================================= #
  # === gtk_scrolled_window
  #
  # This method will create an instance of Gtk::ScrolledWindow.
  #
  # A gtk-widget can be provided to this method. If it is provided
  # then that widget will be used to use as the default viewport
  # for the Gtk::ScrolledWindow instance.
  # ========================================================================= #
  def gtk_scrolled_window(
      optional_widget = nil
    )
    scrolled_window = ::Gtk::ScrolledWindow.new
    # ======================================================================= #
    # === Handle blocks next:
    # ======================================================================= #
    if block_given?
      # ===================================================================== #
      # We use a variable because gtk3 deprecated Gtk::POLICY_ALWAYS.
      # ===================================================================== #
      always = :always
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :left_right
      # ===================================================================== #
      when :left_right
        scrolled_window.set_policy(always, :never)
      # ===================================================================== #
      # === :only_top_down_bar
      #
      # This variant will show the top-to-bottom bar, but not the
      # left-to-right slider. I quite like this variant.
      # ===================================================================== #
      when :only_top_down_bar,
           :only_top_to_bottom,
           :only_up_and_down_scrollbar,
           :only_up_and_down,
           :only_up_and_bottom,
           :only_top_down,
           :only_top_and_bottom_scrollbar,
           :up_and_down_only,
           :up_and_down_always,
           :always_up_and_down,
           :top_to_bottom_always
        # =================================================================== #
        # The API is:
        #
        #              .set_policy(hscrollbar_policy, vscrollbar_policy)
        #
        # =================================================================== #
        scrolled_window.set_policy(:never, always)
      # ===================================================================== #
      # === :never_always
      #
      # This first uses :never, then :always.
      # ===================================================================== #
      when :never_always
        scrolled_window.set_policy(:never, always)
      # ===================================================================== #
      # === :always_scrollbars
      #
      # Usage examples:
      #
      #   gtk_scrolled_window { :always_scrollbars }
      #   gtk_scrolled_window { :always_show_the_scrollbars }
      #
      # ===================================================================== #
      when :always_scrollbars,
           :always_show_the_scrollbars,
           :always
        scrolled_window.set_policy(always, always)
      # ===================================================================== #
      # === :automatic_scrollbars
      #
      # Usage example:
      #
      #   gtk_scrolled_window { :automatic_scrollbars }
      #
      # ===================================================================== #
      when :automatic_scrollbars
        scrolled_window.automatic_scrollbars
      # ===================================================================== #
      # === :automatic_scrolling
      # ===================================================================== #
      when :automatic_scrolling,
           :automatic_policy,
           :automatic
         scrolled_window.set_policy(:automatic, :automatic)
      # ===================================================================== #
      # === :default_policy
      #
      # Usage example:
      #
      #   gtk_scrolled_window { :default_policy }
      #
      # ===================================================================== #
      when :default_policy
        scrolled_window.set_policy(:automatic, always)
      else
        if yielded.is_a? Hash
          # ================================================================= #
          # === :use_this_widget
          # ================================================================= #
          if yielded.has_key? :use_this_widget
            scrolled_window.add(
              yielded.delete(:use_this_widget)
            )
          end
        end
      end
    end
    if optional_widget
      scrolled_window.add_with_viewport(optional_widget)
    end
    return scrolled_window # Return it here.
  end; alias scrolled_window        gtk_scrolled_window # === scrolled_window
       alias scrolling_for          gtk_scrolled_window # === scrolling_for
       alias create_scrolled_window gtk_scrolled_window # === create_scrolled_window

  # ========================================================================= #
  # === gtk_list_store_table2
  #
  # The input to this should be an Array containing Strings.
  # ========================================================================= #
  def gtk_list_store_table2(*i)
    i = i.flatten.compact
    list_store = ::Gtk::ListStore.new(
      *[String] * 2
    )
    tree_view = gtk_tree_view(list_store)
    renderer = ::Gtk::CellRendererText.new
    renderer.editable = true
    tree_view << [i.shift, renderer, text: 0]
    tree_view << [i.shift, renderer, text: 1]
    i.each_slice(2) {|entry|
      iter = list_store.append
      iter.set_value(0, entry[0])
      iter.set_value(1, entry[1]) if entry[1]
    }
    # ======================================================================= #
    # Next, we need to style this tree view a little bit.
    # ======================================================================= #
    tree_view.hadjustment = ::Gtk::BaseModule.default_gtk_adjustment?
    tree_view.clickable_headers
    tree_view.enable_drag_and_drop
    return tree_view
  end

  # ========================================================================= #
  # === gtk_button_with_label
  #
  # This variant is for creating a gtk-button with a label. It has been
  # specifically added if we just want to create a button in ruby-gtk3
  # as-is.
  # ========================================================================= #
  def gtk_button_with_label(
      use_this_label     = '',
      shall_we_underline = true
    )
    return ::Gtk::Button.new(
      label:         use_this_label,
      use_underline: shall_we_underline
    )
  end; alias create_button_with_label gtk_button_with_label # === gtk_button_with_label

  # ========================================================================= #
  # === gtk_left_aligned_button
  # ========================================================================= #
  def gtk_left_aligned_button(
      i             = nil,
      use_underline = false,
      &block
    )
    button = create_button(i, use_underline, &block)
    button.left_align
    return button
  end

  require 'gtk_paradise/core_classes/check_button.rb'
  # ========================================================================= #
  # === gtk_check_button                                   (check_button tag)
  # ========================================================================= #
  def gtk_check_button(
      i             = '',
      use_underline = :do_not_underline,
      &block
    )
    if i.respond_to?(:include?) and i.include?('_')
      use_underline = true
    end
    button = ::Gtk.check_button(i, use_underline, &block)
    # ======================================================================= #
    # Append the button to the @internal_hash next.
    # ======================================================================= #
    if @internal_hash and @internal_hash.has_key?(:check_buttons)
      @internal_hash[:check_buttons] << button
    end
    return button
  end; alias gtk_checkbutton  gtk_check_button # === gtk_checkbutton
       alias gtk_checkbox     gtk_check_button # === gtk_checkbox
       alias gtk_check_box    gtk_check_button # === gtk_check_box
       alias create_checkbox  gtk_check_button # === create_checkbox
       alias create_check_box gtk_check_button # === create_check_box
       alias check_button     gtk_check_button # === check_button
       alias checkbox         gtk_check_button # === checkbox

  # ========================================================================= #
  # === gtk_bold_check_button
  #
  # This method will create a check-button but will also make the button
  # widget bold (the label on it).
  # ========================================================================= #
  def gtk_bold_check_button(
      i             = '',
      use_underline = :do_not_underline
    )
    _ = ::Gtk.check_button(i, use_underline)
    _.make_bold
    return _
  end

  if ::Gtk.use_gtk4? # Add .show_all method next for gtk4.
    # ======================================================================= #
    # === show_all
    #
    # Add .show_all() for compatibility reasons.
    # ======================================================================= #
    def show_all
      show
    end
  end

  # ========================================================================= #
  # === create_a_new_file_chooser_dialog
  #
  # The first argument to this method should be a Gtk::Window instance.
  #
  # Note that this method, since as of February 2022, is mildly
  # deprecated. Use new_open_file_dialog() instead, but keep
  # in mind that this new method is NOT compatible with
  # ruby-gtk2. Only ruby-gtk3 is supported. Eventually the
  # method here will be deprecated one day.
  #
  # Usage examples:
  #
  #   create_a_new_file_chooser_dialog {{ filter: '.pdf' }}
  #   create_a_new_file_chooser_dialog {{ filter_for: '.pdb' }}
  #   create_a_new_file_chooser_dialog {{ filter_for: '.md', additional_folders: '/Depot/j/' }}
  #   create_a_new_file_chooser_dialog {{ filter_for: '.md', additional_folders: '/Depot/j/', start_dir: '/tmp/' }}
  #   create_a_new_file_chooser_dialog {{ filter_for: %w( .md .mp4 ), additional_folders: '/Depot/j/', start_dir: '/tmp/' }}
  #
  # ========================================================================= #
  def create_a_new_file_chooser_dialog(
      this_widget   = self,
      use_this_text = 'Open File',
      &block
    )
    action_open = :open
    case this_widget
    when nil
      this_widget = self
    end
    # ======================================================================= #
    # Assign to the @file_chooser_dialog variable next.
    # ======================================================================= #
    @file_chooser_dialog = ::Gtk::FileChooserDialog.new(
      title:   use_this_text,
      action:  action_open,
      buttons: [[Gtk::Stock::OPEN,   :accept],
                [Gtk::Stock::CANCEL, :cancel]]
    )
    # ===================================================================== #
    # Give it a name - that way people can re-define the looks as-is
    # via CSS.
    # ===================================================================== #
    @file_chooser_dialog.set_name('custom_file_chooser_dialog')
    @file_chooser_dialog.current_folder = return_pwd
    @file_chooser_dialog.show_hidden = true # Do show hidden files.
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === additional_folders
        #
        # This can be used like so:
        #
        #   additional_folders: '/Depot/j/'
        #
        # =================================================================== #
        if yielded.has_key? :additional_folders
          _ = yielded[:additional_folders]
          if _.is_a? Array # Handle arrays here.
            _.each {|this_entry|
              @file_chooser_dialog.add_this_directory(this_entry)
            }
          else
            @file_chooser_dialog.add_this_directory(_)
          end
        # =================================================================== #
        # === additional_directories
        #
        # This is an alias to the above.
        # =================================================================== #
        elsif yielded.has_key? :additional_directories
          _ = yielded[:additional_directories]
          if _.is_a? Array
            _.flatten.sort_by {|entry| File.basename(entry) }.each {|this_entry|
              @file_chooser_dialog.add_this_directory(this_entry)
            }
          else
            @file_chooser_dialog.add_this_directory(_)
          end
        end
        # =================================================================== #
        # === :start_dir
        #
        # This allows us to use another start-directory.
        # =================================================================== #
        if yielded.has_key? :start_dir
          _ = yielded[:start_dir].to_s
          @file_chooser_dialog.cd_to(_)
        # =================================================================== #
        # === :start_directory
        # =================================================================== #
        elsif yielded.has_key? :start_directory
          _ = yielded[:start_directory].to_s
          @file_chooser_dialog.cd_to(_)
        end
        # =================================================================== #
        # === :current_folder
        #
        # This is different to :start_directory, because the former only
        # cds, whereas the entry here does both.
        # =================================================================== #
        if yielded.has_key? :current_folder
          _ = yielded[:current_folder]
          if _ and File.directory?(_)
            @file_chooser_dialog.current_folder = _
            @file_chooser_dialog.cd_to(_)
          end
        end
        # =================================================================== #
        # === :filter
        #
        # This allows us to filter towards a particular file-type.
        # =================================================================== #
        if yielded.has_key? :filter
          _ = yielded[:filter]
          @file_chooser_dialog.add_filter(
            gtk_file_filter(_) # Apply the file-filter here.
          )
        # =================================================================== #
        # === :filter_for
        # =================================================================== #
        elsif yielded.has_key? :filter_for
          _ = yielded[:filter_for]
          if _.is_a? Array
            file_filter = ::Gtk::FileFilter.new
            _.each {|entry|
              this_pattern = ('*.'+entry).squeeze('.')
              file_filter.add_pattern(this_pattern)
            }
            @file_chooser_dialog.add_filter(file_filter)
          else
            @file_chooser_dialog.add_filter(
              gtk_file_filter(_.to_s) # Apply the file-filter here.
            )
          end
        end
        # =================================================================== #
        # === :extra_widget
        # =================================================================== #
        if yielded.has_key? :extra_widget
          _ = yielded[:extra_widget]
          @file_chooser_dialog.extra_widget = _
        end
        # =================================================================== #
        # === :show_hidden
        # =================================================================== #
        if yielded.has_key? :show_hidden
          _ = yielded[:show_hidden]
          @file_chooser_dialog.show_hidden = _
        end
        # =================================================================== #
        # === :show_hidden_files
        # =================================================================== #
        if yielded.has_key? :show_hidden_files
          @file_chooser_dialog.show_hidden = true # Do show hidden files.
        end
      end
    end
    response_accept = Gtk::ResponseType::ACCEPT
    case @file_chooser_dialog.run
    when response_accept 
      filename = @file_chooser_dialog.filename
      # ===================================================================== #
      # We will next store the filename on the toplevel "namespace". This
      # allows the user to query the designated file via a query-method.
      #
      # This can be queried via: Gtk.main_file?
      # ===================================================================== #
      ::Gtk.set_main_file(filename)
    end
    @file_chooser_dialog.destroy
    return @file_chooser_dialog
  end; alias new_file_chooser_dialog create_a_new_file_chooser_dialog # === new_file_chooser_dialog
       alias open_local_file         create_a_new_file_chooser_dialog # === open_local_file
       alias open_file_dialog        create_a_new_file_chooser_dialog # === open_file_dialog
       alias do_open_a_file          create_a_new_file_chooser_dialog # === do_open_a_file
       alias look_for_a_local_file   create_a_new_file_chooser_dialog # === look_for_a_local_file

  # ========================================================================= #
  # === create_paned_vbox
  # ========================================================================= #
  def create_paned_vbox(*i)
    ::Gtk.create_paned_vbox(i)
  end

  # ========================================================================= #
  # === paned_vbox
  # ========================================================================= #
  def paned_vbox(*i)
    ::Gtk.create_paned_vbox(i)
  end

  # ========================================================================= #
  # === unicode_horizontal_spacer
  # ========================================================================= #
  def unicode_horizontal_spacer
    ::Gtk.unicode_horizontal_spacer
  end; alias horizontal_unicode_bar unicode_horizontal_spacer # === horizontal_unicode_bar

  # ========================================================================= #
  # === gtk_application_window
  #
  # This method is specifically available only for gtk3, as it instantiates
  # a new Gtk::ApplicationWindow.
  # ========================================================================= #
  def gtk_application_window(this_application)
    ::Gtk::ApplicationWindow.new(this_application)
  end

  # ========================================================================= #
  # === report_current_dir
  #
  # will report current dir in a colourful way.
  # ========================================================================= #
  def report_current_dir
    efancy 'We are currently in the directory:'
    e sdir("  #{return_pwd}")
  end

  # ========================================================================= #
  # === popup_this_text_over_that_widget
  #
  # This method can be used, in ruby-gtk3, to popup a specific text
  # over an (existing) widget. First pass in the string as the first
  # argument to the method; then designate the widget where the
  # pop-up will show up.
  #
  # Usage example:
  #
  #   pop_up_this_text_over_that_widget('Backup complete!', @button_backup_audio)
  #
  # ========================================================================= #
  def popup_this_text_over_that_widget(
      text,
      widget
    )
    do_popup(
      {
        message:          text,
        over_this_widget: widget
      } 
    )
  end; alias pop_up_this_text_over_that_widget    popup_this_text_over_that_widget # === pop_up_this_text_over_that_widget
       alias pop_up_this_message_over_that_widget popup_this_text_over_that_widget # === pop_up_this_message_over_that_widget

  # ========================================================================= #
  # === gtk_switch
  #
  # This is a "toggle" button in the sense of having a switch state.
  #
  # The two possible states are "on" and "off".
  # ========================================================================= #
  def gtk_switch
    ::Gtk::Switch.new
  end

  # ========================================================================= #
  # === random_colour
  #
  # This method will return a random HTML colour. The functionality depends
  # on the "colours" gem.
  # ========================================================================= #
  def random_colour
    return_the_html_colours.sample # ← This method is defined in this .rb file.
  end; alias random_html_colour random_colour # === random_html_colour

  # ========================================================================= #
  # === swarn
  # ========================================================================= #
  def swarn(i = '')
    return ::Colours.swarn(i)
  end

  # ========================================================================= #
  # === ewarn
  # ========================================================================= #
  def ewarn(i)
    e swarn(i)
  end

  # ========================================================================= #
  # === efancy
  # ========================================================================= #
  def efancy(
      i,
      use_colours = true
    )
    e sfancy(i, use_colours)
  end

  # ========================================================================= #
  # === sfancy
  # ========================================================================= #
  def sfancy(
      i,
      use_colours = true
    )
    ::Colours.sfancy(i, use_colours)
  end

  # ========================================================================= #
  # === sfile
  # ========================================================================= #
  def sfile(i)
    ::Colours.sfile(i)
  end

  # ========================================================================= #
  # === sdir
  # ========================================================================= #
  def sdir(i)
    ::Colours.sdir(i)
  end

  # ========================================================================= #
  # === steelblue
  # ========================================================================= #
  def steelblue(i)
    ::Colours.steelblue(i)
  end

  # ========================================================================= #
  # === mediumseagreen
  # ========================================================================= #
  def mediumseagreen(i)
    ::Colours.mediumseagreen(i)
  end

  # ========================================================================= #
  # === tomato
  # ========================================================================= #
  def tomato(i)
    ::Colours.tomato(i)
  end

  # ========================================================================= #
  # === lightblue
  # ========================================================================= #
  def lightblue(i)
    ::Colours.tomato(i)
  end

  # ========================================================================= #
  # === crimson
  # ========================================================================= #
  def crimson(i)
    ::Colours.crimson(i)
  end

  # ========================================================================= #
  # === add_table3
  #
  # This variant will also pack the list-store, via .maximal().
  # ========================================================================= #    
  def add_table3(*i)
    _ = gtk_list_store_table3(*i)
    maximal(_, 0)
    return _ # And return it here, so that we may want to re-use it again.
  end

  require 'gtk_paradise/core_classes/statusbar.rb'
  # ========================================================================= #
  # === gtk_statusbar
  #
  # Local documentation is available, on my system, via:
  #
  #   rf ruby gtk_statusbar
  #   http://localhost:10000/gtk#gtk_statusbar
  #
  # Variants may include this:
  #
  #   gtk_statusbar { :with_resize_grip }
  #   gtk_statusbar { :has_resize_grip }
  #   gtk_statusbar { :resize_grip }
  #   gtk_statusbar { :with_a_resize_grip }
  #
  # ========================================================================= #
  def gtk_statusbar(
      optional_use_this_as_status_bar = nil
    )
    statusbar = ::Gtk::Statusbar.new
    if optional_use_this_as_status_bar
      @status_bar_context_id = statusbar.get_context_id(optional_use_this_as_status_bar)
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      case yield
      # ===================================================================== #
      # === :with_resize_grip
      # ===================================================================== #
      when :with_resize_grip,
           :with_a_resize_grip,
           :has_resize_grip,
           :resize_grip,
           :with_handle,
           :grip
        statusbar.has_resize_grip = true if respond_to?(:has_resize_grip)
      end
    end
    return statusbar # And return it here.
  end; alias gtk_status_bar            gtk_statusbar # === gtk_status_bar
       alias status_bar                gtk_statusbar # === status_bar
       alias return_default_status_bar gtk_statusbar # === return_default_status_bar
       alias create_statusbar          gtk_statusbar # === create_statusbar
       alias create_status_bar         gtk_statusbar # === create_status_bar

  # ========================================================================= #
  # === gtk_button_xorg_buffer
  #
  # This will return a new instance of the xorg-buffer button.
  # ========================================================================= #
  def gtk_button_xorg_buffer(
      optional_use_this_as_label = nil
    )
    require 'gtk_paradise/widgets/gtk3/button_xorg_buffer/button_xorg_buffer.rb'
    return ::Gtk::ButtonXorgBuffer.new(optional_use_this_as_label)
  end; alias button_xorg_buffer gtk_button_xorg_buffer # === button_xorg_buffer

  # ========================================================================= #
  # === top_left
  #
  # This method may only work on Gtk::Window, I think.
  # ========================================================================= #
  def top_left
    move(0, 0) if respond_to? :move
  end

  # ========================================================================= #
  # === interactive_change_the_font
  #
  # Unfortunately this does not work as I would have expected, in 2021 ...
  # ========================================================================= #
  def interactive_change_the_font
    vbox = create_vbox
    label = create_label('Enter the font:')
    vbox << label
    entry = gtk_entry
    entry.center
    vbox << entry
    a_quit_button = gtk_button('_Quit')
    vbox << a_quit_button
    r = ::Gtk.run
    r << vbox
    r.top_left_then_run
    a_quit_button.on_clicked {
      set_font(entry.text?)
      r.destroy
    }
  end

  # ========================================================================= #
  # === smiley1
  # ========================================================================= #
  def smiley1
    ::Gtk::Emojis::SMILEY_1
  end

  # ========================================================================= #
  # === smiley2
  # ========================================================================= #
  def smiley2
    ::Gtk::Emojis::SMILEY_2
  end

  # ========================================================================= #
  # === smiley3
  # ========================================================================= #
  def smiley3
    ::Gtk::Emojis::SMILEY_3
  end

  # ========================================================================= #
  # === smiley4
  # ========================================================================= #
  def smiley4
    ::Gtk::Emojis::SMILEY_4
  end

  # ========================================================================= #
  # === smiley5
  # ========================================================================= #
  def smiley5
    ::Gtk::Emojis::SMILEY_5
  end

  # ========================================================================= #
  # === is_on_windows?
  #
  # Query whether we are on a windows-platform (from Microsoft) or not.
  #
  # The method will return true if this is the case.
  # ========================================================================= #
  def is_on_windows?
    Gem.win_platform?
  end

  # ========================================================================= #
  # === is_it_an_enter_key?
  #
  # This method will query whether the given event equals the enter-key.
  #
  # Two different keys may be used for that on most keyboard-devices,
  # which explaines why two when-checks are done, ultimately.
  # ========================================================================= #
  def is_it_an_enter_key?(event)
    case Gdk::Keyval.to_name(event.keyval)
    when 'Return','KP_Enter'
      return true
    else
      return false
    end
  end; alias enter_key?    is_it_an_enter_key? # === enter_key?
       alias is_enter_key? is_it_an_enter_key? # === is_enter_key?

  # ========================================================================= #
  # === absolute_file_path
  # ========================================================================= #
  def absolute_file_path(i)
    File.absolute_path(i)
  end

  # ========================================================================= #
  # === add_a_horizontal_spacer
  #
  # This variant will automatically add the horizontal-spacer as-is.
  # ========================================================================= #
  def add_a_horizontal_spacer
    add(horizontal_spacer)
  end

  # ========================================================================= #
  # === vertical_spacer
  # ========================================================================= #
  def vertical_spacer
    ::Gtk.vertical_spacer
  end; alias vspacer     vertical_spacer # === vspacer
       alias gtk_vspacer vertical_spacer # === gtk_vspacer

  # ========================================================================= #
  # === spacer
  # ========================================================================= #
  def spacer
    ::Gtk.spacer
  end; alias gtk_spacer spacer # === gtk_spacer

  # ========================================================================= #
  # === gtk_menu
  # ========================================================================= #
  def gtk_menu
    ::Gtk::Menu.new
  end; alias create_menu gtk_menu # === create_menu

  # ========================================================================= #
  # === return_editable_cell_renderer_text
  # ========================================================================= #
  def return_editable_cell_renderer_text
    ::Gtk.return_editable_cell_renderer_text
  end

  # ========================================================================= #
  # === gtk_size_group
  # ========================================================================= #
  def gtk_size_group(
      i = :horizontal
    )
    orientation_to_use = :horizontal
    case i
    when :horizontal
      orientation_to_use = :horizontal
    end
    return ::Gtk::SizeGroup.new(orientation_to_use)
  end

  # ========================================================================= #
  # === response_type_ok?
  # ========================================================================= #
  def response_type_ok?(i)
    i == ::Gtk::ResponseType::OK
  end

  # ========================================================================= #
  # === gtk_colour_selection_dialog
  # ========================================================================= #
  def gtk_colour_selection_dialog(
      use_this_title = 'Select a color'
    )
    _ = ::Gtk::ColorSelectionDialog.new(title: use_this_title)
    _colour_selection = _.color_selection
    colour = _.current_rgba
    e colour
    return _
  end; alias colour_selection_dialog gtk_colour_selection_dialog # === colour_selection_dialog

  # ========================================================================= #
  # === gtk_menu_item
  #
  # This can be used like so:
  #
  #   filem1 = gtk_menu_item('File') {{
  #     submenu: filemenu
  #   }}
  #
  # Or so:
  #
  #   gtk_menu_item(label: 'Test1')
  #   gtk_menu_item('Test1')
  #
  # ========================================================================= #
  def gtk_menu_item(
      i             = 'File',
      use_underline = false,
      &block
    )
    if i and i.respond_to?(:include?) and i.include?('_')
      use_underline = true
    end
    hash = {}
    if i.is_a? Hash
      # In this case we can just pass-through, as-is.
    else
      i = { label: i } # And in that cased we have to turn it into a Hash first.
    end
    hash.merge!(i)
    hash.merge!({use_underline: use_underline})
    menu_item = ::Gtk::MenuItem.new(hash)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        if yielded.has_key? :submenu
          menu_item.set_submenu(yielded[:submenu])
        end
      elsif yielded.is_a? Symbol
        case yielded
        # =================================================================== #
        # === :on_activate_quit
        # =================================================================== #
        when :on_activate_quit
          menu_item.on_activate { Gtk.main_quit }
        end
      end
    end
    return menu_item
  end; alias menu_item        gtk_menu_item # === menu_item
       alias create_menu_item gtk_menu_item # === create_menu_item

  # ========================================================================= #
  # === window_or_vbox
  #
  # This is currently only a vbox. In the future this may be changed,
  # but for now (November 2021) this is how it is.
  #
  # Note that currently the optional_menu argument is ignored.
  # ========================================================================= #
  def window_or_vbox(
      optional_title  = '',
      optional_width  = 800,
      optional_height = 640,
      optional_menu   =   0 # This may also be a Hash.
    )
    _ = vbox
    if optional_title
      _.set_title(optional_title) if _.respond_to?(:set_title)
    end
    if optional_width or optional_height
      _.set_size_request(optional_width, optional_height)
    end
    return _
  end

  # ========================================================================= #
  # === modify_bold_label
  #
  # Delegate towards Gtk.modify_bold_label() here.
  # ========================================================================= #
  def modify_bold_label(
      name,
      colour    = 'black', # This here is the foreground label.
      justify   = '',
      alignment = ''
    )
    ::Gtk.modify_bold_label(
      name,
      colour,
      justify,
      alignment
    )
  end

  # ========================================================================= #
  # === gtk_text_view
  #
  # Instantiate a new TreeView widget, by delegating towards Gtk.textview().
  #
  # Since as of November 2020 the first entry is the text-buffer that
  # is to be used.
  #
  # Some symbols can be passed into this method, for its first argument,
  # such as :default_buffer.
  # ========================================================================= #
  def gtk_text_view(
      use_this_text_buffer = ::Gtk::TextBuffer.new,
      left_margin          = 2,
      right_margin         = 2,
      is_editable          = true
    )
    ::Gtk.textview(
      use_this_text_buffer,
      left_margin,
      right_margin,
      is_editable
    )
  end; alias gtk_textview     gtk_text_view # === gtk_textview
       alias gtk_textbox      gtk_text_view # === gtk_textbox
       alias gtk_textarea     gtk_text_view # === gtk_textarea
       alias gtk_text_area    gtk_text_view # === gtk_text_area
       alias gtk_text_viewer  gtk_text_view # === gtk_text_viewer
       alias textview         gtk_text_view # === textview
       alias text_view        gtk_text_view # === text_view
       alias create_text_view gtk_text_view # === create_text_view
       alias create_textarea  gtk_text_view # === create_textarea
       alias create_textfield gtk_text_view # === create_textfield

  # ========================================================================= #
  # === default_entry
  #
  # This creates a default gtk-entry, and then returns it. The CSS
  # class is defined in the file project.css.
  # ========================================================================= #
  def default_entry
    _ = entry
    _.css_class('default_entry')
    return _
  end

  # ========================================================================= #
  # === padded_hbox
  # ========================================================================= #
  def padded_hbox
    _ = ::Gtk.hbox
    _.pad5px
    return _
  end

  # ========================================================================= #
  # === paned_hbox
  # ========================================================================= #
  def paned_hbox(*i)
    ::Gtk.create_hbox(i) # This has been changed in January 2024.
  end; alias create_paned_hbox paned_hbox # === create_paned_hbox

  # ========================================================================= #
  # === use_gtk2?
  # ========================================================================= #
  def use_gtk2?
    ::Gtk.use_gtk2?
  end

  # ========================================================================= #
  # === create_clipboard
  # ========================================================================= #
  def create_clipboard
    ::Gtk.create_clipboard
  end

  # ========================================================================= #
  # === clipboard_text?
  # ========================================================================= #
  def clipboard_text?
    ::Gtk.clipboard_text?
  end

  # ========================================================================= #
  # === return_popover
  #
  # The second argument of this method is the widget to which the
  # popover should appear.
  #
  # Do not forget to invoke .popup() to show the widget.
  #
  # Usage example:
  #
  #   return_popover('The exam topic can be set to the right side, or below this text.')
  #   return_popover('Use this title.', use_this_widget, padding_value_to_use_20)
  #   return_popover('The exam topic can be set to the right side, or below this text.', @entry, 20)
  #   return_popover('The exam topic can be set to the right side, or below this text.', @entry, 20).popup
  #
  # ========================================================================= #
  def return_popover(
      title_to_use              = '',          # (1) Title (or widget) to use
      this_widget               = ::Gtk.main_image?, # (2) The widget relative to the popover widget
      padding_to_use            = 15,          # (3) How much to pad
      use_this_container_widget = nil,         # (4) The container widget to use
      &block
    )
    case padding_to_use
    # ======================================================================= #
    # === :default
    # ======================================================================= #
    when :default,
         :default_padding,
         :default_padding_to_use
      padding_to_use = 15
    end
    # ======================================================================= #
    # Create the popover next.
    # ======================================================================= #
    @popover = gtk_popover
    # ======================================================================= #
    # === Handle Hashes next
    # ======================================================================= #
    if this_widget.is_a? Hash
      # ===================================================================== #
      # === :font_to_use
      # ===================================================================== #
      if this_widget.has_key? :font_to_use
        _ = this_widget[:font_to_use]
        if _.is_a? Symbol
          _ = ::Gtk.expand_this_font(_)
        end
        @popover.override_font(Pango::FontDescription.new(_))
      end
      # ===================================================================== #
      # === :use_this_widget
      #
      # This should come last in this if-clause.
      # ===================================================================== #
      if this_widget.has_key? :use_this_widget
        this_widget = this_widget[:use_this_widget]
      end
    end
    use_this_label = title_to_use
    if use_this_label.is_a? String
      # ===================================================================== #
      # Turn it into a gtk-label (Gtk::Label) in this case:
      # ===================================================================== #
      use_this_label = gtk_label(title_to_use)
      use_this_label.do_markify
      if block_given? and (yield == :make_selectable)
        use_this_label.make_selectable
        use_this_label.deselect_everything
      end
    end
    # ======================================================================= #
    # Determine which container-widget to use next.
    # ======================================================================= #
    if use_this_container_widget
      smaller_vbox = gtk_vbox
      smaller_vbox.minimal(use_this_label) # Always use a label nonetheless.
      smaller_vbox.maximal(use_this_container_widget)
      @popover.add(smaller_vbox)
    else
      vbox = create_vbox
      vbox.pack_start(
        use_this_label,
        fill: false,
        expand: true,
        padding: padding_to_use
      )
      @popover.add(vbox)
    end
    @popover.set_position(:bottom)
    @popover.set_relative_to(this_widget)
    @popover.show_all
    # @popover.popup # ← Use this in your application, if you want the popover to show.
    return @popover
  end; alias popover_message return_popover # === popover_message
       alias popover_message return_popover # === popover_message
       alias popover         return_popover # === popover

  # ========================================================================= #
  # === to_font_description
  #
  # Use a simplified wrapper over Pango::FontDescription here.
  # ========================================================================= #
  def to_font_description(i)
    return Pango::FontDescription.new(::Gtk.expand_this_font(i))
  end

  # ========================================================================= #
  # === gtk_hbox
  #
  # Wrapper-method towards Gtk::HBox. In other words, this method will
  # create a new horizontal-box.
  #
  # Note that you can also pass a Hash, as a block.
  #
  # Example:
  #
  #   hbox = gtk_hbox {{ spacing: 10 }}
  #
  # ========================================================================= #
  def gtk_hbox(
      *i,
      &block
    )
    if i and i.is_a?(Array) and i.empty? and block_given?
      i = [yield]
    end
    if i.is_a? Array
      first = i.first
      # ===================================================================== #
      # === Handle the situation when the first element is a Hash
      # ===================================================================== #
      if first.is_a? Hash
        new_result = []
        # =================================================================== #
        # {:label=>"File path:", :entry=>#<Gtk::Entry:0x7fae47a04cc0 ptr=0x7fae481e6450>}
        # =================================================================== #
        first.each_pair {|key, value|
          case key
          # ================================================================= #
          # === :label
          # ================================================================= #
          when :label
            new_result << ::Gtk.label(value)
          # ================================================================= #
          # === :entry
          # ================================================================= #
          when :entry
            new_result << ::Gtk.entry # (value)
          # ================================================================= #
          # === :button
          # ================================================================= #
          when :button
            new_result << ::Gtk.button
          end
        }
        i = new_result unless new_result.empty?
      end
    end
    i.flatten! if i.is_a? Array
    ::Gtk.hbox(i, &block)
  end; alias create_hbox  gtk_hbox # === create_hbox
       alias hbox         gtk_hbox # === hbox
       alias default_hbox gtk_hbox # === default_hbox

  # ========================================================================= #
  # === image_document_open
  # ========================================================================= #
  def image_document_open
    return_this_image_based_on_the_default_icon_theme(:document_open)
  end; alias document_open   image_document_open # === document_open
       alias image_open_file image_document_open # === image_open_file
       alias image_file_open image_document_open # === image_file_open

  # ========================================================================= #
  # === title_width_height_font
  #
  # This variant will delegate to two other methods.
  # ========================================================================= #
  def title_width_height_font(
      title  = TITLE,
      width  = WIDTH,
      height = HEIGHT,
      font   = USE_THIS_FONT
    )
    title_width_height(title, width, height)
    set_use_this_font(font)
  end

  # ========================================================================= #
  # === title_font
  # ========================================================================= #
  def title_font(
      title = TITLE,
      font  = USE_THIS_FONT
    )
    set_my_title(title)
    set_use_this_font(font)
  end

  # ========================================================================= #
  # === title_width_height
  #
  # This variant combines three methods, as well as a prior check before
  # calling the method at hand.
  #
  # The if-checks are spaced out over several lines, in order to allow
  # us to add debug-calls if need be.
  # ========================================================================= #
  def title_width_height(
      title  = TITLE,
      width  = WIDTH,
      height = HEIGHT
    )
    # ======================================================================= #
    # === title
    # ======================================================================= #
    set_my_title(title)
    # ======================================================================= #
    # === width
    # ======================================================================= #
    if respond_to? :set_width
      set_width(width)
    end
    # ======================================================================= #
    # === height
    # ======================================================================= #
    if respond_to? :set_height
      set_height(height)
    end
  end

  # ========================================================================= #
  # === gtk_tree_view_column
  #
  # The second argument to this method should be the gtk cell-renderer,
  # which defaults to a textual representation. This object can then
  # also be manipulated via invocations such as:
  #
  #   renderer.foreground = "#ff0000"
  #
  # To use this, do something like this:
  #
  #   gtk_tree_view_column('Buy', gtk_cell_renderer, text: BUY_IT)
  #   gtk_tree_view_column('Item', gtk_cell_renderer_centered_text, text: BUY_IT)
  #
  # ========================================================================= #
  def gtk_tree_view_column(
      title_of_the_column = 'Testing',
      gtk_renderer_to_use = gtk_cell_renderer_text, # or gtk_cell_renderer_centered_text
      text_position       = 0
    )
    # ======================================================================= #
    # Not sure why the following code queried for a Hash. If I ever find
    # out, I will note it down back here, but for now it is commented
    # out. (06.12.2020)
    #if text_position.is_a? Hash
    #  if text_position.has_key? :text
    #    text_position = text_position.delete(:text)
    #  end
    #elsif text_position.is_a? Integer
    # ======================================================================= #
    if text_position.is_a? Integer
      text_position = { text: text_position }
    end
    ::Gtk::TreeViewColumn.new(
      title_of_the_column,
      gtk_renderer_to_use,
      text_position
    )
  end; alias tree_view_column gtk_tree_view_column # === gtk_tree_view_column

  # ========================================================================= #
  # === check_for_a_manifest_yaml_file
  #
  # This method will query whether a file called manifest.yml exists. If so
  # it will load this dataset and evaluate it. Unfortunately this does
  # not work properly right now (February 2021); I may have to re-evaluate
  # this again in the future.  
  # ========================================================================= #
  def check_for_a_manifest_yaml_file
    _ = 'manifest.yml'
    if @configuration
      this_file = @configuration[1]+'/'+_
      if File.exist? this_file
        dataset = load_this_yaml_file(this_file)
        # =================================================================== #
        # Next, parse it.
        # =================================================================== #
        if dataset.has_key? 'buttons'
          dataset['buttons'].each_pair {|name_of_the_instance_variable, value|
            # =============================================================== #
            # First refer to the Hash, so that we can re-use it a later time.
            # =============================================================== #
            hash = dataset['buttons'][name_of_the_instance_variable]
            if name_of_the_instance_variable.frozen?
              name_of_the_instance_variable = name_of_the_instance_variable.dup
            end
            unless name_of_the_instance_variable.start_with? '@'
              name_of_the_instance_variable[0,0] = '@'
            end
            _ = gtk_button
            instance_variable_set(name_of_the_instance_variable.to_sym, _)
            hash.each_pair {|key, value|
              case key
              when 'text'
                _.set_text(value)
              when 'use_mnemonic'
                old_text = _.text?
                _.set_text(old_text, use_mnemonics: true)
              when 'colour','color'
                override_color(:normal, value.to_sym)
              when 'tooltip'
                _.hint = value
              when 'weight'
                if value == 'bold'
                  _.make_bold
                end
              when 'width'
                _.width_height(value, hash['height'])
              end
            }
          }
        end
      end
    end
  end

  # ========================================================================= #
  # === gtk_bold_frame
  # ========================================================================= #
  def gtk_bold_frame(i)
    frame = gtk_frame(i)
    frame.make_bold
    return frame
  end

  # ========================================================================= #
  # === gdk_screen
  # ========================================================================= #
  def gdk_screen
    Gdk::Screen.default
  end

  # ========================================================================= #
  # === gdk_selection_clipboard
  # ========================================================================= #
  def gdk_selection_clipboard
    ::Gdk::Selection::CLIPBOARD
  end

  # ========================================================================= #
  # === to_utf
  # ========================================================================= #
  def to_utf(i = '')
    ::Gtk.to_utf8(i)
  end

  # ========================================================================= #
  # === gtk_alignment
  #
  # The first argument is the uniform-padding value that is to be used.
  #
  # If more than one argument is given, the method will assign 
  # the corresponding alignment.
  # ========================================================================= #
  def gtk_alignment(
      a = 0.98,
      b = nil,
      c = nil,
      d = nil
    )
    # ======================================================================= #
    # The next three lines are to default to "uniform padding values".
    # ======================================================================= #
    case a
    # ======================================================================= #
    # === :center
    #
    # This is simply an alias really, to allow something like:
    #
    #   gtk_alignment(:center)
    #
    # ======================================================================= #
    when :center
      a = 0.5; b = 0.5; c = 0; d = 0
    end
    b = a if b.nil?
    c = a if c.nil?
    d = a if d.nil?
    ::Gtk::Alignment.new(a, b, c, d)
  end

  # ========================================================================= #
  # === rev                                                         (rev tag)
  # ========================================================================= #
  def rev
    if Object.const_defined?(:Colours)
      return ::Colours.rev
    else
      return ''
    end
  end

  # ========================================================================= #
  # === create_combobox_text
  # ========================================================================= #
  def create_combobox_text(
      array = []
    )
    _ = ::Gtk::ComboBoxText.new
    if array and !array.empty?
      array.each {|entry|
        _.append_text(entry)
      }
    end
    _.set_active(0)
    return _
  end; alias dropdown_box_text create_combobox_text # === dropdown_box_text

  # ========================================================================= #
  # === return_combo_box_entry
  #
  # This method will return a new instance of Gtk::ComboBoxEntry. However
  # had, this widget no longer exists in ruby-gtk3, so for ruby-gtk3 we
  # have to use a slightly different way.
  # ========================================================================= #
  def return_combo_box_entry(i)
    combo_box_entry = ::Gtk::ComboBoxText.new
    if i.is_a? Hash
      if i.has_key? :with_these_elements
        i = i.delete(:with_these_elements)
      end
    end
    i.each { |value|
      combo_box_entry.append_text(value.to_s)
    }
    return combo_box_entry
  end

  # ========================================================================= #
  # === gtk_markup
  #
  # This method will automatically apply .use_markup = true. It returns
  # this Gtk::Label instance as well.
  # ========================================================================= #
  def gtk_markup(
      i = ''
    )
    label = ::Gtk::Label.new(i)
    label.use_markup = true
    return label
  end

  # ========================================================================= #
  # === gtk_source_view
  #
  # Usage example:
  #
  #   gtk_sourceview { :sane_defaults }
  #
  # ========================================================================= #
  def gtk_source_view(
      optional_buffer = nil,
      &block
    )
    require 'gtk_paradise/core_classes/source_view.rb'
    if optional_buffer.nil?
      source_view = ::Gtk::SourceView.new
    else
      source_view = ::Gtk::SourceView.new(optional_buffer)
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :do_use_sane_defaults
      # ===================================================================== #
      when :do_use_sane_defaults,
           :sane_defaults
        source_view.do_use_sane_defaults
      end
    end
    return source_view
  end; alias gtk_sourceview gtk_source_view # === gtk_sourceview

  # ========================================================================= #
  # === create_sourceview
  #
  # Note that this method is different to gtk_sourceview.
  # ========================================================================= #
  def create_sourceview
    text_buffer = create_text_buffer
    text_view   = create_text_view(text_buffer)
    source_view = create_scrolled_window(text_view)
    return source_view
  end; alias create_source_view create_sourceview # === create_source_view

  # ========================================================================= #
  # === return_top_bar
  #
  # This method can be customized via blocks such as:
  #
  #   return_default_header_bar { :with_close_button_and_inferred_title }
  #
  # ========================================================================= #
  def return_top_bar(&block)
    if block_given?
      header_bar = gtk_header_bar(&block)
    else
      header_bar = gtk_header_bar { :with_close_button }
    end
    header_bar.spacing = 2
    return header_bar
  end; alias top_bar                   return_top_bar # === top_bar
       alias default_header_bar        return_top_bar # === default_header_bar
       alias default_header            return_top_bar # === default_header_bar
       alias default_top_bar           return_top_bar # === default_top_bar
       alias default_topbar            return_top_bar # === default_topbar
       alias gtk_top_bar               return_top_bar # === gtk_top_bar
       alias return_default_header_bar return_top_bar # === return_default_header_bar
       alias return_default_top_bar    return_top_bar # === return_default_top_bar

  # ========================================================================= #
  # === reset_the_internal_variables_as_well_as_the_internal_hash
  # ========================================================================= #
  def reset_the_internal_variables_as_well_as_the_internal_hash
    reset_the_internal_variables # This will also invoke reset_the_internal_hash().
  end

  # ========================================================================= #
  # === centered_text
  # ========================================================================= #
  def centered_text(
      i = ''
    )
    _ = text(i)
    _.align_to_the_center # hcenter
    return _
  end

  # ========================================================================= #
  # === pixbuf_loader
  # ========================================================================= #
  def pixbuf_loader
    GdkPixbuf::PixbufLoader.new
  end

  # ========================================================================= #
  # === abort_on_exception
  # ========================================================================= #
  def abort_on_exception
    Thread.abort_on_exception = true
  end

  # ========================================================================= #
  # === image_from_file
  # ========================================================================= #
  def image_from_file(this_file)
    ::Gtk::Image.new(
      file: this_file
    )
  end; alias gtk_image_from_file image_from_file # === gtk_image_from_file

  # ========================================================================= #
  # === image_emblem_urgent
  # ========================================================================= #
  def image_emblem_urgent
    return_this_image_based_on_the_default_icon_theme(:emblem_urgent)
  end

  # ========================================================================= #
  # === image_computer
  #
  # This uses a cute little computer-image, if the icon theme supports
  # it.
  # ========================================================================= #
  def image_computer
    return_this_image_based_on_the_default_icon_theme(:computer)
  end

  # ========================================================================= #
  # === image_application_exit
  # ========================================================================= #
  def image_application_exit
    return_this_image_based_on_the_default_icon_theme(:application_exit)
  end

  # ========================================================================= #
  # === image_document_revert
  # ========================================================================= #
  def image_document_revert
    return_this_image_based_on_the_default_icon_theme(:document_revert)
  end; alias document_revert image_document_revert # === document_revert

  # ========================================================================= #
  # === image_system_search
  # ========================================================================= #
  def image_system_search
    return_this_image_based_on_the_default_icon_theme(:system_search)
  end; alias system_search image_system_search # === system_search

  # ========================================================================= #
  # === image_audio_speakers
  # ========================================================================= #
  def image_audio_speakers
    return_this_image_based_on_the_default_icon_theme(:audio_speakers)
  end

  # ========================================================================= #
  # === image_audio_x_generic_symbolic
  # ========================================================================= #
  def image_audio_x_generic_symbolic
    return_this_image_based_on_the_default_icon_theme('audio-x-generic-symbolic.symbolic')
  end

  # ========================================================================= #
  # === image_from_pixbuf
  # ========================================================================= #
  def image_from_pixbuf(i)
    ::Gtk::Image.new(
      pixbuf: i
    )
  end; alias gtk_image_from_pixbuf image_from_pixbuf # === gtk_image_from_pixbuf
       alias gtk_image_pixbuf      image_from_pixbuf # === gtk_image_pixbuf

  # ========================================================================= #
  # === image_applications_system
  # ========================================================================= #
  def image_applications_system
    return_this_image_based_on_the_default_icon_theme(:image_applications_system)
  end

  # ========================================================================= #
  # === image_accessories_text_editor_symbolic_symbolic
  # ========================================================================= #
  def image_accessories_text_editor_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme(:accessories_text_editor_symbolic_symbolic)
  end

  # ========================================================================= #
  # === image_inode_directory
  # ========================================================================= #
  def image_inode_directory
    return_this_image_based_on_the_default_icon_theme(:inode_directory)
  end; alias image_directory image_inode_directory # === image_directory

  # ========================================================================= #
  # === image_network_cellular_connected_symbolic_symbolic
  # ========================================================================= #
  def image_network_cellular_connected_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme(:network_cellular_connected_symbolic_symbolic)
  end

  # ========================================================================= #
  # === image_cellular_signal_excellent_symbolic_symbolic
  # ========================================================================= #
  def image_cellular_signal_excellent_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme(:network_cellular_signal_excellent_symbolic_symbolic)
  end; alias network_cellular_signal_excellent_symbolic_symbolic image_cellular_signal_excellent_symbolic_symbolic # === network_cellular_signal_excellent_symbolic_symbolic

  # ========================================================================= #
  # === image_trophy_gold
  # ========================================================================= #
  def image_trophy_gold
    return_this_image_based_on_the_default_icon_theme(:trophy_gold)
  end

  # ========================================================================= #
  # === image_avatar_default
  # ========================================================================= #
  def image_avatar_default
    return_this_image_based_on_the_default_icon_theme('avatar-default')
  end

  # ========================================================================= #
  # === image_applications_system_symbolic_symbolic
  # ========================================================================= #
  def image_applications_system_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme('applications-system-symbolic.symbolic')
  end

  # ========================================================================= #
  # === image_play_button
  # ========================================================================= #
  def image_play_button
    return_this_image_based_on_the_default_icon_theme(:video_x_generic)
  end

  # ========================================================================= #
  # === image_software_update_urgent
  # ========================================================================= #
  def image_software_update_urgent
    return_this_image_based_on_the_default_icon_theme(:software_update_urgent)
  end

  # ========================================================================= #
  # === image_accessories_text_editor
  # ========================================================================= #
  def image_accessories_text_editor
    return_this_image_based_on_the_default_icon_theme(:accessories_text_editor)
  end; alias image_text_editor image_accessories_text_editor # === image_text_editor

  # ========================================================================= #
  # === image_bookmark_new
  # ========================================================================= #
  def image_bookmark_new
    return_this_image_based_on_the_default_icon_theme(:bookmark_new)
  end

  # ========================================================================= #
  # === image_list_remove
  # ========================================================================= #
  def image_list_remove
    return_this_image_based_on_the_default_icon_theme('list-remove')
  end

  # ========================================================================= #
  # === image_media_optical
  # ========================================================================= #
  def image_media_optical
    return_this_image_based_on_the_default_icon_theme('media-optical')
  end

  # ========================================================================= #
  # === image_go_up
  # ========================================================================= #
  def image_go_up
    return_this_image_based_on_the_default_icon_theme('go-up')
  end; alias arrow_up image_go_up # === arrow_up

  # ========================================================================= #
  # === image_go_down
  # ========================================================================= #
  def image_go_down
    return_this_image_based_on_the_default_icon_theme('go-down')
  end; alias arrow_down image_go_down # === arrow_down

  # ========================================================================= #
  # === image_dialog_question
  # ========================================================================= #
  def image_dialog_question
    return_this_image_based_on_the_default_icon_theme(:dialog_question)
  end; alias image_question image_dialog_question # === image_question

  # ========================================================================= #
  # === image_information
  # ========================================================================= #
  def image_information
    return_this_image_based_on_the_default_icon_theme(
      'dialog-information'
    )
  end

  # ========================================================================= #
  # === image_network_wireless_signal_excellent_symbolic
  # ========================================================================= #
  def image_network_wireless_signal_excellent_symbolic
    return_this_image_based_on_the_default_icon_theme(
      'network-wireless-signal-excellent-symbolic.symbolic'
    )
  end

  # ========================================================================= #
  # === image_folder_visiting_symbolic
  # ========================================================================= #
  def image_folder_visiting_symbolic
    return_this_image_based_on_the_default_icon_theme(
      'folder-visiting-symbolic.symbolic'
    )
  end

  # ========================================================================= #
  # === image_web_browser
  # ========================================================================= #
  def image_web_browser
    return_this_image_based_on_the_default_icon_theme(
      'web_browser'
    )
  end

  # ========================================================================= #
  # === minimal_spacer
  # ========================================================================= #
  def minimal_spacer
    add(horizontal_spacer)
  end

  # ========================================================================= #
  # === horizontal_spacer
  # ========================================================================= #
  def horizontal_spacer
    ::Gtk.horizontal_spacer
  end; alias hspacer               horizontal_spacer # === hspacer
       alias gtk_hspacer           horizontal_spacer # === gtk_hspacer
       alias gtk_horizontal_spacer horizontal_spacer # === gtk_horizontal_spacer

  # ========================================================================= #
  # === add_horizontal_spacer
  # ========================================================================= #
  def add_horizontal_spacer
    add(gtk_horizontal_spacer)
  end

  # ========================================================================= #
  # === return_inactive_spinner
  # ========================================================================= #
  def return_inactive_spinner
    spinner = default_spinner
    spinner.set_active(false)
    return spinner
  end; alias inactive_spinner return_inactive_spinner # === inactive_spinner

  # ========================================================================= #
  # === return_active_spinner
  # ========================================================================= #
  def return_active_spinner
    spinner = default_spinner
    spinner.set_active(true)
    return spinner
  end; alias active_spinner return_active_spinner # === active_spinner

  # ========================================================================= #
  # === gtk_message_dialog
  #
  # This method will create a new instance of Gtk::MessageDialog,
  # without much further ado.
  # ========================================================================= #
  def gtk_message_dialog(
      show_this_text = '',
      parent_widget  = :new_window
    )
    case parent_widget
    # ======================================================================= #
    # === :new_window
    # ======================================================================= #
    when :new_window,
         :window
      parent_widget = ::Gtk::Window.new
    end
    # ======================================================================= #
    # Instantiate a new message-dialog next.
    # ======================================================================= #
    # if use_gtk2?
    #   message_dialog = ::Gtk::MessageDialog.new(
    #     parent_widget,
    #     Gtk::Dialog::MODAL,
    #     Gtk::MessageDialog::ERROR,
    #     Gtk::MessageDialog::BUTTONS_CLOSE,
    #     show_this_text
    #   )
    # else # This is for ruby-gtk3, but it currently does not work properly.
    message_dialog = ::Gtk::MessageDialog.new(
      parent_widget,
      Gtk::Dialog::MODAL,
      Gtk::MessageDialog::ERROR,
      Gtk::MessageDialog::BUTTONS_CLOSE,
      show_this_text
    )
    # ======================================================================= #
    # Then invoke:
    #
    #   message_dialog.run
    #   message_dialog.destroy
    #
    # ======================================================================= #
    return message_dialog
  end; alias gtk_message    gtk_message_dialog # === gtk_message
       alias message_box    gtk_message_dialog # === message_box
       alias message_dialog gtk_message_dialog # === message_dialog

  # ========================================================================= #
  # === gtk_coloured_bar
  # ========================================================================= #
  def gtk_coloured_bar
    ::Gtk::ColouredBar.new
  end

  # ========================================================================= #
  # === cell_renderer_progress
  # ========================================================================= #
  def cell_renderer_progress(i = nil)
    _ = ::Gtk::CellRendererProgress.new
    _.set_value(i) if i
    return _
  end

  # ========================================================================= #
  # === left_label
  #
  # This will align the text to the left-side of the gtk-label at hand.
  # ========================================================================= #
  def left_label(i = '')
    _ = gtk_label(i)
    _.align_to_the_left
    return _
  end

  # ========================================================================= #
  # === gtk_left_aligned_bold_label
  # ========================================================================= #
  def gtk_left_aligned_bold_label(i = nil)
    _ = gtk_text_left_aligned(i)
    _.make_bold
    return _
  end; alias left_aligned_bold_label gtk_left_aligned_bold_label # === left_aligned_bold_label

  # ========================================================================= #
  # === standard_file_chooser_dialog
  # ========================================================================= #
  def standard_file_chooser_dialog(title, widget)
    ::Gtk.send(__method__, title, widget)
  end

  # ========================================================================= #
  # === directory_content_widget
  # ========================================================================= #
  def directory_content_widget(
      i = return_pwd,
      &block
    )
    ::Gtk::DirectoryContentWidget.new(i, &block)
  end; alias return_directory_widget directory_content_widget # === return_directory_widget

  # ========================================================================= #
  # === gtk_eventbox
  #
  # Wrapper to creating a new Gtk::EventBox instance.
  # ========================================================================= #
  def gtk_eventbox(i = nil)
    ::Gtk.eventbox(i)
  end; alias gtk_event_box    gtk_eventbox # === gtk_event_box
       alias event_box        gtk_eventbox # === event_box
       alias eventbox         gtk_eventbox # === eventbox
       alias create_eventbox  gtk_eventbox # === create_eventbox
       alias create_event_box gtk_eventbox # === create_event_box

  # ========================================================================= #
  # === label_then_entry
  #
  # The second argument, called optional_commands, can be used to
  # e. g. make the label bold.
  # ========================================================================= #
  def label_then_entry(
      i                 = 'Merge these images together:',
      optional_commands = nil
    )
    hbox = create_hbox
    label = gtk_label(i)
    if label
      case optional_commands
      when :make_bold
        label.make_bold
      end
      if use_gtk2?
        hbox.pack_start(label, false, false)
        hbox.pack_start(gtk_entry, true, true)
      else
        hbox.pack_start(label, expand: false, fill: false)
        hbox.pack_start(gtk_entry, expand: true, fill: true)
      end
    end
    hbox
  end

  # ========================================================================= #
  # === create_centered_entry
  # ========================================================================= #
  def create_centered_entry(i = '')
    _ = create_entry(i)
    _.center
    return _
  end

  # ========================================================================= #
  # === gtk_base_module
  # ========================================================================= #
  def gtk_base_module
    ::Gtk::BaseModule
  end

  # ========================================================================= #
  # === set_this_widget
  #
  # Note that this will not work for every widget; the main use case is to
  # leverage it for gtk-frame widgets.
  # ========================================================================= #
  def set_this_widget(i)
    set_label_widget(i)
  end

  # ========================================================================= #
  # === percentage_widget
  # ========================================================================= #
  def percentage_widget(
      lower_bound = 50,
      upper_bound = 100
    )
    current_value = (lower_bound.to_f * 100) / upper_bound
    progress_bar = ::Gtk::ProgressBar.new
    progress_bar.set_name('progressbar1') # Enable default styling here.
    new_value = 100.0 / current_value
    if new_value.is_a? Float
      new_value = new_value.round(2)
    end
    progress_bar.set_fraction(new_value) # n%.
    return progress_bar
  end

  # ========================================================================= #
  # === percentage_widget_with_this_text
  #
  # Note that the :default symbol means to show the percentage value
  # calculated.
  # ========================================================================= #
  def percentage_widget_with_this_text(
      lower_bound = 50,
      upper_bound = 100,
      this_text   = :default
    )
    case this_text
    when :default
      this_text = (
        (lower_bound.to_f * 100) / upper_bound
      ).to_s.to_f.round(2).to_s+'%'
    end
    progress_bar = percentage_widget(lower_bound, upper_bound)
    progress_bar.set_show_text(true)
    progress_bar.set_text(this_text)
    return progress_bar
  end; alias percentage_widget_with_text percentage_widget_with_this_text # === percentage_widget_with_text

  require 'gtk_paradise/core_classes/radio_menu_item.rb'
  # ========================================================================= #
  # === gtk_radio_menu_item
  # ========================================================================= #
  def gtk_radio_menu_item(
      group_to_use = nil, label_to_use = ''
    )
    ::Gtk.gtk_radio_menu_item(
      group_to_use, label_to_use
    )
  end

  # ========================================================================= #
  # === vte_terminal
  #
  # This method will instantiate a new vte-terminal.
  #
  # You can pass in a hash to the block, such as in this way:
  #
  #   terminal = vte_terminal {{ font: USE_THIS_FONT }}
  #
  # ========================================================================= #
  def vte_terminal
    terminal = Vte::Terminal.new
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :font
        # =================================================================== #
        if yielded.has_key? :font
          terminal.use_this_font = yielded[:font]
        end
      end
    end
    return terminal
  end

  # ========================================================================= #
  # === gtk_scrollbar
  # ========================================================================= #
  def gtk_scrollbar(a = :vertical, b = @terminal.vadjustment)
    ::Gtk::Scrollbar.new(a, b)
  end

  # ========================================================================= #
  # === gtk_pixbuf
  #
  # This is essentially just a wrapper over:
  #
  #   image = Gtk::Image.new(pixbuf: icon) 
  #
  # I am not sure whether it will be kept as it is, as the name is a
  # bit of a misnomer.
  # ========================================================================= #
  def gtk_pixbuf(
      i, optional_width = nil, optional_height = nil
    )
    if optional_width
      ::Gtk::Image.new(pixbuf: i, width: optional_width, height: optional_height)
    else
      ::Gtk::Image.new(pixbuf: i)
    end
  end; alias gdk_pixbuf gtk_pixbuf # === gdk_pixbuf

  # ========================================================================= #
  # === pixbuf_from_file
  # ========================================================================= #
  def pixbuf_from_file(this_file)
    GdkPixbuf::Pixbuf.new(file: this_file)
  end

  # ========================================================================= #
  # === button_hover_colour
  # ========================================================================= #
  def button_hover_colour(
      i = '#cef0ea; /* This is a very light blue. */'
    )
    i = i.to_s.dup
    in_file_css '
    button:hover {
      background: '+i+'
    }'
  end

  # ========================================================================= #
  # === gtk_tree_view
  #
  # Usage examples:
  #
  #   @treeview = gtk_tree_view(@liststore)
  #   @treeview = gtk_tree_view(@liststore) { :clickable_headers }
  #
  # ========================================================================= #
  def gtk_tree_view(
      optional_input = nil
    )
    if optional_input
      widget = ::Gtk::TreeView.new(optional_input)
    else
      widget = ::Gtk::TreeView.new
    end
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :clickable_headers
      # ===================================================================== #
      when :clickable_headers
        widget.headers_clickable = true
      end
    end
    widget
  end; alias gtk_treeview     gtk_tree_view # === gtk_treeview
       alias tree_view        gtk_tree_view # === tree_view
       alias create_tree_view gtk_tree_view # === create_tree_view

  # ========================================================================= #
  # === gtk_runner_module
  # ========================================================================= #
  def gtk_runner_module
    ::Gtk::AppModule
  end; alias gtk_app_module gtk_runner_module # === gtk_app_module

  # ========================================================================= #
  # === options
  #
  # This shall be similar to the HTML drop-down options menu. The user
  # will make a selection.
  #
  # This method will return an Array; the first entry is the vbox that
  # can be used, such as when you embed it into another gtk-application.
  # The second entry is the array containing all the radio-buttons.
  # These allow us to query the state.
  #
  # Usage example:
  #
  #    vbox, array_radio_buttons = options()
  #
  # ========================================================================= #
  def options(*i)
    return ::Gtk.options(i)
  end; alias dropdown_menu options # === dropdown_menu

  # ========================================================================= #
  # === ungrab_application
  #
  # Not sure if this is in use right now.
  # ========================================================================= #
  def ungrab_application
    Gdk.pointer_ungrab(Gdk::Event::CURRENT_TIME)
  end

  # ========================================================================= #
  # === xid_value?
  # ========================================================================= #
  def xid_value?
    window.xid.to_s
  end; alias xid? xid_value? # === xid?

  # ========================================================================= #
  # === gtk_tree_store
  # ========================================================================= #
  def gtk_tree_store(*i)
    ::Gtk::TreeStore.new(i)
  end

  require 'gtk_paradise/core_classes/option_menu.rb'
  # ========================================================================= #
  # === gtk_option_menu
  # ========================================================================= #
  def gtk_option_menu
    ::Gtk.gtk_option_menu
  end

  # ========================================================================= #
  # === table4
  #
  # will return a table with a 4 columns layout.
  # ========================================================================= #
  def table4(*all_args)
    n_elements = 4
    table = ::Gtk::Table.new(
      all_args.size / n_elements, 2, true
    )
    counter = 0
    slow_counter = 0
    for element in all_args
      # print (counter % 2),' ',(counter % 2)+1,' ',slow_counter,' ',slow_counter+1,"\n"
      table.attach_defaults( 
        element,
        (counter % n_elements), (counter % n_elements)+1, 
        slow_counter,   slow_counter+1
      )
      counter += 1
      slow_counter += 1 if  counter % n_elements == 0
    end
    return table
  end

  # ========================================================================= #
  # === gtk_vpaned
  # ========================================================================= #
  def gtk_vpaned(*optional_arguments)
    # if use_gtk2?
    #   vpaned = ::Gtk::VPaned.new
    vpaned = ::Gtk::Paned.new(:vertical)
    if optional_arguments and !optional_arguments.empty?
      optional_arguments.each {|widget|
        vpaned.add(widget)
      }
    end
    return vpaned
  end

  # ========================================================================= #
  # === Gtk::BaseModule.included
  #
  # This could be enabled if we want to act on a per-inclusion basis,
  # via a hook.
  # ========================================================================= #
  # def self.included(i)
  #   case i.name.to_s
  #   when 'Gtk::BaseModuleAndAppModule'
  #     # ======================================================================= #
  #     # This is activated whenever Gtk::BaseModuleAndAppModule is 
  #     # included.
  #     # ======================================================================= #
  #     puts "#{i} included #{self} - next work on CSS stuff."
  #   end
  # end

  # ========================================================================= #
  # === mouse_button_double_click?
  #
  # The argument should be an event.
  # ========================================================================= #
  def mouse_button_double_click?(event)
    # if use_gtk2?
    #   event.event_type == Gdk::Event::BUTTON2_PRESS and
    #   event.button == 1
    # else
    event.event_type == Gdk::EventType::BUTTON2_PRESS and
    event.button == 1
    # end
  end; alias double_click? mouse_button_double_click? # === double_click?

  # ========================================================================= #
  # === gtk_dialog
  # ========================================================================= #
  def gtk_dialog
    ::Gtk::Dialog.new
  end

  # ========================================================================= #
  # === gtk_toolbar
  # ========================================================================= #
  def gtk_toolbar
    ::Gtk::Toolbar.new
  end; alias create_toolbar gtk_toolbar # === create_toolbar

  # ========================================================================= #
  # === gtk_state_normal
  # ========================================================================= #
  def gtk_state_normal
    ::Gtk::StateType::NORMAL
  end

  # ========================================================================= #
  # === gtk_colour_black
  # ========================================================================= #
  def gtk_colour_black
    ::Gtk::Colours::BLACK
  end

  # ========================================================================= #
  # === gtk_vruler
  # ========================================================================= #
  def gtk_vruler
    ::Gtk::VRuler.new
  end

  # ========================================================================= #
  # === gtk_drawing_area
  # ========================================================================= #
  def gtk_drawing_area
    ::Gtk::DrawingArea.new
  end

  # ========================================================================= #
  # === create_entry_with_text_and_max_length
  # ========================================================================= #
  def create_entry_with_text_and_max_length(i = '')
    ::Gtk.create_entry_with_text_and_max_length(i)
  end

  # ========================================================================= #
  # === gtk_adjustment
  # ========================================================================= #
  def gtk_adjustment(*i)
    ::Gtk::Adjustment.new(*i)
  end

  # ========================================================================= #
  # === gtk_message_dialog_run_then_destroy
  # ========================================================================= #
  def gtk_message_dialog_run_then_destroy(
      content_to_show = 'This functionality is not yet enabled.'
    )
    _ = gtk_message_dialog(content_to_show)
    _.run
    _.destroy
    return _
  end

  # ========================================================================= #
  # === create_paned_hbox_first_big
  # ========================================================================= #
  def create_paned_hbox_first_big(*i)
    ::Gtk.create_paned_hbox_first_big(i)
  end

  # ========================================================================= #
  # === directory_to_the_images?
  #
  # This method will return the directory to the images.
  # ========================================================================= #
  def directory_to_the_images?
    IMG_DIR
  end

  # ========================================================================= #
  # === gtk_tooltip_set_tip
  # ========================================================================= #
  def gtk_tooltip_set_tip(
      widget,
      text = ''
    )
    ::Gtk::Tooltips.new.set_tip(widget, text, nil)
  end

  # ========================================================================= #
  # === gtk_expanded_expander
  #
  # This variant will expand the newly created gtk-expander at hand.
  # ========================================================================= #
  def gtk_expanded_expander(
      text = '', &block
    )
    expander = gtk_expander(text, &block) # Create a new Gtk::Expander instance here.
    expander.do_expand
    return expander
  end

  # ========================================================================= #
  # === table2
  #
  # Will return a table with a 2 columns layout.
  # ========================================================================= #
  def table2(*all_args)
    table = ::Gtk::Table.new(all_args.size / 2, 2, true)
    counter = 0
    slow_counter = 0
    for element in all_args
      # print (counter % 2),' ',(counter % 2)+1,' ',slow_counter,' ',slow_counter+1,"\n"
      table.attach_defaults( 
        element,
        (counter % 2), (counter % 2)+1, 
        slow_counter,   slow_counter+1
      )
      counter += 1
      slow_counter += 1 if  counter % 2 == 0
    end
    return table
  end

  # ========================================================================= #
  # === gtk_button_box
  #
  # This method will instantiate a new Gtk::ButtonBox.
  #
  # The usage can be quite complex, as the following example shows:
  #
  #  button_box = gtk_button_box(
  #    @button_choose_file,
  #    @button_analyse
  #  ) {{ layout:  :horizontal,
  #       padding: 2,
  #       fill:    true }}
  #
  # But you can also simplify this a bit if you'd like to:
  #
  #   button_box = button_box(:horizontal)
  #   button_box = button_box(:vertical)
  #
  # It seems as if ruby-gtk4 no longer supports this, so we may have
  # to find a solution for that case.
  # ========================================================================= #
  def gtk_button_box(
      *i,
      &block
    )
    orientation_to_use = :horizontal # This is the default orientation to use.
    shall_we_fill_the_elements = false
    padding_to_use = 0
    if i.is_a?(Array) and i.first.is_a?(Symbol)
      orientation_to_use = i.shift # Remove the first element as well.
    end
    button_box = ::Gtk::ButtonBox.new(orientation_to_use)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next, when given as Block variant
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === :layout
        # =================================================================== #
        if yielded.has_key? :layout
          orientation_to_use = yielded[:layout]
        # =================================================================== #
        # === :spread
        # =================================================================== #
        elsif yielded.has_key? :spread
          # orientation_to_use = yielded[:spread]
          button_box.layout = :spread
        end
        # =================================================================== #
        # === :padding
        # =================================================================== #
        if yielded.has_key? :padding
          padding_to_use = yielded[:padding]
        end
        # =================================================================== #
        # === :fill
        # =================================================================== #
        if yielded.has_key? :fill
          shall_we_fill_the_elements = yielded[:fill]
        end
      end
    # ======================================================================= #
    # === Handle Symbols next
    # ======================================================================= #
    elsif yielded.is_a? Symbol
      case yielded
      # ===================================================================== #
      # === :spread
      # ===================================================================== #
      when :spread
        button_box.layout = :spread
      # ===================================================================== #
      # === :center
      # ===================================================================== #
      when :center
        button_box.layout = :center
      end
    end
    # ======================================================================= #
    # Next iterate over the given arguments 1:1 as-is:
    # ======================================================================= #
    i.flatten.compact.each {|this_widget|
      button_box.pack_start(
        this_widget,
        padding: padding_to_use,
        expand:  true,
        fill:    shall_we_fill_the_elements
      )
    }
    # ======================================================================= #
    # Sanitize the orientation a bit, if it is a symbol.
    # ======================================================================= #
    case orientation_to_use
    # ======================================================================= #
    # === :spread
    # ======================================================================= #
    when :spread
      button_box.layout = :spread
    else
      button_box.orientation = orientation_to_use
    end
    return button_box
  end; alias gtk_buttonbox     gtk_button_box # === gtk_buttonbox
       alias button_box        gtk_button_box # === button_box
       alias return_button_box gtk_button_box # === return_button_box
       alias create_button_box gtk_button_box # === create_button_box

  # ========================================================================= #
  # === quit_button
  #
  # This method will create a simple quit-button. The actual code for
  # this is stored in the method called gtk_button().
  #
  # The method must be sufficiently flexible to allow the following use
  # cases as well:
  #
  #   button_quit = quit_button('_Quit_lightgreen')
  #
  # ========================================================================= #
  def quit_button(
      optional_string = '_Quit',
      &block
    )
    case optional_string
    # ======================================================================= #
    # === :use_image
    # ======================================================================= #
    when :use_image
      _ = gtk_button
      _.set_always_show_image(true)
      icon  = ::Gtk.load_icon('application-exit', 32, 0)
      image = gtk_pixbuf(icon)
      hbox  = gtk_hbox
      hbox.add(image)
      hbox.add(create_label('Quit', :do_use_mnemonics)) # The mnemonics currently do not work.
      _.add(hbox)
    # ======================================================================= #
    # === :use_emoji
    # ======================================================================= #
    when :use_emoji
      _ = gtk_button('🛑 Quit', &block) # This is the stop-sign emoji.
    else
      if optional_string and optional_string.empty?
        _ = create_button(optional_string, &block)
      else
        _ = create_button(:quit, &block)
      end
    end
    _.on_clicked { exit_the_application }
    return _
  end; alias gtk_quit_button    quit_button # === gtk_quit_button
       alias create_quit_button quit_button # === create_quit_button

  # ========================================================================= #
  # === gtk_label
  #
  # Instantiate a new gtk-label. Note that you can also use a few
  # emojis. Example for the latter use case:
  #
  #   gtk_label(:arrow_right) # This will use the right-pointed arrow emoji
  #
  # Since as of December 2020, gtk_label() will automatically make use
  # of .set_markup() if the string "<b>" or "<i>" is encountered. This
  # should make it a bit easier for users to embed HTML tags in their
  # Strings. This is handled via Gtk.label() actually.
  # ========================================================================= #
  def gtk_label(
      use_this_text = '',
      use_mnemonics = false
    )
    if use_this_text.is_a? Symbol
      use_this_text = ::Gtk::Emojis.map_emoji(use_this_text)
    end
    # assume gtk3 next.
    label = ::Gtk.label(use_this_text, use_underline: use_mnemonics)
    if label and @internal_hash
      @internal_hash[:label] << label if @internal_hash.has_key?(:label)
    end
    return label
  end; alias label gtk_label # === label
       alias text  gtk_label # === text

  # ========================================================================= #
  # === configuration?
  # ========================================================================= #
  def configuration?
    @configuration
  end; alias configfile? configuration? # === configfile?

  # ========================================================================= #
  # === use_config_file?
  # ========================================================================= #
  def use_config_file?
    @configuration and (@configuration[0] == true)
  end; alias do_we_use_a_config_file? use_config_file? # === do_we_use_a_config_file?

  # ========================================================================= #
  # === return_the_title_from_the_config_file
  #
  # This method will return the title based on the config-file at hand.
  # ========================================================================= #
  def return_the_title_from_the_config_file(
      use_config_file = use_config_file?,
      configuration   = @configuration
    )
    if use_config_file # Will return an Array containing three elements.
      # ======================================================================= #
      # The second entry will have our desired value.
      # ======================================================================= #
      this_file = configuration[1]+'/'+
                  File.basename(configuration[1])+'.config'
      if File.exist? this_file
        dataset = File.readlines(this_file).select {|line|
          line.start_with? 'title:'
        }
        return dataset.flatten.first.strip.split(':').last.strip
      else
        ''
      end
    else
      ''
    end
  end

  # ========================================================================= #
  # === show_debug_information                                    (debug tag)
  #
  # This method can be used to show debug-information.
  #
  # An alias called .debug, as a method, may be used as well.
  # ========================================================================= #
  def show_debug_information
    e
    e rev+
      'Showing potentially useful information about the gtk-widget at hand'
    e rev+
      'as well as the environment:'
    e
    e rev+'  Resolution: '+
      steelblue('max-width:  '+max_width?.to_s)
    e rev+'  Resolution: '+
      steelblue('max-height: '+max_height?.to_s)
    e
    e 'The font name in use is: '+
      steelblue(::Gtk::Settings.default.gtk_font_name)
    e
    e 'If the application makes use of a default .css file, then that file'
    e 'is assumed to reside at '
    e
    e '  '+sfancy(
      project_css_file?
    )
    e
    # ======================================================================= #
    # Next, if we use a .config file then we will show more information.
    # ======================================================================= #
    if do_we_use_a_config_file?
      _ = configuration?
      this_file = _[1]+'/'+File.basename(_[1])+'.config'
      e 'This application '+crimson('DOES')+
        ' make use of a .config file. That file '
      e 'should be located at:'
      e
      e "  #{sfile(this_file)}"
      e
      e
      e 'We will show information deduced from this file next.'
      e
      if File.exist? this_file
        file_content = File.read(this_file).split("\n").map {|entry|
          "  #{entry}"
        }.join("\n")
        e lightblue(
            file_content
          )
      end
      e
      # ===================================================================== #
      # Also show the CSS rules in use:
      # ===================================================================== #
      show_the_CSS_rules_in_use
      e
      # ===================================================================== #
      # Show the user whether a global project.css file is used.
      # ===================================================================== #
      global_file_project_css = return_the_path_to_the_file_called_project_css(_[1])
      e 'Does a global '+sfile('project.css')+' file exist, '\
        'at location `'+global_file_project_css.to_s+'`? '+
         sfancy(
           File.exist?(
             global_file_project_css
           ).to_s
         )
    else
      e 'This application does not make use of a .config file.'
    end
    # e '  Resolution: width:  '+width.to_s
    # e '  Resolution: height: '+height.to_s
    e
  end; alias debug_widget  show_debug_information # === debug
       alias debug_widget? show_debug_information # === debug?

  # ========================================================================= #
  # === increase_font_size
  #
  # The first argument specifies the size of the font, in pixel.
  #
  # The second argument specifies the font family.
  # ========================================================================= #
  def increase_font_size(
      use_this_font_size   = '18',
      use_this_font_family = 'Mono'
    )
    # ======================================================================= #
    # === Hashes
    #
    # First, handle Hashes as input given:
    # ======================================================================= #
    if use_this_font_size.is_a? Hash
      # ===================================================================== #
      # === :to
      # ===================================================================== #
      if use_this_font_size.has_key? :to
        use_this_font_size = use_this_font_size.delete(:to)
      # ===================================================================== #
      # === :size
      # ===================================================================== #
      elsif use_this_font_size.has_key? :size
        use_this_font_size = use_this_font_size.delete(:size)
      end
    end
    set_font_in_use_for_this_application(
      "#{use_this_font_family} #{use_this_font_size}"
    )
  end; alias use_larger_font        increase_font_size # === use_larger_font
       alias increase_the_font_size increase_font_size # === increase_the_font_size

  # ========================================================================= #
  # === all_labels?
  # ========================================================================= #
  def all_labels?
    @internal_hash[:label]
  end; alias return_all_labels all_labels? # === return_all_labels

  # ========================================================================= #
  # === automatic_scrolled_window
  #
  # The first argument should be a gtk-text-view instance.
  # ========================================================================= #
  def automatic_scrolled_window(i)
    return gtk_scrolled_window(i) { :automatic }
  end

  # ========================================================================= #
  # === gtk_header_bar
  #
  # Invocation example:
  #
  #   header_bar = gtk_header_bar { :with_close_button }
  #
  # ========================================================================= #
  def gtk_header_bar(&block)
    _ = ::Gtk::HeaderBar.new
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :with_close_button_and_inferred_title
      # ===================================================================== #
      when :with_close_button_and_inferred_title
        _.set_show_close_button(true) # Show the close-button too.
        _.set_title(my_title?)
      # ===================================================================== #
      # === :with_close_button
      # ===================================================================== #
      when :with_close_button,
           :close_button,
           :close
        _.show_close_button = true # Show the close-button too.
      end
    end
    return _
  end; alias header_bar gtk_header_bar # === header_bar

  # ========================================================================= #
  # === reset                                                     (reset tag)
  #
  # This method will have to initialize a very important instance variables.
  #
  # It was disabled as of January 2021. I initially thought I may bring
  # this method back, but it may be better to not use it - we'll see
  # at a later time.
  # ========================================================================= #
  # def reset
  #   reset_the_internal_variables
  # end

  # ========================================================================= #
  # === use_a_popular_reset_variant
  #
  # This method can be used to quickly "bootstrap" from scratch,
  # without having to rely on reset().
  # ========================================================================= #
  def use_a_popular_reset_variant
    reset_the_internal_variables
  end

  # ========================================================================= #
  # === gtk_status_icon
  #
  # The first argument may be a path to an image.
  # ========================================================================= #
  def gtk_status_icon(
      optional_file_path = nil
    )
    # ======================================================================= #
    # Instantiate a new status-icon next:
    # ======================================================================= #
    status_icon = ::Gtk::StatusIcon.new
    if optional_file_path
      if optional_file_path.is_a? Symbol
        status_icon.stock = ::Gtk::Stock.const_get(
          optional_file_path.to_s.upcase.to_sym
        )
      else
        if use_gtk2?
          status_icon.use_this_image = optional_file_path
        else
          e 'status_icon.use_this_image = has not yet been adapted.'
        end
      end
    end
    return status_icon
  end; alias status_icon        gtk_status_icon # === status_icon
       alias return_status_icon gtk_status_icon # === return_status_icon
       alias create_status_icon gtk_status_icon # === return_status_icon

  # ========================================================================= #
  # === allow_only_numbers
  # ========================================================================= #
  def allow_only_numbers(
      gtk_entry_widget,
      be_not_too_strict = true
    )
    gtk_entry_widget.allow_only_numbers(
      be_not_too_strict
    )
  end

  # ========================================================================= #
  # === is_not_decorated
  # ========================================================================= #
  def is_not_decorated
    set_decorated(false)
  end

  # ========================================================================= #
  # === replace_with_proper_tags
  # ========================================================================= #
  def replace_with_proper_tags(i)
    text = ::Gtk.replace_with_proper_tags(i)
    _ = gtk_text(text)
    _.do_markify
    return _
  end; alias replace_colours_with_html_colours replace_with_proper_tags # === replace_colours_with_html_colours
       alias replace_html_colours              replace_with_proper_tags # === replace_html_colours

  # ========================================================================= #
  # === gtk_themed_icon
  # ========================================================================= #
  def gtk_themed_icon(
      use_this_icon = 'mail-send-receive-symbolic'
    )
    button = create_button
    # ======================================================================= #
    # === Handle symbols next
    # ======================================================================= #
    if use_this_icon.is_a? Symbol
      # ===================================================================== #
      # We handle Symbols a bit differently here. If it is a Symbol and if
      # that Symbol includes a '_' character then it is assumed that the
      # user instead wants to use '-'. Thus, we convert the Symbols to
      # the correct variant in this case.
      #
      # So, for instance, from :open_file to :open-file.
      # ===================================================================== #
      if use_this_icon.to_s.include? '_'
        use_this_icon = use_this_icon.to_s.tr('_','-').to_sym
      end
    end
    icon   = Gio::ThemedIcon.new(use_this_icon) 
    image  = ::Gtk::Image.new(icon: icon, size: :button)
    button.set_image(image)
    return button
  end; alias themed_icon      gtk_themed_icon # === themed_icon
       alias icon_theme_image gtk_themed_icon # === icon_theme_image
       alias icon_theme       gtk_themed_icon # === icon_theme

  # ========================================================================= #
  # === destroy_then_quit
  # ========================================================================= #
  def destroy_then_quit
    signal_connect(:destroy) { ::Gtk.main_quit }
  end; alias signal_destroy_then_quit destroy_then_quit # === signal_destroy_then_quit
       alias exit_on_destroy_signal   destroy_then_quit # === exit_on_destroy_signal
       alias exit_on_delete_event     destroy_then_quit # === exit_on_delete_event

  # ========================================================================= #
  # === exit_program
  # ========================================================================= #
  def exit_program
    ::Gtk.main_quit
  end; alias do_quit              exit_program # === do_quit
       alias do_exit              exit_program # === do_exit
       alias exit_the_application exit_program # === exit_the_application
       alias exit_application     exit_program # === exit_application

  # ========================================================================= #
  # === is_mouse_button_release_event?
  # ========================================================================= #
  def is_mouse_button_release_event?(event)
    event.respond_to?(:name?) and
    event.name? == 'GDK_BUTTON_RELEASE'
  end

  # ========================================================================= #
  # === dd_mm_yy
  #
  # This method can be used to denote the current day, month and year.
  # ========================================================================= #
  def dd_mm_yy
    ::Time.now.strftime('%d.%m.%Y') # => '13.01.2021' # dd.mm.yyyy
  end; alias dd_mm_yy?   dd_mm_yy # === dd_mm_yy?
       alias dd_mm_yyyy? dd_mm_yy # === dd_mm_yyyy?
       alias dd_mm_yyyy  dd_mm_yy # === dd_mm_yyyy

  # ========================================================================= #
  # === return_current_time
  #
  # This method will return the current time, in HH:MM:SS format.
  # ========================================================================= #
  def return_current_time
    ::Time.now.strftime('%H:%M:%S') # => "22:07:16"
  end; alias return_current_time_in_hh_mm_ss return_current_time # === return_current_time_in_hh_mm_ss
       alias hh_mm_ss?                       return_current_time # === hh_mm_ss?

  # ========================================================================= #
  # === return_current_time_hh_mm_only
  # ========================================================================= #
  def return_current_time_hh_mm_only
    ::Time.now.strftime('%H:%M') # => "22:07"
  end; alias return_current_time_hh_mm return_current_time_hh_mm_only # === return_current_time_hh_mm

  # ========================================================================= #
  # === title_height_width
  # ========================================================================= #
  def title_height_width(
      title  = TITLE,
      height = HEIGHT,
      width  = WIDTH
    )
    set_my_title(title)
    set_height(height)
    set_width(width)
  end

  # ========================================================================= #
  # === title?
  # ========================================================================= #
  def title?(
      i = :try_to_infer_the_title
    )
    case i
    # ======================================================================= #
    # === :try_to_infer_the_title
    # ======================================================================= #
    when :try_to_infer_the_title
      i = @internal_hash[:title] if @internal_hash # Must ensure it exists here.
    end
    return i
  end

  # ========================================================================= #
  # === @internal_hash[:title]
  # ========================================================================= #
  def my_title?
    @internal_hash[:title]
  end

  # ========================================================================= #
  # === gtk_input_field
  #
  # This is essentially a wrapper over Gtk::Entry.
  # ========================================================================= #
  def gtk_input_field(&block)
    ::Gtk.input_field(&block) # Defined in this file here.
  end; alias input_field        gtk_input_field # === input_field
       alias create_input_field gtk_input_field # === create_input_field

  # ========================================================================= #
  # === set_font_in_use_for_this_application
  #
  # Use this method to set the default font for GTK-related components
  # in a unified manner.
  #
  # You can also use set_font('Mono 18') as a shorter alias.
  #
  # Examples:
  #
  #   ::Gtk::Settings.default.gtk_font_name = 'Mono 20'
  #   ::Gtk::Settings.default.gtk_font_name = 'Mono'
  #
  # Specific usage example:
  #
  #   set_font_in_use_for_this_application :hack_25
  #   set_font_in_use_for_this_application :hack_25
  #
  # ========================================================================= #
  def set_font_in_use_for_this_application(
      i = USE_THIS_FONT
    )
    i = ::Gtk.expand_this_font(i)
    default_gtk_settings?.gtk_font_name = i
  end; alias use_default_font  set_font_in_use_for_this_application # === use_default_font
       alias set_gtk_font      set_font_in_use_for_this_application # === set_gtk_font
       alias set_gtk_font_size set_font_in_use_for_this_application # === set_gtk_font_size
       alias set_font_globally set_font_in_use_for_this_application # === set_font_globally

  # ========================================================================= #
  # === current_font_in_use?
  # ========================================================================= #
  def current_font_in_use?
    default_gtk_settings?.gtk_font_name
  end

  # ========================================================================= #
  # === raw_cd
  #
  # This is deliberately different from the method called change_directory(),
  # largely to ensure some legacy code to work as-is, but also to keep the
  # change-directory functionality as pure as possible. So raw_cd()
  # will be simpler than change_directory().
  # ========================================================================= #
  def raw_cd(i)
    Dir.chdir(i) if File.directory?(i)
  end

  # ========================================================================= #
  # === change_directory                                             (cd tag)
  # ========================================================================= #
  def change_directory(i)
    if i.is_a? Array
      i = i.first
    end
    raw_cd(i) if File.directory? i # Need to check whether it is a directory.
  end; alias cd    change_directory # === cd
       alias chdir change_directory # === chdir

  # ========================================================================= #
  # === register_sigint
  # ========================================================================= #
  def register_sigint
    ::Gtk.register_sigint
  end

  # ========================================================================= #
  # === write_what_into
  #
  # This method can be used to save content into a file.
  # ========================================================================= #
  def write_what_into(what, into)
    File.open(into, 'w') {|file|
      file.write(what)
    }
  end

  # ========================================================================= #
  # === load_this_yaml_file
  #
  # This method can be used to load data from a .yml file. It exists as
  # a method in the event that we may want to attach some error handling,
  # or checking whether a file exists or does not exist.
  # ========================================================================= #
  def load_this_yaml_file(i)
    YAML.load_file(i)
  end

  # ========================================================================= #
  # === return_corresponding_gtk_stock_item
  #
  # This method will replace a Symbol such as :quit with the corresponding
  # Gtk::Stock item. Make sure that the input is registered in the
  # table here.
  # ========================================================================= #
  def return_corresponding_gtk_stock_item(
      i = :default
    )
    case i # case tag
    # ======================================================================= #
    # === :open_local_file
    # ======================================================================= #
    when :open_local_file
      i = image_document_open
    # ======================================================================= #
    # === :weather_icon
    #
    # This is not a gtk-stock item, but we will retain it here for the
    # time being anyway. At a later time it may be relocated, though.
    # (The above was written at 05.12.2020).
    # ======================================================================= #
    when :weather_icon
      i = "#{PROJECT_BASE_DIRECTORY}images/misc/weather.png"
    # ======================================================================= #
    # === :open
    # ======================================================================= #
    when :open,
         :stock_open
      i = ::Gtk::Stock::OPEN
    # ======================================================================= #
    # === :quit
    # ======================================================================= #
    when :quit,
         :default
      i = ::Gtk::Stock::QUIT 
    # ======================================================================= #
    # === :save
    # ======================================================================= #
    when :save
      i = ::Gtk::Stock::SAVE
    # ======================================================================= #
    # === :delete
    # ======================================================================= #
    when :delete
      i = ::Gtk::Stock::DELETE
    # ======================================================================= #
    # === :close
    #
    # This is the "close-button".
    # ======================================================================= #
    when :close
      i = ::Gtk::Stock::CLOSE
    # ======================================================================= #
    # === :find
    # ======================================================================= #
    when :find
      i = ::Gtk::Stock::FIND
    # ======================================================================= #
    # === :edit
    # ======================================================================= #
    when :edit
      i = ::Gtk::Stock::EDIT
    # ======================================================================= #
    # === :print_preview
    # ======================================================================= #
    when :print_preview,
         :print
      i = ::Gtk::Stock::PRINT_PREVIEW
    # ======================================================================= #
    # === :refresh
    # ======================================================================= #
    when :refresh,
         :update
      i = ::Gtk::Stock::REFRESH
    end
    return i
  end

  # ========================================================================= #
  # === name_of_this_event?
  # ========================================================================= #
  def name_of_this_event?(event)
    return event.event_type.name
  end

  # ========================================================================= #
  # === gtk_file_filter
  #
  # A filter may be provided, such as:
  #
  #    '.pdf'
  #
  # As input.
  # ========================================================================= #
  def gtk_file_filter(
      optional_file_filter_to_use = nil
    )
    _ = ::Gtk::FileFilter.new
    if optional_file_filter_to_use
      if optional_file_filter_to_use.is_a? Symbol
        # Such as: :pdf_files
        optional_file_filter_to_use = optional_file_filter_to_use.to_s
      end
      if optional_file_filter_to_use.include? '_'
        optional_file_filter_to_use = optional_file_filter_to_use.split('_').first
      end
      _.name = "#{optional_file_filter_to_use} files"
      _.add_pattern("*#{optional_file_filter_to_use}")
    end
    return _
  end

  # ========================================================================= #
  # === gtk_source_buffer
  # ========================================================================= #
  def gtk_source_buffer
    begin
      ::Gtk::SourceBuffer.new
    rescue LoadError => error
      puts 'Gtk::SourceBuffer does not appear to be available.'
      pp error
    end
  end; alias return_sourcecode gtk_source_buffer # === return_sourcecode
       alias source_buffer     gtk_source_buffer # === source_buffer

  # ========================================================================= #
  # === gtk_bold_button
  #
  # This is similar to gtk_button() above, but will have a bold label by
  # default.
  # ========================================================================= #
  def gtk_bold_button(
      i             = '',
      use_underline = false,
      &block
    )
    button = create_button(i, use_underline, &block)
    button.make_bold
    return button
  end; alias bold_button        gtk_bold_button # === bold_button
       alias create_bold_button gtk_bold_button # === create_bold_button

  # ========================================================================= #
  # === simplified_button_box
  #
  # Usage example:
  #
  #   simplified_button_box(button1, button2, button3) { :center }
  #
  # ========================================================================= #
  def simplified_button_box(*i, &block)
    if block_given?
      _ = create_button_box(&block)
    else
      _ = create_button_box { :spread }
    end
    i.flatten.each {|entry|
      _.add(entry)
    }
    return _
  end

  require 'gtk_paradise/core_classes/notebook.rb'
  # ========================================================================= #
  # === gtk_notebook
  #
  # Usage example:
  #
  #   @notebook = gtk_notebook(:useful_defaults)
  #
  # ========================================================================= #
  def gtk_notebook(
      optional_arguments = nil, &block
    )
    _ = ::Gtk::Notebook.new
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :make_it_scrollable
      # ===================================================================== #
      when :make_it_scrollable
        _.is_scrollable
      end
    end
    case optional_arguments
    # ======================================================================= #
    # === :useful_default
    #
    # These defaults are useful to have when I create a new Gtk::Notebook
    # instance.
    # ======================================================================= #
    when :useful_default,
         :useful_defaults
      _.focus_on_the_first_tab
      _.do_show_the_border
      _.do_show_the_tabs
      _.the_tabs_appear_on_top
      _.is_scrollable
      _.enable_popup = true
    end
    return _
  end; alias notebook        gtk_notebook # === notebook
       alias create_notebook gtk_notebook # === create_notebook

  # ========================================================================= #
  # === remove_all_pages
  #
  # Removes all pages from a Gtk::Notebook.
  #
  # Usage example:
  #
  #   .remove_all_pages(@notebook)
  #
  # ========================================================================= #
  def remove_all_pages(which_widget)
    which_widget.n_pages.times { |index|
      which_widget.remove_page(index)
    }
  end

  # ========================================================================= #
  # === gtk_markified_text
  #
  # This will create a new gtk-label and then apply the markify-method,
  # in order to enable markdown formatting.
  # ========================================================================= #
  def gtk_markified_text(i)
    _ = gtk_text(i)
    _.do_markify
    return _
  end

  # ========================================================================= #
  # === event_right_mouse_button_click?
  #
  # This method queries whether the right mouse-button was clicked.
  # ========================================================================= #
  def event_right_mouse_button_click?(event)
    event.is_a?(Gdk::EventButton) and
   (event.event_type.name == 'GDK_BUTTON_PRESS') and
   (event.button == 3)
  end; alias on_right_mouse_button_click event_right_mouse_button_click? # === on_right_mouse_button_click
       alias right_mouse_button_clicked? event_right_mouse_button_click? # === right_mouse_button_clicked?
       alias event_right_mouse_button?   event_right_mouse_button_click? # === event_right_mouse_button?
       alias right_mouse_click?          event_right_mouse_button_click? # === right_mouse_click?
       alias right_button_pressed?       event_right_mouse_button_click? # === right_button_pressed?

  # ========================================================================= #
  # === is_mouse_button_event?
  #
  # A query whether the given event is a mouse-button event.
  # ========================================================================= #
  def is_mouse_button_event?(event)
    event.is_a?(Gdk::EventButton) and
    (event.event_type.name == 'GDK_BUTTON_PRESS')
  end; alias button_press_event? is_mouse_button_event? # === button_press_event?

  # ========================================================================= #
  # === image_document_save_as_symbolic_symbolic
  # ========================================================================= #
  def image_document_save_as_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme('document-save-as-symbolic.symbolic')
  end

  # ========================================================================= #
  # === image_view_refresh_symbolic_symbolic
  # ========================================================================= #
  def image_view_refresh_symbolic_symbolic
    return_this_image_based_on_the_default_icon_theme('view-refresh-symbolic.symbolic')
  end

  # ========================================================================= #
  # === return_this_image_based_on_the_default_icon_theme
  #
  # This method can be used to return a default icon from the main
  # IconTheme installed.
  #
  # The method will return a Gtk::Image instance, thus an image.
  # ========================================================================= #
  def return_this_image_based_on_the_default_icon_theme(
      i           = :application_exit,
      size_to_use = 32,
      whatever    = 0
    )
    i = i.to_s.tr('_','-') # target should be like this: application-exit
    # ======================================================================= #
    # Fix "-symbolic-symbolic" into "-symbolic.symbolic" next.
    # ======================================================================= #
    if i.include? '-symbolic-symbolic'
      i = i.reverse.sub(/-/,'.').reverse
    end
    begin
      _ = ::Gtk::IconTheme.default.load_icon(i, size_to_use, whatever)
      result = ::Gtk::Image.new(pixbuf: _)
    rescue Gtk::IconThemeError::NotFound
      result = ::Gtk::Image.new
    end
    return result
  end; alias load_icon   return_this_image_based_on_the_default_icon_theme # === load_icon
       alias gtk_icon    return_this_image_based_on_the_default_icon_theme # === gtk_icon
       alias create_icon return_this_image_based_on_the_default_icon_theme # === create_icon

  # ========================================================================= #
  # === gtk_entry
  #
  # Make a new Gtk::Entry widget.
  #
  # Some code-variants can be used, such as:
  #
  #   gtk_entry { :align_center }
  #
  # This would align the text contained in the Gtk::Entry widget into the
  # center (middle).
  #
  # Another variant would be:
  #
  #   gtk_entry { :read_only }
  #
  # This would mean to make the Gtk::Entry widget read-only - in other
  # words, the user is not allowed to make changes to it.
  # ========================================================================= #
  def gtk_entry(i = '')
    if i.is_a? Float
      i = i.to_s
    end
    gtk_entry = ::Gtk.entry(i)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === Handle Hashes next
      # ===================================================================== #
      if yielded.is_a? Hash
        # =================================================================== #
        # === :set_text
        # =================================================================== #
        if yielded.has_key? :set_text
          gtk_entry.set_text(yielded.delete(:set_text))
        end
        # =================================================================== #
        # === :max_length
        #
        # This entry point allows for setting the max-length value quickly.
        #
        # Usage example:
        #
        #   entry = gtk_entry {{ max_length: 50 }}
        #
        # =================================================================== #
        if yielded.has_key? :max_length
          gtk_entry.set_max_length(yielded.delete(:max_length))
        end
      # ===================================================================== #
      # === Handle Symbols next
      # ===================================================================== #
      elsif yielded.is_a? Symbol
        case yielded
        # =================================================================== #
        # === :read_only
        #
        # This entry point will disable any modifications to the gtk-entry
        # at hand.
        # =================================================================== #
        when :read_only
          gtk_entry.editable = false
        # =================================================================== #
        # === :on_click_select_all
        # =================================================================== #
        when :on_click_select_all
          gtk_entry.on_click_highlight_all
        # =================================================================== #
        # === :align_center
        #
        # This will essentially center the gtk-entry at hand, by making
        # use of .xalign=.
        # =================================================================== #
        when :align_center,
             :align_to_the_center,
             :do_center
          gtk_entry.xalign = 0.5
        end
      end
    end
    @internal_hash[:entries] << gtk_entry if @internal_hash
    return gtk_entry
  end; alias gtk_input    gtk_entry # === gtk_input
       alias input        gtk_entry # === input
       alias entry        gtk_entry # === entry
       alias create_entry gtk_entry # === create_entry

  # ========================================================================= #
  # === gtk_resize_grip
  #
  # This method needs two widgets as input. It will then instantiate a
  # new Gtk::VPaned instance.
  #
  # Typical usage example:
  #
  #   widget = gtk_resize_grip(@tree_view, @text_view_at_bottom)
  #   pack_start(widget, true, true, 4)
  #
  # ========================================================================= #
  def gtk_resize_grip(widget1, widget2)
    # ======================================================================= #
    # Since Gtk::VPaned has been deprecated in ruby-gtk3, we have to query
    # the version of gtk in use next.
    # ======================================================================= #
    # if use_gtk2?
    #   vpaned = ::Gtk::VPaned.new
    vpaned = ::Gtk::Paned.new(:vertical)
    vpaned.add1(widget1)
    vpaned.add2(widget2)
    return vpaned
  end; alias gtk_resize_handle   gtk_resize_grip # === gtk_resize_handle
       alias draggable_spacer    gtk_resize_grip # === draggable_spacer
       alias horizontal_drag     gtk_resize_grip # === horizontal_drag
       alias horizontal_dragger  gtk_resize_grip # === horizontal_dragger
       alias draggable_hpane     gtk_resize_grip # === draggable_hpane
       alias gtk_draggable_hpane gtk_resize_grip # === gtk_draggable_hpane

  # ========================================================================= #
  # === create_paned_vbox_last_big
  # ========================================================================= #
  def create_paned_vbox_last_big(*i)
    ::Gtk.create_paned_vbox_last_big(i)
  end

  # ========================================================================= #
  # === we_use_this_gtk_version?
  # ========================================================================= #
  def we_use_this_gtk_version?
    ::Gtk.we_use_this_gtk_version?
  end; alias we_use_which_gtk_version? we_use_this_gtk_version? # === we_use_which_gtk_version?
       alias use_which_gtk_version?    we_use_this_gtk_version? # === use_which_gtk_version?

  # ========================================================================= #
  # === return_pwd
  # ========================================================================= #
  def return_pwd
    "#{Dir.pwd}/".squeeze('/')
  end

  # ========================================================================= #
  # === add_notebook
  # ========================================================================= #
  def add_notebook(*i)
    ::Gtk.add_notebook(i)
  end

  # ========================================================================= #
  # === gtk_tooltip_widget
  # ========================================================================= #
  def gtk_tooltip_widget(
      tooltip, widget
    )
    ::Gtk.tooltip_widget(
      tooltip, widget
    )
  end; alias gtk_tooltip gtk_tooltip_widget # === gtk_tooltip

  # ========================================================================= #
  # === gtk_colour_grey
  # ========================================================================= #
  def gtk_colour_grey
    ::Gtk::Colours::GREY
  end

  # ========================================================================= #
  # === gtk_window
  # ========================================================================= #
  def gtk_window(
      optional_title = nil
    )
    if optional_title
      Gtk::Window.new(optional_title)
    else
      Gtk::Window.new
    end
  end

  # ========================================================================= #
  # === gtk_vbutton_box
  #
  # This variant is specifically for ruby-gtk2.
  # ========================================================================= #
  def gtk_vbutton_box(
      *optional_arguments
    )
    box = ::Gtk::VButtonBox.new
    if optional_arguments and !optional_arguments.empty?
      optional_arguments.each {|entry| box.pack_start(entry, true, true) }
    end
    return box
  end

  # ========================================================================= #
  # === drop_down_find_box
  #
  # This method will return an Array of the two widgets that may be
  # useful, if the user wants to create a search-bar.
  # ========================================================================= #
  def drop_down_find_box
    search_entry = gtk_search_entry
    search_bar = gtk_search_bar(search_entry)
    return [search_bar, search_entry]
  end

  # ========================================================================= #
  # === modify_foreground
  # ========================================================================= #
  def modify_foreground(
      colour = :steelblue,
      state  = ::Gtk::StateType::NORMAL
    )
    colour = ::Gtk.colour_parse(colour)
    modify_fg(
      state, colour # Flip it here.
    )
  end

  # ========================================================================= #
  # === gtk_image_menu_item
  # ========================================================================= #
  def gtk_image_menu_item(
      use_this_image = ::Gtk::Stock::QUIT
    )
    ::Gtk::ImageMenuItem.new(stock_id: use_this_image)
  end; alias image_menu_item        gtk_image_menu_item # === image_menu_item
       alias create_image_menu_item gtk_image_menu_item # === create_image_menu_item

  # ========================================================================= #
  # === populate_this_list_store
  #
  # Note that this method currently only works for one or two entries,
  # which explains the odd alias-name to this. At a later point we may
  # change this, though.
  #
  # It is recommended to use the aliased name, aka:
  #
  #   populate_this_list_store2(@list_store, dataset_here)
  #
  # This is not as elegant, but at some point in the future we may
  # change the main alias, but the numbered-variants such as _store2
  # or _store3 will always remain backwards-compatible. Keep that
  # in mind when using this method.
  # ========================================================================= #
  def populate_this_list_store(
      this_list_store, this_dataset
    )
    this_dataset.each { |first, second|
      iter = this_list_store.append
      iter.set_value(0, first.to_s)
      iter.set_value(1, second.to_s) if second
    }
  end; alias populate_this_list_store2                populate_this_list_store # === populate_this_list_store2
       alias populate_this_list_store_with_that_array populate_this_list_store # === populate_this_list_store_with_that_array

  # ========================================================================= #
  # === populate_this_list_store4
  #
  # This variant works with 4 entries.
  # ========================================================================= #
  def populate_this_list_store4(
      this_list_store, this_dataset
    )
    this_dataset.each { |first, second, third, fourth|
      iter = this_list_store.append
      iter.set_value(0, first.to_s)
      iter.set_value(1, second.to_s)
      iter.set_value(2, third.to_s)
      iter.set_value(3, fourth.to_s)
    }
  end; alias populate_this_list_store_with_that_array4 populate_this_list_store4 # === populate_this_list_store_with_that_array4

  # ========================================================================= #
  # === populate_this_list_store3
  #
  # This variant works with 3 entries.
  # ========================================================================= #
  def populate_this_list_store3(
      this_list_store, this_dataset
    )
    this_dataset.each { |first, second, third|
      iter = this_list_store.append
      iter.set_value(0, first.to_s)
      iter.set_value(1, second.to_s)
      iter.set_value(2, third.to_s)
    }
  end; alias populate_this_list_store_with_that_array3 populate_this_list_store3 # === populate_this_list_store_with_that_array3

  # ========================================================================= #
  # === default_notebook
  # ========================================================================= #
  def default_notebook
    return create_notebook(:useful_defaults)
  end; alias return_default_notebook default_notebook # === return_default_notebook

  # ========================================================================= #
  # === return_pixbuf_based_on_the_default_icon_theme
  #
  # The first argument to this method should be the name for the icon
  # at hand.
  # ========================================================================= #
  def return_pixbuf_based_on_the_default_icon_theme(
      i           = :application_exit,
      size_to_use = 32,
      whatever    = 0
    )
    # ======================================================================= #
    # Do some ad-hoc sanitizing next.
    # ======================================================================= #
    case i
    when :image_computer
      i = :computer
    end
    pixbuf = ::Gtk::IconTheme.default.load_icon(i, size_to_use, whatever)
    return pixbuf
  end

  # ========================================================================= #
  # === check_button_is_checked
  # ========================================================================= #
  def check_button_is_checked(
      i             = '',
      use_underline = :do_not_underline,
      &block
    )
    check_button = gtk_check_button(i)
    check_button.is_checked
    return check_button
  end

  # ========================================================================= #
  # === gtk_radio_button
  # ========================================================================= #
  def gtk_radio_button(
      text               = '',
      member_of          = nil,
      shall_we_underline = true
    )
    if member_of
      _ = ::Gtk::RadioButton.new(
        label: text,
        use_underline: shall_we_underline
      )
    else
      _ = ::Gtk::RadioButton.new(
        member: member_of,
        label: text,
        use_underline: shall_we_underline
      )
    end
    return _
  end

  # ========================================================================= #
  # === gtk_colour_button
  # ========================================================================= #
  def gtk_colour_button
    ::Gtk::ColorButton.new
  end; alias gtk_color_button gtk_colour_button # === gtk_color_button

  # ========================================================================= #
  # === gtk_button_accel
  #
  # This is like gtk_button, except that the first character will be
  # guaranteed to be a '_', if the input argument is a String. That
  # way we ensure that the gtk-button at hand will be an accel-button,
  # meaning that alt+first_character key press event will trigger
  # their default action.
  # ========================================================================= #
  def gtk_button_accel(
      i = 'Example for an accelerated button',
      &block
    )
    if i.is_a?(String) and !i.start_with?('_')
      i = "_#{i}" # Prepend the _ in this case.
    end
    create_button(
      i, true, &block
    )
  end; alias gtk_button_underline gtk_button_accel # === gtk_button_underline

  # ========================================================================= #
  # === gtk_hbox_with_vertical_pane
  #
  # This method will generate a hbox (horizontal box) which is separated
  # by a vertical drag-pane.
  # ========================================================================= #
  def gtk_hbox_with_vertical_pane(
      *i,
      &block
    )
    # if use_gtk2?
    #   _ = ::Gtk::VPaned.new
    _ = Gtk::Paned.new(:vertical)
    _.add1(i.first)
    _.add2(i.last)
    return _
  end; alias vpane_split     gtk_hbox_with_vertical_pane # === vpane_split
       alias draggable_vbox  gtk_hbox_with_vertical_pane # === draggable_vbox
       alias horizontal_pane gtk_hbox_with_vertical_pane # === horizontal_pane

  # ========================================================================= #
  # === gtk_bold_entry
  # ========================================================================= #
  def gtk_bold_entry(i = '')
    _ = create_entry(i)
    _.make_bold
    return _
  end; alias create_bold_entry gtk_bold_entry # === create_bold_entry

  # ========================================================================= #
  # === gtk_entry_with_secondary_icon_name
  # ========================================================================= #
  def gtk_entry_with_secondary_icon_name(this_icon_name)
    _ = create_entry
    _.set_secondary_icon_name(this_icon_name)
    return _
  end; alias entry_with_secondary_icon_name gtk_entry_with_secondary_icon_name # === entry_with_secondary_icon_name

  # ========================================================================= #
  # === gtk_entry_with_icon_name
  # ========================================================================= #
  def gtk_entry_with_icon_name(this_icon_name)
    _ = create_entry
    _.set_primary_icon_name(this_icon_name)
    return _
  end; alias entry_with_icon_name gtk_entry_with_icon_name # === entry_with_icon_name

  # ========================================================================= #
  # === gtk_search_entry
  # ========================================================================= #
  def gtk_search_entry
    ::Gtk::SearchEntry.new
  end; alias search_entry gtk_search_entry # === search_entry

  # ========================================================================= #
  # === gtk_list_store
  #
  # This method can be invoked like this:
  #
  #   gtk_list_store(String, Float)
  #
  # ========================================================================= #
  def gtk_list_store(*i)
    i.flatten!
    ::Gtk::ListStore.new(i)
  end; alias gtk_liststore     gtk_list_store # === gtk_liststore
       alias create_liststore  gtk_list_store # === create_liststore
       alias create_list_store gtk_list_store # === create_list_store
       alias liststore         gtk_list_store # === liststore

  # ========================================================================= #
  # === two_strings
  #
  # This is simply a list-store with two Strings per row.
  # ========================================================================= #
  def two_strings
    gtk_list_store(
      String, String
    )
  end

  # ========================================================================= #
  # === three_strings
  #
  # This is simply a list-store with three Strings per row.
  # ========================================================================= #
  def three_strings
    gtk_list_store(
      String, String, String
    )
  end

  # ========================================================================= #
  # === four_strings
  # ========================================================================= #
  def four_strings
    gtk_list_store(
      String, String, String, String
    )
  end

  # ========================================================================= #
  # === five_strings
  #
  # This is simply a list-store with five Strings per row.
  # ========================================================================= #
  def five_strings
    gtk_list_store(
      String, String, String, String, String
    )
  end

  # ========================================================================= #
  # === six_strings
  #
  # This is simply a list-store with six Strings per row.
  # ========================================================================= #
  def six_strings
    gtk_list_store(
      String, String, String, String, String, String
    )
  end

  # ========================================================================= #
  # === seven_strings
  #
  # This is simply a list-store with seven Strings per row.
  # ========================================================================= #
  def seven_strings
    gtk_list_store(
      String, String, String, String, String, String, String
    )
  end

  # ========================================================================= #
  # === eight_strings
  #
  # This is simply a list-store with eight Strings per row.
  # ========================================================================= #
  def eight_strings
    gtk_list_store(
      String, String, String, String, String, String, String, String, String, String
    )
  end

  # ========================================================================= #
  # === string_list_store
  # ========================================================================= #
  def string_list_store
    ::Gtk::ListStore.new(String)
  end

  # ========================================================================= #
  # === default_renderer
  # ========================================================================= #
  def default_renderer
    gtk_cell_renderer_text
  end

  # ========================================================================= #
  # === create_bold_cell_renderer
  # ========================================================================= #
  def create_bold_cell_renderer(
      i             = '',
      make_editable = false
    )
    bold_renderer = gtk_cell_renderer_text
    bold_renderer.editable = true if make_editable
    bold_renderer.set_markup(
      '<span weight="bold">'+i.to_s+'</span>'
    )
    return bold_renderer
  end; alias bold_renderer create_bold_cell_renderer # === bold_renderer

  # ========================================================================= #
  # === emoji_image
  #
  # This is just a wrapper over gtk_image really; but I think it may
  # be helpful when designing GUIs, as a simple memory-mapper.
  # ========================================================================= #
  def emoji_image(i = :right_arrow)
    gtk_image(i) # This will handle special symbols properly.
  end

  # ========================================================================= #
  # === icon_theme_default
  #
  # This method will return an instance of class GdkPixbuf::Pixbuf, if
  # the image is found.
  # ========================================================================= #
  def icon_theme_default(
      i, size_to_use = 16, whatever = 0
    )
    _ = ::Gtk::IconTheme.default.load_icon(
      i.to_s.tr('_','-'), size_to_use, whatever
    )
    return _
  end

  # ========================================================================= #
  # === gtk_right_aligned_label
  # ========================================================================= #
  def gtk_right_aligned_label(i = nil)
    _ = gtk_text(i)
    _.right_align
    return _
  end; alias right_aligned_label gtk_right_aligned_label # === right_aligned_label

  # ========================================================================= #
  # === gtk_bold_label
  # ========================================================================= #
  def gtk_bold_label(
      use_this_text = '',
      use_mnemonics = false
    )
    return create_label(use_this_text, use_mnemonics).make_bold
  end; alias gtk_bold_text     gtk_bold_label # === gtk_bold_text
       alias gtk_bold          gtk_bold_label # === gtk_bold
       alias bold_text         gtk_bold_label # === bold_text
       alias bold_label        gtk_bold_label # === bold_label
       alias create_bold_label gtk_bold_label # === create_bold_label
       alias text_bold         gtk_bold_label # === text_bold
       alias bold              gtk_bold_label # === bold

  # ========================================================================= #
  # === gtk_popover
  #
  # Usage example:
  #
  #   gtk_popover(self)
  #
  # ========================================================================= #
  def gtk_popover(
      this_widget = nil
    )
    if ::Gtk.const_defined? :Popover # This check is for ruby-gtk3.
      return ::Gtk::Popover.new(this_widget)
    end
  end

  # ========================================================================= #
  # === gtk_about_dialog
  # ========================================================================= #
  def gtk_about_dialog
    ::Gtk::AboutDialog.new
  end; alias create_about_dialog gtk_about_dialog # === create_about_dialog
       alias about_dialog        gtk_about_dialog # === about_dialog

  # ========================================================================= #
  # === width_height_font
  # ========================================================================= #
  def width_height_font(
      width  = WIDTH,
      height = HEIGHT,
      font   = USE_THIS_FONT
    )
    width_height(width, height)
    set_use_this_font(font)
  end

  # ========================================================================= #
  # === width_title_height
  # ========================================================================= #
  def width_title_height(
      width  = WIDTH,
      title  = TITLE,
      height = HEIGHT
    )
    set_width(width)    if respond_to? :set_width
    set_my_title(title) if respond_to? :set_my_title
    set_height(height)  if respond_to? :set_height
  end

  # ========================================================================= #
  # === width_height
  # ========================================================================= #
  def width_height(
      width  = WIDTH,
      height = HEIGHT
    )
    set_width(width)   if respond_to? :set_width
    set_height(height) if respond_to? :set_height
  end

  # ========================================================================= #
  # === delayed_css_change
  #
  # The first argument to this method is the target CSS that you wish to
  # set.
  #
  # The second argument should be the gtk-widget, such as a gtk-entry.
  #
  # The third argument specifies how long to sleep, to indicate a delay,
  # before restoring to the old colour (CSS) in use again.
  # ========================================================================= #
  def delayed_css_change(
      from   = 'BG_floralwhite',
      widget = nil,
      delay  = 0.4
    )
    old_name = widget.name
    set_name(from) # Set the new "css class" in use.
    Thread.new {
      sleep(delay)
      # ===================================================================== #
      # This "reinstates" the old colour.
      # ===================================================================== #
      set_name(old_name)
    }
  end

  # ========================================================================= #
  # === use_lean_CSS_rules
  #
  # Use only the registered CSS classes here.
  # ========================================================================= #
  def use_lean_CSS_rules
    _ = ::Gtk.css_classes_in_use?
    pp _.class
    pp _
    e
    e '^^^ This method called .use_lean_CSS_rules() is currently unfinished.'
    e
    e 'The idea was that we would only use registered CSS classes and skip'
    e 'loading the rest. This should speed up the gtk-application '\
      'a little bit.'
  end

  # ========================================================================= #
  # === load_rules_from_the_project_css_file
  #
  # This method will always use the file "project.css".
  # ========================================================================= #
  def load_rules_from_the_project_css_file(
      use_this_file = 'project.css',
      &block
    )
    yielded = nil
    if block_given?
      yielded = yield
    end
    unless use_this_file.include? '/'
      # Here we should find out the absolute path.
      use_this_file = use_this_file.dup
      use_this_file.prepend(return_pwd)
    end
    if File.exist? use_this_file
      case yielded
      # ===================================================================== #
      # === :be_verbose_then_exit
      # ===================================================================== #
      when :be_verbose_then_exit
        e 'Loading from the existing file at '+use_this_file+' before exiting.'
        exit
      end
      dataset = File.read(use_this_file)
      apply_these_CSS_rules(dataset)
    else
      case yielded
      # ===================================================================== #
      # === :be_verbose_then_exit
      # ===================================================================== #
      when :be_verbose_then_exit
        e 'Can not load from the file at '+use_this_file+'. Will exit next.'
        exit
      end
      no_file_exists_at(use_this_file)
    end
  end

  # ========================================================================= #
  # === use_these_css_classes
  #
  # This allows for convenience in regards to using CSS classes.
  #
  # It is important to separate these CSS classes via a ' ' character.
  #
  # Usage example:
  #
  #   @frame.use_these_css_classes('no_box_shadow bblack2 BG_black pad0px mar0px')
  #
  # ========================================================================= #
  def use_these_css_classes(i)
    if i and i.include?(' ')
      i.split(' ').each {|this_css_class|
        css_class(this_css_class)
      }
    else
      css_class(this_css_class)
    end
  end; alias use_these_CSS_classes use_these_css_classes # === use_these_CSS_classes

  # ========================================================================= #
  # === do_apply_the_CSS_rules
  #
  # This method will simply apply the defined CSS rules, stored in the
  # @internal_hash. It depends on class Gtk::CssProvider.
  # ========================================================================= #
  def do_apply_the_CSS_rules(
      _     = ::Gtk.css_rules?.to_s,
      debug = debug?
    )
    _ = _.to_s.dup # This will be our dataset.
    # This is only available for ruby-gtk3 or higher, not for ruby-gtk2. Old code was: if use_gtk3? or use_gtk4?
    if debug
      e 'The following CSS rules will be added next:'
      e '-'*80
      e _
      e '-'*80
    end
    return ::Gtk.css_provider(_)
  end; alias apply_the_css_string do_apply_the_CSS_rules # === apply_the_css_string
       alias apply_the_css_rules  do_apply_the_CSS_rules # === apply_the_css_rules
       alias apply_the_CSS_rules  do_apply_the_CSS_rules # === apply_the_CSS_rules
       alias apply_the_CSS        do_apply_the_CSS_rules # === apply_the_CSS
       alias trigger_CSS_addition do_apply_the_CSS_rules # === trigger_CSS_addition
       alias trigger_CSS          do_apply_the_CSS_rules # === trigger_CSS

  # ========================================================================= #
  # === append_gtk_paradise_project_css_file
  # ========================================================================= #
  def append_gtk_paradise_project_css_file(
      array = ARRAY_CSS_FILES
    )
    dataset = ''.dup
    array.each {|file| dataset << File.read(file)+"\n" }
    append_these_new_CSS_rules(dataset)
  end; alias add_gtk_paradise_project_css_file append_gtk_paradise_project_css_file # === add_gtk_paradise_project_css_file
       alias append_gtk_paradise_css_rules     append_gtk_paradise_project_css_file # === append_gtk_paradise_css_rules
       alias use_the_default_css_rules         append_gtk_paradise_project_css_file # === use_the_default_css_rules

  # ========================================================================= #
  # === append_the_default_css_rules
  # ========================================================================= #
  def append_the_default_css_rules(
      i = :project_internal_files
    )
    ::Gtk.append_css(
      return_default_CSS_rules(i)
    )
  end

  # ========================================================================= #
  # === apply_the_default_CSS_rules
  #
  # This variant will first load the project-specific CSS rules.
  # ========================================================================= #
  def apply_the_default_CSS_rules(
      i = return_default_CSS_rules
    )
    # @internal_hash[:css_rules_to_apply] = dataset
    append_these_CSS_rules(i)
    do_apply_the_CSS_rules
  end; alias apply_the_default_css_rules apply_the_default_CSS_rules # === apply_the_default_css_rules
       alias load_default_css                                                               apply_the_default_CSS_rules # === load_default_css
       alias read_in_from_this_css_file_and_apply_the_content_as_the_new_rules_to_adhere_to apply_the_default_CSS_rules # === read_in_from_this_css_file_and_apply_the_content_as_the_new_rules_to_adhere_to
       alias load_dataset_from_this_css_file                                                apply_the_default_CSS_rules # === load_dataset_from_this_css_file

  # ========================================================================= #
  # === return_default_CSS_rules
  # ========================================================================= #
  def return_default_CSS_rules(
      i = :project_internal_files
    )
    return ::Gtk.return_default_CSS_rules(i)
  end

  # ========================================================================= #
  # === use_gtk_paradise_project_css_file
  #
  # This method can be used to use the default CSS file that is
  # distributed via the gtk_paradise gem. It contains mostly
  # CSS rules that I re-use among different web-related
  # projects, but other users may find some CSS rules used therein
  # useful as well.
  # ========================================================================= #
  def use_gtk_paradise_project_css_file(
      i = :project_internal_files
    )
    append_the_default_css_rules(i)
  end; alias use_gtk_paradise_css_file         use_gtk_paradise_project_css_file # === use_gtk_paradise_css_file
       alias use_gtk_project_css_file          use_gtk_paradise_project_css_file # === use_gtk_project_css_file
       alias gtk_paradise_project_file         use_gtk_paradise_project_css_file # === gtk_paradise_project_file
       alias gtk_paradise_css_file             use_gtk_paradise_project_css_file # === gtk_paradise_css_file
       alias use_gtk_paradise_project_CSS_file use_gtk_paradise_project_css_file # === use_gtk_paradise_project_CSS_file
       alias use_gtk_paradise_project          use_gtk_paradise_project_css_file # === use_gtk_paradise_project
       alias use_the_gtk_paradise_css_file     use_gtk_paradise_project_css_file # === use_the_gtk_paradise_css_file

  # ========================================================================= #
  # === css_rules_to_apply?
  # ========================================================================= #
  def css_rules_to_apply?
    # @internal_hash[:css_rules_to_apply]
    return ::Gtk.css_rules_to_apply?
  end; alias css_string? css_rules_to_apply? # === css_string?
       alias css_rules?  css_rules_to_apply? # === css_rules?

  # ========================================================================= #
  # === css_background1
  #
  # Hardcoded background for now. One day we should split this into a
  # separate page.
  # ========================================================================= #
  def css_background1
    'background: linear-gradient(153deg, #151515, #151515 5px, transparent 5px) 0 0,
                 linear-gradient(333deg, #151515, #151515 5px, transparent 5px) 10px 5px,
                 linear-gradient(153deg, #222, #222 5px, transparent 5px) 0 5px,
                 linear-gradient(333deg, #222, #222 5px, transparent 5px) 10px 10px,
                 linear-gradient(90deg, #1b1b1b, #1b1b1b 10px, transparent 10px),
                 linear-gradient(#1d1d1d, #1d1d1d 25%, #1a1a1a 25%, #1a1a1a 50%, transparent 50%, transparent 75%, #242424 75%, #242424);
     background-color: #131313;
     background-size: 20px 20px;'
  end

  # ========================================================================= #
  # === store_the_CSS_rules_into_a_local_file
  #
  # This method can be used to store the CSS rules into a local file. It is
  # mostly meant as a debug-helper.
  # ========================================================================= #
  def store_the_CSS_rules_into_a_local_file
    require 'fileutils'
    begin
      require 'save_file'
    rescue LoadError; end
    if Object.const_defined? :SaveFile
      use_this_css_file = '/tmp/gtk_paradise/css_rules.css'
      FileUtils.mkdir_p('/tmp/gtk_paradise/')
      ::SaveFile.write_what_into(css_rules_to_apply?, use_this_css_file)
      e 'Saving into the following file: '+use_this_css_file
    else
      e 'SaveFile is not available.'
    end
  end

  # ========================================================================= #
  # === generate_css_class
  #
  # This can be used to quickly, ad-hoc, generate CSS rules.
  # ========================================================================= #
  def generate_css_class(i)
    case i
    # ======================================================================= #
    # === :bblack1
    # ======================================================================= #
    when :bblack1
      i = '.bblack1 { border: 1px solid black; }'
    # ======================================================================= #
    # === :bblack3
    # ======================================================================= #
    when :bblack2
      i = '.bblack2 { border: 2px solid black; }'
    # ======================================================================= #
    # === :bblack3
    # ======================================================================= #
    when :bblack3
      i = '.bblack3 { border: 3px solid black; }'
    end
    ::Gtk.css_provider(i)
  end; alias generate_css_rule generate_css_class # === generate_css_rule

  # ========================================================================= #
  # === css_provider?
  # ========================================================================= #
  def css_provider?
    ::Gtk.css_provider?
  end

  # ========================================================================= #
  # === read_in_from_this_css_theme
  #
  # Usage example:
  #
  #   read_in_from_this_css_theme(:rainbow_theme)
  #
  # ========================================================================= #
  def read_in_from_this_css_theme(
      i = :rainbow_theme
    )
    case i
    # ======================================================================= #
    # === :rainbow_theme
    # ======================================================================= #
    when :rainbow_theme
      i = Gtk.return_path_to_the_sitelibdir+
          'gtk_paradise/css_files/themes/rainbow_theme.css'
      if File.exist? i
        append_these_CSS_rules(File.read(i))
      end
    end
  end

  # ========================================================================= #
  # === return_the_path_to_the_file_called_project_css
  # ========================================================================= #
  def return_the_path_to_the_file_called_project_css(i)
    ::Gtk.return_the_path_to_the_file_called_project_css(i)
  end

  # ========================================================================= #
  # === add_css_rules
  # ========================================================================= #
  def add_css_rules(i)
    ::Gtk.css_provider(i)
  end

  # ========================================================================= #
  # === return_scrollable_CSS_in_use_widget
  #
  # This method will return a widget that can be used to show the
  # CSS in use, via a scrollable interface.
  # ========================================================================= #
  def return_scrollable_CSS_in_use_widget
    text_buffer = gtk_text_buffer
    # ======================================================================= #
    # Fill the buffer with the CSS rules in use next:
    # ======================================================================= #
    text_buffer.set_text(
      css_rules_to_apply?.to_s
    )
    text_view   = gtk_text_view(text_buffer)
    _ = gtk_scrolled_window(text_view) { :only_up_and_down }
    _.pad10px
    _.mar8px
    _.bblack1
    _.set_size_request(240, 300)
    _.set_border_width(20)
    return _
  end

  # ========================================================================= #
  # === display_the_CSS_rules
  # ========================================================================= #
  def display_the_CSS_rules(
      i = css_rules_to_apply?
    )
    unless Object.const_defined? :Colours
      begin
        require 'colours'
      rescue LoadError; end
    end
    e steelblue(i)
    # ======================================================================= #
    # Report the number of newlines found as well - this helps in
    # debugging, so we can quickly determine whether everything
    # worked correctly.
    # ======================================================================= #
    e ::Colours.tomato(
        'Number of newlines in this assumed .css file was: '
      )+
      ::Colours.royalblue(i.count("\n"))
    registered_CSS_classes = ::Gtk.css_classes_in_use?
    e 'The registered CSS classes are ('+registered_CSS_classes.size.to_s+'):'
    e
    pp registered_CSS_classes
    e
  end; alias debug_CSS                 display_the_CSS_rules # === debug_CSS
       alias debug_CSS?                display_the_CSS_rules # === debug_CSS?
       alias show_the_CSS_rules_in_use display_the_CSS_rules # === show_the_CSS_rules_in_use
       alias show_the_CSS_in_effect    display_the_CSS_rules # === show_the_CSS_in_effect
       alias show_the_CSS_rules        display_the_CSS_rules # === show_the_CSS_rules
       alias do_show_the_CSS_in_use    display_the_CSS_rules # === do_show_the_CSS_in_use
       alias do_show_CSS_rules_in_use  display_the_CSS_rules # === do_show_CSS_rules_in_use
       alias show_CSS?                 display_the_CSS_rules # === show_CSS?
       alias show_CSS                  display_the_CSS_rules # === show_CSS
       alias show_CSS_in_use           display_the_CSS_rules # === show_CSS_in_use
       alias show_CSS_rules            display_the_CSS_rules # === show_CSS_rules

  # ========================================================================= #
  # === more_css_then_apply_the_CSS_rules
  #
  # This method will simply combine two other method calls - it can be
  # used to add new CSS rules, and apply them at once (onto the
  # CSS provider).
  # ========================================================================= #
  def more_css_then_apply_the_CSS_rules(i)
    more_css(i)
    # apply_the_CSS_rules
  end; alias apply_these_CSS_rules              more_css_then_apply_the_CSS_rules # === apply_these_CSS_rules
       alias add_custom_CSS_rules               more_css_then_apply_the_CSS_rules # === add_custom_CSS_rules
       alias more_CSS_rules_then_apply_them     more_css_then_apply_the_CSS_rules # === more_CSS_rules_then_apply_them
       alias more_CSS_then_apply_these_rules    more_css_then_apply_the_CSS_rules # === more_CSS_then_apply_these_rules
       alias more_CSS_then_apply_it             more_css_then_apply_the_CSS_rules # === more_CSS_then_apply_it
       alias custom_CSS_then_apply_it           more_css_then_apply_the_CSS_rules # === custom_CSS_then_apply_it
       alias add_more_CSS_rules_then_apply_them more_css_then_apply_the_CSS_rules # === add_more_CSS_rules_then_apply_them
       alias apply_more_CSS_rules               more_css_then_apply_the_CSS_rules # === apply_more_CSS_rules
       alias apply_more_CSS                     more_css_then_apply_the_CSS_rules # === apply_more_CSS

  # ========================================================================= #
  # === create_context_menu
  #
  # This method can be used to create dynamic context-menus
  # "on the fly". These must be sufficiently flexible.
  # ========================================================================= #
  def create_context_menu(
      parent_widget = nil
    )
    context_menu = create_menu
    # ======================================================================= #
    # === Handle blocks given to this method next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :actions
        #
        # This entry point associates the context-menu entries with
        # specific "actions". These actions are typically simply
        # methods that you can call.
        # =================================================================== #
        if yielded.has_key? :actions
          yielded[:actions].each_pair {|title_to_use, method_to_call|
            # =============================================================== #
            # Example how this may look:
            #
            #   'FoobarAction1': :foobar_action1
            #
            # =============================================================== #
            menu_item = gtk_menu_item(title_to_use)
            context_menu << menu_item
            menu_item.signal_connect(:activate) { |widget|
              if parent_widget
                case method_to_call
                when :interactively_create_directory
                  parent_widget.send(method_to_call.to_sym, parent_widget)
                else # This is the default.
                  parent_widget.send(method_to_call.to_sym)
                end
              end
            }
          }
        # =================================================================== #
        # === :numbered_actions
        #
        # This entry point will simply iterate over the elements and add
        # a number, such as "1", "2" and so forth, to denote which number
        # is associated with which particular Gtk::Menu entry.
        # =================================================================== #
        elsif yielded.has_key? :numbered_actions
          index = 0
          # ================================================================= #
          # Iterate over the Hash next, where the key is the title that
          # is displayed, and the value is the method that will be
          # called (via .send()).
          # ================================================================= #
          yielded[:numbered_actions].each_pair {|title_to_use, method_to_call| index += 1
            menu_item = gtk_menu_item("(#{index}) #{title_to_use}")
            # =============================================================== #
            # Append each menu-item onto the context menu:
            # =============================================================== #
            context_menu << menu_item
            menu_item.signal_connect(:activate) { |widget|
              if parent_widget
                case method_to_call
                when :interactively_create_directory
                  parent_widget.send(method_to_call.to_sym, parent_widget)
                else # This is the default.
                  parent_widget.send(method_to_call.to_sym)
                end
              end
            }
          }
        end
        # =================================================================== #
        # === n_entries
        # =================================================================== #
        if yielded.has_key? :n_entries
          yielded[:n_entries].times {|n_run|
            menu_item = gtk_menu_item("Test#{n_run}")
            context_menu << menu_item
            menu_item.signal_connect(:activate) { |widget|
              puts "#{widget.class} - #{menu_item.text?}"
            }
          }
        end
      end
    end
    context_menu.show_all
    return context_menu
  end; alias return_context_menu create_context_menu # === return_context_menu

  # ========================================================================= #
  # === window_manager_name?
  #
  # This method may return a String such as "IceWM 2.7.0 (Linux/x86_64)".
  # ========================================================================= #
  def window_manager_name?
    Gdk::Screen.default.window_manager_name
  end; alias window_manager_name window_manager_name? # === window_manager_name

  # ========================================================================= #
  # === return_debug_widget
  #
  # This method will return a new instance of the main debug-widget in
  # use. The idea behind this widget is to allow us to quickly debug
  # stuff.
  #
  # To require the ruby-gtk3 version, try:
  #
  #   require 'gtk_paradise/widgets/gtk3/debug_widget/debug_widget.rb'
  #
  # ========================================================================= #
  def return_debug_widget
    require 'gtk_paradise/widgets/gtk3/debug_widget/debug_widget.rb'
    ::Gtk::DebugWidget.new
  end; alias gtk_debug_widget return_debug_widget # === gtk_debug_widget

  # ========================================================================= #
  # === debug?
  #
  # This query-method tells us whether the gtk-widget at hand is running
  # in debug-mode or not. By default this is not the case.
  # ========================================================================= #
  def debug?
    if @internal_hash and @internal_hash.has_key?(:debug)
      @internal_hash[:debug]
    else
      false
    end
  end; alias debug debug? # === debug

  # ========================================================================= #
  # === do_enable_debug
  #
  # This method can be used to activate debugging. This is not the default,
  # so you may have to call it explicitely if you need debug-support.
  # ========================================================================= #
  def do_enable_debug
    @internal_hash[:debug] = true
  end; alias enable_debug do_enable_debug # === enable_debug

  # ========================================================================= #
  # === gtk_separator_menu_item
  # ========================================================================= #
  def gtk_separator_menu_item
    ::Gtk::SeparatorMenuItem.new
  end; alias separator_menu_item gtk_separator_menu_item # === separator_menu_item

  # ========================================================================= #
  # === scroll_up?
  # ========================================================================= #
  def scroll_up?(event)
    ::Gtk.scroll_up_event?(event)
  end

  # ========================================================================= #
  # === scroll_down?
  # ========================================================================= #
  def scroll_down?(event)
    ::Gtk.scroll_down_event?(event)
  end

  # ========================================================================= #
  # === gtk_vseparator
  # ========================================================================= #
  def gtk_vseparator
    ::Gtk::Separator.new(:vertical)
  end; alias top_to_bottom_separator gtk_vseparator # === top_to_bottom_separator
       alias gtk_vertical_separator  gtk_vseparator # === gtk_vertical_separator
       alias gtk_vsep                gtk_vseparator # === gtk_vsep
       alias create_vseparator       gtk_vseparator # === create_vseparator

  # ========================================================================= #
  # === gtk_button_box_spread
  # ========================================================================= #
  def gtk_button_box_spread(*i)
    _ = button_box(:horizontal) {{ layout: :spread }}
    _.spacing = 1
    i.flatten.each {|widget|
      _.add(widget)
    }
    return _
  end

  # ========================================================================= #
  # === hbutton_box
  #
  # This is for the horizontal variant of a Gtk::ButtonBox.
  # ========================================================================= #
  def hbutton_box
    return button_box(:horizontal)
  end

  # ========================================================================= #
  # === vbutton_box
  #
  # This is for the vertical variant of a Gtk::ButtonBox.
  # ========================================================================= #
  def vbutton_box
    return button_box(:vertical)
  end

  require 'gtk_paradise/core_classes/frame.rb'
  # ========================================================================= #
  # === gtk_frame
  #
  # This must be able to support a String, which is then used as the
  # label, and a widget, in which case it will be embedded.
  #
  # Alternative usage example:
  #
  #   gtk_frame(small_scrolled_window) {{ title: 'Last exam question' }}
  #
  # ========================================================================= #
  def gtk_frame(
      optional_title = ''
    )
    if optional_title.to_s.start_with?('#<Gtk::') and
      !optional_title.is_a?(String)
      frame = ::Gtk::Frame.new
      frame.add(optional_title)
    else
      frame = ::Gtk::Frame.new(optional_title)
    end
    # ======================================================================= #
    # === Handle blocks given next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :title
        # =================================================================== #
        if yielded.has_key? :title
          frame.set_text(yielded.delete(:title))
        end
      end
    end
    return frame
  end; alias create_frame gtk_frame # === create_frame

  # ========================================================================= #
  # === create_font_chooser_dialog
  # ========================================================================= #
  def create_font_chooser_dialog
    return ::Gtk::FontChooserDialog.new
  end

  # ========================================================================= #
  # === gtk_vbox
  #
  # The alias vbox exists, but I am not entirely sure whether this will
  # be retained or not. The second variant, thus, for the time being,
  # allows something like this:
  #
  #   vbox {
  #     label 'Hello'
  #     button 'World !'
  #   }
  #
  # ========================================================================= #
  def gtk_vbox(*i, &block)
    i.flatten!
    if i.empty?
      vbox = ::Gtk.vbox
    elsif block_given? and i.empty?
      vbox = ::Gtk.vbox(&block)
    else
      vbox = ::Gtk.vbox(i)
    end
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      if yielded.is_a?(Hash) and yielded.has_key?(:spacing)
        vbox.spacing = yielded[:spacing]
      end
    end
    return vbox 
  end; alias vbox        gtk_vbox # === vbox
       alias gtk_box     gtk_vbox # === gtk_box
       alias create_vbox gtk_vbox # === create_vbox

  # ========================================================================= #
  # === image_left_arrow
  #
  # There are several left-arrows, at the least 3, in adwaita-icon-theme.
  #
  # This method here refers to the, in my opinion, "best" looking left
  # arrow image.
  # ========================================================================= #
  def image_left_arrow
    return_this_image_based_on_the_default_icon_theme(
      'application-exit-rtl-symbolic.symbolic' # 'application-exit-symbolic.symbolic'
    )
  end

  # ========================================================================= #
  # === coloured_quit_button
  #
  # This variant can be called like this:
  #
  #   coloured_quit_button(colour: 'honeydew')
  #
  # ========================================================================= #
  def coloured_quit_button(
      optional_hash = {}
    )
    use_this_colour = ''.dup # No colour by default.
    unless use_this_colour.empty?
      use_this_colour.prepend('_') unless use_this_colour.start_with? '_'
    end
    _ = coloured_button("_Quit#{use_this_colour}", true)
    # ======================================================================= #
    # Next, add the quit-specific action to this button.
    # ======================================================================= #
    _.signal_connect(:clicked) { |widget| ::Gtk.main_quit }
    return _
  end

  # ========================================================================= #
  # === gtk_table
  #
  # To use this method, try something like this:
  #
  #   gtk_table(2, 1)
  #   table = gtk_table(2, 1) {{ border_width: 2 }}
  #
  # This can often be similar to something like this:
  #
  #   ::Gtk::Table.new(2, 1, false)
  #
  # ========================================================================= #
  def gtk_table(
      n_rows      = 2,
      n_columns   = 1,
      homogeneous = false
    )
    the_table = ::Gtk::Table.new(n_rows, n_columns, homogeneous)
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        # =================================================================== #
        # === :border_width
        # =================================================================== #
        if yielded.has_key? :border_width
          the_table.set_border_width(yielded.delete(:border_width))
        end
      end
    end
    the_table
  end

  # ========================================================================= #
  # === no_file_exists_at
  # ========================================================================= #
  def no_file_exists_at(i)
    e "No file could be found at `#{sfile(i)}`."
  end; alias no_file_was_found_at no_file_exists_at # === no_file_was_found_at

  # ========================================================================= #
  # === popup_over_this_widget
  #
  # The third argument is the container-widget. That way you can build
  # up your own custom widgets quickly.
  #
  # Usage examples:
  #
  #   popup_over_this_widget(main_entry?, 'Please assign a file to work with first.')
  #   popup_over_this_widget(main_entry?, 'Please assign a file to work with first.', container_widget)
  #   popover_on(this_widget: @search_entry, use_this_message: 'ABC <b>'+_+'</b>.')
  #   pop_up_widget(over_this_widget: @entry_consensus_sequence, text: 'abc def')
  #   pop_up_widget(over_this_widget: @entry_consensus_sequence, text: 'abc def', custom_widget: hbox)
  #   pop_up_widget(over_this_widget: @entry_consensus_sequence, text: 'abc def', custom_widget: hbox) { :make_selectable }
  #   popup_over_this_widget( commit_button?, "The new file can now be found at:\n  <b>#{result}</b> \n" ) { :make_selectable }
  #
  # ========================================================================= #
  def popup_over_this_widget(
      this_widget,
      text                      = '',
      use_this_container_widget = nil,
      &block
    )
    # ======================================================================== #
    # === Handle Hashes first
    # ======================================================================== #
    if this_widget.is_a? Hash
      # ====================================================================== #
      # === :custom_widget
      # ====================================================================== #
      if this_widget.has_key? :custom_widget
        use_this_container_widget = this_widget.delete(:custom_widget)
      end
      # ====================================================================== #
      # === :use_this_message
      #
      # Only check for a few things in this case.
      # ====================================================================== #
      if this_widget.has_key? :use_this_message
        text = this_widget.delete(:use_this_message)
      # ====================================================================== #
      # === :text
      # ====================================================================== #
      elsif this_widget.has_key? :text
        text = this_widget.delete(:text)
      end
      # ====================================================================== #
      # === :this_widget
      # ====================================================================== #
      if this_widget.has_key? :this_widget
        this_widget = this_widget.delete(:this_widget)
      # ====================================================================== #
      # === :over_this_widget
      # ====================================================================== #
      elsif this_widget.has_key? :over_this_widget
        this_widget = this_widget.delete(:over_this_widget)
      end
    end
    return_popover(
      text,
      this_widget,
      :default_padding, # This is the default padding to use.
      use_this_container_widget,
      &block
    ).popup
  end; alias popover_on    popup_over_this_widget # === popover_on
       alias pop_up_widget popup_over_this_widget # === pop_up_widget

  # ========================================================================= #
  # === word_wrap
  #
  # This method has to be fairly flexible, to allow for use cases such as:
  #
  #   word_wrap(i, threshold: WORD_WRAP_AT_N_CHARACTERS)
  #
  # Usage example:
  #
  #   long_string.word_wrap(75)
  #
  # ========================================================================= #
  def word_wrap(
      string,
      line_width       = 80,
      optional_padding = ''
    )
    if line_width.is_a? Hash
      # ===================================================================== #
      # === :threshold
      # ===================================================================== #
      if line_width.has_key? :threshold
        line_width = line_width[:threshold].to_i
      end
    end
    new_text = string.dup.gsub(
      /(.{1,#{line_width}})(\s+|$)/, optional_padding+"\\1\n"
    )
    return new_text
  end

  # ========================================================================= #
  # === draggable_left_to_right_pane
  #
  # This method was added on 21.01.2021.
  #
  # The main idea for this method is to return a draggable hbox, but I keep
  # on mixing up the name with a draggable vbox. Thus, the mental model for
  # "left to right" is much easier to remember, hence the name. In the
  # long run I will re-alias onto this method here, but for now it just
  # is the way it is.
  # ========================================================================= #
  def draggable_left_to_right_pane(
      widget1,
      widget2
    )
    # for gtk3 and beyond
    vpaned = ::Gtk::Paned.new(:vertical)
    vpaned.add1(widget1)
    vpaned.add2(widget2)
    return vpaned
  end; alias gtk_left_to_right_dragger                 draggable_left_to_right_pane # === gtk_left_to_right_dragger
       alias gtk_left_to_right_pane                    draggable_left_to_right_pane # === gtk_left_to_right_pane
       alias gtk_drag_left_to_right                    draggable_left_to_right_pane # === gtk_drag_left_to_right
       alias gtk_draggable_left_to_right               draggable_left_to_right_pane # === gtk_draggable_left_to_right
       alias draggable_left_to_right                   draggable_left_to_right_pane # === draggable_left_to_right
       alias draggable_horizontal_bar                  draggable_left_to_right_pane # === draggable_horizontal_bar
       alias add_an_up_and_down_dragger                draggable_left_to_right_pane # === add_an_up_and_down_dragger
       alias draggable_via_up_and_down_movement        draggable_left_to_right_pane # === draggable_via_up_and_down_movement
       alias draggable_via_up_and_down_movements       draggable_left_to_right_pane # === draggable_via_up_and_down_movements
       alias can_be_dragged_up_and_down                draggable_left_to_right_pane # === can_be_dragged_up_and_down
       alias draggable_up_and_down                     draggable_left_to_right_pane # === draggable_up_and_down
       alias drag_via_up_and_down_movement             draggable_left_to_right_pane # === draggable_from_the_left_to_the_right_side
       alias gtk_drag_up_and_down                      draggable_left_to_right_pane # === gtk_drag_up_and_down
       alias mouse_draggable_via_up_and_down_movement  draggable_left_to_right_pane # === mouse_draggable_via_up_and_down_movement
       alias draggable_up_to_bottom_movement           draggable_left_to_right_pane # === draggable_up_to_bottom_movement
       alias drag_via_up_and_down_mouse_movement       draggable_left_to_right_pane # === drag_via_up_and_down_mouse_movement
       alias up_and_down_dragging                      draggable_left_to_right_pane # === up_and_down_dragging

  # ========================================================================= #
  # === gtk_button
  #
  # The first input-argument to this method should be a String. It does,
  # however had, also support Hashes or block arguments.
  #
  # The block argument can look like this, for a tooltip:
  #
  #   {{ tooltip: "Clicking this button will generate a random aminoacid sequence."}}
  #
  # ========================================================================= #
  def gtk_button(
      i             = nil,
      use_underline = false,
      &block
    )
    # ======================================================================= #
    # The next case menu has some symbol-shortcuts for stock items.
    # That way we can avoid having to type (and remember) these long
    # constants.
    # ======================================================================= #
    if i.is_a? Symbol
      i = return_corresponding_gtk_stock_item(i)
    end
    if i and i.is_a?(String)
      if i.start_with? '_'
        use_underline = true
      end
    end
    # ======================================================================= #
    # === Handle hashes next - this should be fine for gtk2 at the least.
    # ======================================================================= #
    if i.is_a? Hash
      # ===================================================================== #
      # === :label
      # ===================================================================== #
      if i.has_key? :label
        i = i.delete(:label) # Keep in mind that this line here is destructive.
      end
    end
    button = ::Gtk.button(i, use_underline, &block)
    # ======================================================================= #
    # Append the button to the @internal_hash next.
    # ======================================================================= #
    @internal_hash[:buttons] << button if @internal_hash
    return button
  end; alias gtk_button_new gtk_button # === gtk_button
       alias button         gtk_button # === button
       alias create_button  gtk_button # === create_button

  # ========================================================================= #
  # === create_skeleton_then_connect_skeleton_then_apply_the_CSS_rules
  # ========================================================================= #
  def create_skeleton_then_connect_skeleton_then_apply_the_CSS_rules
    create_skeleton_then_connect_skeleton
    apply_the_CSS_rules
  end

  # ========================================================================= #
  # === sbr
  #
  # This is normally aliased to e as well, but e is already an alias to
  # "puts", so we can not use this. The "br" in sbr stands for <br>, so
  # a newline ultimately - hence the "\n" that is appended in the method.
  # ========================================================================= #
  def sbr(i = '')
    create_label("#{i}\n")
  end

  # ========================================================================= #
  # === return_p_tag
  #
  # This method shall ultimately simulate a <p> tag. We will assume
  # that a gtk-hbox can handle a paragraph properly.
  # ========================================================================= #
  def return_p_tag(
      optional_css_class = '',
      &block
    )
    optional_css_class = optional_css_class.to_s # Need a String past this point.
    p_paragraph = create_hbox(&block) # Create the hbox here.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| p_paragraph.add_css_class(inner_entry) }
        else
          p_paragraph.add_css_class(entry) # Add the CSS class to the paragraph.
        end
      }
    end
    return p_paragraph
  end

  # ========================================================================= #
  # === gtk_cell_renderer_combo
  # ========================================================================= #
  def gtk_cell_renderer_combo
    ::Gtk::CellRendererCombo.new
  end

  # ========================================================================= #
  # === gtk_cell_renderer_pixbuf
  # ========================================================================= #
  def gtk_cell_renderer_pixbuf
    ::Gtk::CellRendererPixbuf.new
  end; alias cell_renderer_pixbuf gtk_cell_renderer_pixbuf # === cell_renderer_pixbuf

  # ========================================================================= #
  # === gtk_cell_renderer_centered_text
  # ========================================================================= #
  def gtk_cell_renderer_centered_text
    _ = gtk_cell_renderer_text
    _.center
    return _
  end

  # ========================================================================= #
  # === gtk_cell_renderer_text
  # ========================================================================= #
  def gtk_cell_renderer_text
    ::Gtk::CellRendererText.new
  end; alias gtk_cellrenderer_text     gtk_cell_renderer_text # === gtk_cellrenderer_text
       alias gtk_cell_renderer         gtk_cell_renderer_text # === gtk_cell_renderer
       alias cell_renderer_text        gtk_cell_renderer_text # === cell_renderer_text
       alias create_cell_renderer_text gtk_cell_renderer_text # === create_cell_renderer_text

  # ========================================================================= #
  # === centered_cell_renderer_text
  # ========================================================================= #
  def centered_cell_renderer_text
    _ = cell_renderer_text
    _.xalign = 0.5 # Center it here.
    return _
  end

  # ========================================================================= #
  # === gtk_cell_renderer_right_aligned
  # ========================================================================= #
  def gtk_cell_renderer_right_aligned
    _ = ::Gtk::CellRendererText.new
    _.set_xalign(1.0)
    return _
  end

  # ========================================================================= #
  # === gtk_menu_bar
  # ========================================================================= #
  def gtk_menu_bar
    ::Gtk::MenuBar.new
  end; alias create_menu_bar gtk_menu_bar # === create_menu_bar

  # ========================================================================= #
  # === gtk_dropdown
  #
  # This creates a "drop-down" box, a combo-box entry.
  # ========================================================================= #
  def gtk_dropdown(*i)
    i.flatten!
    dropdown_box = gtk_combo_box(true)
    # Next, append every element given to this method.
    i.each { |entry| dropdown_box.append_text(entry) }
    return dropdown_box
  end; alias gtk_dropdown_box gtk_dropdown # === gtk_dropdown_box

  # ========================================================================= #
  # === app_chooser_button
  # ========================================================================= #
  def app_chooser_button(
      i = 'x-scheme-handler/mailto'
    )
    ::Gtk::AppChooserButton.new(i)
  end; alias choose_app app_chooser_button # === choose_app

  # ========================================================================= #
  # === gtk_main_quit
  # ========================================================================= #
  def gtk_main_quit
    ::Gtk.main_quit
  end; alias gtk_quit      gtk_main_quit # === gtk_quit
       alias gtk_main_exit gtk_main_quit # === gtk_main_exit

  # ========================================================================= #
  # === resize_based_on_width_and_height_attributes
  #
  # This method will resize based on .width? and .height? results.
  # ========================================================================= #
  def resize_based_on_width_and_height_attributes(
      optional_arguments = nil
    )
    width  = width?
    height = height?
    if width.is_a?(String) and width.include?('%')
      # ===================================================================== #
      # Handle something like '80%'.
      # ===================================================================== #
      width = (max_width? * width.delete('%').to_f) / 100.0
    end
    if height.is_a?(String) and height.include?('%')
      # ===================================================================== #
      # Handle something like '80%'.
      # ===================================================================== #
      height = (max_height? * height.delete('%').to_f) / 100.0
    end
    case optional_arguments
    # ======================================================================= #
    # === :debug
    # ======================================================================= #
    when :debug
      e 'Width and Height are: '+width.to_s+', '+height.to_s
    end
    set_size_request(width.to_i, height.to_i)
  end; alias infer_the_size_automatically resize_based_on_width_and_height_attributes # === infer_the_size_automatically
       alias infer_size_request           resize_based_on_width_and_height_attributes # === infer_size_request
       alias determine_the_default_size   resize_based_on_width_and_height_attributes # === determine_the_default_size

  # ========================================================================= #
  # === padded_vbox
  # ========================================================================= #
  def padded_vbox
    _ = ::Gtk.vbox
    _.pad5px
    return _
  end

  # ========================================================================= #
  # === set_background
  # ========================================================================= #
  def set_background(
      use_this_colour = :grey
    )
    use_this_colour = ::Gtk.colour_parse(use_this_colour)
    # if use_gtk2?
    #   modify_bg(
    #     ::Gtk::StateType::NORMAL, use_this_colour
    #   )
    if use_gtk4?
      e 'Gdk::Color is not available in ruby-gtk4. We have to replace'
      e 'this at a later time.' 
    else
      if use_this_colour.is_a? Gdk::Color
        # =================================================================== #
        # Convert it into RGB values next.
        # =================================================================== #
        array = [
          (use_this_colour.red   * 100.0 / 65535) / 100,
          (use_this_colour.green * 100.0 / 65535) / 100,
          (use_this_colour.blue  * 100.0 / 65535) / 100,
          1.0
        ]
        use_this_colour = Gdk::RGBA.new(*array)
      end
      override_background_color(:normal, use_this_colour)
    end
  end; alias background=           set_background # === background=
       alias set_bg                set_background # === set_bg
       alias background_colour     set_background # === background_colour
       alias set_background_colour set_background # === set_background_colour

  # ========================================================================= #
  # === override_normal_background_color
  # ========================================================================= #
  def override_normal_background_color(
      colour_to_use = Gdk::RGBA.new(0.999, 0.999, 0.999, 1)
    )
    override_background_color(:normal, colour_to_use)
  end

  # ========================================================================= #
  # === cliner
  # ========================================================================= #
  def cliner
    e '=' * 80
  end

  # ========================================================================= #
  # === Gtk::BaseModule.extended
  #
  # Whenever this module is extended, .reset_the_internal_variables()
  # is called, in order to initialize @configuration and other
  # important instance-variables.
  # ========================================================================= #
  def self.extended(i)
    i.reset_the_internal_variables
  end

  # ========================================================================= #
  # === gtk_text_buffer
  #
  # The argument to this method can be used as argument to set_text(), aka
  # you can set a default (startup) text for this text-buffer instance.
  # ========================================================================= #
  def gtk_text_buffer(
      optional_input = nil
    )
    text_buffer = ::Gtk::TextBuffer.new
    if optional_input and optional_input.is_a?(String)
      text_buffer.set_text(optional_input)
    end
    return text_buffer
  end; alias text_buffer        gtk_text_buffer # === text_buffer
       alias create_text_buffer gtk_text_buffer # === create_text_buffer
       alias create_buffer      gtk_text_buffer # === create_buffer

  # ========================================================================= #
  # === missing_the_internal_hash?
  # ========================================================================= #
  def missing_the_internal_hash?
    _ = false
    _ = true if defined?(:@internal_hash)
    _ = true if @internal_hash.nil?
    return _
  end

  # ========================================================================= #
  # === font_chooser_dialog
  # ========================================================================= #
  def font_chooser_dialog(
      use_this_title = nil
    )
    _ = create_font_chooser_dialog
    if use_this_title
      _.set_title(use_this_title)
      # Could also do: _.move(0, 0)
    end
    response = _.run # Run it at once.
    if response == Gtk::ResponseType::OK
      ::Gtk.set_main_file(_.font.to_s)
    end
    return _
  end; alias gtk_font_chooser_dialog font_chooser_dialog # === gtk_font_chooser_dialog

  # ========================================================================= #
  # === gtk_combo_box_text
  #
  # A wrapper-method over Gtk::ComboBoxText, specifically for ruby-gtk3.
  # ========================================================================= #
  def gtk_combo_box_text(
      i = nil
    )
    _ = ::Gtk::ComboBoxText.new
    if i
      i.each {|entry|
        _.append_text(entry)
      }
    end
    # ======================================================================= #
    # === Handle blocks given to this method next
    # ======================================================================= #
    if block_given?
      yielded = yield
      case yielded
      # ===================================================================== #
      # === :first_element_is_active
      # ===================================================================== #
      when :first_element_is_active,
           :first
        _.set_active(0)
      end
    end
    return _
  end; alias combo_box_text    gtk_combo_box_text # === combo_box_text
       alias combo_box         gtk_combo_box_text # === combo_box_text
       alias create_combo_box  gtk_combo_box_text # === create_combo_box
       alias gtk_combobox_text gtk_combo_box_text # === gtk_combobox_text

  # ========================================================================= #
  # === simple_quit_button
  # ========================================================================= #
  def simple_quit_button
    _ = button('_Quit')
    _.on_clicked { exit_the_application }
    return _
  end

  # ========================================================================= #
  # === image_input_mouse
  # ========================================================================= #
  def image_input_mouse
    return_this_image_based_on_the_default_icon_theme(:input_mouse)
  end

  # ========================================================================= #
  # === image_audio_volume_high
  # ========================================================================= #
  def image_audio_volume_high
    return_this_image_based_on_the_default_icon_theme(:audio_volume_high)
  end

  # ========================================================================= #
  # === image_audio_volume_low
  # ========================================================================= #
  def image_audio_volume_low
    return_this_image_based_on_the_default_icon_theme(:audio_volume_low)
  end

  # ========================================================================= #
  # === image_quit_action
  #
  # This shows a "quit-image", or at the least it seems to be related
  # to a quit-action.
  # ========================================================================= #
  def image_quit_action
    return_this_image_based_on_the_default_icon_theme(:application_exit)
  end

  require 'gtk_paradise/core_classes/vscale.rb'
  # ========================================================================= #
  # === gtk_vscale
  # ========================================================================= #
  def gtk_vscale(
      a = 0.5, b = 0, c = 0
    )
    if a.is_a? Hash
      if a.has_key? :n_digits
        a = a[:n_digits]
        a = ::Gtk::Adjustment.new(a, b, c, 0, 0, 0)
      end
    end
    # if use_gtk2?
    #  ::Gtk::VScale.new(a, b, c)
    ::Gtk::Scale.new(:vertical, a, b, c)
  end; alias vscale gtk_vscale # === vscale

  # ========================================================================= #
  # === gtk_progress_bar
  #
  # The progress-bar can indicate progress, from a range of 1% to 100%
  # typically.
  # ========================================================================= #
  def gtk_progress_bar
    _ = ::Gtk::ProgressBar.new
    return _
  end; alias progress_bar gtk_progress_bar # === progress_bar

  # ========================================================================= #
  # === text_then_entry
  #
  # This method will return a hbox, which in turn contains a text, and then
  # an entry for user input.
  #
  # Three elements will be returned: the hbox, then the text (a gtk-label)
  # and finally the entry (a gtk-entry).
  #
  # The method may be called with some variation, such as:
  #
  #   text_then_entry('Name of the card', padding: 5)
  #
  # Note that since as of 2023, it may be better to use a Gtk::Grid rather
  # than a hbox. For now the method here will remain as it is, but
  # perhaps in the future we may also add the option to use a gtk-grid
  # rather than a gtk-hbox.
  # ========================================================================= #
  def text_then_entry(
      this_text        = '',
      use_this_padding = 0
    )
    hbox  = create_hbox
    text  = create_text(this_text)
    entry = create_entry
    hbox.minimal(text,  use_this_padding)
    hbox.minimal(entry, use_this_padding)
    return [hbox, text, entry]
  end

  require 'gtk_paradise/core_classes/hscale.rb'
  # ========================================================================= #
  # === gtk_hscale
  # ========================================================================= #
  def gtk_hscale(a = 0.5, b = 0, c = 0)
    if a.is_a? Hash
      if a.has_key? :n_digits
        a = a[:n_digits]
        a = ::Gtk::Adjustment.new(a, b, c, 0, 0, 0)
      end
    end
    ::Gtk::Scale.new(:horizontal, a, b, c)
  end; alias hscale        gtk_hscale # === hscale
       alias create_hscale gtk_hscale # === create_hscale

  # ========================================================================= #
  # === gtk_coloured_button
  #
  # The first argument to this method should be a String.
  #
  # A typical usage example goes like this:
  #
  #   coloured_button('_DBGET_ivory', false)
  #
  # ========================================================================= #
  def gtk_coloured_button(
      i                        = '',
      optional_second_argument = false
    )
    i = i.dup if i.frozen?
    unless i.empty?
      if i.count('_') < 2
        # =================================================================== #
        # In this case we must append the default colour, via a leading
        # '_' token.
        # =================================================================== #
        i << "_#{Gtk::ColouredButton.default_colour?}"
      end
    end
    begin
      require 'gtk_paradise/widgets/gtk3/coloured_button/coloured_button.rb'
    rescue LoadError => error
      pp error
    end
    coloured_button = ::Gtk::ColouredButton.new(i, optional_second_argument)
    @internal_hash[:buttons] << coloured_button if @internal_hash
    return coloured_button
  end; alias coloured_button gtk_coloured_button # === coloured_button

  # ========================================================================= #
  # === esystem                                                 (esystem tag)
  #
  # This method will first output the argument, then run system() on it.
  # It is thus a verbose way to run system-commands.
  # ========================================================================= #
  def esystem(i)
    puts i
    system i
  end

  # ========================================================================= #
  # === gtk_fixed
  #
  # Simpler API wrapper over instantiating a new Gtk::Fixed object.
  # ========================================================================= #
  def gtk_fixed
    ::Gtk::Fixed.new
  end

  # ========================================================================= #
  # === pango_font_description
  #
  # This is simply a wrapper over Pango::FontDescription.
  # ========================================================================= #
  def pango_font_description(
      i = USE_THIS_FONT
    )
    Pango::FontDescription.new(i)
  end; alias font_description pango_font_description # === font_description

  # ========================================================================= #
  # === all_entries?
  # ========================================================================= #
  def all_entries?
    @internal_hash[:entries]
  end; alias return_all_entries all_entries? # === return_all_entries

  # ========================================================================= #
  # === all_check_buttons?
  # ========================================================================= #
  def all_check_buttons?
    @internal_hash[:check_buttons]
  end; alias return_all_check_buttons all_check_buttons? # === return_all_check_buttons
       alias check_buttons?           all_check_buttons? # === check_buttons?

  # ========================================================================= #
  # === all_toggle_buttons?
  # ========================================================================= #
  def all_toggle_buttons?
    @internal_hash[:toggle_buttons]
  end; alias return_all_toggle_buttons all_toggle_buttons? # return_all_toggle_buttons
       alias toggle_buttons?           all_toggle_buttons? # === toggle_buttons?

  # ========================================================================= #
  # === all_buttons?
  # ========================================================================= #
  def all_buttons?
    @internal_hash[:buttons]
  end; alias return_all_buttons  all_buttons? # === return_all_buttons
       alias return_all_buttons? all_buttons? # === return_all_buttons?

  # ========================================================================= #
  # === left_arrow_emoji
  # ========================================================================= #
  def left_arrow_emoji
    emoji(:arrow_left)
  end; alias gtk_label_arrow_left left_arrow_emoji # === gtk_label_arrow_left

  # ========================================================================= #
  # === gtk_table2
  #
  # This method will create a table with 2 rows per line, based on
  # ListStore, TreeView and TreeViewColumn.
  #
  # As input you should provide an Array with even numbers, e. g.
  # 2, 4, 6 elements and so forth.
  #
  # Note that this method return both the tree-view as well as the
  # list-store.
  #
  # Usage example:
  #
  #   use_this_dataset = return_dataset_as_expanded_array
  #   _ = gtk_table2(use_this_dataset)
  #
  # ========================================================================= #
  def gtk_table2(*i)
    i.flatten! # Must keep our Array flat.
    list_store = ::Gtk::ListStore.new(String, String)
    # ======================================================================= #
    # === Handle blocks next
    # ======================================================================= #
    if block_given?
      yielded = yield
      # ===================================================================== #
      # === :list_store
      # ===================================================================== #
      if yielded.is_a?(Hash) and yielded.has_key?(:list_store)
        list_store = yielded[:list_store]
      end
    end
    tree_view = ::Gtk::TreeView.new(list_store)
    # ======================================================================= #
    # Get the first two members and make them the column-headers.
    # ======================================================================= #
    column1 = ::Gtk::TreeViewColumn.new(i.shift.to_s, ::Gtk::CellRendererText.new, { text: 0 })
    column2 = ::Gtk::TreeViewColumn.new(i.shift.to_s, ::Gtk::CellRendererText.new, { text: 1 })
    # ======================================================================= #
    # Add the two columns to the tree-view next.
    # ======================================================================= #
    tree_view.append_column(column1)
    tree_view.append_column(column2)
    # ======================================================================= #
    # Next we will feed the members of that Array in pairs-of-2 into
    # the TreeView widget, which explains why we use .each_slice().
    # ======================================================================= #
    i.each_slice(2) { |first, second|
      iter = list_store.append
      iter.set_value(0, first)
      iter.set_value(1, second)
    } unless i.empty?
    tree_view.enable_model_drag_source(:button1_mask,
      [ ['GTK_TREE_MODEL_ROW', 0, 0] ], 
      Gdk::DragAction::COPY|Gdk::DragAction::MOVE
    )
    tree_view.enable_model_drag_dest(
      [ ['GTK_TREE_MODEL_ROW', 0, 0] ],
      Gdk::DragAction::COPY|Gdk::DragAction::MOVE
    )
    # ======================================================================= #
    # Two results will be returned next, via an Array.
    # ======================================================================= #
    return [tree_view, list_store] # ← Last but not least, return the tree-view here, together with the list store.
  end; alias tree_view_table2                     gtk_table2 # === table2
       alias tree_widget_with_two_columns_per_row gtk_table2 # === tree_widget_with_two_columns_per_row
       alias create_table2                        gtk_table2 # === create_table2

  # ========================================================================= #
  # === p                                                             (p tag)
  # ========================================================================= #
  def p(
      optional_css_class = '',
      &block
    )
    minimal(
      return_p_tag(optional_css_class, &block)
    )
  end

  # ========================================================================= #
  # === div
  #
  # This method shall ultimately simulate a <div> tag. For now this will
  # be assumed to be a gtk-vbox.
  # ========================================================================= #
  def div(&block)
    create_hbox(&block)
  end

  # ========================================================================= #
  # === pre                                                         (pre tag)
  #
  # This shall be an approximation of the <pre> tag in HTML.
  # ========================================================================= #
  def pre(
      i = ''
    )
    label = create_label(i)
    label.set_line_wrap(true) # See: https://ruby-gnome2.osdn.jp/hiki.cgi?Gtk%3A%3ALabel
    return label
  end

  # ========================================================================= #
  # === h1                                                           (h1 tag)
  # ========================================================================= #
  def h1(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    h1_font_size = HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h1]
    label.increase_the_font_size_to(h1_font_size) # Make the <h1> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === h2                                                           (h2 tag)
  # ========================================================================= #
  def h2(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    label.increase_the_font_size_to(HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h2]) # Make the <h2> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === h3                                                           (h3 tag)
  # ========================================================================= #
  def h3(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    label.increase_the_font_size_to(HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h3]) # Make the <h3> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === h4                                                           (h4 tag)
  # ========================================================================= #
  def h4(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    label.increase_the_font_size_to(HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h4]) # Make the <h4> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === h5                                                           (h5 tag)
  # ========================================================================= #
  def h5(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    label.increase_the_font_size_to(HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h5]) # Make the <h5> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === h6                                                           (h6 tag)
  # ========================================================================= #
  def h6(
      i                  = '',
      optional_css_class = ''
    )
    label = create_label(i)
    label.increase_the_font_size_to(HASH_USE_THESE_PERCENTAGE_VALUES_FOR_THE_HEADERS[:h6]) # Make the <h6> tag larger.
    if optional_css_class and !optional_css_class.empty?
      _ = [optional_css_class]
      _.each {|entry|
        if entry.include? ' '
          entry.split(' ').each {|inner_entry| label.add_css_class(inner_entry) }
        else
          label.add_css_class(entry)
        end
      }
    end
    label.left_align
    return label
  end

  # ========================================================================= #
  # === try_to_set_the_xorg_buffer
  # ========================================================================= #
  def try_to_set_the_xorg_buffer(i)
    begin
      require 'xorg_buffer'
      XorgBuffer.copy(i) if Object.const_defined?(:XorgBuffer)
    rescue LoadError; end
  end

  # ========================================================================= #
  # === use_libui?
  #
  # Whether this object is making use of libui. This is logically false if
  # we use gtk, but the reason this method was added was to be able to
  # use different GUIs, so a simple check is useful for that purpose.
  # ========================================================================= #
  def use_libui?
    false
  end

  # ========================================================================= #
  # === icon_theme?
  # ========================================================================= #
  def icon_theme?
    ::Gtk::IconTheme.default
  end

  # ========================================================================= #
  # === use_gtk?
  #
  # This is an ad-hoc method; it was used primarily to simplify
  # transitioning into UniversalWidgets.
  # ========================================================================= #
  def use_gtk?
    true
  end

  # ========================================================================= #
  # === gtk_statusbar_with_handle
  # ========================================================================= #
  def gtk_statusbar_with_handle
    gtk_statusbar { :with_handle }
  end

  # ========================================================================= #
  # === create_skeleton_then_connect_skeleton
  #
  # Note that you may have to create your own methods called
  # connect_skeleton() and connect_skeleton().
  # ========================================================================= #
  def create_skeleton_then_connect_skeleton
    create_skeleton
    connect_skeleton
  end

  # ========================================================================= #
  # === connect_the_skeleton
  # ========================================================================= #
  def connect_the_skeleton
  end; alias connect_skeleton connect_the_skeleton # === connect_skeleton

  # ========================================================================= #
  # === create_skeleton
  # ========================================================================= #
  def create_skeleton
  end; alias create_the_skeleton create_skeleton # === create_the_skeleton

  # ========================================================================= #
  # === add_or_append
  # ========================================================================= #
  def add_or_append(i)
    if use_gtk3?
      add(i)
    elsif use_gtk4?
      append(i)
    end
  end

  # ========================================================================= #
  # === set_font_in_use
  # ========================================================================= #
  def set_font_in_use(i)
    ::Gtk.set_font(i)
  end

  # ========================================================================= #
  # === upon_delete_event_quit_the_application
  #
  # This method can be used to "cleanly" exit a Gtk-application again.
  #
  # Note that the :delete_event no longer exists in ruby-gtk4.
  # ========================================================================= #
  def upon_delete_event_quit_the_application
    if use_gtk3?
      signal_connect(:delete_event) { ::Gtk.main_quit }
    elsif use_gtk4?
      signal_connect(:destroy) { ::Gtk.main_quit }
    end
  end; alias on_delete_event_quit_the_application upon_delete_event_quit_the_application # === on_delete_event_quit_the_application
       alias destroy_on_exit                      upon_delete_event_quit_the_application # === destroy_on_exit

  # ========================================================================= #
  # === enable_these_key_combinations
  #
  # Right now this method requires a Hash. At some point in the future
  # we may ease up on that constraint.
  # ========================================================================= #
  def enable_these_key_combinations(
      hash,
      work_on_this_internal_hash = @internal_hash
    )
    unless hash.is_a? Hash
      e 'We need a Hash for now. This may change at a later moment in'
      e 'time, though.'
      exit
    end
    # ======================================================================= #
    # The Hash that is passed into this method may look like this:
    #
    #   { "ctrl+s"=>"save_the_buffer", :"strg+s"=>"save_the_buffer"}
    #
    # or:
    #
    #   { 'Alt+w' => 'report_return_date' }
    #
    # ======================================================================= #
    hash.each_pair {|this_shortcut_is_to_be_used, call_this_method|
      # ===================================================================== #
      # Remove ' ' characters.
      # ===================================================================== #
      if this_shortcut_is_to_be_used.is_a?(String) and
         this_shortcut_is_to_be_used.include?(' ')
        # =================================================================== #
        # Since as of 12.05.2021 we will remove ' '.
        # =================================================================== #
        if this_shortcut_is_to_be_used.frozen?
          this_shortcut_is_to_be_used = this_shortcut_is_to_be_used.dup
        end
        this_shortcut_is_to_be_used.delete!(' ')
      end
      if work_on_this_internal_hash[:parent_widget]
        work_on_this_internal_hash[:parent_widget].keybinding(
          this_shortcut_is_to_be_used,
          call_this_method
        )
      else
        e '@internal_hash[:parent_widget] has not been initialized yet.'
      end
    }
  end

  # ========================================================================= #
  # === return_numbered_lines_widget
  #
  # This method will yield - and return - the numbered-lines widget.
  # ========================================================================= #
  def return_numbered_lines_widget(
      pass_this_hash = {
        upper_range: 100
      },
      &block
    )
    unless Gtk.const_defined? :NumberedLines
      require 'gtk_paradise/requires/numbered_lines.rb'
    end
    if block_given?
      yielded = yield
      if yielded.is_a? Hash
        pass_this_hash = yielded
      end
    end
    return ::Gtk::NumberedLines.new { pass_this_hash }
  end; alias return_widget_with_line_numbers return_numbered_lines_widget # === return_widget_with_line_numbers
       alias return_numbered_lines_widget    return_numbered_lines_widget # === return_numbered_lines_widget
       alias numbered_text_view_widget       return_numbered_lines_widget # === numbered_text_view_widget
       alias numbered_lines_widget           return_numbered_lines_widget # === numbered_lines_widget

  # ========================================================================= #
  # === main_font?
  # ========================================================================= #
  def main_font?(i = USE_THIS_FONT)
    i
  end

  # ========================================================================= #
  # === project_base_directory?
  # ========================================================================= #
  def project_base_directory?
    ::Gtk.project_base_directory?
  end; alias base_dir?         project_base_directory? # === base_dir?
       alias project_base_dir? project_base_directory? # === project_base_dir?

  # ========================================================================= #
  # === project_yaml_directory?
  #
  # Pointer towards the gtk_paradise project's yaml-directory.
  # ========================================================================= #
  def project_yaml_directory?
    "#{project_base_directory?}yaml/"
  end

  # ========================================================================= #
  # === set_font
  # ========================================================================= #
  def set_font(
      this_font = 'sans 22'
    )
    # ======================================================================= #
    # === Handle Symbols first
    # ======================================================================= #
    if this_font.is_a? Symbol
      this_font = ::Gtk.expand_this_font(this_font)
    end
    if this_font.is_a? Numeric
      this_font = this_font.to_s
    end
    if this_font.is_a? String
      this_font = Pango::FontDescription.new(this_font)
    end
    if use_gtk3?
      # ===================================================================== #
      # Note that gtk_widget_override_font() is deprecated since as of
      # GTK 3.16. We may have to transition into CSS rules, which should
      # work for both gtk3 and gtk4.
      # ===================================================================== #
      override_font(this_font)
    elsif use_gtk4?
      # ===================================================================== #
      # Set a CSS class here.
      # ===================================================================== #
      string = this_font.to_s
      use_this_new_CSS_rule = '.font_'+string.downcase.tr(' ','_')+' {
  font-family: '+string+';
}
'
      css_class(use_this_new_CSS_rule)
    else
      modify_font(this_font)
    end
  end; alias set_use_this_gtk_font                    set_font # === set_use_this_gtk_font
       alias set_use_this_font                        set_font # === set_use_this_font
       alias set_font_size                            set_font # === set_font_size
       alias set_font_to_use                          set_font # === set_font_to_use
       alias set_the_font_in_use_for_this_application set_font # === set_the_font_in_use_for_this_application
       alias use_this_font                            set_font # === use_this_font
       alias fonty                                    set_font # === fonty

  # ========================================================================= #
  # === cairo_context
  # ========================================================================= #
  def cairo_context(surface)
    return Cairo::Context.new(surface)
  end

  # ========================================================================= #
  # === gtk_entry_completion
  #
  # This will create a new instance of Gtk::EntryCompletion.
  #
  # The first argument can be used to attach the completion-object to
  # the widget. An Array can be passed as well.
  # ========================================================================= #
  def gtk_entry_completion(
      i = nil
    )
    entry_completion = ::Gtk::EntryCompletion.new
    entry_completion.populate_with_this_array(i) if i and i.is_a?(Array)
    return entry_completion
  end; alias create_entry_completion gtk_entry_completion # === create_entry_completion

  # ========================================================================= #
  # === draggable_from_the_left_to_the_right_side
  #
  # This method will return a draggable horizontal pane. 
  #
  # Simply pass the two widgets that you wish to have separated into
  # this method, as first and second argument, respectively.
  # ========================================================================= #
  def draggable_from_the_left_to_the_right_side(
      widget1 = nil,
      widget2 = nil
    )
    # This clause is specifically for ruby-gtk3 and ruby-gtk4
    paned = ::Gtk::Paned.new(:horizontal)
    if widget1 and widget1.is_a?(Array) and (widget1.size > 1)
      widget2 = widget1.last # This must come first.
      widget1 = widget1.first
    end
    paned.add1(widget1) if widget1
    paned.add2(widget2) if widget2
    return paned
  end; alias draggable_top_to_bottom_pane                  draggable_from_the_left_to_the_right_side # === draggable_top_to_bottom_pane
       alias up_and_down_pane                              draggable_from_the_left_to_the_right_side # === up_and_down_pane
       alias draggable_vertical_pane                       draggable_from_the_left_to_the_right_side # === draggable_vertical_pane
       alias draggable_up_and_down_pane                    draggable_from_the_left_to_the_right_side # === draggable_up_and_down_pane
       alias gtk_pane_top_to_bottom                        draggable_from_the_left_to_the_right_side # === gtk_pane_top_to_bottom
       alias gtk_draggable_top_to_bottom                   draggable_from_the_left_to_the_right_side # === gtk_draggable_top_to_bottom
       alias gtk_top_to_bottom_pane                        draggable_from_the_left_to_the_right_side # === gtk_top_to_bottom_pane
       alias gtk_drag_top_to_bottom                        draggable_from_the_left_to_the_right_side # === gtk_drag_top_to_bottom
       alias gtk_drag_top_to_bottom_pane                   draggable_from_the_left_to_the_right_side # === gtk_drag_top_to_bottom_pane
       alias gtk_hbox_with_horizontal_pane                 draggable_from_the_left_to_the_right_side # === gtk_hbox_with_horizontal_pane
       alias vertical_draggable_pane                       draggable_from_the_left_to_the_right_side # === vertical_draggable_pane
       alias gtk_vertical_dragger                          draggable_from_the_left_to_the_right_side # === gtk_vertical_dragger
       alias vertical_dragger                              draggable_from_the_left_to_the_right_side # === vertical_dragger
       alias vertical_pane                                 draggable_from_the_left_to_the_right_side # === vertical_pane
       alias allow_left_to_right_dragging                  draggable_from_the_left_to_the_right_side # === allow_left_to_right_dragging
       alias drag_via_left_and_right_movement              draggable_from_the_left_to_the_right_side # === allow_left_to_right_dragging
       alias drag_via_left_to_right_mouse_movement         draggable_from_the_left_to_the_right_side # === drag_via_left_to_right_mouse_movement
       alias drag_via_left_and_right_movement_of_the_mouse draggable_from_the_left_to_the_right_side # === drag_via_left_and_right_movement_of_the_mouse
       alias drag_left_to_right                            draggable_from_the_left_to_the_right_side # === drag_left_to_right
       alias drag_from_left_to_right                       draggable_from_the_left_to_the_right_side # === drag_from_left_to_right
       alias drag_it_left_to_right                         draggable_from_the_left_to_the_right_side # === drag_it_left_to_right
       alias left_to_right_dragger                         draggable_from_the_left_to_the_right_side # === left_to_right_dragger
       alias draggable_from_left_to_right                  draggable_from_the_left_to_the_right_side # === draggable_from_left_to_right
       alias drag_via_left_to_right                        draggable_from_the_left_to_the_right_side # === drag_via_left_to_right
       alias draggable_via_left_to_right_movement          draggable_from_the_left_to_the_right_side # === draggable_via_left_to_right_movement
       alias draggable_via_left_and_right_movement         draggable_from_the_left_to_the_right_side # === draggable_via_left_and_right_movement

  require 'gtk_paradise/core_classes/combo_box.rb'
  # ========================================================================= #
  # === gtk_combo_box
  # ========================================================================= #
  def gtk_combo_box(
      array_or_nil = nil,
      i            = :strings_only
    )
    if array_or_nil and 
       array_or_nil.is_a?(Array) and 
       array_or_nil.respond_to?(:sort)
      array_or_nil = array_or_nil.sort
    end
    _ = ::Gtk.combo_box(array_or_nil, i)
    _.set_active(0)
    return _
  end; alias create_combobox gtk_combo_box # === create_combobox

  # ========================================================================= #
  # === run_in_background_if_is_on_roebe?
  # ========================================================================= #
  def run_in_background_if_is_on_roebe?
    run_in_background if is_on_roebe?
  end

  # ========================================================================= #
  # === run_main
  #
  # This is simply a wrapper over Gtk.main. It was added
  # ========================================================================= #
  def run_main
    Gtk.main
  end

  # ========================================================================= #
  # === run_in_background
  #
  # This method can be used to put the application in the background,
  # via Process.daemon().
  #
  # Only call it if you are absolutely certain that you want to have this
  # application run in the background, though.
  # ========================================================================= #
  def run_in_background
    Process.daemon
  end; alias run_in_the_background   run_in_background # === run_in_the_background
       alias start_in_the_background run_in_background # === start_in_the_background

  # ========================================================================= #
  # === right_arrow_emoji
  #
  # This is guaranteed to return the right-arrow emoji. The usual symbol
  # for the right-arrow emoji is '➡️'.
  # ========================================================================= #
  def right_arrow_emoji
    emoji(:arrow_right)
  end; alias gtk_label_arrow_right right_arrow_emoji # === gtk_label_arrow_right

  # ========================================================================= #
  # === emoji
  #
  # This convenient method can be used to quickly use a Gtk::Label
  # instance to add an emoji.
  #
  # Usage exmaples:
  #
  #   emoji(:pin)
  #   emoji(:arrow_right)
  #   emoji_label(:croatia)
  #
  # ========================================================================= #
  def emoji(
      i = :arrow_right
    )
    create_label(
      return_emoji(i)
    )
  end; alias emoji_label emoji # === emoji_label
       alias gtk_emoji   emoji # === gtk_emoji

  # ========================================================================= #
  # === return_emoji
  # ========================================================================= #
  def return_emoji(i)
    ::Gtk::Emojis.map_emoji(i)
  end; alias map_emoji return_emoji # === map_emoji

  # ========================================================================= #
  # === create_and_add_gtk_accel_group
  # ========================================================================= #
  def create_and_add_gtk_accel_group
    accel_group = gtk_accel_group
    add_accel_group(accel_group) if respond_to? :add_accel_group
    return accel_group
  end

  # ========================================================================= #
  # === gtk_accel_group
  # ========================================================================= #
  def gtk_accel_group
    ::Gtk::AccelGroup.new
  end; alias gtk_accelgroup gtk_accel_group # === gtk_accelgroup

  # ========================================================================= #
  # === entry_with_this_completion
  #
  # This method requires an Array as its input.
  # ========================================================================= #
  def entry_with_this_completion(
      array
    )
    this_entry = entry
    this_entry.completion = create_entry_completion(array)
    return this_entry
  end

  require 'gtk_paradise/core_classes/label.rb'
  # ========================================================================= #
  # === gtk_text
  # ========================================================================= #
  def gtk_text(
      i             = '',
      make_bold     = false,
      use_mnemonics = false
    )
    ::Gtk.text(i, make_bold, use_mnemonics)
  end; alias create_text  gtk_text # === create_text
       alias create_label gtk_text # === create_label

  # ========================================================================= #
  # === selectable_text
  # ========================================================================= #
  def selectable_text(
      use_this_text = '',
      make_bold     = false,
      use_mnemonics = false
    )
    _ = create_label(
      use_this_text,
      make_bold,
      use_mnemonics
    )
    _.make_selectable
    return _
  end; alias selectable_label selectable_text # === selectable_label

  # ========================================================================= #
  # === gtk_hseparator
  #
  # This is a horizontal separator.
  # ========================================================================= #
  def gtk_hseparator
    ::Gtk::Separator.new(:horizontal)
  end; alias horizontal_separator gtk_hseparator # === horizontal_separator
       alias gtk_separator        gtk_hseparator # === gtk_separator
       alias gtk_hsep             gtk_hseparator # === gtk_hsep

  # ========================================================================= #
  # === group_maximal
  # ========================================================================= #
  def group_maximal(*i)
    i.flatten.each {|entry|
      maximal(entry)
    }
  end

  # ========================================================================= #
  # === run                                                         (run tag)
  # ========================================================================= #
  def run
    create_skeleton_then_connect_skeleton_then_apply_the_CSS_rules
  end

end

# =========================================================================== #
# === Gtk.directory_to_the_images?
# =========================================================================== #
def self.directory_to_the_images?
  ::Gtk::BaseModule::IMG_DIR
end

end