# Copyright (C) 2018  Matthew Pitkin
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.


## \defgroup lal_py_antenna Antenna
## \ingroup lal_python
"""This module provides a Python class for generating antenna response
functions. By default the class uses its own implementation of the antenna
response functions, but it also provides a wrapper to the equivalent
functions within LAL.
"""
# \author Matthew Pitkin (<matthew.pitkin@ligo.org>)
#
# ### Synopsis ###
#
# ~~~
# from lal import antenna
# ~~~
#
# ### Examples ###
#
# A simple example of just getting the antenna response for LHO at a single
# time for a single sky position and polarization angle would be:
#
# \code
# from lal import antenna
# ra = 1.2  # right ascension in radians
# dec = -0.3  # declination in radians
# psi = 1.5  # polarization angle in radians
# times = 1000000000.  # time (GPS seconds)
# resp = antenna.AntennaResponse('H1', ra, dec, psi=psi, times=times)
# print(resp.plus)
# \endcode
#
# To produce plots of the antenna responses over the sky for the different
# polarizations, assuming a detector at the North Pole, one could use:
#
# \code
# import numpy as np
# import lal
# from lal.antenna import AntennaResponse
# from mpl_toolkits.mplot3d import Axes3D
# import matplotlib.pyplot as pl
#
# # set a LALFrDetector object for the North Pole
# frdt = lal.FrDetector()
#
# frdt.name = 'NORTHPOLE'
# frdt.prefix = 'N1'
# frdt.vertexLongitudeRadians = 0.
# frdt.vertexLatitudeRadians = np.pi/2.
# frdt.vertexElevation = 0.
# frdt.xArmAltitudeRadians = 0.
# frdt.xArmAzimuthRadians = 0.
# frdt.yArmAltitudeRadians = 0.
# frdt.yArmAzimuthRadians = np.pi/2.  # y-arm 90 degs from x-arm
# frdt.xArmMidpoint = 2000.  # 4km long arms
# frdt.yArmMidpoint = 2000.
#
# # create lal.Detector object
# det = lal.Detector()
# lal.CreateDetector(det, frdt, lal.LALDETECTORTYPE_IFODIFF)
#
# # create RA/dec grids uniform on sphere
# ras = np.linspace(0, 2.*np.pi, 150)
# decs = np.arcsin(np.linspace(-1, 1, 150))
#
# # create antenna response on the sky grid
# resp = AntennaResponse(det, ras, decs, psi=0.0, times=900000000.0, vector=True,
#                        scalar=True)
#
# # get equatorial to cartesian factors
# xfac = np.cos(resp.dec_mesh)*np.cos(resp.ra_mesh)
# yfac = np.cos(resp.dec_mesh)*np.sin(resp.ra_mesh)
# zfac = np.sin(resp.dec_mesh)
#
# fig = pl.figure(figsize=(12, 7))
#
# for i, mode in enumerate(['plus', 'x', 'b', 'cross', 'y', 'l']):
#     r = np.abs(resp.response[mode])  # get absolute values of response
#
#     # convert from equatorial coordinates to cartesian
#     X, Y, Z = r*xfac, r*yfac, r*zfac
#
#     ax = fig.add_subplot(2, 3, i+1, projection='3d')
#     ax.plot_surface(X, Y, Z, cmap=pl.cm.viridis, linewidth=0, rstride=1, cstride=1)
#     ax.set_aspect('equal')
#     ax.set_title(mode)
#
# fig.tight_layout()
#
# pl.show()
# \endcode
#
# which will display:
# <img src="data:image/jpeg;base64,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"/>
#
# <!--
# To produce the above jpeg byte string for the image I have used:
# from io import BytesIO
# import base64
# figfile = BytesIO()
# fig.savefig(figfile, dpi=50, format='jpg')
# figfile.seek(0)  # rewind to beginning of file
# figdata_jpg = base64.b64encode(figfile.getvalue()).decode('utf8')
# -->
## @{


import numpy as np


# import SWIG-wrapped LAL
from lal import (LALDetectorIndexLHODIFF, LALDetectorIndexLLODIFF,
                 LALDetectorIndexGEO600DIFF, LALDetectorIndexVIRGODIFF,
                 LALDetectorIndexTAMA300DIFF, LALDetectorIndexKAGRADIFF,
                 LALDetectorIndexLIODIFF, LALDetectorIndexE1DIFF,
                 LALDetectorIndexE2DIFF, LALDetectorIndexE3DIFF,
                 CachedDetectors, LIGOTimeGPS, GreenwichMeanSiderealTime,
                 ComputeDetAMResponse, ComputeDetAMResponseExtraModes,
                 DAYSID_SI, Detector, TranslateHMStoRAD, TranslateDMStoRAD)

from . import git_version

__author__ = "Matthew Pitkin <matthew.pitkin@ligo.org>"
__version__ = git_version.verbose_msg
__date__ = git_version.date


## mapping between detector names and LALCachedDetectors
DETMAP = {'H1': LALDetectorIndexLHODIFF,
          'H2': LALDetectorIndexLHODIFF,
          'LHO': LALDetectorIndexLHODIFF,
          'L1': LALDetectorIndexLLODIFF,
          'LLO': LALDetectorIndexLLODIFF,
          'G1': LALDetectorIndexGEO600DIFF,
          'GEO': LALDetectorIndexGEO600DIFF,
          'GEO600': LALDetectorIndexGEO600DIFF,
          'V1': LALDetectorIndexVIRGODIFF,
          'VIRGO': LALDetectorIndexVIRGODIFF,
          'T1': LALDetectorIndexTAMA300DIFF,
          'TAMA': LALDetectorIndexTAMA300DIFF,
          'TAMA300': LALDetectorIndexTAMA300DIFF,
          'K1': LALDetectorIndexKAGRADIFF,
          'KAGRA': LALDetectorIndexKAGRADIFF,
          'LCGT': LALDetectorIndexKAGRADIFF,
          'I1': LALDetectorIndexLIODIFF,
          'LIO': LALDetectorIndexLIODIFF,
          'E1': LALDetectorIndexE1DIFF,
          'E2': LALDetectorIndexE2DIFF,
          'E3': LALDetectorIndexE3DIFF}


## The ranges of variables required for the antenna response (for look-up table generation)
VAR_RANGES = {'psi': [0., 2.*np.pi],
              'time': [0., DAYSID_SI],
              'ra': [0., 2.*np.pi],
              'dec': [-1., 1.]}  # this range is actually in sin(dec)


class AntennaResponse(object):
    def __init__(self, detector, ra, dec, psi=0., times=None, tensor=True,
                 vector=False, scalar=False, use_lal=False, lookup=False,
                 lookuppair=None, bins1=100, bins2=100):
        """
        Calculate the long-wavelength limit antenna response functions for a given
        ground-based gravitational wave detector. The response can include tensor,
        vector, and scalar modes.

        @param detector: (str) a valid detector name (e.g., 'H1') or
            lal.Detector object.
        @param ra: (array_like) the right ascension of the source in radians,
            or a string of the format 'hh:mm:ss.s'.
        @param dec: (array_like) the declination of the source in radians, or a
            string of the format 'dd:mm:ss.s'.
        @param psi: (array_like) the polarization angle in radians. Defaults to
            zero.
        @param times: (array_like) an array of GPS time values at which to
            calculate the response function.
        @param tensor: (bool) set to calculate and store the tensor
            polarization components (plus and cross). Defaults to True.
        @param vector: (bool) set to calculate and store the vector
            polarization components ("x" and "y"). Defaults to False.
        @param scalar: (bool) set to calculate and store the scalar
            polarization components (longitudinal and breathing). Defaults to
            False.
        @param use_lal: (bool) set to internally use the
            XLALComputeDetAMResponse() and XLALComputeDetAMResponseExtraModes()
            functions. Defaults to False.
        @param lookup: (bool) set to generate and use a look-up table in a pair
            of parameters for computing the antenna responses. Defaults to
            False. If using the look-up table, the arrays of values being
            "looked-up" must be in ascending order.
        @param lookuppair: (list, tuple) a list of the two parameters that will
            be used in the look-up table interpolation. Defaults to
            <tt>['psi', 'time']</tt> (allowed values are <tt>'ra'</tt>,
            <tt>'dec'</tt>, <tt>'psi'</tt>, or <tt>'time'</tt>)
        @param bins1: (int) the number of bins in the grid in the first look-up
            table parameter. Defaults to 100.
        @param bins2: (int) the number of bins in the grid in the second
            look-up table parameter. Defaults to 100.

        Example usage for tensor polarizations:
        ~~~
        >>> from lal.antenna import AntennaResponse
        >>> # compute tensor response for a single time
        >>> resp = AntennaResponse('H1', ra=1.2, dec=-0.3, psi=2.9,
        ...times=1000000000)
        >>> print('Fplus: {}'.format(resp.plus))
        Fplus: [0.32427018]
        >>> print('Fcross: {}'.format(resp.cross))
        Fcross: [-0.79809163]
        >>> # re-use class to get response at multiple new times
        >>> resp.compute_response([1010101010., 1234567890.])
        >>> print('Fplus: {}'.format(resp.plus))
        Fplus: [ 0.09498567 -0.45495654]
        >>> print('Fcross: {}'.format(resp.cross))
        Fcross: [0.1706959  0.21690418]
        ~~~

        Example usage for tensor, vector and scalar polarizations (at a series
        of times):
        ~~~
        >>> import numpy as np
        >>> times = np.linspace(1000000000.0, 1000086340.0, 1440)
        >>> resp = AntennaResponse('H1', ra=1.2, dec=-0.3, psi=2.9,
        ...scalar=True, vector=True, times=times)
        >>> resp.plus
        array([0.32427018, 0.32805983, 0.3318344 , ..., 0.32780195, 0.33157755,
               0.33533786])
        >>> resp.cross
        array([-0.79809163, -0.79607858, -0.79404097, ..., -0.7962166 ,
               -0.79418066, -0.79212028])
        >>> resp.x  # vector "x" polarization
        array([-0.46915186, -0.46773594, -0.46627224, ..., -0.46783399,
               -0.46637354, -0.46486538])
        >>> resp.y  # vector "y" polarization
        array([-0.17075718, -0.17475991, -0.17875012, ..., -0.17448742,
               -0.17847849, -0.18245689])
        >>> resp.b  # scalar "breathing" mode
        array([0.05365678, 0.05573073, 0.05780282, ..., 0.05558939, 0.05766162,
               0.05973181])
        >>> resp.l  # scalar "longitudinal mode"
        array([-0.05365678, -0.05573073, -0.05780282, ..., -0.05558939,
               -0.05766162, -0.05973181])
        ~~~
        """

        # response function dictionary
        self.response = {'plus': None,   # tensor plus polarization
                         'cross': None,  # tensor cross polarization
                         'x': None,      # vector "x" polarization
                         'y': None,      # vector "y" polarization
                         'b': None,      # scalar breathing mode polarizarion
                         'l': None}      # scalar longitudinal mode polarization

        # parameter names and order in arrays
        self.parameters = ['ra', 'dec', 'psi', 'time']

        # set values
        self.detector = detector
        self.ra = ra
        self.dec = dec
        self.psi = psi
        self.times = times

        # polarization modes
        self.tensor = tensor
        self.vector = vector
        self.scalar = scalar

        # set whether to use internal LAL functions
        self.use_lal = use_lal

        # set whether to allocate and use a look-up table
        if lookup:
            if lookuppair is None:  # use default pair
                self.set_lookup_pair(bins1=bins1, bins2=bins2)
            else:
                self.set_lookup_pair(pair=lookuppair, bins1=bins1, bins2=bins2)
        self.lookup = lookup

        # calculate antenna responses
        self.compute_response()

    @property
    def detector(self):
        return self._detector

    @detector.setter
    def detector(self, det):
        """
        Set the detector for which to calculate the antenna response.

        @param det: (str) a valid detector name.
        """

        if isinstance(det, Detector):
            self._detector = det.frDetector.prefix
            self.laldetector = det
        elif isinstance(det, str):
            if det.upper() not in DETMAP.keys():
                raise ValueError("Detector is not a valid detector name")

            self._detector = det.upper()

            # set the LAL detector object
            self.laldetector = self.detector
        else:
            raise TypeError("Detector must be a string or lal.Detector object")

    @property
    def laldetector(self):
        return self._laldetector

    @laldetector.setter
    def laldetector(self, det):
        """
        Set the lal.Detector.

        @param det: (str) a valid detector name.
        """

        if isinstance(det, Detector):
            self._laldetector = det.response
        elif isinstance(det, str):
            try:
                detector = DETMAP[det.upper()]
            except KeyError:
                raise KeyError("Key {} is not a valid detector name.".format(det))

            self._laldetector = CachedDetectors[detector].response
        else:
            raise TypeError("Detector must be a string or lal.Detector object")

    @property
    def ra(self):
        return self._ra

    @ra.setter
    def ra(self, raval):
        """
        Set the right ascension.
        """

        if raval is None:
            self._ra = None
            return
        elif isinstance(raval, float) or isinstance(raval, int):
            self._ra = np.array([raval], dtype='float64')
        elif isinstance(raval, list) or isinstance(raval, np.ndarray):
            self._ra = np.copy(raval).astype('float64')
        elif isinstance(raval, str):
            try:
                rarad = TranslateHMStoRAD(raval)
                self._ra = np.array([rarad], dtype='float64')
            except RuntimeError:
                raise ValueError("Could not convert '{}' to a right "
                                 "ascension".format(raval))
        else:
            raise TypeError("Right ascension must be an array")

    @property
    def dec(self):
        return self._dec

    @property
    def costheta(self):
        return self._costheta

    @property
    def sintheta(self):
        return self._sintheta

    @dec.setter
    def dec(self, decval):
        """
        Set the declination.
        """

        if decval is None:
            self._dec = None
            self._costheta = None
            self._sintheta = None
            return
        elif isinstance(decval, float) or isinstance(decval, int):
            self._dec = np.array([decval], dtype='float64')
        elif isinstance(decval, list) or isinstance(decval, np.ndarray):
            self._dec = np.copy(decval).astype('float64')
        elif isinstance(decval, str):
            try:
                decrad = TranslateDMStoRAD(decval)
                self._dec = np.array([decrad], dtype='float64')
            except RuntimeError:
                raise ValueError("Could not convert '{}' to a "
                                 "declination".format(decval))
        else:
            raise TypeError("Declination must be an array")

        self._costheta = np.cos(0.5*np.pi - self._dec)
        self._sintheta = np.sin(0.5*np.pi - self._dec)

    @property
    def psi(self):
        return self._psi

    @property
    def cospsi(self):
        return self._cospsi

    @property
    def sinpsi(self):
        return self._sinpsi

    @psi.setter
    def psi(self, psival):
        """
        Set the value of the gravitational wave polarization angle psi.

        @param psival: (float) the polarization angle (radians)
        """

        if psival is None:
            self._psi = None
            self._cospsi = None
            self._sinpsi = None
            return
        elif isinstance(psival, float) or isinstance(psival, int):
            self._psi = np.array([psival], dtype='float64')
        elif isinstance(psival, list) or isinstance(psival, np.ndarray):
            self._psi = np.copy(psival).astype('float64')
        else:
            raise TypeError("Polarization must be an array")

        self._psi = np.mod(self._psi, 2.*np.pi)  # wrap at 0 and 2pi
        self._cospsi = np.cos(self._psi)
        self._sinpsi = np.sin(self._psi)

    @property
    def times(self):
        return self._times

    @property
    def gmsttimes(self):
        try:
            return self._gmsttimes
        except AttributeError:
            return None

    @times.setter
    def times(self, timearr):
        """
        Set array of times and GPS times.
        """

        # check if times is just a float or int, and if so convert into an array
        if timearr is None:
            self._times = None
            self._gmsttimes = None
            return
        elif isinstance(timearr, float) or isinstance(timearr, int):
            self._times = np.array([timearr], dtype='float64')
        elif isinstance(timearr, list) or isinstance(timearr, np.ndarray):
            self._times = np.copy(timearr).astype('float64')
        else:
            raise TypeError("Times must be an array")

        # get an array of GMSTs
        self._gmsttimes = np.ones_like(self._times) * np.nan  # Greenwich Mean Siderial time (radians)
        for i, time in enumerate(self._times):
            gps = LIGOTimeGPS(time)  # GPS time
            gmstrad = GreenwichMeanSiderealTime(gps)
            self._gmsttimes[i] = gmstrad

    @property
    def shape(self):
        try:
            return self._ra_mesh.shape
        except Exception:
            if (self.ra is not None and self.dec is not None and
                    self.psi is not None and self.times is not None):
                return (len(self.ra), len(self.dec), len(self.psi),
                        len(self.times))
            else:
                return ()

    @property
    def cosphi(self):
        return self._cosphi

    @property
    def sinphi(self):
        return self._sinphi

    @property
    def tensor(self):
        return self._tensor

    @tensor.setter
    def tensor(self, tensorval):
        """
        Set whether to include tensor polarizations.
        """

        if not isinstance(tensorval, bool):
            raise TypeError("Must be boolean value")

        self._tensor = tensorval

    @property
    def vector(self):
        return self._vector

    @vector.setter
    def vector(self, vectorval):
        """
        Set whether to include vector polarizations.
        """

        if not isinstance(vectorval, bool):
            raise TypeError("Must be boolean value")

        self._vector = vectorval

    @property
    def scalar(self):
        return self._scalar

    @scalar.setter
    def scalar(self, scalarval):
        """
        Set whether to include scalar polarizations.
        """

        if not isinstance(scalarval, bool):
            raise TypeError("Must be boolean value")

        self._scalar = scalarval

    @property
    def use_lal(self):
        return self._use_lal

    @use_lal.setter
    def use_lal(self, val):
        """
        Set whether to use LAL antenna response functions.
        """

        if not isinstance(val, bool):
            raise TypeError("Must be a boolean value")

        self._use_lal = val

    @property
    def lookup_pair(self):
        return self._lookup_pair

    @lookup_pair.setter
    def lookup_pair(self, pair):
        if not isinstance(pair, (list, tuple)):
            raise TypeError('Pair must be a list or tuple')

        if len(pair) != 2:
            raise ValueError('Pair must only contain two values')

        for val in pair:
            if val.lower() not in self.parameters:
                raise ValueError('Parameter {} in pair not '
                                 'recognized'.format(val))

        self._lookup_pair = [val.lower() for val in pair]

        # make sure pair is in same order as parameters
        self._lookup_pair = [val for val in self.parameters
                             if val in self._lookup_pair]

        # set parameters that are not in the look-up pair
        self._not_lookup_pair = [val.lower() for val in self.parameters
                                 if val not in self._lookup_pair]
        self._not_lookup_pair_idx = [self.parameters.index(val)
                                     for val in self._not_lookup_pair]

    def set_lookup_pair(self, pair=['psi', 'time'], bins1=100, bins2=100):
        """
        Set the pair of parameters to use for the look-up table.
        """

        self.lookup_pair = pair
        self._lookup_pair_dict = {}

        for val, nbins in zip(pair, [bins1, bins2]):
            if not isinstance(nbins, int):
                raise TypeError("Value must be an integer")

            if nbins < 2:
                raise ValueError("There must be at least 2 bins")

            self._lookup_pair_dict[val] = {}

            vrange = VAR_RANGES[val]

            # set array of bins
            if val == 'dec':
                # range is in sin(dec) to give uniform grid on sky sphere, so
                # convert into dec
                vararray = np.arcsin(np.linspace(vrange[0], vrange[1], nbins))
            else:
                vararray = np.linspace(vrange[0], vrange[1], nbins)

            self._lookup_pair_dict[val]['array'] = vararray
            self._lookup_pair_dict[val]['nbins'] = nbins

    @property
    def bins1(self):
        return self._lookup_pair_dict[self.lookup_pair[0]]['nbins']

    @property
    def array1(self):
        return self._lookup_pair_dict[self.lookup_pair[0]]['array']

    @property
    def bins2(self):
        return self._lookup_pair_dict[self.lookup_pair[1]]['nbins']

    @property
    def array2(self):
        return self._lookup_pair_dict[self.lookup_pair[1]]['array']

    @property
    def lookup(self):
        return self._lookup

    @lookup.setter
    def lookup(self, val):
        """
        Set the 2d look-up table.
        """

        self._timeepoch = 1000000000.  # a GPS epoch

        if not isinstance(val, bool):
            raise TypeError("Value must be a boolean")

        self._lookup = False  # set to False so look-up table can be calculated

        # set the look-up table
        if val:
            try:
                from scipy.interpolate import RectBivariateSpline
            except ImportError:
                raise ImportError("Cannot import scipy")

            curra = self.ra        # save current RA value
            curdec = self.dec      # save current dec value
            curpsi = self.psi      # save current psi value
            try:
                curtimes = self.times  # save current times
            except AttributeError:
                raise AttributeError("A time must be set for the look-up table")

            if 'psi' in self.lookup_pair:
                self.psi = self._lookup_pair_dict['psi']['array']
            if 'time' in self.lookup_pair:
                self.times = (self._timeepoch +
                              self._lookup_pair_dict['time']['array'])
            if 'ra' in self.lookup_pair:
                self.ra = self._lookup_pair_dict['ra']['array']
            if 'dec' in self.lookup_pair:
                self.dec = self._lookup_pair_dict['dec']['array']

            self._not_lookup_pair_lens = []  # lengths of the non-look-up table values
            for val in self._not_lookup_pair:
                if val == 'ra':
                    self._not_lookup_pair_lens.append(len(curra))
                if val == 'dec':
                    self._not_lookup_pair_lens.append(len(curdec))
                if val == 'psi':
                    self._not_lookup_pair_lens.append(len(curpsi))
                if val in 'time':
                    self._not_lookup_pair_lens.append(len(curtimes))

            # set look-up table functions
            self._lookup_func = {}
            if self.tensor:
                self._lookup_func['plus'] = np.empty(self._not_lookup_pair_lens, dtype=object)
                self._lookup_func['cross'] = np.empty(self._not_lookup_pair_lens, dtype=object)

            if self.vector:
                self._lookup_func['x'] = np.empty(self._not_lookup_pair_lens, dtype=object)
                self._lookup_func['y'] = np.empty(self._not_lookup_pair_lens, dtype=object)

            if self.scalar:
                self._lookup_func['b'] = np.empty(self._not_lookup_pair_lens, dtype=object)
                self._lookup_func['l'] = np.empty(self._not_lookup_pair_lens, dtype=object)

            for i in range(self._not_lookup_pair_lens[0]):
                if self._not_lookup_pair[0] == 'ra':
                    self.ra = curra[i]
                if self._not_lookup_pair[0] == 'dec':
                    self.dec = curdec[i]
                if self._not_lookup_pair[0] == 'psi':
                    self.psi = curpsi[i]
                if self._not_lookup_pair[0] == 'time':
                    self.times = curtimes[i]

                for j in range(self._not_lookup_pair_lens[1]):
                    if self._not_lookup_pair[1] == 'ra':
                        self.ra = curra[j]
                    if self._not_lookup_pair[1] == 'dec':
                        self.dec = curdec[j]
                    if self._not_lookup_pair[1] == 'psi':
                        self.psi = curpsi[j]
                    if self._not_lookup_pair[1] == 'time':
                        self.times = curtimes[j]

                    self.compute_response()
                    if self.tensor:
                        self._lookup_func['plus'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                             self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                             self.plus)
                        self._lookup_func['cross'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                              self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                              self.cross)

                    if self.vector:
                        self._lookup_func['x'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                          self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                          self.x)
                        self._lookup_func['y'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                          self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                          self.y)

                    if self.scalar:
                        self._lookup_func['b'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                          self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                          self.b)
                        self._lookup_func['l'][i, j] = RectBivariateSpline(self._lookup_pair_dict[self.lookup_pair[0]]['array'],
                                                                          self._lookup_pair_dict[self.lookup_pair[1]]['array'],
                                                                          self.l)

            # reset values
            self.psi = curpsi
            self.times = curtimes
            self.ra = curra
            self.dec = curdec
            self._lookup = True

    @property
    def plus(self):
        return self.response['plus']

    @property
    def tensor_plus(self):
        return self.plus

    @plus.setter
    def plus(self, resp):
        if self.tensor:
            self.response['plus'] = resp

    @property
    def cross(self):
        return self.response['cross']

    @property
    def tensor_cross(self):
        return self.cross

    @cross.setter
    def cross(self, resp):
        if self.tensor:
            self.response['cross'] = resp

    @property
    def x(self):
        return self.response['x']

    @property
    def vector_x(self):
        return self.x

    @x.setter
    def x(self, resp):
        if self.vector:
            self.response['x'] = resp

    @property
    def y(self):
        return self.response['y']

    @property
    def vector_y(self):
        return self.y

    @y.setter
    def y(self, resp):
        if self.vector:
            self.response['y'] = resp

    @property
    def b(self):
        return self.response['b']

    @property
    def scalar_b(self):
        return self.b

    @b.setter
    def b(self, resp):
        if self.scalar:
            self.response['b'] = resp

    @property
    def l(self):
        return self.response['l']

    @property
    def scalar_l(self):
        return self.l

    @l.setter
    def l(self, resp):
        if self.scalar:
            self.response['l'] = resp

    def _set_mesh(self):
        """
        Convert one-dimensional arrays into a mesh. The meshes dimensions are
        ordered as right ascension, declination, psi, and time.
        """

        # mesh order RA, dec, psi, time
        if (self.ra is None or self.dec is None or
                self.psi is None or self.times is None):
            return

        ramesh, decmesh, psimesh, timemesh = np.meshgrid(self.ra, self.dec,
                                                         self.psi,
                                                         self.gmsttimes,
                                                         indexing='ij')

        self.ra_mesh = ramesh
        self.time_mesh = timemesh
        self.dec_mesh = decmesh
        self.psi_mesh = psimesh

    @property
    def ra_mesh(self):
        return self._ra_mesh

    @ra_mesh.setter
    def ra_mesh(self, val):
        if isinstance(val, np.ndarray):
            # remove singleton dimemsions
            self._ra_mesh = val.squeeze()
        else:
            self._ra_mesh = val

        try:
            self._phi_mesh = np.mod(self._ra_mesh, 2.*np.pi) - self._time_mesh
        except Exception:
            return

        self._cosphi_mesh = np.cos(self._phi_mesh)
        self._sinphi_mesh = np.sin(self._phi_mesh)

    @property
    def time_mesh(self):
        return self._time_mesh

    @time_mesh.setter
    def time_mesh(self, val):
        if isinstance(val, np.ndarray):
            # remove singleton dimemsions
            self._time_mesh = val.squeeze()
        else:
            self._time_mesh = val

        try:
            self._phi_mesh = self._ra_mesh - self._time_mesh
        except Exception:
            return

        self._cosphi_mesh = np.cos(self._phi_mesh)
        self._sinphi_mesh = np.sin(self._phi_mesh)

    @property
    def dec_mesh(self):
        return self._dec_mesh

    @dec_mesh.setter
    def dec_mesh(self, val):
        if isinstance(val, np.ndarray):
            # remove singleton dimemsions
            self._dec_mesh = val.squeeze()
        else:
            self._dec_mesh = val

        self._costheta_mesh = np.cos(0.5*np.pi - self.dec_mesh)
        self._sintheta_mesh = np.sin(0.5*np.pi - self.dec_mesh)

    @property
    def psi_mesh(self):
        return self._psi_mesh

    @psi_mesh.setter
    def psi_mesh(self, val):
        if isinstance(val, np.ndarray):
            # remove singleton dimemsions
            self._psi_mesh = val.squeeze()
        else:
            self._psi_mesh = val

        self._cospsi_mesh = np.cos(self._psi_mesh)
        self._sinpsi_mesh = np.sin(self._psi_mesh)

    def compute_response(self, times=None):
        """
        Compute the detector response.

        @param times: (array_like) an array of GPS times at which to compute
            the response function. If not set the times set at initialization
            of the class, or using the <tt>times</tt> property.
        """

        if times is not None:
            self.times = times

        if (self.ra is None or self.dec is None or self.psi is None or
                self.times is None):
            return

        if self.use_lal:  # use the internal LAL functions
            self._compute_response_lal()
        elif self.lookup:  # use look-up table
            self._compute_response_lookup()
        else:  # use default response
            self._compute_response()

    def _compute_response(self):
        """
        Compute antenna pattern.
        """

        self._set_mesh()

        # set any required additional einsum indices
        einsum_indices = ''
        eindices = 'klmn'  # up to four indices
        for i in range(len(self.shape)):
            einsum_indices += eindices[i]

        # numpy einsum string inputs
        einstr1 = 'i{},j{}->ij{}'.format(*3*[einsum_indices])
        einstr2 = 'ij{},ij->{}'.format(*2*[einsum_indices])

        M = np.array([self._sinphi_mesh*self._cospsi_mesh - self._cosphi_mesh*self._costheta_mesh*self._sinpsi_mesh,
                      -self._cosphi_mesh*self._cospsi_mesh - self._sinphi_mesh*self._costheta_mesh*self._sinpsi_mesh,
                      self._sintheta_mesh*self._sinpsi_mesh])
        N = np.array([-self._sinphi_mesh*self._sinpsi_mesh - self._cosphi_mesh*self._costheta_mesh*self._cospsi_mesh,
                      self._cosphi_mesh*self._sinpsi_mesh - self._sinphi_mesh*self._costheta_mesh*self._cospsi_mesh,
                      self._sintheta_mesh*self._cospsi_mesh])

        mm = np.einsum(einstr1, M, M)
        mn = np.einsum(einstr1, M, N)
        nm = np.einsum(einstr1, N, M)
        nn = np.einsum(einstr1, N, N)

        if self.tensor:
            # set tensor polarization components
            self.plus = np.einsum(einstr2, mm - nn, self.laldetector)
            self.cross = np.einsum(einstr2, mn + nm, self.laldetector)

        if self.vector or self.scalar:
            # set scalar and/or vector polarization components
            Q = np.array([-self._sintheta_mesh*self._cosphi_mesh,
                          -self._sintheta_mesh*self._sinphi_mesh,
                          -self._costheta_mesh])

            if self.vector:
                mq = np.einsum(einstr1, M, Q)
                qm = np.einsum(einstr1, Q, M)
                nq = np.einsum(einstr1, N, Q)
                qn = np.einsum(einstr1, Q, N)

                self.x = np.einsum(einstr2, mq + qm, self.laldetector)
                self.y = np.einsum(einstr2, nq + qn, self.laldetector)

            if self.scalar:
                qq = np.einsum(einstr1, Q, Q)

                self.b = np.einsum(einstr2, mm + nn, self.laldetector)
                self.l = np.einsum(einstr2, qq, self.laldetector)

    def _compute_response_lal(self):
        """
        Compute antenna pattern using LAL functions.
        """

        self._set_mesh()

        slen = np.prod(self.shape) if len(self.shape) != 0 else 1

        # allocate memory
        if slen != 1:
            fp = np.zeros(self.shape)
            fc = np.zeros(self.shape)

        if self.tensor and not self.vector and not self.scalar:
            # only requiring tensor mode
            antenna_func = ComputeDetAMResponse

            if slen == 1:
                fp, fc = antenna_func(self.laldetector,
                                      self._ra_mesh.item(),
                                      self._dec_mesh.item(),
                                      self._psi_mesh.item(),
                                      self._time_mesh.item())
            else:
                for i in range(slen):
                    idxs = np.unravel_index(i, self.shape)

                    fp[idxs], fc[idxs] = antenna_func(self.laldetector,
                                                      self._ra_mesh[idxs],
                                                      self._dec_mesh[idxs],
                                                      self._psi_mesh[idxs],
                                                      self._time_mesh[idxs])

            self.plus = fp
            self.cross = fc
        else:
            antenna_func = ComputeDetAMResponseExtraModes

            if slen == 1:
                fp, fc, fb, fl, fx, fy = antenna_func(self.laldetector,
                                                      self._ra_mesh.item(),
                                                      self._dec_mesh.item(),
                                                      self._psi_mesh.item(),
                                                      self._time_mesh.item())
            else:
                fb = np.zeros(self.shape)
                fl = np.zeros(self.shape)
                fx = np.zeros(self.shape)
                fy = np.zeros(self.shape)

                for i in range(slen):
                    idxs = np.unravel_index(i, self.shape)

                    F = antenna_func(self.laldetector,
                                     self._ra_mesh[idxs],
                                     self._dec_mesh[idxs],
                                     self._psi_mesh[idxs],
                                     self._time_mesh[idxs])

                    fp[idxs] = F[0]
                    fc[idxs] = F[1]
                    fb[idxs] = F[2]
                    fl[idxs] = F[3]
                    fx[idxs] = F[4]
                    fy[idxs] = F[5]

            if self.tensor:
                self.plus = fp
                self.cross = fc

            if self.vector:
                self.x = fx
                self.y = fy

            if self.scalar:
                self.b = fb
                self.l = fl

    def _compute_response_lookup(self):
        # allocate memory
        lushape = (len(self.ra), len(self.dec), len(self.psi), len(self.times))

        if self.tensor:
            fp = np.zeros(lushape)
            fc = np.zeros(lushape)

        if self.scalar:
            fb = np.zeros(lushape)
            fl = np.zeros(lushape)

        if self.vector:
            fx = np.zeros(lushape)
            fy = np.zeros(lushape)

        # set pair of parameters in look-up table
        pairs = []
        unsorted_idxs = None
        for i, val in enumerate(self.lookup_pair):
            if val == 'ra':
                pairs.append(self.ra)
            if val == 'dec':
                pairs.append(self.dec)
            if val == 'psi':
                pairs.append(self.psi)
            if val == 'time':
                # times mod-ed by a sidereal day
                mtimes = np.mod(self.times - self._timeepoch, DAYSID_SI)
                unsorted_idxs = np.argsort(np.argsort(mtimes))  # unsorted indices
                mtimes.sort()  # sort times for interpolation
                pairs.append(mtimes)

        # get reshape tuple
        sshape = [len(self.ra), len(self.dec), len(self.psi), len(self.times)]
        sshape[self._not_lookup_pair_idx[0]] = 1
        sshape[self._not_lookup_pair_idx[1]] = 1
        sshape = tuple(sshape)

        # create slice
        pos = 4*[slice(None)]
        for i in range(self._not_lookup_pair_lens[0]):
            # set slice
            pos[self._not_lookup_pair_idx[0]] = slice(i, i+1)
            for j in range(self._not_lookup_pair_lens[1]):
                # set slice
                pos[self._not_lookup_pair_idx[1]] = slice(j, j+1)

                if self.tensor:
                    fp[tuple(pos)] = self._lookup_func['plus'][i, j](*pairs).reshape(sshape)
                    fc[tuple(pos)] = self._lookup_func['cross'][i, j](*pairs).reshape(sshape)

                if self.scalar:
                    fb[tuple(pos)] = self._lookup_func['b'][i, j](*pairs).reshape(sshape)
                    fl[tuple(pos)] = self._lookup_func['l'][i, j](*pairs).reshape(sshape)

                if self.vector:
                    fx[tuple(pos)] = self._lookup_func['x'][i, j](*pairs).reshape(sshape)
                    fy[tuple(pos)] = self._lookup_func['y'][i, j](*pairs).reshape(sshape)

        if unsorted_idxs is not None:
            if self.tensor:
                self.plus = fp[:, :, :, unsorted_idxs].squeeze()
                self.cross = fc[:, :, :, unsorted_idxs].squeeze()

            if self.scalar:
                self.b = fb[:, :, :, unsorted_idxs].squeeze()
                self.l = fl[:, :, :, unsorted_idxs].squeeze()

            if self.vector:
                self.x = fx[:, :, :, unsorted_idxs].squeeze()
                self.y = fy[:, :, :, unsorted_idxs].squeeze()
        else:
            if self.tensor:
                self.plus = fp.squeeze()
                self.cross = fc.squeeze()

            if self.scalar:
                self.b = fb.squeeze()
                self.l = fl.squeeze()

            if self.vector:
                self.x = fx.squeeze()
                self.y = fy.squeeze()

    def __len__(self):
        return len(self.times)

    def __call__(self, times, ra=None, dec=None, psi=None, detector=None):
        """
        Return the antenna response function as a dictionary.
        """

        if detector is not None:
            self.detector = detector

        if ra is not None:
            self.ra = ra

        if dec is not None:
            self.dec = dec

        if psi is not None:
            self.psi = psi

        self.times = times

        # compute response
        self.compute_response()

        return self.response

## @}
